/*
 * arch/ppc/kernel/gt64260_pic.c
 * 
 * Interrupt controller support for Galileo's GT64260.
 *
 * Author: Chris Zankel <chris@mvista.com>
 * Modified by: Mark A. Greer <mgreer@mvista.com>
 *
 * Based on sources from Rabeeh Khoury / Galileo Technology
 *
 * 2001-2003 (c) MontaVista, Software, Inc.  This file is licensed under
 * the terms of the GNU General Public License version 2.  This program
 * is licensed "as is" without any warranty of any kind, whether express
 * or implied.
 */

/*
 * This file contains the specific functions to support the GT64260
 * interrupt controller.
 *
 * The GT64260 has two main interrupt registers (high and low) that
 * summarizes the interrupts generated by the units of the GT64260.
 * Each bit is assigned to an interrupt number, where the low register
 * are assigned from IRQ0 to IRQ31 and the high cause register
 * from IRQ32 to IRQ63 
 * The GPP (General Purpose Port) interrupts are assigned from IRQ64 (GPP0) 
 * to IRQ95 (GPP31). 
 * get_irq() returns the lowest interrupt number that is currently asserted.
 *
 * Note: 
 *  - This driver does not initialize the GPP when used as an interrupt 
 *    input.
 *  - *** WARNING *** Only level sensitive GPP interrupts are supported
 */

#include <linux/stddef.h>
#include <linux/init.h>
#include <linux/sched.h>
#include <linux/signal.h>
#include <linux/stddef.h>
#include <linux/delay.h>
#include <linux/irq.h>

#include <asm/io.h>
#include <asm/processor.h>
#include <asm/system.h>
#include <asm/irq.h>
#include <asm/gt64260.h>


#define GPP_LEVEL_INTERRUPTS

/* #define ENABLE_ECC_INT_HANDLER */	/* currently broken.. causes deadlock of some kind */
/* #define ECC_DEBUG */

/* ========================== forward declaration ========================== */

static void gt64260_unmask_irq(unsigned int);
static void gt64260_mask_irq(unsigned int);
static void gt64260_end_irq(unsigned int irq);
static void gt64260_set_irq_affinity(unsigned int irq, unsigned long mask);

/* ========================== local declarations =========================== */

struct hw_interrupt_type gt64260_pic = {
	" GT64260_PIC ",		/* typename */
	NULL,				/* startup */
	NULL,				/* shutdown */
	gt64260_unmask_irq,		/* enable */
	gt64260_mask_irq,		/* disable */
	gt64260_mask_irq,		/* mask and ack */
	gt64260_end_irq,		/* end */
	gt64260_set_irq_affinity	/* set_affinity */
};

u32 gt64260_irq_base = 0;      /* GT64260 handles the next 96 IRQs from here */

    				/* intr_mask_lo, intr_mask_hi, intr_mask_gpp */
#define GT64260_VALID_IRQ_MASK	{ 0x3dfffffe,	0x0f000db7,	0xffffffff }

static u32 gt64260_valid_irqs[] = GT64260_VALID_IRQ_MASK;

static u32 gt64260_cpu_intr_mask[NR_CPUS][3] = {
#if defined(CONFIG_SMP) && defined(CONFIG_IRQ_ALL_CPUS)
	[0 ... NR_CPUS-1] =		/* set masks for each cpu */
#endif
	GT64260_VALID_IRQ_MASK
};

/*
 * These are the default interrupt mask register offsets.
 * Since the interrupt lines going to each cpu is board-specific,
 * they may be changed by calling gt64260_set_intr_mask_reg_offsets().
 */
static u32 gt64260_intr_mask_reg_offset[NR_CPUS][2] = {
    { GT64260_IC_CPU_INTR_MASK_LO, GT64260_IC_CPU_INTR_MASK_HI },
#ifdef CONFIG_SMP
    { GT64260_IC_PCI_1_INTR_MASK_LO, GT64260_IC_PCI_1_INTR_MASK_HI }
#endif
};

#ifdef ENABLE_ECC_INT_HANDLER
static u32
gt_ecc_irq_ack(void)
{
    /* NOTE: If ecc_irq_ack is called during mask_and_ack,
     * we lose data_hi and data_lo if we get a new ECC error between
     * the time ack gets called and gt_ecc_irq gets called since reading addr
     * allows new data to get latched in. FIXME!!
     * See GT PIC documentation. */
    /* Currently, we DON'T have a mask_and_ack_irq, so this code is fine */
    static u32 last_addr=0;	/* store local copy since irq_ack overwrites */
    u32 addr=gt_read(GT64260_SDRAM_ERR_ADDR);
    gt_write(GT64260_SDRAM_ERR_ADDR,addr & (~3));

    if(addr&3) last_addr=addr;
    return last_addr;
}

static void
gt_ecc_irq(s32 irq, void *dev_id, struct pt_regs *regs)
{

    u32 addr;
#ifdef ECC_DEBUG
    u32 data_lo, data_hi,type;
    printk("ecc ERR: NIP=%08x LINK=%08x\n", regs->nip, regs->link);

    data_lo=gt_read(GT64260_SDRAM_ERR_DATA_LO);
    data_hi=gt_read(GT64260_SDRAM_ERR_DATA_HI);
#endif

    addr=gt_ecc_irq_ack();

#ifdef ECC_DEBUG
    type=addr&3;

    printk("data:%08x:%08x addr:%08x type:%d\n", data_hi, data_lo, addr&(~3), type);

    printk("rcvd:%08x calc:%08x cntl:%08x cnt:%08x\n",
	    gt_read(GT64260_SDRAM_ERR_ECC_RCVD),
	    gt_read(GT64260_SDRAM_ERR_ECC_CALC),
	    gt_read(GT64260_SDRAM_ERR_ECC_CNTL),
	    gt_read(GT64260_SDRAM_ERR_ECC_ERR_CNT));

    gt_write(GT64260_SDRAM_ERR_ECC_ERR_CNT,0);
#endif /* ECC_DEBUG */

}
#endif /* ENABLE_ECC_INT_HANDLER */

#ifdef CONFIG_SMP

static inline void
gt_write_intr_mask_lo(void)
{
	gt_write(gt64260_intr_mask_reg_offset[0][0],
		gt64260_cpu_intr_mask[0][0] & ppc_cached_irq_mask[0]);
	gt_write(gt64260_intr_mask_reg_offset[1][0],
		gt64260_cpu_intr_mask[1][0] & ppc_cached_irq_mask[0]);
}

static inline void
gt_write_intr_mask_hi(void)
{
	gt_write(gt64260_intr_mask_reg_offset[0][1],
		gt64260_cpu_intr_mask[0][1] & ppc_cached_irq_mask[1]);
	gt_write(gt64260_intr_mask_reg_offset[1][1],
		gt64260_cpu_intr_mask[1][1] & ppc_cached_irq_mask[1]);
}

#else /* !CONFIG_SMP */

static inline void
gt_write_intr_mask_lo(void)
{
	gt_write(gt64260_intr_mask_reg_offset[0][0], ppc_cached_irq_mask[0]);
}

static inline void
gt_write_intr_mask_hi(void)
{
	gt_write(gt64260_intr_mask_reg_offset[0][1], ppc_cached_irq_mask[1]);
}
#endif /* CONFIG_SMP */

/* gt64260_set_intr_mask_reg_offsets()
 *
 * This function initializes the register offsets used to reference
 * the cpu-specific interrupt mask registers
 *
 * Input Variables(s):
 *  int			index of cpu that mask registers are used for
 *  u32			offset of mask_lo register
 *  u32			offset of mask_hi register
 *
 * Output Variable(s):
 *  None.
 *
 * Returns:
 *  void
 *
 * Note:
 *  Must be called before gt64260_init_irq().
 */
__init void
gt64260_set_intr_mask_reg_offsets(int cpu, u32 lo_reg_offset, u32 hi_reg_offset)
{
    	gt64260_intr_mask_reg_offset[cpu][0] = lo_reg_offset;
    	gt64260_intr_mask_reg_offset[cpu][1] = hi_reg_offset;
}

/* gt64260_init_irq()
 *
 *  This function initializes the interrupt controller. It assigns
 *  all interrupts from IRQ0 to IRQ95 to the gt64260 interrupt controller.
 *
 * Input Variable(s):
 *  None.
 *
 * Output Variable(s):
 *  None.
 *
 * Returns:
 *  void
 *
 * Note:
 *  We register all GPP inputs as interrupt source, but disable them.
 */

__init void
gt64260_init_irq(void)
{
	int i;

	if ( ppc_md.progress ) ppc_md.progress("gt64260_init_irq: enter", 0x0);

	ppc_cached_irq_mask[0] = 0;
	ppc_cached_irq_mask[1] = 0x0f000000; /* Enable GPP intrs */
	ppc_cached_irq_mask[2] = 0;

	/* disable all interrupts and clear current interrupts */
	gt_write(GT64260_GPP_INTR_MASK, ppc_cached_irq_mask[2]);
	gt_write(GT64260_GPP_INTR_CAUSE,0);
	gt_write_intr_mask_lo();
	gt_write_intr_mask_hi();

	/* use the gt64260 for all (possible) interrupt sources */
	for( i = gt64260_irq_base;  i < (gt64260_irq_base + 96);  i++ )  {
		irq_desc[i].handler = &gt64260_pic;
	}

#ifdef GPP_LEVEL_INTERRUPTS
        if(gt64260_revision != GT64260)  {
		if ( ppc_md.progress )
			ppc_md.progress("gt64260_init_irq: GPP -> levelint ", 0x0);
                /* #FEr MISC-4 */
                /* Mark GPP intrs level sensitive */
                gt_set_bits(GT64260_COMM_ARBITER_CNTL, (1<<10));

                for(i=gt64260_irq_base+64;i<gt64260_irq_base+64+32;i++)
			irq_desc[i].status |= IRQ_LEVEL;
        }
#endif

#ifdef ENABLE_ECC_INT_HANDLER
	request_irq(gt64260_irq_base + 17, gt_ecc_irq, 0, "GT64260 SDRAM ECC Error", 0);
#endif

	if ( ppc_md.progress ) ppc_md.progress("gt64260_init_irq: exit", 0x0);
}

static inline u32
gt_read_main_cause_lo(int cpu)
{
	return gt_read(GT64260_IC_MAIN_CAUSE_LO) &
		gt64260_cpu_intr_mask[cpu][0] & ppc_cached_irq_mask[0];
}

static inline u32
gt_read_main_cause_hi(int cpu)
{
	return gt_read(GT64260_IC_MAIN_CAUSE_HI) &
		gt64260_cpu_intr_mask[cpu][1] & ppc_cached_irq_mask[1];
}

static inline u32
gt_read_gpp_cause(int cpu)
{
	return gt_read(GT64260_GPP_INTR_CAUSE) &
		gt64260_cpu_intr_mask[cpu][2] & ppc_cached_irq_mask[2];
}
 
/* gt64260_get_irq()
 *
 *  This function returns the lowest interrupt number of all interrupts that
 *  are currently asserted. 
 *
 * Input Variable(s):
 *  struct pt_regs*	not used
 *
 * Output Variable(s):
 *  None.
 *
 * Returns:
 *  int	<interrupt number> or -2 (bogus interrupt)
 * 
 */
int
gt64260_get_irq(struct pt_regs *regs)
{
	int irq;
	int cpu = smp_processor_id();

	irq = __ilog2(gt_read_main_cause_lo(cpu));

	if (irq == -1) {
		irq = __ilog2(gt_read_main_cause_hi(cpu));

		if (irq == -1) {
#ifdef CONFIG_SMP
/*
 * When irq are shared between two cpus, both may trap.
 * The second one to respond finds no irq pending.
 * Set return value to -2, so it's not recorded as a spurious interrupt.
 */
			irq = -2;
#endif
		}
		else if (irq < 24)
			irq += 32;
		else {
			irq = __ilog2(gt_read_gpp_cause(cpu));

			if (irq == -1) {
#ifdef CONFIG_SMP
				irq = -2; /* assume this cpu lost the race */
#endif
			}
			else
				irq += 64;
		}
	}

	if (irq < 0)
		return irq;
	else
		return gt64260_irq_base + irq;
}

/* gt64260_unmask_irq()
 *
 *  This function enables an interrupt.
 *
 * Input Variable(s):
 *  unsigned int	interrupt number (IRQ0...IRQ95).
 *
 * Output Variable(s):
 *  None.
 *
 * Returns:
 *  void
 */

static void
gt64260_unmask_irq(unsigned int irq)
{
	irq -= gt64260_irq_base;

	if (irq > 31) {
		if (irq > 63) {
			/* unmask GPP irq */
			set_bit(irq-64, &ppc_cached_irq_mask[2]);
			gt_write(GT64260_GPP_INTR_MASK, ppc_cached_irq_mask[2]);
		} else {
			/* unmask high interrupt register */
			set_bit(irq-32, &ppc_cached_irq_mask[1]);
			gt_write_intr_mask_hi();
		}
	} else {
		/* unmask low interrupt register */
		set_bit(irq, &ppc_cached_irq_mask[0]);
		gt_write_intr_mask_lo();
	}
}


/* gt64260_mask_irq()
 *
 *  This function disables the requested interrupt.
 *
 * Input Variable(s):
 *  unsigned int	interrupt number (IRQ0...IRQ95).
 *
 * Output Variable(s):
 *  None.
 *
 * Returns:
 *  void
 */

static void
gt64260_mask_irq(unsigned int irq)
{
	irq -= gt64260_irq_base;

	if (irq > 31) {
		if (irq > 63) {
			/* mask GPP irq */
			clear_bit(irq-64, &ppc_cached_irq_mask[2]);
			gt_write(GT64260_GPP_INTR_MASK, ppc_cached_irq_mask[2]);
		} else {
			/* mask high interrupt register */
			clear_bit(irq-32, &ppc_cached_irq_mask[1]);
			gt_write_intr_mask_hi();
		}
	} else {
		/* mask low interrupt register */
		clear_bit(irq, &ppc_cached_irq_mask[0]);
		gt_write_intr_mask_lo();
	}
	/*
	 * Need to drain the gt64260 pipeline to ensure that the irq has
	 * been masked before returning.  Reading any gt64260 register
	 * is sufficient.
	 */
	(void) gt_read(GT64260_GPP_INTR_MASK);
}


/* gt64260_end_irq()
 *
 * This function unmasks the irq, if its not disabled or in progress
 *
 * Input Variable(s):
 *  unsigned int	interrupt number (IRQ0...IRQ95).
 *
 * Output Variable(s):
 *  None.
 *
 * Returns:
 *  void
 */
static void
gt64260_end_irq(unsigned int irq)
{
	int local_irq = irq - gt64260_irq_base;

	if (!(irq_desc[irq].status & (IRQ_DISABLED|IRQ_INPROGRESS))) {
		/* FEr MISC-4 */
		if ((local_irq > 63) && (irq_desc[irq].status & IRQ_LEVEL)) {
			gt_write(GT64260_GPP_INTR_CAUSE,
				 ~(1<<(local_irq-64)));
		}

		gt64260_unmask_irq(irq);
	}
}

static inline void
gt_set_intr_mask_hi_for_gpp(unsigned int irq)
{
	int cpu;
    	unsigned int gpp_bit;
	unsigned int hi_bit;
	unsigned int gpp_group_mask;

	/* each group of 8 gpp irqs shares one bit of intr_mask_hi */
	gpp_bit = irq - 64;
	hi_bit = (gpp_bit >> 3) + 24;
	gpp_group_mask = 0xff << (gpp_bit & ~7);

	for (cpu=0; cpu<NR_CPUS; cpu++) {
		if (gpp_group_mask & gt64260_cpu_intr_mask[cpu][2])
			set_bit(hi_bit, &gt64260_cpu_intr_mask[cpu][1]);
		else
			clear_bit(hi_bit, &gt64260_cpu_intr_mask[cpu][1]);
	}
	gt_write_intr_mask_hi();
}

static void
gt64260_set_irq_affinity(unsigned int irqa, unsigned long mask)
{
	int cpu;
	unsigned int irq = irqa - gt64260_irq_base;

	if (!test_bit(irq, &gt64260_valid_irqs)) {
		printk(KERN_WARNING "smp_affinity: invalid irq: %d\n", irqa);
		return;
	}

	for (cpu=0; cpu<NR_CPUS; cpu++) {
		if (mask & (1<<cpu))
			set_bit(irq, &gt64260_cpu_intr_mask[cpu]);
		else
			clear_bit(irq, &gt64260_cpu_intr_mask[cpu]);
	}

	if (irq < 32)
		gt_write_intr_mask_lo();
	else if (irq < 64)
		gt_write_intr_mask_hi();
	else
		gt_set_intr_mask_hi_for_gpp(irq);
}
