/*
 *  PowerPC version derived from arch/arm/mm/consistent.c
 *    Copyright (C) 2001 Dan Malek (dmalek@jlc.net)
 *
 *  linux/arch/arm/mm/consistent.c
 *
 *  Copyright (C) 2000 Russell King
 *
 * Consistent memory allocators.  Used for DMA devices that want to
 * share uncached memory with the processor core.  The function return
 * is the virtual address and 'dma_handle' is the physical address.
 * Mostly stolen from the ARM port, with some changes for PowerPC.
 *						-- Dan
 * Modified for 36-bit support.  -Matt
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 */

#include <linux/config.h>
#include <linux/module.h>
#include <linux/signal.h>
#include <linux/sched.h>
#include <linux/kernel.h>
#include <linux/errno.h>
#include <linux/string.h>
#include <linux/types.h>
#include <linux/ptrace.h>
#include <linux/mman.h>
#include <linux/mm.h>
#include <linux/swap.h>
#include <linux/stddef.h>
#include <linux/vmalloc.h>
#include <linux/init.h>
#include <linux/delay.h>
#include <linux/bootmem.h>
#include <linux/highmem.h>
#include <linux/pci.h>
#include <linux/interrupt.h>

#include <asm/pgalloc.h>
#include <asm/prom.h>
#include <asm/io.h>
#include <asm/hardirq.h>
#include <asm/mmu_context.h>
#include <asm/pgtable.h>
#include <asm/mmu.h>
#include <asm/uaccess.h>
#include <asm/smp.h>
#include <asm/machdep.h>

#include "mmu_decl.h"

extern int map_page(unsigned long va, phys_addr_t pa, int flags);

/* This function will allocate the requested contiguous pages and
 * map them into the kernel's vmalloc() space.  This is done so we
 * get unique mapping for these pages, outside of the kernel's 1:1
 * virtual:physical mapping.  This is necessary so we can cover large
 * portions of the kernel with single large page TLB entries, and
 * still get unique uncached pages for consistent DMA.
 */
void *consistent_alloc(int gfp, size_t size, dma_addr_t *dma_handle)
{
	int order, err, i;
	unsigned long page, va, flags;
	phys_addr_t pa;
	struct vm_struct *area;
	void	 *ret;

	if (in_interrupt())
		BUG();

	/* Only allocate page size areas.
	*/
	size = PAGE_ALIGN(size);
	order = get_order(size);

	page = __get_free_pages(gfp, order);
	if (!page) {
		BUG();
		return NULL;
	}

	/*
	 * we need to ensure that there are no cachelines in use,
	 * or worse dirty in this area.
	 */
	invalidate_dcache_range(page, page + size);

	/* Allocate some common virtual space to map the new pages.
	*/
	area = get_vm_area(size, VM_ALLOC);
	if (area == 0) {
		free_pages(page, order);
		return NULL;
	}
	va = VMALLOC_VMADDR(area->addr);
	ret = (void *)va;

	/* This gives us the real physical address of the first page.
	*/
	*dma_handle = pa = virt_to_bus((void *)page);

	flags = _PAGE_KERNEL | _PAGE_NO_CACHE;

	/*
	 * Set refcount=1 on all pages in an order>0
	 * allocation so that vfree() will actually
	 * free all pages that were allocated. 
	 */
	if (order > 0)
	{
		struct page *rpage = virt_to_page(page);
		for (i = 1; i < (1 << order); i++)
			set_page_count(rpage+i, 1);
	}

	err = 0;
	for (i = 0; i < size && err == 0; i += PAGE_SIZE)
		err = map_page(va+i, pa+i, flags);
	
	if (err) {
		vfree((void *)va);
		return NULL;
	}

	return ret;
}

/*
 * free page(s) as defined by the above mapping.
 */
void consistent_free(void *vaddr)
{
	if (in_interrupt())
		BUG();
	vfree(vaddr);
}

/*
 * make an area consistent.
 */
void consistent_sync(void *vaddr, size_t size, int direction)
{
	unsigned long start = (unsigned long)vaddr;
	unsigned long end   = start + size;

	switch (direction) {
	case PCI_DMA_NONE:
		BUG();
	case PCI_DMA_FROMDEVICE:	/* invalidate only */
#if defined(CONFIG_4xx) && (defined(CONFIG_USB) || defined(CONFIG_USB_MODULE))
		/* 
		 * USB code does DMA off the stack, so when a driver
		 * uses the stock pci DMA routines, they end up calling
		 * invalidate_dcache_range(). 
		 * The cache invalidate code in the 405 port invalidates 
		 * an entire cache line, regardless of what addresses are 
		 * passed in - it 'rounds out' to the surrounding cache line.
		 * While other ports will notice if part of the cache line
		 * is NOT part of the range being invalidated and 'do
		 * the right thing', the 405 code will invalidate the 
		 * entire line, potentially trashing the stack, causing a
		 * panic.
		 * Until all the USB drivers are fixed, we flush the data
		 * buffer, a much safer operation.
		 */
		flush_dcache_range(start, end);
#else
		invalidate_dcache_range(start, end);
#endif
		break;
	case PCI_DMA_TODEVICE:		/* writeback only */
		clean_dcache_range(start, end);
		break;
	case PCI_DMA_BIDIRECTIONAL:	/* writeback and invalidate */
		flush_dcache_range(start, end);
		break;
	}
}

#ifdef CONFIG_HIGHMEM
/*
 * consistent_sync_page() implementation for non cache coherent
 * systems using highmem. In this case, each page of a buffer
 * must be kmapped/kunmapped in order to have a virtual address
 * for consistent_sync(). This must be interrupt safe so hard
 * interrupts are disabled and kmap_atomic() are used when in an
 * interrupt context.
 */
static inline void __consistent_sync_page(struct page *page,
		unsigned long offset, size_t size, int direction,
		int in_int)
{
	size_t seg_size = min((size_t)PAGE_SIZE - offset, size);
	size_t cur_size = seg_size;
	unsigned long flags, start, seg_offset = offset;
	int nr_segs = PAGE_ALIGN(size + (PAGE_SIZE - offset))/PAGE_SIZE;
	int seg_nr = 0;

	if (in_int) local_irq_save(flags);

	do {
		if (in_int)
			start = (unsigned long)kmap_atomic(page + seg_nr, KM_PPC_SYNC_PAGE) + seg_offset; 
		else
			start = (unsigned long)kmap(page + seg_nr) + seg_offset; 
		/* Sync this buffer segment */
		consistent_sync((void *)start, seg_size, direction);
		if (in_int)
			kunmap_atomic((void *)(start - seg_offset), KM_PPC_SYNC_PAGE);
		else
			kunmap(page + seg_nr);
		seg_nr++;
		/* Calculate next buffer segment size */
		seg_size = min((size_t)PAGE_SIZE, size - cur_size);
		/* Add the segment size to our running total */
		cur_size += seg_size;
		seg_offset = 0;
	} while (seg_nr < nr_segs);

	if (in_int) local_irq_restore(flags);
}
#endif /* CONFIG_HIGHMEM */

/*
 * consistent_sync_page makes memory consistent. identical
 * to consistent_sync, but takes a struct page instead of a
 * virtual address
 */
void consistent_sync_page(struct page *page, unsigned long offset,
	size_t size, int direction)
{
#ifdef CONFIG_HIGHMEM
	__consistent_sync_page(page, offset, size, direction, in_interrupt());
#else
	unsigned long start = (unsigned long)page_address(page) + offset; 
	consistent_sync((void *)start, size, direction);
#endif
}
