/* ibm405lp_pm.c:	Power management for the 405LP
 *
 * Copyright (C) 2002, 2003 Bishop Brock, Hollis Blanchard, IBM Corporation.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License.
 */

#include <linux/config.h>
#include <linux/module.h>
#include <linux/init.h>

#include <linux/kernel.h>	/* printk */
#include <linux/ctype.h>	/* isdigit */

#include <linux/serial_reg.h>	/* UART_LCR, etc */
#include <linux/pm.h>
#include <linux/sysctl.h>	/* ctl_table */
#include <linux/delay.h>	/* udelay */

#include <asm/uaccess.h>	/* get_user */
#include <asm/ibm4xx.h>		/* mtdcr */
#include <asm/time.h>		/* get_tbu/l, tb_ticks_per_jiffy */
#include <asm/machdep.h>	/* ppc_md */
#include <asm/pgtable.h>	/* ptep stuff */

#include <platforms/ibm405lp.h>			/* mtdcr_interlock */
#include <platforms/ibm405lp_pm.h>		/* struct wakeup_info */


#define WATCHDOG_CYCLES		64	/* number of RTC SQW cycles */
#define SLEEP_DELAY		100000	/* post-sleep spin */
#define PREFIX		"405LP APM: "	/* for printk */

/* eventually disable MSR:EE and MSR:CE . Currently CE is 0 kernel-wide. */
#define save_and_crit_cli	save_and_cli

/* BIN_TO_BCD is stupid; can only use it on variables */
#define BIN2BCD(val)	((((val)/10)<<4) + (val)%10)
#define BCD2BIN(val)	(((val)&15) + ((val)>>4)*10)

/* inspiration from arch/arm/mach-sa1100/pm.c */
#define SPR_SAVE(array, reg)	\
	array[SLEEP_SAVE_##reg] = mfspr(SPRN_##reg)
#define SPR_RESTORE(array, reg)	\
	mtspr(SPRN_##reg, array[SLEEP_SAVE_##reg])
#define DCR_SAVE(array, reg)	\
	array[SLEEP_SAVE_##reg] = mfdcr(DCRN_##reg)
#define DCR_RESTORE(array, reg)	\
	mtdcr(DCRN_##reg, array[SLEEP_SAVE_##reg])
#define DCRI_SAVE(array, base, reg)	\
	array[SLEEP_SAVE_##base##_##reg] = mfdcri(DCRN_##base, reg)
#define DCRI_RESTORE(array, base, reg) \
	mtdcri(DCRN_##base, reg, array[SLEEP_SAVE_##base##_##reg])

enum apm_sleep_type {
	clocksuspend,
	powerdown,
	standby
};

const char *sleep_type_name[] = {
	[clocksuspend] = "clock-suspend",
	[powerdown] = "power-down",
	[standby] = "standby",
};

/* from head_4xx.S */
extern unsigned long ibm405lp_wakeup_info;

/* from ibm405lp_pmasm.S */
unsigned long ibm405lp_asm_suspend(unsigned long cmd,
				   unsigned long sleep_delay,
				   struct ibm405lp_wakeup_info *wakeup_info);
unsigned long ibm405lp_pm_command(unsigned long cmd,
				  unsigned long sleep_delay);

static int rtc_freq_hz;	/* RTC input clock, in Hz */
static int alarm_secs;
static int cdiv;
static enum apm_sleep_type sleep_mode = standby;
static int attempts;	/* debugging */
static int watchdog = 1;

enum {
	/* SPRs */
	SLEEP_SAVE_DAC1,
	SLEEP_SAVE_DAC2,
	SLEEP_SAVE_DBCR0,
	SLEEP_SAVE_DBCR1,
	SLEEP_SAVE_DCWR,
	SLEEP_SAVE_DVC1,
	SLEEP_SAVE_DVC2,
	SLEEP_SAVE_EVPR,
	SLEEP_SAVE_IAC1,
	SLEEP_SAVE_IAC2,
	SLEEP_SAVE_IAC3,
	SLEEP_SAVE_IAC4,
	SLEEP_SAVE_PID,
	SLEEP_SAVE_SGR,
	SLEEP_SAVE_SLER,
	SLEEP_SAVE_SPRG0,
	SLEEP_SAVE_SPRG1,
	SLEEP_SAVE_SPRG2,
	SLEEP_SAVE_SPRG3,
	SLEEP_SAVE_SPRG4,
	SLEEP_SAVE_SPRG5,
	SLEEP_SAVE_SPRG6,
	SLEEP_SAVE_SPRG7,
	SLEEP_SAVE_SRR0,
	SLEEP_SAVE_SRR1,
	SLEEP_SAVE_SRR2,
	SLEEP_SAVE_SRR3,
	SLEEP_SAVE_SU0R,
	SLEEP_SAVE_ZPR,

	/* DCRs */
	SLEEP_SAVE_PLB0_ACR,
	SLEEP_SAVE_CPC0_PLBAPR,
	SLEEP_SAVE_CPMER,
	SLEEP_SAVE_CPMFR,
	SLEEP_SAVE_UIC0_ER,
	SLEEP_SAVE_UIC0_CR,
	SLEEP_SAVE_UIC0_PR,
	SLEEP_SAVE_UIC0_TR,

	SLEEP_SAVE_EBC0_B0AP,
	SLEEP_SAVE_EBC0_B1AP,
	SLEEP_SAVE_EBC0_B2AP,
	SLEEP_SAVE_EBC0_B3AP,
	SLEEP_SAVE_EBC0_B4AP,
	SLEEP_SAVE_EBC0_B5AP,
	SLEEP_SAVE_EBC0_B6AP,
	SLEEP_SAVE_EBC0_B7AP,
	SLEEP_SAVE_EBC0_B0CR,
	SLEEP_SAVE_EBC0_B1CR,
	SLEEP_SAVE_EBC0_B2CR,
	SLEEP_SAVE_EBC0_B3CR,
	SLEEP_SAVE_EBC0_B4CR,
	SLEEP_SAVE_EBC0_B5CR,
	SLEEP_SAVE_EBC0_B6CR,
	SLEEP_SAVE_EBC0_B7CR,
	SLEEP_SAVE_EBC0_CFG,
};

/* do_suspend()
 * called *only* in power-down or standby mode
 */
static int do_suspend(apm0_cfg_t command)
{
	u64 timebase;
	struct ibm405lp_wakeup_info my_wakeup_info = {
		.magic = IBM405LP_WAKEUP_MAGIC,
	};
	unsigned long *sleep_save;
	u32 pre_rtc_time, rtc_secs_elapsed;
	int ret;

	printk(KERN_DEBUG "CFG = 0x%.8x\n", command.reg);

	/* remember what time it is */
	timebase = ((u64)get_tbu() << 32) | get_tbl();
	pre_rtc_time = ppc_md.get_rtc_time();

	sleep_save = (unsigned long *)__get_free_page(GFP_ATOMIC); /* irqs are disabled */
	if (sleep_save == NULL) {
		printk(KERN_ERR PREFIX "__get_free_page failed!\n");
		return -ENOMEM;
	}

	/* SPRs aren't saved/restored by hardware even in standby mode */
	SPR_SAVE(sleep_save, DAC1);
	SPR_SAVE(sleep_save, DAC2);
	SPR_SAVE(sleep_save, DBCR0);
	SPR_SAVE(sleep_save, DBCR1);
	SPR_SAVE(sleep_save, DCWR);
	SPR_SAVE(sleep_save, DVC1);
	SPR_SAVE(sleep_save, DVC2);
	SPR_SAVE(sleep_save, IAC1);
	SPR_SAVE(sleep_save, IAC2);
	SPR_SAVE(sleep_save, IAC3);
	SPR_SAVE(sleep_save, IAC4);
	SPR_SAVE(sleep_save, PID);
	SPR_SAVE(sleep_save, SGR);
	SPR_SAVE(sleep_save, SLER);
	SPR_SAVE(sleep_save, SPRG0);
	SPR_SAVE(sleep_save, SPRG1);
	SPR_SAVE(sleep_save, SPRG2);
	SPR_SAVE(sleep_save, SPRG3);
	SPR_SAVE(sleep_save, SPRG4);
	SPR_SAVE(sleep_save, SPRG5);
	SPR_SAVE(sleep_save, SPRG6);
	SPR_SAVE(sleep_save, SPRG7);
	SPR_SAVE(sleep_save, SRR0);
	SPR_SAVE(sleep_save, SRR1);
	SPR_SAVE(sleep_save, SRR2);
	SPR_SAVE(sleep_save, SRR3);
	SPR_SAVE(sleep_save, SU0R);
	SPR_SAVE(sleep_save, ZPR);

	/* EBC DCRs - in standby mode hardware *does* save/restore these, but
	 * firmware may override on wakeup */
	DCRI_SAVE(sleep_save, EBC0, B0AP);
	DCRI_SAVE(sleep_save, EBC0, B1AP);
	DCRI_SAVE(sleep_save, EBC0, B2AP);
	DCRI_SAVE(sleep_save, EBC0, B3AP);
	DCRI_SAVE(sleep_save, EBC0, B4AP);
	DCRI_SAVE(sleep_save, EBC0, B5AP);
	DCRI_SAVE(sleep_save, EBC0, B6AP);
	DCRI_SAVE(sleep_save, EBC0, B7AP);
	DCRI_SAVE(sleep_save, EBC0, B0CR);
	DCRI_SAVE(sleep_save, EBC0, B1CR);
	DCRI_SAVE(sleep_save, EBC0, B2CR);
	DCRI_SAVE(sleep_save, EBC0, B3CR);
	DCRI_SAVE(sleep_save, EBC0, B4CR);
	DCRI_SAVE(sleep_save, EBC0, B5CR);
	DCRI_SAVE(sleep_save, EBC0, B6CR);
	DCRI_SAVE(sleep_save, EBC0, B7CR);
	DCR_SAVE(sleep_save, EBC0_CFG);

	if (command.fields.sm == APM0_CFG_SM_POWER_DOWN) {
		/* powerdown suspend. Most state will be saved/restored by drivers via
		 * the pm_send notifier.
		 */
		DCR_SAVE(sleep_save, PLB0_ACR);
		DCR_SAVE(sleep_save, CPC0_PLBAPR);
		DCR_SAVE(sleep_save, CPMER);
		DCR_SAVE(sleep_save, CPMFR);
		DCR_SAVE(sleep_save, UIC0_ER);
		DCR_SAVE(sleep_save, UIC0_CR);
		DCR_SAVE(sleep_save, UIC0_PR);
		DCR_SAVE(sleep_save, UIC0_TR);
	}

	/* the sleep call */
	ret = ibm405lp_asm_suspend(command.reg, SLEEP_DELAY, &my_wakeup_info);
	/* apm_wakeup (called by the firmware on wake) returns here */

	/* restore SPRs unhandled in apm_wakeup */
	SPR_RESTORE(sleep_save, DAC1);
	SPR_RESTORE(sleep_save, DAC2);
	SPR_RESTORE(sleep_save, DBCR0);
	SPR_RESTORE(sleep_save, DBCR1);
	SPR_RESTORE(sleep_save, DCWR);
	SPR_RESTORE(sleep_save, DVC1);
	SPR_RESTORE(sleep_save, DVC2);
	SPR_RESTORE(sleep_save, IAC1);
	SPR_RESTORE(sleep_save, IAC2);
	SPR_RESTORE(sleep_save, IAC3);
	SPR_RESTORE(sleep_save, IAC4);
	SPR_RESTORE(sleep_save, PID);
	SPR_RESTORE(sleep_save, SGR);
	SPR_RESTORE(sleep_save, SLER);
	SPR_RESTORE(sleep_save, SPRG0);
	SPR_RESTORE(sleep_save, SPRG1);
	SPR_RESTORE(sleep_save, SPRG2);
	SPR_RESTORE(sleep_save, SPRG3);
	SPR_RESTORE(sleep_save, SPRG4);
	SPR_RESTORE(sleep_save, SPRG5);
	SPR_RESTORE(sleep_save, SPRG6);
	SPR_RESTORE(sleep_save, SPRG7);
	SPR_RESTORE(sleep_save, SRR0);
	SPR_RESTORE(sleep_save, SRR1);
	SPR_RESTORE(sleep_save, SRR2);
	SPR_RESTORE(sleep_save, SRR3);
	SPR_RESTORE(sleep_save, SU0R);
	SPR_RESTORE(sleep_save, ZPR);

	/* restore EBC registers */
	DCRI_RESTORE(sleep_save, EBC0, B0AP);
	DCRI_RESTORE(sleep_save, EBC0, B1AP);
	DCRI_RESTORE(sleep_save, EBC0, B2AP);
	DCRI_RESTORE(sleep_save, EBC0, B3AP);
	DCRI_RESTORE(sleep_save, EBC0, B4AP);
	DCRI_RESTORE(sleep_save, EBC0, B5AP);
	DCRI_RESTORE(sleep_save, EBC0, B6AP);
	DCRI_RESTORE(sleep_save, EBC0, B7AP);
	DCRI_RESTORE(sleep_save, EBC0, B0CR);
	DCRI_RESTORE(sleep_save, EBC0, B1CR);
	DCRI_RESTORE(sleep_save, EBC0, B2CR);
	DCRI_RESTORE(sleep_save, EBC0, B3CR);
	DCRI_RESTORE(sleep_save, EBC0, B4CR);
	DCRI_RESTORE(sleep_save, EBC0, B5CR);
	DCRI_RESTORE(sleep_save, EBC0, B6CR);
	DCRI_RESTORE(sleep_save, EBC0, B7CR);
	DCR_RESTORE(sleep_save, EBC0_CFG);

	if (command.fields.sm == APM0_CFG_SM_POWER_DOWN) {
		DCR_RESTORE(sleep_save, PLB0_ACR);
		DCR_RESTORE(sleep_save, CPC0_PLBAPR);
		DCR_RESTORE(sleep_save, CPMER);
		DCR_RESTORE(sleep_save, CPMFR);
		DCR_RESTORE(sleep_save, UIC0_ER);
		DCR_RESTORE(sleep_save, UIC0_CR);
		DCR_RESTORE(sleep_save, UIC0_PR);
		DCR_RESTORE(sleep_save, UIC0_TR);

		/* so we know we did a full restore */
		printk(KERN_DEBUG PREFIX "full restore\n");
	}

	/* free our memory */
	free_page((unsigned long)sleep_save);

	/* Update timebase. We're stuck with 1-second accuracy from the RTC. */
	rtc_secs_elapsed = ppc_md.get_rtc_time() - pre_rtc_time;
	printk(KERN_DEBUG PREFIX "%u seconds elapsed\n", rtc_secs_elapsed);
	timebase += rtc_secs_elapsed * HZ * tb_ticks_per_jiffy;
	set_tb(timebase >> 32, (unsigned long)timebase);

	return ret;
}

/* set RTC alarm to go off at <now> + secs_offset
 * XXX assumes BCD mode */
static void rtc_alarm(const int secs_offset)
{
	u32 secal, minal, hral;
	u32 rtc_cr1;
	int secs=0;

	rtc_cr1 = mfdcr(DCRN_RTC0_CR1);
	mtdcr(DCRN_RTC0_CR1, rtc_cr1 | 0x80);	/* disable RTC updates */

	secs += BCD2BIN(mfdcr(DCRN_RTC0_SEC));
	secs += BCD2BIN(mfdcr(DCRN_RTC0_MIN)) * 60;
	secs += BCD2BIN(mfdcr(DCRN_RTC0_HR) & 0x1f) * 60 * 60;

	secs += secs_offset;

	secal = BIN2BCD(secs % 60);
	minal = BIN2BCD((secs / 60) % 60);
	hral = BIN2BCD((secs / 60 / 60) % 24);

	if (((rtc_cr1 & 0x2) == 0) && (hral > 0x12)) {
		/* 12hr mode */
		hral -= 0x12;
	}

	mtdcr(DCRN_RTC0_SECAL, secal);
	mtdcr(DCRN_RTC0_MINAL, minal);
	mtdcr(DCRN_RTC0_HRAL, hral);

	mtdcr(DCRN_RTC0_CR1, (rtc_cr1|0x20)&0x7f); /* enable RTC updates, alarm */

	printk(KERN_DEBUG PREFIX "time = %.2x:%.2x:%.2x; alarm = %.2x:%.2x:%.2x\n",
			mfdcr(DCRN_RTC0_HR),
			mfdcr(DCRN_RTC0_MIN),
			mfdcr(DCRN_RTC0_SEC),
			hral, minal, secal);
}

/* approximate the time required for cryo scan out + scan in */
static unsigned long calc_rtc_sqw(int cdiv, int rtc_freq)
{
	const int data_bits = 36400;    /* bits of data transferred (max 64Kbits) */
	int iic_freq;
	int transfer_ms;                /* time in ms to complete transfer */
	unsigned long rtc_rs;

	iic_freq = rtc_freq / (2 * cdiv);
	transfer_ms = 2 * data_bits * 1000 / iic_freq;
	transfer_ms = 2 * transfer_ms; /* double it to be safe */
	rtc_rs = __ilog2((transfer_ms * 0x10000) / (WATCHDOG_CYCLES * 1000));

	return rtc_rs;
}

static int ibm405lp_suspend(enum apm_sleep_type mode)
{
	apm0_cfg_t command;
	unsigned long flags;
	u32 der;
	u32 ier = 0x4; /* wakeup-2, button on Beech */
	int	retval;

	printk(KERN_DEBUG PREFIX "attempt #%i\n", ++attempts);

	save_and_crit_cli(flags); /* disable critical exceptions too */

	command.reg = mfdcr(DCRN_APM0_CFG) & 0xfffffffe;
	command.fields.isp = 1;

	/* turn off LCD */
	der = mfdcri(DCRN_LCD0, DER);
	mtdcri(DCRN_LCD0, DER, 0);

	mfdcr(DCRN_RTC0_CR2); /* clear pending RTC irqs */

	if (alarm_secs != 0) {
		rtc_alarm(alarm_secs); /* set up the RTC alarm */
		ier |= 0x2;	/* enable RTC wakeup */
	}

	/* initialize for wakeup */
	mtdcr_interlock(DCRN_APM0_IER, ier, 0xffffffe0);
	mtdcr_interlock(DCRN_APM0_IPR, 0x1e, 0xffffffe0); /* active high */
	mtdcr_interlock(DCRN_APM0_ITR, 0x00, 0xffffffe0); /* level triggered */

	/* magic do not touch! */
	mtdcr_interlock(DCRN_APM0_ISR, 0x1e, 0xffffffe0); /* clear pending irqs */
	udelay((4*1000000)/rtc_freq_hz);	/* allow 3-4 RTC cycles */
	mtdcr(DCRN_APM0_ISR, 0);		/* must leave a 0 in ISR */

	mtdcr_interlock(DCRN_APM0_SR, 0xffffffff, 0); /* vital: leave a 1 in SR */

	/* wait for ISR & SR writes to complete */
	udelay((4*1000000)/rtc_freq_hz); 	/* allow 3-4 RTC cycles */
	/* end magic */

	printk(KERN_DEBUG "ISR = 0x%.8x\n", mfdcr(DCRN_APM0_ISR));
	printk(KERN_DEBUG "SR  = 0x%.8x\n", mfdcr(DCRN_APM0_SR));

	switch (mode) {
		case clocksuspend:
			command.fields.sm = APM0_CFG_SM_CLOCK_SUSPEND;
			printk(KERN_DEBUG "CFG = 0x%.8x\n", command.reg);
			retval = ibm405lp_pm_command(command.reg,
						     SLEEP_DELAY);
			break;

		case powerdown:
			command.fields.sm = APM0_CFG_SM_POWER_DOWN;
			/* Power-down requires a slow I2C clock to allow the
			   power-up reset pulse to propagate correctly on
			   Beech. */ 
			command.fields.cdiv = 31;

			retval = do_suspend(command);
			break;

		case standby:
			command.fields.sm = APM0_CFG_SM_CRYO_STANDBY;
			command.fields.rst = 1;

			/* set cdiv such that I2C freq is ~100 KHz */
			if (cdiv == 0) {
				command.fields.cdiv = rtc_freq_hz / (100000 * 2);
				command.fields.cdiv++; /* round up to be safe */
			} else {
				/* user override */
				command.fields.cdiv = cdiv;
			}

			/* enable cryo watchdog if user requested it */
			if (watchdog) {
				ibm405lp_set_rtc_sqw(calc_rtc_sqw(cdiv, rtc_freq_hz));
				command.fields.ewt = 1;
			}

			retval = do_suspend(command);
			break;
	}

	mtdcri(DCRN_LCD0, DER, der);	/* restore LCD */

	restore_flags(flags);

	return retval; /* success */
}

static int suspend_proc_handler(ctl_table *ctl, int write, struct file * filp,
				void *buffer, size_t *lenp)
{
	int retval;

	switch (sleep_mode) {
	case clocksuspend:
	case standby:
		retval = ibm405lp_suspend(sleep_mode);
		break;
		
	case powerdown:
		retval = pm_send_all(PM_SUSPEND, (void *)3);	/* notify drivers */
		if (! retval) {
			retval = ibm405lp_suspend(powerdown);
			pm_send_all(PM_RESUME, (void *)0);
		}
		break;
	}

	printk(KERN_DEBUG "SR = 0x%.8x\n", retval);

	return 0;	/* success */
}

/* whose BRILLIANT idea was it to have the sysctl "strategy" routine NOT
 * called when accessed via /proc ? That renders it useless AFAICS.
 *
 * allow only ("clock-suspend" | "power-down" | "standby") to be written
 */
static int parse_mode(ctl_table *table, int write, struct file *filp,
		      void *buffer, size_t *lenp)
{
	size_t len;
	
	if (!table->maxlen || !*lenp || (filp->f_pos && !write)) {
		*lenp = 0;
		return 0;
	}

	if (write) {
		char string[table->maxlen];
		int i;

		len = *lenp;
		if (len > table->maxlen)
			len = table->maxlen;

		if (copy_from_user(string, buffer, len))
			return -EFAULT;

		for (i = 0; i < table->maxlen; i++) {
			/* Chop the string off at first nul or \n */
			if ( (i < len) && 
			     ( (string[i] == '\n') || (string[i] == '\0') ) )
				len = i;

			/* Clear the tail of the buffer */
			if (i >= len)
				string[i] = '\0';
		}

		for (i = 0; i < ARRAY_SIZE(sleep_type_name); i++) {
			const char *name = sleep_type_name[i];

			if (name && (strncmp(string, name, len) == 0)) {
				sleep_mode = i;
				filp->f_pos += *lenp;
				return 0;
			}
		}

		/* Didn't find it */
		return -EINVAL;
	} else {
		const char *name = sleep_type_name[sleep_mode];

		len = strlen(name);

		if (len > table->maxlen)
			len = table->maxlen;
		if (len > *lenp)
			len = *lenp;
		if (len > 0)
			if (copy_to_user(buffer, name, len))
				return -EFAULT;
		if (len < *lenp) {
			if (put_user('\n', ((char *) buffer) + len))
				return -EFAULT;
			len++;
		}
		*lenp = len;
		filp->f_pos += len;
	}

	return 0;
}

static struct ctl_table pm_table[] =
{
	{
		.ctl_name = PM_405LP_SLEEP_CMD,
		.procname = "suspend",
		.mode = 0200, /* write-only to trigger sleep */
		.proc_handler = &suspend_proc_handler,
	},
	{
		.ctl_name = PM_405LP_SLEEP_MODE,
		.procname = "mode",
		.maxlen = 16,
		.mode = 0644,
		.proc_handler = &parse_mode,
	},
	{
		.ctl_name = PM_405LP_SLEEP_ALARM,
		.procname = "rtc_alarm",
		.data = &alarm_secs,
		.maxlen = sizeof(int),
		.mode = 0644,
		.proc_handler = &proc_dointvec,
	},
	{
		.ctl_name = PM_405LP_SLEEP_DEBUG_CDIV,
		.procname = "cdiv",
		.data = &cdiv,
		.maxlen = sizeof(int),
		.mode = 0644,
		.proc_handler = &proc_dointvec,
	},
	{
		.ctl_name = PM_405LP_SLEEP_DEBUG_WATCHDOG,
		.procname = "watchdog",
		.data = &watchdog,
		.maxlen = sizeof(int),
		.mode = 0644,
		.proc_handler = &proc_dointvec,
	},
	{0}
};

static struct ctl_table pm_dir_table[] =
{
	{
		.ctl_name = CTL_PM_405LP,
		.procname = "sleep",
		.mode = 0555,
		.child = pm_table,
	},
	{0}
};

/*
 * Initialize power interface
 */
static int __init ibm405lp_pm_init(void)
{
	apm0_cfg_t command;

	/* Since CDIV determines the speed of the entire APM unit, the
	 * delay required after an mtdcr (before it takes effect) is
	 * dependent on it.  We don't want to have to wait very long
	 * at all, so we *always* zero CDIV (causing the APM to run at
	 * RTC frequency), both at boot and in wakeup code.
	 */
	command.reg = mfdcr(DCRN_APM0_CFG) & 0xfffffffe;
	command.fields.cdiv = 0;
	mtdcr_interlock(DCRN_APM0_CFG, command.reg, 0xfffc0000);

	/* APM0_ID wasn't updated from 405LP 1.0 -> 1.1, so is
	 * probably useless (use PVR instead). OTOH it would be nice
	 * if it was used. :(
	 */
	printk(KERN_INFO PREFIX "APM version %x\n", mfdcr(DCRN_APM0_ID));

	/* we need to know the RTC frequency in a few places, so read
	 * it here */
	switch ((mfdcr(DCRN_RTC0_CR0) >> 4) & 0x7) {
		/* check DV0, DV1, DV3 */
		case RTC_DVBITS_4MHZ:
			rtc_freq_hz = 4194304;
			break;
		case RTC_DVBITS_33KHZ:
			rtc_freq_hz = 33768;
			break;
		default:
			printk(KERN_ERR PREFIX
					"unknown RTC clock rate! defaulting to 1048576 Hz\n");
			/* fall through */
		case RTC_DVBITS_1MHZ:
			rtc_freq_hz = 1048576;
			break;
	}

	register_sysctl_table(pm_dir_table, 1);

	return 0;
}
__initcall(ibm405lp_pm_init);

