/*
 * arch/ppc/platforms/ibm440gp.c
 *
 * PPC440GP I/O descriptions
 *
 * Matt Porter <mporter@mvista.com>
 * Armin Kuster <akuster@mvista.com>
 *
 * Copyright 2002 MontaVista Software Inc.
 *
 * Eugene Surovegin <eugene.surovegin@zultys.com> or <ebs@ebshome.net>
 * Copyright (c) 2003 Zultys Technologies
 *
 * This program is free software; you can redistribute  it and/or modify it
 * under  the terms of  the GNU General Public License as published by the
 * Free Software Foundation;  either version 2 of the  License, or (at your
 * option) any later version.
 *
 */

#include <linux/config.h>
#include <linux/init.h>
#include <linux/smp.h>
#include <linux/threads.h>
#include <linux/param.h>
#include <linux/string.h>
#include <asm/ibm440.h>
#include <asm/mmu.h>
#include <asm/ocp.h>
#include "ebony.h"

phys_addr_t fixup_bigphys_addr(phys_addr_t addr, phys_addr_t size)
{
	phys_addr_t page_4gb = 0;

        /*
	 * Trap the least significant 32-bit portions of an
	 * address in the 440's 36-bit address space.  Fix
	 * them up with the appropriate ERPN
	 */
	if ((addr >= PPC440_IO_LO) && (addr < PPC440_IO_HI))
		page_4gb = PPC440_IO_PAGE;
	else if ((addr >= PPC440_PCICFG_LO) && (addr < PPC440_PCICFG_HI))
		page_4gb = PPC440_PCICFG_PAGE;
	else if ((addr >= PPC440_PCIMEM_LO) && (addr < PPC440_PCIMEM_HI))
		page_4gb = PPC440_PCIMEM_PAGE;

	return (page_4gb | addr);
};

/*
 * We only have onboard EMAC support for now.  I2C, DMA, timers,
 * etc. have yet to be enabled.
 */

struct ocp_def core_ocp[] = {
	{UART, PPC440GP_UART0_ADDR, UART0_INT, IBM_CPM_UART0},
	{UART, PPC440GP_UART1_ADDR, UART1_INT, IBM_CPM_UART1},
	{IIC, PPC440GP_IIC0_ADDR, IIC0_IRQ, IBM_CPM_IIC0},
	{IIC, PPC440GP_IIC1_ADDR, IIC1_IRQ, IBM_CPM_IIC1},
	{GPIO, PPC440GP_GPIO0_ADDR, OCP_IRQ_NA, IBM_CPM_GPIO0},
	{EMAC, PPC440GP_EMAC0_ADDR, BL_MAC_ETH0, OCP_CPM_NA},
	{EMAC, PPC440GP_EMAC1_ADDR, BL_MAC_ETH1, OCP_CPM_NA},
	{ZMII, PPC440GP_ZMII_ADDR, OCP_IRQ_NA, OCP_CPM_NA},
	{OCP_NULL_TYPE, 0x0, OCP_IRQ_NA, OCP_CPM_NA},

};

/*
 * Calculate 440GP clocks
 */
void ibm440gp_get_clocks(struct ibm440gp_clocks* p, unsigned int sys_clk, 
	unsigned int ser_clk)
{
	u32 cpc0_sys0 = mfdcr(DCRN_CPC0_SYS0);
	u32 cpc0_cr0 = mfdcr(DCRN_CPC0_CR0);
	u32 opdv, epdv;
	
	if (cpc0_sys0 & 0x2){
		/* Bypass system PLL */
		p->cpu = p->plb = sys_clk;
	}
	else {
		u32 fbdv, fwdva, fwdvb, m, vco;
		
		fbdv = (cpc0_sys0 >> 18) & 0x0f;
		if (!fbdv)
			fbdv = 16;
    
		fwdva = 8 - ((cpc0_sys0 >> 15) & 0x7);
		fwdvb = 8 - ((cpc0_sys0 >> 12) & 0x7);
    
    		/* Feedback path */	    
		if (cpc0_sys0 & 0x00000080){
			/* PerClk */
			m = fwdvb * opdv * epdv;
		}
		else {
			/* CPU clock */
			m = fbdv * fwdva;
    		}
		vco = sys_clk * m;
		p->cpu = vco / fwdva;
		p->plb = vco / fwdvb;
	}
    
	opdv = ((cpc0_sys0 >> 10) & 0x3) + 1;
	epdv = ((cpc0_sys0 >> 8) & 0x3) + 1;
	
	p->opb = p->plb / opdv;
	p->ebc = p->opb / epdv;
    
	if (cpc0_cr0 & 0x00400000){
		/* External UART clock */
		p->uart = ser_clk;
	}
	else {
		/* Internal UART clock */    
    		u32 uart_div = ((cpc0_cr0 >> 16) & 0x1f) + 1;
		p->uart = p->plb / uart_div;
	}
}

