/*
 * arch/ppc/platforms/sandpoint_setup.c
 * 
 * Board setup routines for the Motorola SPS Sandpoint Test Platform.
 *
 * Author: Mark A. Greer
 *         mgreer@mvista.com
 *
 * Copyright 2000-2002 MontaVista Software Inc.
 *
 * This program is free software; you can redistribute  it and/or modify it
 * under  the terms of  the GNU General Public License as published by the
 * Free Software Foundation;  either version 2 of the  License, or (at your
 * option) any later version.
 */

/*
 * This file adds support for the Motorola SPS Sandpoint Test Platform.
 * These boards have a PPMC slot for the processor so any combination
 * of cpu and host bridge can be attached.  This port is for an 8240 PPMC
 * module from Motorola SPS and other closely related cpu/host bridge
 * combinations (e.g., 750/755/7400 with MPC107 host bridge).
 * The sandpoint itself has a Windbond 83c553 (PCI-ISA bridge, 2 DMA ctlrs, 2
 * cascaded 8259 interrupt ctlrs, 8254 Timer/Counter, and an IDE ctlr), a
 * National 87308 (RTC, 2 UARTs, Keyboard & mouse ctlrs, and a floppy ctlr),
 * and 4 PCI slots (only 2 of which are usable; the other 2 are keyed for 3.3V
 * but are really 5V).
 *
 * The firmware on the sandpoint is called DINK (not my acronym :).  This port
 * depends on DINK to do some basic initialization (e.g., initialize the memory
 * ctlr) and to ensure that the processor is using MAP B (CHRP map).
 *
 * The switch settings for the Sandpoint board MUST be as follows:
 * 	S3: down
 * 	S4: up
 * 	S5: up
 * 	S6: down
 *
 * 'down' is in the direction from the PCI slots towards the PPMC slot;
 * 'up' is in the direction from the PPMC slot towards the PCI slots.
 * Be careful, the way the sandpoint board is installed in XT chasses will
 * make the directions reversed.
 *
 * Since Motorola listened to our suggestions for improvement, we now have
 * the Sandpoint X3 board.  All of the PCI slots are available, it uses
 * the serial interrupt interface (just a hardware thing we need to
 * configure properly).
 *
 * Use the default X3 switch settings.  The interrupts are then:
 *		EPIC	Source
 *		  0	SIOINT 		(8259, active low)
 *		  1	PCI #1
 *		  2	PCI #2
 *		  3	PCI #3
 *		  4	PCI #4
 *		  7	Winbond INTC	(IDE interrupt)
 *		  8	Winbond INTD	(IDE interrupt)
 *
 */
#include <linux/config.h>
#include <linux/stddef.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/errno.h>
#include <linux/reboot.h>
#include <linux/pci.h>
#include <linux/kdev_t.h>
#include <linux/major.h>
#include <linux/blk.h>
#include <linux/console.h>
#include <linux/delay.h>
#include <linux/irq.h>
#include <linux/ide.h>
#include <linux/irq.h>
#include <linux/seq_file.h>
#include <linux/serial.h>

#include <asm/system.h>
#include <asm/pgtable.h>
#include <asm/page.h>
#include <asm/time.h>
#include <asm/dma.h>
#include <asm/io.h>
#include <asm/machdep.h>
#include <asm/prom.h>
#include <asm/smp.h>
#include <asm/keyboard.h>
#include <asm/vga.h>
#include <asm/open_pic.h>
#include <asm/i8259.h>
#include <asm/todc.h>
#include <asm/bootinfo.h>
#include <asm/mpc10x.h>
#include <asm/pci-bridge.h>

#include "sandpoint.h"
#include "sandpoint_serial.h"

#include "buffalo/5c372.h"
#include "buffalo/ls_hwctl.h"

extern int pckbd_setkeycode(unsigned int scancode, unsigned int keycode);
extern int pckbd_getkeycode(unsigned int scancode);
extern int pckbd_translate(unsigned char scancode, unsigned char *keycode,
		                           char raw_mode);
extern char pckbd_unexpected_up(unsigned char keycode);
extern void pckbd_leds(unsigned char leds);
extern void pckbd_init_hw(void);
extern unsigned char pckbd_sysrq_xlate[128];

extern void gen550_progress(char *, unsigned short);
extern void gen550_init(int, struct serial_struct *);

unsigned char __res[sizeof(bd_t)];

static void	sandpoint_halt(void);


/*
 * Define all of the IRQ senses and polarities.  Taken from the
 * Sandpoint X3 User's manual.
 */
static u_char sandpoint_openpic_initsenses[] __initdata = {
	(IRQ_SENSE_LEVEL | IRQ_POLARITY_NEGATIVE),	/* EPIC IRQ 0 */
	(IRQ_SENSE_LEVEL | IRQ_POLARITY_NEGATIVE),	/* EPIC IRQ 1 */
	(IRQ_SENSE_LEVEL | IRQ_POLARITY_NEGATIVE),	/* EPIC IRQ 2 */
	(IRQ_SENSE_LEVEL | IRQ_POLARITY_NEGATIVE),	/* EPIC IRQ 3 */
	(IRQ_SENSE_LEVEL | IRQ_POLARITY_NEGATIVE)	/* EPIC IRQ 4 */
};

#ifdef CONFIG_SERIAL
static void __init
sandpoint_early_serial_map(void)
{
	struct serial_struct serial_req;

	/* Setup serial port access */
	memset(&serial_req, 0, sizeof(serial_req));
	serial_req.baud_base = BASE_BAUD;
	serial_req.line = 0;
	serial_req.port = 0;
//	serial_req.irq = 26;
	serial_req.irq = 25;
	serial_req.flags = STD_COM_FLAGS;
	serial_req.io_type = SERIAL_IO_MEM;
	serial_req.iomem_base = (u_char *)SANDPOINT_SERIAL_1;
	serial_req.iomem_reg_shift = 0;

#if defined(CONFIG_SERIAL_TEXT_DEBUG) || defined(CONFIG_KGDB)
	gen550_init(0, &serial_req);
#endif

	if (early_serial_setup(&serial_req) != 0)
		printk("Early serial init of port 0 failed\n");

	/* Assume early_serial_setup() doesn't modify serial_req */
	serial_req.line = 1;
	serial_req.port = 1;
//	serial_req.irq = 25; /* XXXX */
	serial_req.irq = 24; /* XXXX */
	serial_req.iomem_base = (u_char *)SANDPOINT_SERIAL_0;

#if defined(CONFIG_SERIAL_TEXT_DEBUG) || defined(CONFIG_KGDB)
//	gen550_init(1, &serial_req);
#endif

//	if ((&serial_req) != 0)
//		printk("Early serial init of port 1 failed\n");

}
#endif

static void __init
sandpoint_setup_arch(void)
{
	loops_per_jiffy = 100000000 / HZ;

#ifdef CONFIG_BLK_DEV_INITRD
	if (initrd_start)
		ROOT_DEV = MKDEV(RAMDISK_MAJOR, 0);
	else
#endif
#ifdef	CONFIG_ROOT_NFS
		ROOT_DEV = to_kdev_t(0x00FF);	/* /dev/nfs pseudo device */
#else
		ROOT_DEV = to_kdev_t(0x0301);	/* /dev/hda1 IDE disk */
#endif

	/* Lookup PCI host bridges */
	sandpoint_find_bridges();

#ifdef CONFIG_SERIAL
	sandpoint_early_serial_map();
#endif

#ifdef CONFIG_DUMMY_CONSOLE
	conswitchp = &dummy_con;
#endif

	printk(KERN_INFO "BUFFALO Network Attached Storage Series\n");
	printk(KERN_INFO "(C) 2002-2004 BUFFALO INC.\n");

	/* DINK32 12.3 and below do not correctly enable any caches.
	 * We will do this now with good known values.  Future versions
	 * of DINK32 are supposed to get this correct.
	 */
	if (cur_cpu_spec[0]->cpu_features & CPU_FTR_SPEC7450)
		/* 745x is different.  We only want to pass along enable. */
		_set_L2CR(L2CR_L2E);
	else if (cur_cpu_spec[0]->cpu_features & CPU_FTR_L2CR)
		/* All modules have 1MB of L2.  We also assume that an
		 * L2 divisor of 3 will work.
		 */
		_set_L2CR(L2CR_L2E | L2CR_L2SIZ_1MB | L2CR_L2CLK_DIV3
				| L2CR_L2RAM_PIPE | L2CR_L2OH_1_0 | L2CR_L2DF);
}

#define	SANDPOINT_87308_CFG_ADDR		0x15c
#define	SANDPOINT_87308_CFG_DATA		0x15d

#define	SANDPOINT_87308_CFG_INB(addr, byte) {				\
	outb((addr), SANDPOINT_87308_CFG_ADDR);				\
	(byte) = inb(SANDPOINT_87308_CFG_DATA);				\
}

#define	SANDPOINT_87308_CFG_OUTB(addr, byte) {				\
	outb((addr), SANDPOINT_87308_CFG_ADDR);				\
	outb((byte), SANDPOINT_87308_CFG_DATA);				\
}

#define SANDPOINT_87308_SELECT_DEV(dev_num) {				\
	SANDPOINT_87308_CFG_OUTB(0x07, (dev_num));			\
}

#define	SANDPOINT_87308_DEV_ENABLE(dev_num) {				\
	SANDPOINT_87308_SELECT_DEV(dev_num);				\
	SANDPOINT_87308_CFG_OUTB(0x30, 0x01);				\
}

static void __init
sandpoint_init2(void)
{
	return;
}

/*
 * Interrupt setup and service.  Interrrupts on the Sandpoint come
 * from the four PCI slots plus the 8259 in the Winbond Super I/O (SIO).
 * These interrupts are sent to one of four IRQs on the EPIC.
 * The SIO shares its interrupt with either slot 2 or slot 3 (INTA#).
 * Slot numbering is confusing.  Sometimes in the documentation they
 * use 0,1,2,3 and others 1,2,3,4.  We will use slots 1,2,3,4 and
 * map this to IRQ 16, 17, 18, 19.
 * For Sandpoint X3, this has been better designed.  The 8259 is
 * cascaded from EPIC IRQ0, IRQ1-4 map to PCI slots 1-4, IDE is on
 * EPIC 7 and 8.
 */
static void __init
sandpoint_init_IRQ(void)
{
	int i;

	OpenPIC_InitSenses = sandpoint_openpic_initsenses;
	OpenPIC_NumInitSenses = sizeof(sandpoint_openpic_initsenses);

	/*
	 * We need to tell openpic_set_sources where things actually are.
	 * mpc10x_common will setup OpenPIC_Addr at ioremap(EUMB phys base +
	 * EPIC offset (0x40000));  The EPIC IRQ Register Address Map -
	 * Interrupt Source Configuration Registers gives these numbers
	 * as offsets starting at 0x50200, we need to adjust occordinly.
	 */

	/* Map serial interrupt 0 */
	openpic_set_sources(0, 5, OpenPIC_Addr + 0x10200);
	/* Map serial interrupts 2-5 */
#if 0
//	openpic_set_sources(1, 4, OpenPIC_Addr + 0x10220);
	/* Map serial interrupts 8-9 */
//	openpic_set_sources(5, 2, OpenPIC_Addr + 0x10300);
	/* Skip reserved space and map i2c and DMA Ch[01] */
#endif
//	openpic_set_sources(5, 3, OpenPIC_Addr + 0x11020);
	openpic_set_sources(5, 2, OpenPIC_Addr + 0x11040);
	/* Skip reserved space and map Message Unit Interrupt (I2O) */
//	openpic_set_sources(8, 1, OpenPIC_Addr + 0x110C0);
	openpic_set_sources(7, 1, OpenPIC_Addr + 0x110C0);
	/* Map Internal Duart Ch 1 and Ch 2 */
//	openpic_set_sources(9, 1, OpenPIC_Addr + 0x11120);
//	openpic_set_sources(10, 1, OpenPIC_Addr + 0x11140);
	openpic_set_sources(8, 2, OpenPIC_Addr + 0x11120);

	openpic_init(NUM_8259_INTERRUPTS);

}

/*
 * Because the Sandpoint X2 has the i8259 cascade sharing an IRQ with a
 * PCI device, if we have a possible cascade IRQ we need to see if the
 * i8259 has something pending.  The only issue here is that the cascade
 * IRQs will get a higher priority than an OpenPIC one, but this should be
 * OK.
 */
static int
sandpoint_get_irq(struct pt_regs *regs)
{
        int irq, cascade_irq;

	irq = openpic_irq();

	if (irq == OPENPIC_VEC_SPURIOUS+16)
		irq = -1;

	return irq;
}

static u32
sandpoint_irq_cannonicalize(u32 irq)
{
	if (irq == 2)
		return 9;
	else
		return irq;
}

static unsigned long __init
sandpoint_find_end_of_memory(void)
{
#if 0
	bd_t *bp = (bd_t *)__res;

	if (bp->bi_memsize)
		return bp->bi_memsize;
#endif

	/* This might be fixed in DINK32 12.4, or we'll have another
	 * way to determine the correct memory size anyhow. */
	/* return mpc10x_get_mem_size(MPC10X_MEM_MAP_B); */
#if defined(CONFIG_HGLAN) || defined(CONFIG_HTGL)
	return 128*1024*1024;
#elif defined(CONFIG_HLAN) || defined(CONFIG_LAN)
	return 64*1024*1024;
#else
	return 128*1024*1024;
#endif

}

static void __init
sandpoint_map_io(void)
{
	// PCI I/O, ROM area
	io_block_mapping(0xfe000000, 0xfe000000, 0x02000000, _PAGE_IO);
	// EUMB area
	io_block_mapping(MPC10X_MAPB_EUMB_BASE, MPC10X_MAPB_EUMB_BASE, 
						MPC10X_EUMB_SIZE, _PAGE_IO);
}

/*
 * Due to Sandpoint X2 errata, the Port 92 will not work.
 */
static void
sandpoint_restart(char *cmd)
{
	__cli();

	/* Reset system via Port 92 */
	for(;;)  /* Spin until reset happens */
		out_8(AVR_PORT, 0x47);	// send shutdown messages
}

static void
sandpoint_power_off(void)
{
	__cli();
	for(;;)  /* No way to shut power off with software */
		out_8(AVR_PORT, 0x47);	// send shutdown messages
	/* NOTREACHED */
}

static void
sandpoint_halt(void)
{
	sandpoint_power_off();
	/* NOTREACHED */
}

static int
sandpoint_show_cpuinfo(struct seq_file *m)
{
	seq_printf(m, "vendor\t\t: Motorola SPS\n");
	seq_printf(m, "machine\t\t: Sandpoint\n");

	return 0;
}

/*
 * Set BAT 3 to map 0xf8000000 to end of physical memory space 1-to-1.
 */
static __inline__ void
sandpoint_set_bat(void)
{
	unsigned long bat2u, bat2l;
	unsigned long bat3u, bat3l;

#if 0
	static int	mapping_set = 0;

	if (!mapping_set) {
		__asm__ __volatile__(
		" lis %0,0x8000\n \
		  ori %1,%0,0x002a\n \
		  ori %0,%0,0x000e\n \
		  isync\n \
		  mtspr 0x21c,%0\n \
		  isync\n \
		  mtspr 0x21d,%1\n \
		  isync\n \
		  sync "
		:: "r" (bat2u), "r" (bat2l));

		__asm__ __volatile__(
		" lis %0,0xf800\n \
		  ori %1,%0,0x002a\n \
		  ori %0,%0,0x0ffe\n \
		  isync\n \
		  mtspr 0x21f,%1\n \
		  isync\n \
		  mtspr 0x21e,%0\n \
		  isync\n \
		  sync "
		:: "r" (bat3u), "r" (bat3l));

		mapping_set = 1;
	}
#endif
	__asm__ __volatile__(
			" lis %0,0x8000\n	\
			ori %1,%0,0x002a\n	\
			ori %0,%0,0x000e\n	\
			mtspr 0x21c,%0\n	\
			mtspr 0x21d,%1\n	\
			isync\n			\
			sync "
			: "=r" (bat2u), "=r" (bat2l));
	__asm__ __volatile__(
			" lis %0,0xf800\n	\
			ori %1,%0,0x002a\n	\
			ori %0,%0,0x0ffe\n	\
			mtspr 0x21e,%0\n	\
			mtspr 0x21f,%1\n	\
			isync\n			\
			sync "
			: "=r" (bat3u), "=r" (bat3l));

}

TODC_ALLOC();

#define PARM_ADDR 0xfff70000
void __init
parse_kernelparm(void) {
	unsigned char *cmdparm = cmd_line;
	unsigned char *addr;
	unsigned int magic = 0, i;

	addr = (unsigned char *)PARM_ADDR;

	for (i = 0; i < 4; i++) {
		magic += (unsigned int)*addr << (i*8);
		addr++;
	}
	/* not found magic number */
	if (magic != 0x82410000)
		return;

	addr = (unsigned char *)PARM_ADDR + 0x10;

	for (;;) {
		unsigned char tmp_parm[0x20], *parm = tmp_parm;
		unsigned int terminate;

		terminate = 0;
		memset(tmp_parm, (char)NULL, sizeof(tmp_parm));

		/* read kernel parameter from flash */
		for (i = 0; i < 0x20; i++) {
			if (i < 4) terminate += (unsigned int)*addr << (i*8);
			tmp_parm[i] = *addr++;
		}

		if (strncmp(tmp_parm, "bootparm", 8) == 0) {
			parm += 9;
			if (strncmp(parm, "root=/dev/ram0", 14) == 0) {
				memcpy(cmdparm, parm, 14);
				cmdparm += 14;
				*cmdparm++ = ' ';
				*cmdparm++ = '\0';
			}
			else if (strncmp(parm, "root=/dev/hda1", 14) == 0) {
				memcpy(cmdparm, parm, 14);
				cmdparm += 14;
				*cmdparm++ = ' ';
				*cmdparm++ = '\0';
			}
			else if (strncmp(parm, "root=/dev/md0", 13) == 0) {
				memcpy(cmdparm, parm, 13);
				cmdparm += 13;
				*cmdparm++ = ' ';
				*cmdparm++ = '\0';
			}
		}

		if (terminate == 0xffffffff)
			/* not written in flash */
			break;
	}
}

void __init
platform_init(unsigned long r3, unsigned long r4, unsigned long r5,
	      unsigned long r6, unsigned long r7)
{
	parse_bootinfo(find_bootinfo());


	/* ASSUMPTION:  If both r3 (bd_t pointer) and r6 (cmdline pointer)
	 * are non-zero, then we should use the board info from the bd_t
	 * structure and the cmdline pointed to by r6 instead of the
	 * information from birecs, if any.  Otherwise, use the information
	 * from birecs as discovered by the preceeding call to
	 * parse_bootinfo().  This rule should work with both PPCBoot, which
	 * uses a bd_t board info structure, and the kernel boot wrapper,
	 * which uses birecs.
	 */
#if 0
	if ( r3 && r6) {
		/* copy board info structure */
		memcpy( (void *)__res,(void *)(r3+KERNELBASE), sizeof(bd_t) );
		/* copy command line */
		*(char *)(r7+KERNELBASE) = 0;
		strcpy(cmd_line, (char *)(r6+KERNELBASE));
	}
#endif

#ifdef CONFIG_BLK_DEV_INITRD
	/* take care of initrd if we have one */
	if ( r4 )
	{
		initrd_start = r4 + KERNELBASE;
		initrd_end = r5 + KERNELBASE;
	}
#endif /* CONFIG_BLK_DEV_INITRD */

	/* Map in board regs, etc. */
	sandpoint_set_bat();

	parse_kernelparm();

	isa_io_base = MPC10X_MAPB_ISA_IO_BASE;
	isa_mem_base = MPC10X_MAPB_ISA_MEM_BASE;
	pci_dram_offset = MPC10X_MAPB_DRAM_OFFSET;
	ISA_DMA_THRESHOLD = 0x00ffffff;
	DMA_MODE_READ = 0x44;
	DMA_MODE_WRITE = 0x48;

	ppc_md.setup_arch = sandpoint_setup_arch;
	ppc_md.show_cpuinfo = sandpoint_show_cpuinfo;
	ppc_md.irq_cannonicalize = sandpoint_irq_cannonicalize;
	ppc_md.init_IRQ = sandpoint_init_IRQ;
	ppc_md.get_irq = sandpoint_get_irq;
	ppc_md.init = sandpoint_init2;

	ppc_md.restart = sandpoint_restart;
	ppc_md.power_off = sandpoint_power_off;
	ppc_md.halt = sandpoint_halt;

	ppc_md.find_end_of_memory = sandpoint_find_end_of_memory;
	ppc_md.setup_io_mappings = sandpoint_map_io;

#ifdef CONFIG_CHAR_DEV_BUFFALO_RTC
	ppc_md.time_init      = melco_rtc_init;
	ppc_md.set_rtc_time   = melco_set_rtc;
	ppc_md.get_rtc_time   = melco_get_rtc;
	ppc_md.calibrate_decr = melco_caribrate_decr;
#else
	ppc_md.time_init = NULL;
	ppc_md.set_rtc_time = NULL;
	ppc_md.get_rtc_time = NULL;
	ppc_md.calibrate_decr = NULL;
#endif

	ppc_md.nvram_read_val = NULL;
	ppc_md.nvram_write_val = NULL;

#ifdef CONFIG_SERIAL
#ifdef CONFIG_SERIAL_TEXT_DEBUG
	ppc_md.progress = gen550_progress;
#endif
	ppc_md.early_serial_map = sandpoint_early_serial_map;
#endif

#ifdef CONFIG_VT
	ppc_md.kbd_setkeycode    = pckbd_setkeycode;
	ppc_md.kbd_getkeycode    = pckbd_getkeycode;
	ppc_md.kbd_translate     = pckbd_translate;
	ppc_md.kbd_unexpected_up = pckbd_unexpected_up;
	ppc_md.kbd_leds          = pckbd_leds;
	ppc_md.kbd_init_hw       = pckbd_init_hw;
#ifdef CONFIG_MAGIC_SYSRQ
	ppc_md.ppc_kbd_sysrq_xlate       = pckbd_sysrq_xlate;
	SYSRQ_KEY = 0x54;
#endif
#endif

	return;
}
