/*
 * arch/ppc/platforms/powerk2.c
 *
 * Support for configuring the NP4GS3 carrier boards and the Interposer card
 * in the NP4GS3 chassis.
 *
 * "Portions of this file are partially based on earlier efforts by Ralph
 * Blach, John F Davis and Burt Silverman from IBM Corp"
 *
 * Other portions of this file are derived from arch/ppc/kernel/pci.c

 * This program is free software; you can redistribute  it and/or modify it
 * under  the terms of  the GNU General  Public License as published by the
 * Free Software Foundation;  either version 2 of the  License, or (at your
 * option) any later version.
 */

#include <linux/config.h>
#include <linux/kernel.h>
#include <linux/pci.h>

#include <asm/pci-bridge.h>
#include <asm/machdep.h>

#include <platforms/k2.h>
#include <platforms/powerk2.h>

#undef DEBUG

#ifdef DEBUG
#define DBG(x...) printk(x)
#else
#define DBG(x...)
#endif


void powerk2_pcibios_fixup_resources(struct pci_dev *dev)
{
  struct pci_controller* hose = (struct pci_controller *)dev->sysdata;
  int i;
  unsigned long offset;
  static unsigned d6;
  static unsigned rainier[2] =    {K2_PCI64_UPPER_MEM + 1 -  0x8000000,
                                   K2_PCI64_UPPER_MEM + 1 - 0x18000000};
  unsigned int bar_response, testdw;
  unsigned short testw;

  if (dev->bus->number == 1 && PCI_SLOT(dev->devfn)==PLXI) {
    /* Configure the PLX9030 on the interposer card. */

    void *register_space;

    pcibios_read_config_dword(1,dev->devfn,PCI_BASE_ADDRESS_1,
                                &bar_response);
    bar_response = (bar_response & PCI_BASE_ADDRESS_IO_MASK) +
                                                        K2_PCI64_IO_BASE;
    writel(-LAS0SIZE,bar_response+0);    /*LAS0RR is 0x10000 bytes  MEM */
    writel(-LAS1SIZE,bar_response+4);    /*LAS1RR is 0x20000 bytes  MEM */
    writel(-LAS2SIZE,bar_response+8);    /*LAS2RR is 0x10000 bytes  MEM */
    writel(-LAS3SIZE,bar_response+0xc);  /*LAS3RR is 0x20000 bytes  MEM */
    writel(0x00000001,bar_response+0x14); /*LAS0BA is at 0x000 */
    writel(0x00020001,bar_response+0x18); /*LAS1BA is at 0x00020000 */
    writel(0x00000001,bar_response+0x1c); /*LAS2BA is at 0x00000000 */

    /*WRCYCLHLD(1)|BUSWIDTH(8)|NWADWAIT(3)|NRADWAIT(3)*/
    writel(0x400180C0, bar_response+0x28); /*LAS0BRD */
    writel(0x00007FC0, bar_response+0x2C); /*LAS1BRD */
    writel(0x400180C0, bar_response+0x30); /*LAS2BRD */

    /*****************         RTC  (GPIO4 Chip Select) ***********************/
    /*CS2[bit6](SMACK) and CS3[bit9](Interposer Flash), not GP, pin defnition */
    /* also we desire: GPIO4, and GPIO4 Output*/
    writel(SMACK_ADRS|SMACK_SIZE|CSENABLE,bar_response+0x44);/*CS2BASEforSMACK*/
    writel(0x00030001, bar_response+0x48); /* CS3BASE for Interposer Flash */
    testdw = readl(bar_response+GPIOC);
    writel( (testdw&GPIO4)|GPIO4OUT|CS3|CS2,bar_response+GPIOC);
    writel(CNTRL_DELREAD|0x00780000, bar_response+PCI9030_CNTRL);

    /* Configure resource 0 away, else we have resource conflicts when
       rebooting without cycling power.  */
    pcibios_write_config_dword(1,dev->devfn,PCI_BASE_ADDRESS_0,0x0);
    dev->resource[0].start = 0x0;
    dev->resource[0].end   = 0x0;

    // Restore to the wise firmware choice
    pcibios_write_config_dword(1,dev->devfn,PCI_BASE_ADDRESS_1,0x00000101);
    dev->resource[1].start = 0x00000100;
    dev->resource[1].end   = dev->resource[1].start + 0x0000007F;

    /* UART, SMACK, and I2C in memory space */
    pcibios_write_config_dword(1,dev->devfn,PCI_BASE_ADDRESS_2,
			       K2_PCI64_UPPER_MEM+1 - 2*LAS0SIZE);
    register_space=ioremap(K2_PCI64_UPPER_MEM+1 - 2*LAS0SIZE +
			   hose->pci_mem_offset, 0x400);
    if (register_space)
	    writew(0,register_space+SMACK_ADRS+SMACKBLADERESETENABLE);
    else
	    printk("Cannot remap\n");

    iounmap (register_space);
    dev->resource[2].start = K2_PCI64_UPPER_MEM+1 - 2*LAS0SIZE;
    dev->resource[2].end   = K2_PCI64_UPPER_MEM - LAS0SIZE;
    dev->resource[2].flags &= ~IORESOURCE_IO;
    dev->resource[2].flags |= IORESOURCE_MEM;

    /* HW not used by reference SW, but must configure for proper
       operation of the embedded 405 */
    /* Interposer Flash, accessible from EPPC only */
    //BMS FLUKE: You have to write this register twice
    pcibios_write_config_dword(1,dev->devfn, PCI_BASE_ADDRESS_3, 0xFFFE0000);
    pcibios_write_config_dword(1,dev->devfn, PCI_BASE_ADDRESS_3, 0xFFFE0000);

    //BMS just to give some extra protection
    pcibios_read_config_word(1,dev->devfn,PCI_COMMAND,
			     &testw);
    pcibios_write_config_word(1,dev->devfn,PCI_COMMAND, 
			      testw|PCI_COMMAND_SERR|PCI_COMMAND_PARITY);

#if 1
    /* The interposer flash is not used by the system SW of the NP4GS3,
       but the BAR register above must be configured for the embedded 405
       to boot.  However, set the kernel's knowledge of it's resources to
       0 so the kernel will ignore it.  Otherwise, we get warnings when
       booting the PowerK2 if DEBUG is defined above.  If you really
       want access to this device, the code below in the #else clause
       should be a start for getting it working.*/
    dev->resource[3].start = 0x0;
    dev->resource[3].end   = 0x0;
    dev->resource[3].flags = 0x0;
#else
    dev->resource[3].start = 0xFFFE0000 - hose->pci_mem_offset;
    dev->resource[3].end   = dev->resource[3].start+ LAS1SIZE - 1;
    dev->resource[3].flags &= ~IORESOURCE_IO;
    dev->resource[3].flags |= IORESOURCE_MEM;
#endif

    /* Interposer Real Time Clock, accessible from EPPC */
    pcibios_write_config_dword(1,dev->devfn, PCI_BASE_ADDRESS_4, 0x80000000);
    dev->resource[4].start = 0x00000000;
    dev->resource[4].end   = 0x00000000;

    /* Configure resource 5 away.  It is not used, and causes warnings on
       boot if DEBUG is defined above */
    pcibios_write_config_dword(1,dev->devfn,PCI_BASE_ADDRESS_5,0x0);
    dev->resource[5].start = 0x0;
    dev->resource[5].end   = 0x0;

  } else if (dev->bus->number == 1 && ( PCI_SLOT(dev->devfn)==NP1)) {
    /* Configure the NP4GS3 device in slot 1 of the NP4GS3 chassis */
    /*move VxWorks CarrierCard Rainier BARS until warm reset issue is fixed*/

    pcibios_write_config_dword(1,dev->devfn,PCI_BASE_ADDRESS_1,
			       rainier[0]|PCI_BASE_ADDRESS_MEM_PREFETCH);
    dev->resource[1].start = rainier[0];
    dev->resource[1].end   = rainier[0] + 0x1FFFF;
    dev->resource[1].flags |= IORESOURCE_MEM;
    d6 = rainier[0]- 0x8000000;
    pcibios_write_config_dword(1,dev->devfn,PCI_BASE_ADDRESS_2,
			       d6|PCI_BASE_ADDRESS_MEM_PREFETCH);
    dev->resource[2].start = d6;
    dev->resource[2].end   = d6 + 0x7FFFFFF;
    dev->resource[2].flags |= IORESOURCE_MEM;
    pcibios_read_config_dword(1,dev->devfn,PCI_COMMAND,&bar_response);
    pcibios_write_config_dword( 1, dev->devfn, PCI_COMMAND, bar_response |
		PCI_COMMAND_IO | PCI_COMMAND_MEMORY | PCI_COMMAND_MASTER);
  } else if (dev->bus->number == 1 && (PCI_SLOT(dev->devfn)== NP2)) {
    /* Configure the NP4GS3 device in slot 2 of the NP4GS3 chassis */

    pcibios_write_config_dword(1,dev->devfn,PCI_BASE_ADDRESS_1,
			       rainier[1]|PCI_BASE_ADDRESS_MEM_PREFETCH);
    dev->resource[1].start = rainier[1];
    dev->resource[1].end   = rainier[1] + 0x1FFFF;
    dev->resource[1].flags |= IORESOURCE_MEM;
    d6 = rainier[1] - 0x8000000;
    pcibios_write_config_dword(1,dev->devfn,PCI_BASE_ADDRESS_2,
                                      d6|PCI_BASE_ADDRESS_MEM_PREFETCH);
    dev->resource[2].start = d6;
    dev->resource[2].end   = d6 + 0x7FFFFFF;
    dev->resource[2].flags |= IORESOURCE_MEM;
    pcibios_read_config_dword(1,dev->devfn,PCI_COMMAND,&bar_response);
    pcibios_write_config_dword( 1, dev->devfn, PCI_COMMAND, bar_response |
		PCI_COMMAND_IO | PCI_COMMAND_MEMORY | PCI_COMMAND_MASTER);
  }
  if(dev->bus->number==1 && PCI_SLOT(dev->devfn)==PLX1){
    unsigned int plx[2]     = {0x1000, 0x1300};

    /* Configure the PLX9030 chip on the NP4GS3 carrier card in slot 1
       of the NP4GS3 chassis to enable access to the UART behind it */
    pcibios_write_config_dword(1,dev->devfn,PCI_BASE_ADDRESS_1, plx[0]);
    writel(-8+1,plx[0] +0 +K2_PCI64_IO_BASE);   /*LAS0RR is 8 bytes  I/O */
    writel(0x00000301,plx[0]+0x14+K2_PCI64_IO_BASE);/*LAS0BA is at 0x300 */
    writel(0x400180C0, plx[0]+0x28+K2_PCI64_IO_BASE);          /*LAS0BRD */
    writel(CNTRL_DELREAD|0x00780000, plx[0]+PCI9030_CNTRL+K2_PCI64_IO_BASE);
    dev->resource[1].start = plx[0];
    dev->resource[1].end   = dev->resource[1].start+127 ;
    pcibios_write_config_dword(1,dev->devfn,PCI_BASE_ADDRESS_2, plx[1]);
    dev->resource[2].start = plx[1];
    dev->resource[2].end   = dev->resource[2].start+7 ;
    dev->resource[2].flags = IORESOURCE_IO;
  }
  if(dev->bus->number==1 && PCI_SLOT(dev->devfn)==PLX2){
    unsigned int plx[2]     = {0x2000, 0x2300};

    /* Configure the PLX9030 chip on the NP4GS3 carrier card in slot 2
       of the NP4GS3 chassis to enable access to the UART behind it */
    pcibios_write_config_dword(1,dev->devfn,PCI_BASE_ADDRESS_1, plx[0]);
    writel(-8+1,plx[0] +0 +K2_PCI64_IO_BASE);   /*LAS0RR is 8 bytes  I/O */
    writel(0x00000301,plx[0]+0x14+K2_PCI64_IO_BASE);/*LAS0BA is at 0x300 */
    writel(0x400180C0, plx[0]+0x28+K2_PCI64_IO_BASE);          /*LAS0BRD */
    writel(CNTRL_DELREAD|0x00780000, plx[0]+PCI9030_CNTRL+K2_PCI64_IO_BASE);
    dev->resource[1].start = plx[0];
    dev->resource[1].end   = dev->resource[1].start+127 ;
    pcibios_write_config_dword(1,dev->devfn,PCI_BASE_ADDRESS_2, plx[1]);
    dev->resource[2].start = plx[1];
    dev->resource[2].end   = dev->resource[2].start+7 ;
    dev->resource[2].flags = IORESOURCE_IO;
  }

	for (i = 0; i < DEVICE_COUNT_RESOURCE; i++) {
		struct resource *res = dev->resource + i;

		if ((!res->start || !res->flags) &&
		   /* Don't skip if we are looking at the PLX9030 Interposer */
                   !(i==1 && PCI_SLOT(dev->devfn)==PLXI && dev->bus->number==1))
			continue;
		if (res->end == 0xffffffff) {
			DBG("PCI:%s Resource %d [%08lx-%08lx] is unassigned\n",
			    dev->slot_name, i, res->start, res->end);
			res->end -= res->start;
			res->start = 0;
			continue;
		}
		offset = 0;
		if (res->flags & IORESOURCE_MEM) {
			offset = hose->pci_mem_offset;
		} else if (res->flags & IORESOURCE_IO) {
			/* Temporary HACK to get the Power K2 working */
			if ((unsigned long)hose->io_base_virt == isa_io_base) {
				offset = 0;
			} else {
			  	offset = (unsigned long) hose->io_base_virt;
			}
		}
		if (offset != 0) {
			res->start += offset;
			res->end += offset;
#ifdef DEBUG
			printk("Fixup res %d (%lx) of dev %s: %lx -> %lx\n",
			       i, res->flags, dev->slot_name,
			       res->start - offset, res->start);
#endif
		}
	}

	/* Call machine specific resource fixup */
	if (ppc_md.pcibios_fixup_resources)
		ppc_md.pcibios_fixup_resources(dev);

}
