/*
 * linux/drivers/char/innovator-keypad.c
 *
 * TI Innovator/OMAP1510 keypad driver
 *
 * Author: MontaVista Software, Inc.
 *         <gdavis@mvista.com> or <source@mvista.com>
 *
 * 2003 (c) MontaVista Software, Inc. This file is licensed under the
 * terms of the GNU General Public License version 2. This program is
 * licensed "as is" without any warranty of any kind, whether express
 * or implied.
 *
 * History:
 *
 * 20030529: George G. Davis <gdavis@mvista.com>
 *	Initial release
 * 20031029: George G. Davis <gdavis@mvista.com>
 *	Convert to EV_MSC based device since EV_KEY was not appropriate.
 *
 */

#include <linux/config.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/stddef.h>
#include <linux/timer.h>
#include <linux/input.h>
#include <linux/interrupt.h>
#include <linux/delay.h>
#include <linux/sched.h>
#include <linux/ioport.h>
#include <linux/slab.h>

#include <asm/hardware.h>
#include <asm/uaccess.h>
#include <asm/arch/irqs.h>
#include <asm/io.h>
#include <asm/errno.h>
#include <asm/irq.h>

#undef	DEBUG
#ifdef	DEBUG
int debug = 0;
#define	DPRINTK(fmt, args...) \
	if (debug) {						\
		printk(KERN_DEBUG "%s:%d: " fmt,		\
		       __FUNCTION__ , __LINE__ , ## args);	\
	}
#else
#define	DPRINTK(fmt, args...)
#endif

static struct input_dev keypad_dev;
static int innovator_keypad_pid;
static DECLARE_COMPLETION(keypadt);
static wait_queue_head_t keypadt_wait;
struct task_struct *innovator_keypad_task;


static u32
innovator_keypad_read(void)
{
	int i;
	u32 keys = 0;

	outw(0xff, OMAP1510_MPUIO_KBC_REG);
	udelay(1);

	for (i = 0; i < 4; i += 1) {
		outw(~(1 << i) & 0xff, OMAP1510_MPUIO_KBC_REG);
		udelay(1);
		keys |= ((~inw(OMAP1510_MPUIO_KBR_LATCH)) & 0xf) << (i << 2);
		inw(OMAP1510_32kHz_TIMER_BASE);	/* BTS_Errata.22 */
	}

	outw(0x0, OMAP1510_MPUIO_KBC_REG);
	udelay(1);

	DPRINTK("keys: %0x\n", keys);

	return keys;
}

static void
innovator_keypad_scan(void)
{
	int timeout;
	u32 old = 0;
	u32 new = 0;

	do {
		new = innovator_keypad_read();
		if (old != new) {
			DPRINTK("old: %0x, new: %0x\n", old, new);
			input_event(&keypad_dev, EV_MSC, MSC_KEYPAD, new);
			old = new;
		}
		timeout = interruptible_sleep_on_timeout(&keypadt_wait, 10);
		if (timeout) {
			break;
		}
	} while (new);

	return;
}

static int
innovator_keypad_thread(void *null)
{
	struct task_struct *tsk = current;
	unsigned long flags;

	daemonize();
	reparent_to_init();
	strcpy(tsk->comm, "keypadt");
	tsk->tty = NULL;

	/* only want to receive SIGKILL */
	spin_lock_irq(&tsk->sigmask_lock);
	siginitsetinv(&tsk->blocked, sigmask(SIGKILL));
	recalc_sigpending(tsk);
	spin_unlock_irq(&tsk->sigmask_lock);

	innovator_keypad_task = tsk;

	complete(&keypadt);

	DPRINTK("init\n");

	do {
		/* Careful, a dead lock can occur here if an keypad
		 * interrupt occurs before we're ready and waiting on
		 * the keypadt_wait queue. So we disable interrupts
		 * while unmasking device interrupts prior to putting
		 * the thread to sleep.
		 */
		local_irq_save(flags);
		DPRINTK("sleep\n");
		outw(0x0, OMAP1510_MPUIO_KBD_MASKIT);
		interruptible_sleep_on(&keypadt_wait);
		local_irq_restore(flags);
		if (signal_pending(tsk))
			break;
		DPRINTK("wake\n");
		innovator_keypad_scan();
	} while (!signal_pending(tsk));

	outw(0x1, OMAP1510_MPUIO_KBD_MASKIT);
	complete_and_exit(&keypadt, 0);
}

static void
innovator_keypad_interrupt(int irq, void *dev_id, struct pt_regs *regs)
{
	outw(1, OMAP1510_MPUIO_KBD_MASKIT);

	DPRINTK("interrupt\n");

	/* Wake up thread to scan keypad. */
	wake_up(&keypadt_wait);
}

static int __init
innovator_keypad_init(void)
{
	printk(KERN_INFO "TI Innovator/OMAP1510 keypad driver.\n");

	outw(1, OMAP1510_MPUIO_KBD_MASKIT);
	outw(0xff, OMAP1510_GPIO_DEBOUNCING_REG);
	outw(0x0, OMAP1510_MPUIO_KBC_REG);

	keypad_dev.name = "Innovator/OMAP1510 Keypad";
	keypad_dev.idbus = BUS_ONCHIP;
	keypad_dev.idvendor = 0x0451;	/* Texas Instruments, Inc. */
	keypad_dev.idproduct = 0x1510;	/* total fabrication here */
	keypad_dev.idversion = 0x0100;

	keypad_dev.evbit[LONG(EV_MSC)] |= BIT(EV_MSC);
	keypad_dev.mscbit[LONG(MSC_KEYPAD)] |= BIT(MSC_KEYPAD);

	input_register_device(&keypad_dev);

	init_waitqueue_head(&keypadt_wait);
	init_completion(&keypadt);
	innovator_keypad_pid =
	    kernel_thread(innovator_keypad_thread, NULL,
			  CLONE_FS | CLONE_FILES | CLONE_SIGHAND);
	if (innovator_keypad_pid <= 0) {
		return -1;
	}
	wait_for_completion(&keypadt);

	if (request_irq(INT_KEYBOARD, innovator_keypad_interrupt,
			SA_INTERRUPT, "keypad", 0) < 0) {
		/* Kill the thread */
		init_completion(&keypadt);
		send_sig(SIGKILL, innovator_keypad_task, 1);
		wake_up(&keypadt_wait);
		wait_for_completion(&keypadt);
		input_unregister_device(&keypad_dev);
		return -EINVAL;
	}

	return 0;
}

static void __exit
innovator_keypad_exit(void)
{
	outw(1, OMAP1510_MPUIO_KBD_MASKIT);

	/* Kill the thread */
	init_completion(&keypadt);
	send_sig(SIGKILL, innovator_keypad_task, 1);
	wake_up(&keypadt_wait);
	wait_for_completion(&keypadt);

	input_unregister_device(&keypad_dev);
	free_irq(INT_KEYBOARD, 0);

	return;
}

module_init(innovator_keypad_init);
module_exit(innovator_keypad_exit);

MODULE_AUTHOR("George G. Davis <gdavis@mvista.com>");
MODULE_DESCRIPTION("TI Innovator/OMAP1510 Keypad driver");
MODULE_LICENSE("GPL");
