/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Copyright 2002, 2003 Motorola, Inc. All Rights Reserved.
 *
 */

/*MMA_MAC device driver*/
#include <linux/version.h>
#include <linux/module.h>
#include <linux/errno.h>
#include <linux/signal.h>
#include <linux/sched.h>

#include <linux/vmalloc.h>
#include <linux/interrupt.h>
#include <linux/tty.h>
#include <linux/tty_flip.h>
#include <linux/config.h>
#include <linux/major.h>
#include <linux/slab.h>
#include <linux/string.h>
#include <linux/fcntl.h>
#include <linux/mm.h>
#include <linux/kernel.h>
#include <linux/console.h>
#include <linux/reboot.h>
#include <linux/keyboard.h>
#include <linux/init.h>

#include <linux/spinlock.h>
#include <asm/hardirq.h>

#include <asm/io.h>
#include <asm/irq.h>
#include <asm/system.h>
#include <asm/segment.h>
#include <asm/bitops.h>
#include <asm/delay.h>
#include <asm/uaccess.h>
#include <asm/dma.h>
#include <asm/pgalloc.h>
#include <asm/page.h>
#include <linux/wrapper.h>

#include "mma_dct.h"
#include "Dct.h"

#define DCT_PM
#ifdef DCT_PM
#include <linux/pm.h>
#define PM_OPT " [pm]"
static struct pm_dev *pm;
#endif				// DCT_PM

/*******************************************************
*DCT macros
* 	
********************************************************/
#define Enable_Dct_Irq(x) (MMA_DCTIRQENA |= (x))
#define Disable_Dct_Irq(x) (MMA_DCTIRQENA &= (~(x)))

//static wait_queue_head_t gDctWaitQueue;
static DECLARE_WAIT_QUEUE_HEAD(gDctWaitQueue);	/* waiting for buffer */
UINT8 gDctWaitFlag;
UINT32 *gDctResultBaseAddr;
void
dct_delay(UINT16 time)
{
	UINT16 i, count;
	for (i = 0; i < time; i++) {
		count++;
	}
}

/*******************************************************
*DCT open routine
* return : 0 -- success
*		  - 1 -- fail
********************************************************/
static int
dct_open(struct inode *inode, struct file *file)
{
	if (!gDctAvail)
		return -1;
	gDctDmaInCount = 0;
	gDctDmaOutCount = 0;
	gDctTransBlock = 0;
	gDctAvail = FALSE;
	MOD_INC_USE_COUNT;
	gDctComplete = FALSE;
	gDctResultBaseAddr = NULL;
	gDctTransMode = DCT_TRANS_INTR;
	gDctDmaAddr = 0;
	gDctTransCount = 0x0;
	gDctUseMC = TRUE;	//the default mode is memory controler
	gDctWaitFlag = 1;
	/*reset the DCT module */
	MMA_DCTCTRL = DCT_CLK;
	MMA_DCTCTRL = DCT_CLK | DCT_RESET;
	memset(&gDctSrc, 0, sizeof (struct DCT));
	memset(&gDctDest, 0, sizeof (struct DCT));
	return 0;
}

/*******************************************************
*DCT close routine
* return : 0 -- success
*		  - 1 -- fail
********************************************************/

static int
dct_release(struct inode *inode, struct file *filp)
{
	if (gDctAvail)
		return -1;
	gDctAvail = TRUE;

//should free the memory have request here??????
	if (gDctSrc.baseaddr != NULL) {
		kfree(gDctSrc.baseaddr);
		gDctSrc.baseaddr = NULL;
	}
	if (gDctDest.baseaddr != NULL) {
		//Add here if using consistent_alloc, then should use consistent_free()
		if (gDctUseMC == TRUE)
			consistent_free(gDctDest.baseaddr, gDctDest.datalen,
					gDctDmaAddr);
		else
			kfree(gDctDest.baseaddr);
		gDctDest.baseaddr = NULL;
		gDctDest.datalen = 0;
	}

	MOD_DEC_USE_COUNT;
	MMA_DCTCTRL = 0x0;
	return 0;
}

/*******************************************************
*DCT master routine
*
********************************************************/
void
dct_operation(UINT32 arg)
{
	UINT32 *srcphysical;
	UINT32 *destphysical;
	UINT32 *srcaddr;
	UINT32 *destaddr;
	UINT32 *testaddr;
	UINT32 incount;
	UINT32 outcount;
	int temp_timeout;
	int j;
	int blockcount;
	UINT32 i;
	UINT16 tempcount;
	UINT32 dctflags;
	gDctComplete = FALSE;

	if ((MMA_DCTCTRL & 0x0000000C) == DCT_MC_MC) {
		srcphysical = (UINT32 *) virt_to_bus(gDctSrc.baseaddr);
		destphysical = (UINT32 *) gDctDmaAddr;

		gDctSrcAddrDma = srcphysical;
		gDctDestAddrDma = destphysical;

		testaddr = gDctDest.baseaddr;
		srcaddr = gDctSrc.baseaddr;
		destaddr = gDctDest.baseaddr;

		gDctSrcAddr = gDctSrc.baseaddr;
		gDctDestAddr = gDctDest.baseaddr;
		incount = (gDctSrc.datalen) / 4;
		outcount = incount;
		blockcount = (gDctSrc.xCount) * (gDctSrc.yCount);

		MMA_DCTCTRL = DCT_CLK;
		MMA_DCTSRCADD = (UINT32) srcphysical;
		MMA_DCTDESADD = (UINT32) destphysical;
		MMA_DCTXOFF = gDctSrc.xOffset;
		MMA_DCTYOFF = gDctSrc.yOffset;
		tempcount = (gDctSrc.yCount) << 8;
		MMA_DCTXYCNT = tempcount + gDctSrc.xCount;
		MMA_DCTSKIP = gDctSrc.skip;
		switch (gDctTransMode) {
		case DCT_TRANS_POLL:
			MMA_DCTCTRL = DCT_CLK | DCT_START | arg;
			temp_timeout = 0;
			while ((MMA_DCTIRQSTAT & DCT_COMP_IRQ) == 0x00000000) {
			}
			MMA_DCTIRQSTAT |= DCT_COMP_IRQ;
			copy_to_user(gDctResultBaseAddr, gDctDest.baseaddr,
				     gDctDest.datalen);
			break;
		default:
			MMA_DCTCTRL = DCT_CLK;
			Enable_Dct_Irq(DCT_COMP_IRQ_ENA);
			MMA_DCTCTRL = DCT_CLK | DCT_START | arg;
			if (gDctWaitFlag == 1) {
				while (gDctComplete == FALSE) {
					interruptible_sleep_on(&gDctWaitQueue);
				}
				copy_to_user(gDctResultBaseAddr,
					     gDctDest.baseaddr,
					     gDctDest.datalen);
			}
			break;
		}
	} else if ((MMA_DCTCTRL & 0X0000000C) == DCT_ARC_ARC) {

		srcphysical = (UINT32 *) virt_to_bus(gDctSrc.baseaddr);
		destphysical = (UINT32 *) virt_to_bus(gDctDest.baseaddr);

		gDctSrcAddrDma = srcphysical;
		gDctDestAddrDma = destphysical;

		testaddr = gDctDest.baseaddr;
		srcaddr = gDctSrc.baseaddr;
		destaddr = gDctDest.baseaddr;

		gDctSrcAddr = gDctSrc.baseaddr;
		gDctDestAddr = gDctDest.baseaddr;
		incount = (gDctSrc.datalen) / 4;
		outcount = incount;
		blockcount = (gDctSrc.xCount) * (gDctSrc.yCount);

		switch (gDctTransMode) {
		case DCT_TRANS_POLL:
			MMA_DCTCTRL = DCT_CLK;
			tempcount = (gDctSrc.yCount) << 8;
			MMA_DCTXYCNT = tempcount + gDctSrc.xCount;
			MMA_DCTCTRL = DCT_CLK | DCT_ARC_ARC | DCT_START | arg;
			for (j = 0; j < blockcount; j++) {
				//when the FIFO is empty, write data
				while ((MMA_DCTIRQSTAT & DCT_FIFO_EMP_IRQ) ==
				       0x0) {
				}
				for (i = 0; i < 32; i++) {
					MMA_DCTFIFO = *srcaddr;
					srcaddr++;
				}

				//when the DCT operation finished, read result
				while ((MMA_DCTIRQSTAT & DCT_FIFO_FULL_IRQ) ==
				       0x0) {
				}
				for (i = 0; i < 32; i++) {
					*destaddr = MMA_DCTFIFO;
					destaddr++;
					outcount--;
				}
			}
			copy_to_user(gDctResultBaseAddr, gDctDest.baseaddr,
				     gDctDest.datalen);
			gDctComplete = FALSE;

			break;
		case DCT_TRANS_INTR:
			gDctTransCount = (gDctSrc.xCount) * (gDctSrc.yCount) * 32;	//for isr routine
			MMA_DCTCTRL = DCT_CLK;
			Enable_Dct_Irq(DCT_DATA_OUT_IRQ_ENA |
				       DCT_DATA_IN_IRQ_ENA);
			MMA_DCTSRCADD = 0x0;
			MMA_DCTDESADD = 0x0;
			MMA_DCTXOFF = 0x0;
			MMA_DCTYOFF = 0x0;
			MMA_DCTXYCNT = 0x0;
			MMA_DCTSKIP = 0x0;
			temp_timeout = 0;
			if (gDctWaitFlag == 1) {
				if (gDctTransCount >= 64) {
					save_flags(dctflags);
					cli();
					MMA_DCTCTRL =
					    DCT_CLK | DCT_ARC_ARC | DCT_START |
					    arg;
					while (gDctComplete == FALSE) {
						interruptible_sleep_on
						    (&gDctWaitQueue);
						restore_flags(dctflags);
					}
				} else {
					MMA_DCTCTRL =
					    DCT_CLK | DCT_ARC_ARC | DCT_START |
					    arg;
					while ((gDctComplete == FALSE)
					       && (temp_timeout < 200)) {
						temp_timeout++;
					}
				}
				copy_to_user(gDctResultBaseAddr,
					     gDctDest.baseaddr,
					     gDctDest.datalen);
				gDctComplete = FALSE;

			} else {
				MMA_DCTCTRL =
				    DCT_CLK | DCT_ARC_ARC | DCT_START | arg;

			}
			break;
		default:
			printk("transfer error in DCT********* \n");
			break;
		}
	} else {
		printk("not support this mem method \n");
	}
}

/*******************************************************
*DCT ioctl routine
* return : 0 -- success
*		  - 1 -- fail
********************************************************/
static int
dct_ioctl(struct inode *inode, struct file *filp, u_int cmd, u_long arg)
{
	UINT32 *tempaddr;
	UINT32 dctflags;
	switch (cmd) {

	case DCT_CMD_NO_WAIT:
		gDctWaitFlag = 0;
		break;
	case DCT_CMD_SET_WAIT:
		gDctWaitFlag = 1;
		break;
	case DCT_CMD_GET_STAT:
		save_flags(dctflags);
		cli();
		if (gDctComplete == FALSE) {
			*((UINT8 *) arg) = 0;
			restore_flags(dctflags);
		} else {
			*((UINT8 *) arg) = 1;
			restore_flags(dctflags);
		}
		break;
	case DCT_CMD_GET_DATA:
		copy_to_user(gDctResultBaseAddr, gDctDest.baseaddr,
			     gDctDest.datalen);
		break;
	case DCT_CMD_SET_SRC:
		if (gDctSrc.datalen >= ((struct DCT *) arg)->datalen) {
			tempaddr = gDctSrc.baseaddr;
			copy_from_user(&gDctSrc, (void *) arg,
				       sizeof (struct DCT));
			copy_from_user(tempaddr, gDctSrc.baseaddr,
				       gDctSrc.datalen);
			gDctSrc.baseaddr = tempaddr;
			return 0;
		}
		if (gDctSrc.baseaddr != NULL) {
			kfree(gDctSrc.baseaddr);
			gDctSrc.baseaddr = NULL;
		}
		copy_from_user(&gDctSrc, (void *) arg, sizeof (struct DCT));
		tempaddr = kmalloc(gDctSrc.datalen, GFP_KERNEL | GFP_DMA);
		copy_from_user(tempaddr, gDctSrc.baseaddr, gDctSrc.datalen);
		gDctSrc.baseaddr = tempaddr;
		break;
	case DCT_CMD_SET_DES:
		if (gDctDest.datalen >= ((struct DCT *) arg)->datalen) {
			tempaddr = gDctDest.baseaddr;
			copy_from_user(&gDctDest, (void *) arg,
				       sizeof (struct DCT));
			gDctResultBaseAddr = gDctDest.baseaddr;	//record the base addr
			gDctDest.baseaddr = tempaddr;
			return 0;
		}
		if ((gDctDest.baseaddr != NULL) && (gDctUseMC == TRUE)) {
			consistent_free(gDctDest.baseaddr, gDctDest.datalen,
					gDctDmaAddr);
			gDctDest.baseaddr = NULL;
		} else if (gDctDest.baseaddr != NULL) {
			kfree(gDctDest.baseaddr);
			gDctDest.baseaddr = NULL;
		}
		copy_from_user(&gDctDest, (void *) arg, sizeof (struct DCT));
		gDctResultBaseAddr = gDctDest.baseaddr;	//record the base addr
			/**********add for bug of M_M*****************************/
		if (gDctUseMC == TRUE) {
			tempaddr =
			    consistent_alloc(GFP_KERNEL | GFP_DMA,
					     gDctDest.datalen, &gDctDmaAddr);
		} else		//not using MC_MC
		{
			tempaddr =
			    kmalloc(gDctDest.datalen, GFP_KERNEL | GFP_DMA);
		}
			/**********end of add for bug of M_M*****************************/
		gDctDest.baseaddr = tempaddr;
		memset(gDctDest.baseaddr, 0, gDctDest.datalen);
		break;
	case DCT_SET_MEMMODE:	//set use ARC_ARC,M_M
			/**********add for bug of M_M*****************************/
		if (arg == DCT_ARC_ARC) {
			if ((gDctUseMC == TRUE) && (gDctDest.baseaddr != NULL))	//Should consistent free the allocated mem here
			{
				consistent_free(gDctDest.baseaddr,
						gDctDest.datalen, gDctDmaAddr);
				gDctDest.baseaddr = NULL;
				gDctDest.datalen = 0;
			}
			gDctUseMC = FALSE;
			MMA_DCTCTRL |= DCT_CLK | arg;
		} else if (arg == DCT_MC_MC) {
			if ((gDctUseMC == FALSE) && (gDctDest.baseaddr != NULL))	//should free the allocated mem here
			{
				kfree(gDctDest.baseaddr);
				gDctDest.baseaddr = NULL;
				gDctDest.datalen = 0;
			}
			gDctUseMC = TRUE;
			MMA_DCTCTRL |= DCT_CLK | arg;
		} else {
			printk("not support this method in DCT *** \n");
		}
			/**********end of add for bug of M_M*****************************/
		break;
	case DCT_CMD_START:
		dct_operation(arg);
		break;

	}
	return 0;
}

/*******************************************************
DCT isr routine
 return : 0 -- success
		  - 1 -- fail
********************************************************/
void
dct_isr(UINT16 irq, void *dev_id, struct pt_regs *regs)
{
	UINT32 i;

	UINT32 reg_value;
	reg_value = MMA_DCTIRQSTAT;
	if (reg_value & DCT_DATA_IN_IRQ)	//intr in ARC_ARC
	{
		MMA_DCTIRQSTAT |= DCT_DATA_IN_IRQ;
		if (gDctTransCount != 0) {
			for (i = 0; i < 32; i++) {
				MMA_DCTFIFO = *gDctSrcAddr;
				gDctSrcAddr++;
			}
		}
	} else if (reg_value & DCT_DATA_OUT_IRQ) {
		MMA_DCTIRQSTAT |= DCT_DATA_OUT_IRQ;
		for (i = 0; i < 32; i++) {
			*gDctDestAddr = MMA_DCTFIFO;
			gDctDestAddr++;
			gDctTransCount--;
		}

		if (gDctTransCount == 0) {
			MMA_DCTIRQSTAT |= 0xf;
			Disable_Dct_Irq(DCT_DATA_IN_IRQ);
			Disable_Dct_Irq(DCT_DATA_OUT_IRQ);
			gDctComplete = TRUE;
			if (gDctWaitFlag == 1)
				wake_up_interruptible(&gDctWaitQueue);

		}
	} else if (reg_value & DCT_COMP_IRQ) {
		MMA_DCTIRQSTAT |= DCT_COMP_IRQ;
		Disable_Dct_Irq(DCT_COMP_IRQ);
		gDctComplete = TRUE;
		if (gDctWaitFlag == 1)
			wake_up_interruptible(&gDctWaitQueue);
	} else {
		printk("other inter \n");
		MMA_DCTIRQSTAT |= 0xf;
	}
}

int
mma_dct_pm_handler(struct pm_dev *dev, pm_request_t rqst, void *data)
{
	switch (rqst) {
	case PM_RESUME:
//                      MMA_DCTCTRL = DCT_CLK;
		break;
	case PM_SUSPEND:
//                      MMA_DCTCTRL = 0x0;
		break;
	default:
		break;
	}
	return 0;
}

/************************************************************************
	Init/Cleanup Module, for module operation
*************************************************************************/
//#ifdef MODULE
MODULE_DESCRIPTION("DCT Module");
//MODULE_PARM(mac_debug, "i");
//MODULE_PARM_DESC(mac_debug,"debug level");
MODULE_LICENSE("GPL");

/************************************************************
 provide some device operation for test
*************************************************************/

struct file_operations dct_fops = {
	read:NULL,
	poll:NULL,
	ioctl:dct_ioctl,
	open:dct_open,
	release:dct_release,
};
/*******************************************************
*MMA init function
*Parameters:None
*Return	
*	0	indicates SUCCESS
* 	-1	indicates FAILURE
* 	
********************************************************/
SINT16 __init
dct_init(void)
{
	int tmp;
	int result;
	/*Initialize its global variable */
	gDctAvail = TRUE;
	/*Request for MAC ISR handler */
	tmp = request_irq(MMA_DCT_IRQ,
			  (void *) dct_isr, MMA_DCT_INTR_MODE, "dct_Mx1", NULL);
	if (tmp < 0)
		return tmp;

	//Register to char device tab
	result = register_chrdev(0, "dct_Mx1", &dct_fops);
	if (result < 0) {
		printk(KERN_WARNING "dct_Mx1: unable to get major \n");
		return result;
	} else
		printk("make node for DCT with 'mknod dct_Mx1 c %d 0'\n",
		       result);

	gDctMajor = result;
//      init_waitqueue_head(&gDctWaitQueue);    

	pm = pm_register(PM_SYS_DEV, PM_SYS_UNKNOWN, mma_dct_pm_handler);

	return 0;
}

/*******************************************************
*MMA cleanup function
*Parameters:None
*Return	
* 	
********************************************************/
void __exit
dct_cleanup(void)
{
	/*Do some cleanup work */
	/*release MAC ISR handler */
	free_irq(MMA_DCT_IRQ, NULL);
	pm_unregister_all(mma_dct_pm_handler);
	unregister_chrdev(gDctMajor, "dct_Mx1");
	printk("this is DCT module release successfully \n");

}

module_init(dct_init);
module_exit(dct_cleanup);

/*
int init_module(void)
{
	return dct_init();
}

void cleanup_module(void)
{
	dct_cleanup();
}
*/

//#endif
