/******************************************************************************

	mx1ads-cam.c
	driver for Motorola MX1ADS on-board CMOS Image Sensor

	Author: MontaVista Software, Inc. <source@mvista.com>
	Copyright (c) 2003 MontaVista Software, Inc.

	Based on: csi2c.c
	Copyright (c) 2001 Motorola Semiconductors HK Ltd

	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
	
	Modifications:
	Nov 2003 - ver 1.1, MontaVista Software, Inc: added i2c search for dev-fs

********************************************************************************/
/*the camera has a lot of HW problems,
and because of that the driver openly supports only:
size: 320*240 fixed,
output format: GRBG raw,
decimation: 1,2,4
brightness: 0-100
access via read function.

the driver is set for 96MHz System PLL freq
*/

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/init.h>

#include <linux/delay.h>
#include <asm/io.h>

#include <asm/dma.h>
#include <asm/arch/dma.h>
#include <asm/mach/dma.h>

#ifdef CONFIG_PM
#include <linux/pm.h>
#endif

#include <linux/i2c.h>
#include <asm/arch/mx1ads-gpio.h>
#include <asm/arch/platform.h>
#include <linux/ioport.h>

#include "mx1ads-cam.h"

//#define DEBUG_MX1IM
#ifdef DEBUG_MX1IM
#define debugprintk(fmt, args...) printk("%s: " fmt, __FUNCTION__ , ## args)
#else
#define debugprintk(fmt, args...)
#endif

//#define MX1IM_USE_LOCK
#ifdef MX1IM_USE_LOCK
#define mx1cam_do_lock()   down(&mx1cam_lock);
#define mx1cam_do_unlock()  up(&mx1cam_lock);
#else
#define mx1cam_do_lock()
#define mx1cam_do_unlock()
#endif

#define mx1_csi_reg_out(r,x) outl(x,IO_ADDRESS((CSI_BASE + r)))
#define mx1_csi_reg_in(r) inl(IO_ADDRESS((CSI_BASE + r)))
#define mx1_dma_chan_reg_out(r,c,x) outl(x,IO_ADDRESS((DMA_CH0_BASE + r + c * DMA_REG_SET_OFS * sizeof(VU32))))

/*image sensor address*/
#define MX1IM_I2C_ADDR (0x66>>1)

static struct mx1cam_pict_struct mx1cam_pict = {
	.x = 0,
	.y = 0,
	.width = MX1CAM_MAX_WIDTH,
	.height = MX1CAM_MAX_HEIGHT,
	.sensitivity = 20,
	.decimation = 1,
};

static struct mx1cam_vframe_struct mx1cam_vframe = {
	.width = MX1CAM_MAX_WIDTH + MX1CAM_VFRAME_WIDTH_DIFF,
	.height = MX1CAM_MAX_HEIGHT + MX1CAM_VFRAME_HEIGHT_DIFF,
};

static struct mx1cam_gain_struct mx1cam_gain = {
.global = 5,.green_gr = 5,.red = 5,.blue = 5,.green_bg = 5,};

static int mx1cam_initstate_csi;
static int mx1cam_initstate_gpio_A_3;
static int mx1cam_initstate_gpio_A_23;
static int mx1cam_initstate_gpio_A_CSI;
static int mx1cam_initstate_gpio_B;
static int mx1cam_initstate_irq;
static int mx1cam_initstate_dev;
static int mx1cam_initstate_i2c;

static struct file *mx1cam_i2c;
static struct i2c_client *mx1cam_client;

static unsigned long *mx1cam_data_buf = 0;
static unsigned long mx1cam_phys_buf_addr;
static unsigned long mx1cam_image_size;

static int mx1cam_inuse = 0;

static int mx1cam_continuous_capture;

static dmach_t mx1cam_dma_chan;

static DECLARE_WAIT_QUEUE_HEAD(dma_wait);
static DECLARE_WAIT_QUEUE_HEAD(intr_wait);
static DECLARE_WAIT_QUEUE_HEAD(sync_wait);
static DECLARE_MUTEX(mx1cam_lock);

#ifdef CONFIG_PM
static struct pm_dev *mx1cam_pmdev;
#endif

static void mx1cam_start_capture(void);
static void mx1cam_stop_capture(void);
static void mx1cam_dma_intr_handler(void);
static void mx1cam_dma_err_handler(int error_type);
static void mx1cam_intr_handler(int irq, void *dev_id, struct pt_regs *regs);

int __init mx1ads_image_init(void);
void __init mx1ads_image_exit(void);
static void mx1cam_i2c_write(char reg, int data);
static int mx1cam_i2c_read(char reg);
static int mx1cam_i2c_init(void);
static void mx1cam_i2c_exit(void);

static int mx1cam_open(struct video_device *dev, int mode);
static void mx1cam_close(struct video_device *dev);
static int mx1cam_ioctl(struct video_device *dev, unsigned int cmd, void *arg);
static long mx1cam_read(struct video_device *dev, char *buf,
			unsigned long count, int flags);
static void mx1cam_set_window_size(struct mx1cam_pict_struct *pict,
				   unsigned int x, unsigned int y,
				   unsigned int width, unsigned int height);
static void mx1cam_set_sensitivity(struct mx1cam_pict_struct *pict,
				   unsigned int sensitivity);
static void mx1cam_set_decimation(struct mx1cam_pict_struct *pict,
				  unsigned int decimation);
static unsigned long int mx1cam_get_image_size(struct mx1cam_pict_struct
					       *pict);

static void mx1cam_set_vframe(struct mx1cam_vframe_struct *vframe,
			      unsigned int width, unsigned int height);
static void mx1cam_set_gains(struct mx1cam_gain_struct *gain,
			     unsigned char global, unsigned char green_gr,
			     unsigned char red, unsigned char blue,
			     unsigned char green_bg);

static void mx1cam_reg_init(void);
static void mx1cam_reg_clear(void);

#ifdef CONFIG_PM
static int mx1cam_pm_callback(struct pm_dev *pmdev,
			      pm_request_t rqst, void *data);
#endif

extern void mx1ads_request_dma_intr(dmach_t channel,
				    callback_t dma_callback,
				    err_callback_t dma_err_callback);
extern void mx1ads_free_dma_intr(dmach_t channel);

static struct video_device mx1cam_device = {
	.owner = THIS_MODULE,
	.name = "MX1ADS Image Sensor",
	.type = VID_TYPE_CAPTURE,
	.open = mx1cam_open,
	.close = mx1cam_close,
	.read = mx1cam_read,
	.ioctl = mx1cam_ioctl,
};

static int
mx1cam_i2c_init(void)
{
	char filename[20];
	int tmp;
	if (mx1cam_initstate_i2c)
		return 0;
	/*find the I2C driver we need */
	for (tmp = 0; tmp < I2C_ADAP_MAX; tmp++) {
#ifdef CONFIG_DEVFS_FS
		sprintf(filename, "/dev/i2c/%d", tmp);
#else
		sprintf(filename, "/dev/i2c-%d", tmp);
#endif

		if (!IS_ERR(mx1cam_i2c = filp_open(filename, O_RDWR, 0))) {
			/*found some driver */
			mx1cam_client =
			    (struct i2c_client *) mx1cam_i2c->private_data;
			if (strlen(mx1cam_client->adapter->name) >= 9) {
				if (!memcmp
				    (mx1cam_client->adapter->name,
				     "DBMX1 I2C", 9))
					break;	/*we found our driver! */
			}
			filp_close(mx1cam_i2c, NULL);
		}
	}

	if (tmp == I2C_ADAP_MAX) {	/*no matching I2C driver found */
		printk(KERN_ERR
		       "IMAGE SENSOR ERROR: cannot find DBMX1 I2C driver\n");
		mx1cam_do_unlock();
		return -EPERM;
	}

	mx1cam_client->addr = MX1IM_I2C_ADDR;
	mx1cam_initstate_i2c = 1;
	return 0;
}

static void
mx1cam_i2c_exit(void)
{
	if (mx1cam_initstate_i2c)
		filp_close(mx1cam_i2c, NULL);
	mx1cam_initstate_i2c = 0;
}

static void
mx1cam_reg_init(void)
{
	unsigned long MCLKDIV;
	if (mx1cam_i2c_init() < 0)
		return;
/*CSI_CTRL_REG1
bit 25: statistical FIFO overrun int enable
bit 24: RX FIFO overrun int enable
bits 23:22: stat FIFO full level
bit 21: stat FIFO full int enable
bit 20:19: RX FIFO full level
bit 18: RX FIFO Full Int En
bit 17: SOF int on rising edge vs falling
bit 16: enable SOF int
bits 12-15: MCLKDIV
SYSCLK freq divider: MCLK=SYSCLK/(MCLKDIV+1)*2
bits 11-10: reserved
bit 9: MCLK enable
bit 8: sync FIFO clear with SOF
bit 7: big endian data
bit 6: clear statistical FIFO
bit 5: clear RX FIFO
bit 4: ignore HCLK
bit 3: invert data
bit 2: invert SOF
bit 1: latch data on rising edge of SOF
bit 0: CSI enable
*/

	MCLKDIV = 0x3000;	/*assuming 96MHz System PLL freq */
	mx1_csi_reg_out(CSI_CTRL_REG1, 0);	/*module reset */
	mx1_csi_reg_out(CSI_CTRL_REG1, 1);	/*module enable */
	mx1_csi_reg_out(CSI_CTRL_REG1, mx1_csi_reg_in(CSI_CTRL_REG1) | 0x200);	/*MCLK enable */
	mx1_csi_reg_out(CSI_CTRL_REG1, mx1_csi_reg_in(CSI_CTRL_REG1) | MCLKDIV);	/*set clock divider */
	mx1_csi_reg_out(CSI_CTRL_REG1, mx1_csi_reg_in(CSI_CTRL_REG1) | 0x00020000);	/*SOF int disabled, SOF = rising edge */

	mx1_csi_reg_out(CSI_CTRL_REG1, mx1_csi_reg_in(CSI_CTRL_REG1) | 0x00000100);	/*sync clear FIFOs */

	/* set FIFO full level */
	mx1_csi_reg_out(CSI_CTRL_REG1, mx1_csi_reg_in(CSI_CTRL_REG1) & ~0x180000);	/*full = 16 words */
	mx1_csi_reg_out(CSI_CTRL_REG1, mx1_csi_reg_in(CSI_CTRL_REG1) | 0x100000);	/*full = 16 words */

	/* toggle INIT pin of sensor to hard reset it */
	mx1_gpio_set_bit(PORT_B, 18, 1);	/* RESET asserted */
	mx1_gpio_set_bit(PORT_B, 18, 0);	/* RESET released */

	/*reset: clear all programmable and non-programmable registers */
	mx1cam_i2c_write(0x0E, 0x03);	/*reset state */
	mx1cam_i2c_write(0x0E, 0x00);	/*normal state */

	/* Power Configuration */
	mx1cam_i2c_write(0x0C, 0x40);	/*enable external resistor, no tristate, no sw stby */

	mx1cam_i2c_write(0x60, 0x00);	/* Internal Timing */
	/*restore settings */
	mx1cam_set_window_size(&mx1cam_pict, mx1cam_pict.x, mx1cam_pict.y,
			  mx1cam_pict.width, mx1cam_pict.height);      /* set WOI */
	mx1cam_set_vframe(&mx1cam_vframe, mx1cam_vframe.width,
			  mx1cam_vframe.height);
	mx1cam_set_sensitivity(&mx1cam_pict, mx1cam_pict.sensitivity);	/*set integration time */
	mx1cam_set_decimation(&mx1cam_pict, mx1cam_pict.decimation);
	mx1cam_set_gains(&mx1cam_gain, mx1cam_gain.global,
			 mx1cam_gain.green_gr, mx1cam_gain.red,
			 mx1cam_gain.blue, mx1cam_gain.green_bg);
	mx1cam_image_size = mx1cam_get_image_size(&mx1cam_pict);

	mx1_csi_reg_out(CSI_STS_REG, 0x10000);	/*clear last SOF irq */
	mx1_csi_reg_out(CSI_CTRL_REG1, mx1_csi_reg_in(CSI_CTRL_REG1) | 0x60);	/*clear FIFOs */
	mx1_csi_reg_out(CSI_CTRL_REG1, mx1_csi_reg_in(CSI_CTRL_REG1) | 0x10000);	/*enable SOF intr */
	mx1cam_continuous_capture = 1;	/*the hardware defaults to continuous capture */
}

static void
mx1cam_reg_clear(void)
{
	disable_dma(mx1cam_dma_chan);
	if (!mx1cam_i2c_init())
		mx1cam_i2c_write(0x0C, 0x03);	/*tristate, sw stby */
	mx1_csi_reg_out(CSI_CTRL_REG1, 0);
}

static int
mx1cam_open(struct video_device *dev, int mode)
{
	debugprintk("*** open ***\n");
	mx1cam_do_lock();
	if (mx1cam_inuse) {
		mx1cam_do_unlock();
		return -EBUSY;
	}

	if (mx1cam_i2c_init() < 0) {
		mx1cam_do_unlock();
		return -EPERM;
	}
	mx1cam_reg_init();
	mx1cam_stop_capture();
	mx1cam_inuse = 1;
	mx1cam_do_unlock();
	return 0;
}

static void
mx1cam_close(struct video_device *dev)
{
	debugprintk("*** close ***\n");
	mx1cam_do_lock();
	if (mx1cam_inuse) {
		mx1cam_inuse = 0;
		mx1cam_reg_clear();
	}
	mx1cam_do_unlock();
}

static long
mx1cam_read(struct video_device *dev, char *buf, unsigned long count, int flags)
{
	debugprintk("*** read ***\n");
	if (count < mx1cam_image_size)
		return -EINVAL;

	mx1cam_do_lock();
	if (!mx1cam_continuous_capture) {
		disable_dma(mx1cam_dma_chan);
		set_dma_addr(mx1cam_dma_chan, mx1cam_phys_buf_addr);
		set_dma_count(mx1cam_dma_chan, mx1cam_image_size);
		enable_dma(mx1cam_dma_chan);	/*reset DMA */
		/*trigger frame output in single-frame mode */
		mx1_gpio_set_bit(PORT_A, 23, 1);
		mx1_gpio_set_bit(PORT_A, 23, 0);

		if (!interruptible_sleep_on_timeout(&dma_wait, HZ)) {
			/*HW bug workaroud: sometimes less number of bytes is routinely received from the camera,
			   what prevents DMA interrupts from happening. Full reset and initialization of the camera
			   seems to be the only solution. */
			mx1cam_reg_clear();
			mx1cam_reg_init();
			mx1cam_stop_capture();
		}

	} else
		interruptible_sleep_on_timeout(&dma_wait, HZ);

	copy_to_user(buf, (char *) mx1cam_data_buf, mx1cam_image_size);
	mx1cam_do_unlock();
	return mx1cam_image_size;
}

static int
mx1cam_ioctl(struct video_device *dev, unsigned int cmd, void *arg)
{
	debugprintk("cmd: 0x%08x, arg: 0x%08x\n", cmd, (int) arg);
	switch (cmd) {
	case VIDIOCGCAP:	/*get device capabilities */
		{
			struct video_capability capab;
			memset(&capab, 0, sizeof (capab));
			strcpy(capab.name, "MX1ADS Image Sensor");
			capab.type = VID_TYPE_CAPTURE;	/*subcapture doesn't work properly on this camera */
			capab.channels = 1;
			capab.audios = 0;
			capab.maxwidth = MX1CAM_MAX_WIDTH;
			capab.maxheight = MX1CAM_MAX_HEIGHT;
			capab.minwidth = MX1CAM_MAX_WIDTH;
			capab.minheight = MX1CAM_MAX_WIDTH;
			return copy_to_user(arg, &capab,
					    sizeof (capab)) ? -EFAULT : 0;
		}

	case VIDIOCGCHAN:	/*get channel description */
		{
			struct video_channel chan;
			if (copy_from_user(&chan, arg, sizeof (chan)))
				return -EFAULT;
			if (chan.channel != 0)
				return -EINVAL;
			chan.flags = 0;
			chan.tuners = 0;
			chan.type = VIDEO_TYPE_CAMERA;
			strcpy(chan.name, "MX1ADS Image Sensor");
			if (copy_to_user(arg, &chan, sizeof (chan)))
				return -EFAULT;
			return 0;
		}

	case VIDIOCGCAPTURE:	/*get window size */
		{
			struct video_capture capt;
			memset(&capt, 0, sizeof (capt));
			capt.y = mx1cam_pict.y;
			capt.height = mx1cam_pict.height;
			capt.x = mx1cam_pict.x;
			capt.width = mx1cam_pict.width;
			capt.decimation = mx1cam_pict.decimation;

			return copy_to_user(arg, &capt,
					    sizeof (capt)) ? -EFAULT : 0;
		}

	case VIDIOCSCAPTURE:	/*set window size */
		{
			struct video_capture capt;
			int tmp_capture = mx1cam_continuous_capture;

			if (copy_from_user(&capt, arg, sizeof (capt)))
				return -EFAULT;

			switch (capt.decimation) {
			case 1:	/* full sampling */
			case 2:	/* divide by 2 */
			case 4:	/* divide by 4 */
				mx1cam_do_lock();
				if (tmp_capture)
					mx1cam_stop_capture();
				mx1cam_set_decimation(&mx1cam_pict,
						      capt.decimation);
				break;
			default:

				return -EINVAL;
			}
			mx1cam_image_size = mx1cam_get_image_size(&mx1cam_pict);
			if (tmp_capture)
				mx1cam_start_capture();
			mx1cam_do_unlock();
			return 0;
		}

	case VIDIOCGWIN:	/*get Window of Interest */
		{
			struct video_window win;
			memset(&win, 0, sizeof (win));
			win.y = mx1cam_pict.y;
			win.height = mx1cam_pict.height;
			win.x = mx1cam_pict.x;
			win.width = mx1cam_pict.width;
			return copy_to_user(arg, &win,
					    sizeof (win)) ? -EFAULT : 0;
		}

	case VIDIOCSWIN:	/*set Window of Interest */
		{
			return 0;
		}

	case VIDIOCGPICT:	/*get sensitivity */
		{
			struct video_picture pict;
			memset(&pict, 0, sizeof (pict));

			pict.brightness = mx1cam_pict.sensitivity;
			pict.palette = VIDEO_PALETTE_RAW;

			return copy_to_user(arg, &pict,
					    sizeof (pict)) ? -EFAULT : 0;
		}

	case VIDIOCSPICT:	/*set sensitivity */
		{
			struct video_picture pict;
			if (copy_from_user(&pict, arg, sizeof (pict)))
				return -EFAULT;

			if (pict.brightness > 100)
				return -EINVAL;
			mx1cam_do_lock();
			mx1cam_set_sensitivity(&mx1cam_pict, pict.brightness);
			mx1cam_do_unlock();
			return 0;
		}

		/*Private IOCTLs, for testing only, use at your own risk */
	case MX1CAM_IOC_CAPTURE:	/*start(1)/stop(0) continuous capture. repeat start will resync DMA */
		{
			int do_capture;
			if (copy_from_user(&do_capture, arg, sizeof (int)))
				return -EFAULT;
			mx1cam_do_lock();
			if (do_capture) {	/*start continuous capture */
				mx1cam_start_capture();
			} else {	/*stop continuous capture */

				mx1cam_stop_capture();
			}
			mx1cam_do_unlock();
			return 0;
		}

	case MX1CAM_IOC_GET_PICT:	/*get picture dimensions, decimation, sensitivity */
		{
			return copy_to_user(arg, &mx1cam_pict,
					    sizeof (mx1cam_pict)) ? -EFAULT : 0;
		}

	case MX1CAM_IOC_SET_PICT:	/*set picture dimensions, decimation, sensitivity */
		{
			struct mx1cam_pict_struct pict;
			int tmp_capture = mx1cam_continuous_capture;

			if (copy_from_user(&pict, arg, sizeof (pict)))
				return -EFAULT;
			if (pict.sensitivity > 100)
				return -EINVAL;
			if ((pict.x > MX1CAM_MAX_WIDTH)
			    || (pict.width > MX1CAM_MAX_WIDTH)
			    || (pict.y > MX1CAM_MAX_HEIGHT)
			    || (pict.height > MX1CAM_MAX_HEIGHT))
				return -EINVAL;
			if (pict.x + pict.width > MX1CAM_MAX_WIDTH)
				return -EINVAL;
			if (pict.y + pict.height > MX1CAM_MAX_HEIGHT)
				return -EINVAL;

			switch (pict.decimation) {
			case 1:	/* full sampling */
			case 2:	/* divide by 2 */
			case 4:	/* divide by 4 */
			case 8:	/* divide by 8 */
				mx1cam_do_lock();
				if (tmp_capture)
					mx1cam_stop_capture();
				mx1cam_set_decimation(&mx1cam_pict,
						      pict.decimation);
				break;
			default:

				return -EINVAL;
			}
			mx1cam_set_sensitivity(&mx1cam_pict, pict.sensitivity);
			mx1cam_set_window_size(&mx1cam_pict, pict.x, pict.y, pict.width, pict.height);	/* set WOI */
			mx1cam_image_size = mx1cam_get_image_size(&mx1cam_pict);
			if (tmp_capture)
				mx1cam_start_capture();
			mx1cam_do_unlock();
			return 0;
		}

	case MX1CAM_IOC_GET_GAIN:	/*get color gains */
		{
			return copy_to_user(arg, &mx1cam_gain,
					    sizeof (mx1cam_gain)) ? -EFAULT : 0;
		}

	case MX1CAM_IOC_SET_GAIN:	/*set color gains */
		{
			struct mx1cam_gain_struct gain;
			if (copy_from_user(&gain, arg, sizeof (gain)))
				return -EFAULT;
			if ((gain.global >MX1CAM_MAX_GAIN)
			    ||(gain.green_gr > MX1CAM_MAX_GAIN)
			    || (gain.red > MX1CAM_MAX_GAIN)
			    || (gain.blue > MX1CAM_MAX_GAIN)
			    || (gain.green_bg > MX1CAM_MAX_GAIN))
				return -EINVAL;

			mx1cam_do_lock();
			mx1cam_set_gains(&mx1cam_gain, gain.global,
					 gain.green_gr, gain.red,
					 gain.blue, gain.green_bg);
			mx1cam_do_unlock();
			return 0;
		}

	case MX1CAM_IOC_GET_VFRAME:	/*get Virtual Frame Size */
		{
			return copy_to_user(arg, &mx1cam_vframe,
					    sizeof (mx1cam_vframe)) ?
			    -EFAULT : 0;
		}

	case MX1CAM_IOC_SET_VFRAME:	/*set Virtual Frame Size */
		{
			struct mx1cam_vframe_struct vframe;
			if (copy_from_user(&vframe, arg, sizeof (vframe)))
				return -EFAULT;
			if ((vframe.width > MX1CAM_MAX_VFRAME_WTH)
			    || (vframe.height > MX1CAM_MAX_VFRAME_HGT))
				return -EINVAL;
			if (vframe.height <
			    mx1cam_pict.height + MX1CAM_VFRAME_HEIGHT_DIFF)
				return -EINVAL;
			if (vframe.width <
			    mx1cam_pict.width + MX1CAM_VFRAME_WIDTH_DIFF)
				return -EINVAL;

			mx1cam_do_lock();
			mx1cam_set_vframe(&mx1cam_vframe, vframe.width,
					  vframe.height);
			mx1cam_do_unlock();
			return 0;
		}

	case MX1CAM_IOC_GET_I2C_REG:	/*get I2C register value */
		{
			struct mx1cam_i2creg_struct i2creg;
			if (copy_from_user(&i2creg, arg, sizeof (i2creg)))
				return -EFAULT;
			mx1cam_do_lock();
			i2creg.val = mx1cam_i2c_read(i2creg.reg);
			mx1cam_do_unlock();
			return copy_to_user(arg, &i2creg,
					    sizeof (i2creg)) ? -EFAULT : 0;
		}

	case MX1CAM_IOC_SET_I2C_REG:	/*set I2C register value */
		{
			struct mx1cam_i2creg_struct i2creg;
			if (copy_from_user(&i2creg, arg, sizeof (i2creg)))
				return -EFAULT;
			mx1cam_do_lock();
			mx1cam_i2c_write(i2creg.reg, i2creg.val);
			mx1cam_do_unlock();
			return 0;
		}

	default:
		return -ENOIOCTLCMD;

	}

}

/*retruns image size in bytes*/
static unsigned long int
mx1cam_get_image_size(struct mx1cam_pict_struct
		      *pict)
{
	switch (pict->decimation) {
	case 2:
		return pict->width * pict->height;
	case 4:
		return (pict->width >> 1) * (pict->height >> 1);
	case 8:
		return (pict->width >> 2) * (pict->height >> 2);
	default:
		return pict->width * pict->height * 4;
	}
}

static void
mx1cam_set_decimation(struct mx1cam_pict_struct *pict, unsigned int decimation)
{
	pict->decimation = decimation;
	switch (decimation) {
	case 2:		/* divide by 2 */
		mx1cam_i2c_write(0x41, 0x25);
		break;
	case 4:		/* divide by 4 */
		mx1cam_i2c_write(0x41, 0x2A);
		break;
	case 8:		/* divide by 8 */
		mx1cam_i2c_write(0x41, 0x2F);
		break;
	default:		/* full subsampling */
		mx1cam_i2c_write(0x41, 0);	/* "cm" bit must be cleared for full-sampling */
	}
}

static void
mx1cam_set_window_size(struct mx1cam_pict_struct *pict,
		       unsigned int x, unsigned int y,
		       unsigned int width, unsigned int height)
{
	pict->x = x;
	pict->y = y;
	pict->width = width;
	pict->height = height;
	x = x * 2 + 49;		/*should be even- camera HW bug workaround */
	y = y * 2 + 12;
	height = height * 2 - 1;	/*should be even- camera HW bug workaround */
	width = width * 2 - 1;	/*should be even- camera HW bug workaround */
	mx1cam_i2c_write(0x45, (y >> 8) & 0x1);	/* WOI row pointer */
	mx1cam_i2c_write(0x46, y & 0xFF);
	mx1cam_i2c_write(0x47, (height >> 8) & 0x1);	/* WOI Depth */
	mx1cam_i2c_write(0x48, height & 0xFF);
	mx1cam_i2c_write(0x49, (x >> 8) & 0x3);	/* WOI column pointer */
	mx1cam_i2c_write(0x4A, x & 0xFF);
	mx1cam_i2c_write(0x4B, (width >> 8) & 0x3);	/* WOI width */
	mx1cam_i2c_write(0x4C, width & 0xFF);
}

static void
mx1cam_set_sensitivity(struct mx1cam_pict_struct *pict,
		       unsigned int sensitivity)
{
	pict->sensitivity = sensitivity;
	sensitivity *= 10;

	/*set integration time */
	mx1cam_i2c_write(0x4E, (sensitivity >> 8) & 0xFF);
	mx1cam_i2c_write(0x4F, sensitivity & 0xFF);
}

static void
mx1cam_start_capture(void)
{
	if (mx1cam_continuous_capture)
		mx1cam_stop_capture();	/*synchronize DMA */
	mx1cam_continuous_capture = 1;
	disable_dma(mx1cam_dma_chan);
	set_dma_addr(mx1cam_dma_chan, mx1cam_phys_buf_addr);
	set_dma_count(mx1cam_dma_chan, mx1cam_image_size);
	enable_dma(mx1cam_dma_chan);	/*reset DMA */
	mx1cam_i2c_write(0x40, 0x35);	/* Continuous capture, continuous streaming */
}

static void
mx1cam_stop_capture(void)
{
	if (mx1cam_continuous_capture) {
		interruptible_sleep_on(&intr_wait);	/*wait util start of frame, exclude boundary situations */
		mx1cam_i2c_write(0x40, 0x75);	/* 1 frame mode, stop continuous output */
		interruptible_sleep_on_timeout(&sync_wait, HZ / 2);	/*wait for all data to dry up */
		mx1cam_continuous_capture = 0;
	}
}

static void
mx1cam_set_vframe(struct mx1cam_vframe_struct *vframe,
		  unsigned int width, unsigned int height)
{
	vframe->height = height;
	vframe->width = width;
	height *= 2;
	width *= 2;
	/*set VF height and width */
	mx1cam_i2c_write(0x50, (height >> 8) & 0x3F);
	mx1cam_i2c_write(0x51, height & 0xFF);
	mx1cam_i2c_write(0x52, (width >> 8) & 0x3F);
	mx1cam_i2c_write(0x53, width & 0xFF);
}

static void
mx1cam_set_gains(struct mx1cam_gain_struct *gain,
		 unsigned char global, unsigned char green_gr,
		 unsigned char red, unsigned char blue, unsigned char green_bg)
{
	gain->global = global;
	gain->green_gr = green_gr;
	gain->red = red;
	gain->blue = blue;
	gain->green_bg = green_bg;

	/*set global gain, range 0x0 .. 0x3F */
	mx1cam_i2c_write(0x10, global);
	/*set color gain, range 0x0 .. 0x3F */
	mx1cam_i2c_write(0x00, green_gr);	/* Green of Green-Red Row */
	mx1cam_i2c_write(0x01, red);	/* Red */
	mx1cam_i2c_write(0x02, blue);	/* Blue */
	mx1cam_i2c_write(0x03, green_bg);	/* Green of Blue-Green Row */
}

/*write to a camera's i2c register*/
static void
mx1cam_i2c_write(char reg, int data)
{
	char buf[2];

	buf[0] = (char) reg;	/*register number */
	buf[1] = (char) data;	/*register data */
	i2c_master_send(mx1cam_client, &buf[0], 2);
}

/*read a camera's i2c register*/
static int
mx1cam_i2c_read(char reg)
{
	struct i2c_msg msg[2];
	char buf;

	buf = reg;

	/*a message structure to write the register number */
	msg[0].addr = mx1cam_client->addr;
	msg[0].flags = mx1cam_client->flags;
	msg[0].len = 1;
	msg[0].buf = &buf;
	/*a message structure to read the register data */
	msg[1].addr = mx1cam_client->addr;
	msg[1].flags = mx1cam_client->flags | I2C_M_RD;
	msg[1].len = 1;
	msg[1].buf = &buf;

	i2c_transfer(mx1cam_client->adapter, &msg[0], 2);
	return ((int) buf);
}

static void
mx1cam_dma_intr_handler(void)
{
	wake_up_interruptible(&dma_wait);
}

static void
mx1cam_dma_err_handler(int error_type)
{
	printk(KERN_ERR
	       "IMAGE SENSOR ERROR: CSI DMA error, type = 0x%08x\n",
	       error_type);
}

static void
mx1cam_intr_handler(int irq, void *dev_id, struct pt_regs *regs)
{
	if (mx1_csi_reg_in(CSI_STS_REG) & 0x10000) {	/* SOF intr ? */
		wake_up_interruptible(&intr_wait);
		mx1_csi_reg_out(CSI_STS_REG, 0x10000);	/* clear SOF intr */
		return;
	}
	printk(KERN_ERR "IMAGE SENSOR ERROR: unknown interrupt\n");
	mx1_csi_reg_out(CSI_STS_REG, mx1_csi_reg_in(CSI_STS_REG));	/* clear all remaining interrupt flags */
}

#ifdef CONFIG_PM
/*power management event handling*/
static int
mx1cam_pm_callback(struct pm_dev *pmdev, pm_request_t rqst, void *data)
{
	char tmp;
	switch (rqst) {
	case PM_SUSPEND:
		mx1cam_do_lock();
		mx1cam_reg_clear();
		mx1cam_do_unlock();
		break;
	case PM_RESUME:
		mx1cam_do_lock();
		if (mx1cam_inuse) {
			tmp = mx1cam_continuous_capture;
			mx1cam_reg_init();
			if (!tmp)
				mx1cam_stop_capture();
			else
				mx1cam_start_capture();
		}
		mx1cam_do_unlock();
		break;
	}
	return 0;
}
#endif

/*initialize*/
int __init
mx1ads_image_init(void)
{
	int tmp;

	mx1cam_dma_chan = -1;
	mx1cam_initstate_dev = 0;
	mx1cam_data_buf = 0;
	mx1cam_initstate_csi = 0;
	mx1cam_initstate_gpio_A_3 = 0;
	mx1cam_initstate_gpio_A_23 = 0;
	mx1cam_initstate_gpio_A_CSI = 0;
	mx1cam_initstate_gpio_B = 0;
	mx1cam_initstate_irq = 0;
	mx1cam_initstate_i2c = 0;

	printk(KERN_INFO "MX1ADS CMOS Image Sensor Driver Ver. 1.1\n");
	debugprintk("%s %s\n", __TIME__, __DATE__);

	tmp = (int) request_region(IO_ADDRESS(CSI_BASE), 0x14, "mx1ads_image");
	if (!tmp) {
		printk(KERN_ERR "IMAGE SENSOR ERROR: CSI is already in use\n");
		mx1ads_image_exit();
		return -1;
	}
	mx1cam_initstate_csi = 1;

	tmp = video_register_device(&mx1cam_device, VFL_TYPE_GRABBER, -1);
	if (tmp < 0) {
		printk(KERN_ERR
		       "IMAGE SENSOR ERROR: unable to register video device\n");
		mx1ads_image_exit();
		return tmp;
	}
	mx1cam_initstate_dev = 1;

/*
NVDD2 N13 CSI_PIXCLK I PA14 69K PA14
NVDD2 M13 CSI_HSYNC I PA13 69K PA13
NVDD2 M14 CSI_VSYNC I PA12 69K PA12
NVDD2 N14 CSI_D7 I PA11 69K PA11
NVDD2 M15 CSI_D6 I PA10 69K PA10
NVDD2 M16 CSI_D5 I PA9 69K PA9
NVDD2 M12 CSI_D4 I PA8 69K PA8
NVDD2 L16 CSI_D3 I PA7 69K PA7
NVDD2 L15 CSI_D2 I PA6 69K PA6
NVDD2 L14 CSI_D1 I PA5 69K PA5
NVDD2 L13 CSI_D0 I PA4 69K PA4
NVDD2 L12 CSI_MCLK O PA3 69K PA3
*/

	tmp = mx1_register_gpios(PORT_A, 0x7FF0, PRIMARY | INPUT | TRISTATE);
	if (tmp < 0) {
		printk(KERN_ERR
		       "IMAGE SENSOR ERROR: PORTA mask 0x7FF0 is already in use\n");
		mx1ads_image_exit();
		return tmp;
	}
	mx1cam_initstate_gpio_A_CSI = 1;

	tmp = mx1_register_gpio(PORT_A, 3, PRIMARY | OUTPUT | TRISTATE);
	if (tmp < 0) {
		printk(KERN_ERR
		       "IMAGE SENSOR ERROR: PORTA pin 3 is already in use\n");
		mx1ads_image_exit();
		return tmp;
	}
	mx1cam_initstate_gpio_A_3 = 1;

	/*enable Sync Signal (CSI Daughter Board -> JP1)
	   NVDD1 T4 CS5 O PA23 69K PA23
	   register as data register output GPIO PORT A pin 23 initialize to 0 */
	tmp =
	    mx1_register_gpio(PORT_A, 23,
			      GPIO | OUTPUT | OCR_DATA | TRISTATE |
			      INIT_DATA_0);
	if (tmp < 0) {
		printk(KERN_ERR
		       "IMAGE SENSOR ERROR: PORTA pin 23 is already in use\n");
		mx1ads_image_exit();
		return tmp;
	}
	mx1cam_initstate_gpio_A_23 = 1;

	/*enable Reset Signal (CSI Daughter Board -> JP2)
	   manual: NVDD4 F6 SIM_RST O SSI_TXFS I/O PB18 69K PB18
	   register as data register output GPIO PORT B pin 18 initialize to 0 */
	tmp =
	    mx1_register_gpio(PORT_B, 18,
			      GPIO | OUTPUT | OCR_DATA | TRISTATE |
			      INIT_DATA_0);
	if (tmp < 0) {
		printk(KERN_ERR
		       "IMAGE SENSOR ERROR: PORTB mask 0xC0000 is already in use\n");
		mx1ads_image_exit();
		return tmp;
	}
	mx1cam_initstate_gpio_B = 1;

	/*interrupt request */
	tmp =
	    request_irq(CSI_INT, mx1cam_intr_handler, SA_INTERRUPT,
			"Image Sensor", "camirq");
	if (tmp) {
		printk(KERN_ERR "IMAGE SENSOR ERROR: could not get IRQ\n");
		mx1ads_image_exit();
		return tmp;
	}
	mx1cam_initstate_irq = 1;

	/*allocate buffers in memory */
	mx1cam_data_buf = (unsigned long *) __get_free_pages(GFP_ATOMIC | GFP_DMA, 7);	/* 128 (1<<8) pages * 4k = 512KB */
	if (!mx1cam_data_buf) {
		printk(KERN_ERR
		       "IMAGE SENSOR ERROR: could not allocate memory\n");
		mx1ads_image_exit();
		return -1;
	}

	mx1cam_phys_buf_addr = virt_to_phys((void *) mx1cam_data_buf);

	debugprintk("Buffer start: 0x%08x, DMA addr: 0x%08x\n",
		    (int) mx1cam_data_buf, (int) mx1cam_phys_buf_addr);

	/* request DMA channel for RxFIFO data */
	for (mx1cam_dma_chan = 0; mx1cam_dma_chan < MAX_DMA_CHANNELS;
	     mx1cam_dma_chan++) {
		if (!request_dma(mx1cam_dma_chan, "CMOS Sensor"))
			break;
	}

	if (mx1cam_dma_chan == MAX_DMA_CHANNELS) {
		printk(KERN_ERR "IMAGE SENSOR ERROR: could not allocate DMA\n");
		mx1cam_dma_chan = -1;
		mx1ads_image_exit();
		return -1;
	}

	mx1_dma_chan_reg_out(DMA_SAR, mx1cam_dma_chan, 0x00224010);	/* CSI RxFIFO register */
	mx1_dma_chan_reg_out(DMA_RSSR, mx1cam_dma_chan, 7);	/* CSI data */
	mx1_dma_chan_reg_out(DMA_BLR, mx1cam_dma_chan, 64);	/* burst length : 16 words = 64 bytes */
	mx1_dma_chan_reg_out(DMA_CCR, mx1cam_dma_chan, (0x808 | 0x4));	/*repeat added */
	mx1_dma_chan_reg_out(DMA_RTOR, mx1cam_dma_chan, 0);	/* burst timeout, not used */
	mx1_dma_chan_reg_out(DMA_BUCR, mx1cam_dma_chan, 0);	/* bus utilization, not used */
	set_dma_mode(mx1cam_dma_chan, DMA_MODE_READ);	/*this is for set_dma_addr() */
	mx1ads_request_dma_intr(mx1cam_dma_chan,
				(callback_t) mx1cam_dma_intr_handler,
				(err_callback_t) mx1cam_dma_err_handler);
#ifdef CONFIG_PM
	mx1cam_pmdev =
	    pm_register(PM_UNKNOWN_DEV, PM_SYS_UNKNOWN, mx1cam_pm_callback);

	if (!mx1cam_pmdev)
		printk(KERN_WARNING
		       "IMAGE SENSOR WARNING: failed to init power management\n");
#endif

	return 0;
}

/*deinitialize*/
void __init
mx1ads_image_exit(void)
{
	if (mx1cam_dma_chan != -1) {
#ifdef CONFIG_PM
		pm_unregister(mx1cam_pmdev);
#endif
		mx1cam_reg_clear();
		mx1cam_i2c_exit();
		mx1ads_free_dma_intr(mx1cam_dma_chan);
		free_dma(mx1cam_dma_chan);
	}

	if (mx1cam_data_buf)
		free_pages((unsigned long) mx1cam_data_buf, 7);	/* 512K */

	if (mx1cam_initstate_irq)
		free_irq(CSI_INT, "camirq");
	if (mx1cam_initstate_gpio_B)
		mx1_unregister_gpio(PORT_B, 18);
	if (mx1cam_initstate_gpio_A_23)
		mx1_unregister_gpio(PORT_A, 23);
	if (mx1cam_initstate_gpio_A_3)
		mx1_unregister_gpio(PORT_A, 3);
	if (mx1cam_initstate_gpio_A_CSI)
		mx1_unregister_gpios(PORT_A, 0x7FF0);
	if (mx1cam_initstate_dev)
		video_unregister_device(&mx1cam_device);
	if (mx1cam_initstate_csi)
		release_region(IO_ADDRESS(CSI_BASE), 0x14);
}

MODULE_AUTHOR("MontaVista Software, Inc. <source@mvista.com>");
MODULE_DESCRIPTION("MX1ADS Image Sensor Driver");
MODULE_LICENSE("GPL");

module_init(mx1ads_image_init);
module_exit(mx1ads_image_exit);
