/*
 * drivers/net/mb86977/camelot.c
 *
 * Fujitsu MB86977 driver
 *
 * Author: <source@mvista.com>
 *
 * Copyright (c) 2002-2003 by Fujitsu LSI Solution Ltd..  All Rights Reserved.
 *
 * 2003 (c) MontaVista Software, Inc. This file is licensed under
 * the terms of the GNU General Public License version 2. This program
 * is licensed "as is" without any warranty of any kind, whether express
 * or implied.
 *
 */

#include <linux/config.h>
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/skbuff.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/spinlock.h>
#include <linux/init.h>
#include <asm/uaccess.h>

#include "camelot_var.h"
#include "camelot_reg.h"
#include "camelot_func.h"

extern struct common_softc came_var;
extern NAT_IPF_TBL  natipf_table[1];
static int txbfull;

extern struct cm_softc *cm_unit(int index);
extern u_char came_mac[6];
extern int came_dmz_mode;
extern u_char came_link[4];

//#define SHOW_CONFIG_AT_INIT

extern void disable_MAC(struct camelot_softc *sc);

void
camelot_reset(sc)
	struct camelot_softc *sc;
{
	unsigned long flags;

	printk(KERN_NOTICE "camelot: reset\n");

	local_irq_save(flags);
	disable_MAC(sc);

	init_camelot(sc);

	local_irq_restore(flags);
}


/*
 * camelot_read -- pull packet off interface and forward to appropriate
 * protocol handler
 */
void
camelot_read(struct cm_softc *cm, int stp, int blen, int mark)
{
	struct camelot_softc *sc = &cm->camelot;
	int addr;
	int wlen, rwlen; 
	int rdp;
	struct sk_buff *skb;
	int status;
	u_char *sdata;

#define ETHER_CRC_LEN	4
#define ETHER_HDR_LEN	14
#define ETHERMTU	1536
#define ETHERMIN	64

	/* get input data length */
	blen  = blen - ETHER_CRC_LEN;
	if (blen <= ETHER_HDR_LEN ||
	    blen > ETHERMTU + ETHER_HDR_LEN)
	{
		printk(KERN_ERR "cm%d: bad packet length received, %d bytes\n",
		       cm->unit, blen);
		cm->stats.rx_errors++;
		return;
	}

	if (blen % 4)
		wlen = blen/4 + 1;
	else
		wlen = blen/4;

	/* get an skb for packet */
	skb = alloc_skb(blen+4/*2*/, GFP_ATOMIC);
	if (skb == 0) {
		cm->stats.rx_dropped++;
		return;
	}

	skb_reserve(skb, 2);

	sdata = skb_put(skb, blen);

	/* copy packet */
	rdp  = (stp + 1) % RXBUFSIZE;
	addr = (int)(CAME_BANK1 + (rdp * 4));
	if (rdp + wlen > RXBUFSIZE){
		rwlen = rdp + wlen - RXBUFSIZE;
		(*sc->sc_copyfrombuf)(sc, sdata/*skb->data*/, addr, (wlen - rwlen));
		addr = (int)(CAME_BANK1 + 0);
		(*sc->sc_copyfrombuf)(sc, (sdata/*skb->data*/ + 4*(wlen - rwlen)),
				      addr, rwlen);
	}
	else{
		(*sc->sc_copyfrombuf)(sc, sdata/*skb->data*/, addr, wlen);
	}

	/* set up skb */
	skb->len = blen;
	skb->dev = &cm->net;
	skb->protocol = eth_type_trans(skb, &cm->net);

	cm->stats.rx_packets++;
	cm->stats.rx_bytes += blen;

	/* pass the packet up */
	status = netif_rx(skb);

	if (status != NET_RX_SUCCESS)
		printk(KERN_ERR "camelot: bad rx status 0x%x", status);
}

int camelot_put(struct cm_softc *cm, struct sk_buff *skb)
{
	struct camelot_softc *sc = &cm->camelot;
	int retval = NET_XMIT_SUCCESS;
	int length = skb->len;
	u_int strp, clrp;
	int wemp = 0;
	int wlen, rwlen;
	u_int desc_data = 0;
	u_int ds_add;

#if 0
	/* first check the tramsmited packet total length */
	if (length > ETHERMTU + ETHER_HDR_LEN) {
		printk("camelot: transmit packet too large (>1514 bytes)\n");
	}

	if (length < (ETHERMIN + ETHER_HDR_LEN)) {
		/*printk("Send packet runt %d\n", pblen);*/
		length = 60;
	}  
#else
	/* first check the tramsmited packet total length */
	if (length > ETHERMTU) {
		printk(KERN_ERR
		       "camelot: transmit packet too large (>1514 bytes)\n");
	}

	if (length < ETHERMIN) {
		/*printk("Send packet runt %d\n", pblen);*/
		length = ETHERMIN;
	}  
#endif

	if (length % 4)
		wlen = (length / 4) + 1;
	else
		wlen = length / 4;

	/* Check transmiter buffer capacity */
	strp = (*sc->sc_read)(sc, HOST_TXCPUWRSTP_OFFSET) & 0x03ff;
	clrp = (*sc->sc_read)(sc, HOST_TXCMBCLR_OFFSET) & 0x03ff;

	if (strp == clrp) {
		if (txbfull)
			wemp = 0;
		else
			wemp = TXBUFSIZE;
	} else
		if (strp < clrp)
			wemp = clrp - strp;
		else
			if (strp > clrp)
				wemp = TXBUFSIZE - (strp - clrp);

	if ((wlen + 1) > wemp) {
		printk(KERN_ERR "camelot: waiting TX, not enough space!\n");
		cm->stats.tx_dropped++;
		dev_kfree_skb_any(skb);
		return NET_XMIT_DROP;
	}

	if (0) printk("start strp %x clrp %x, unit %d\n",
		      strp, clrp, cm->unit);

	/* fill up the Currrent Tranmsmit Descriptor and buffer */
	switch (cm->unit) {
	case 0:
		desc_data = 0x03000000 | (INTF_WAN << 16);
		break;
	case 1:
		desc_data = 0x02000000;
		if (came_dmz_mode)
			desc_data |= ((INTF_LAN0 | INTF_LAN1) << 16);
		else
			desc_data |= ((INTF_LAN0 | INTF_LAN1 |INTF_DMZ) << 16);
		break;

	case 2:
        	desc_data = 0x03000000 | (INTF_DMZ << 16);
		break;
	default:
		desc_data = 0;
		printk(KERN_ERR
		       "camelot: TX error; unknown ethernet unit %d\n",
		       cm->unit);
		dev_kfree_skb_any(skb);
		return -ENODEV;
	}

	desc_data |= length & 0x0000ffff;

	(*sc->sc_write)(sc, CAME_BANK0 + (strp * 4), desc_data);
	strp = (strp + 1) % TXBUFSIZE;

	memcpy(&skb->data[6], came_mac, 6);

	ds_add = (u_int32_t)(CAME_BANK0 + (strp * 4));
	if (strp + wlen > TXBUFSIZE){
		rwlen = strp + wlen - TXBUFSIZE;
		(*sc->sc_copytobuf)(sc, skb->data, ds_add, (wlen - rwlen));
		ds_add = (int)(CAME_BANK0 + 0);
		(*sc->sc_copytobuf)(sc, (skb->data + 4*(wlen - rwlen)), ds_add, rwlen);
	} else {
		(*sc->sc_copytobuf)(sc, skb->data, ds_add, wlen);
	}

	dev_kfree_skb_any(skb);

	strp = (strp + wlen) % TXBUFSIZE;
	if ((wlen + 1)== wemp)
		txbfull = 1;

	(*sc->sc_write)(sc, HOST_TXCPUWRSTP_OFFSET, strp);

	if (0) printk("fini strp %x clrp %x desc %x, unit %d\n",
		      strp, clrp, desc_data, cm->unit);

	/* kick transmitter */
	(*sc->sc_write)(sc, HOST_TXSTEN_OFFSET, 0x1);

	cm->stats.tx_packets++;
	cm->stats.tx_bytes += length;

	return retval;
}

/*
 * Receive interrupt routine
 */
void
camelot_rint(struct camelot_softc *sc)
{
	int rxpcknum, reread;
	int rendp, stp;
	int blen;
	u_int rx_stat, rx_port;

	rxpcknum = (*sc->sc_read)(sc, HOST_RXPACKNUM_OFFSET) & 0x3f;

	/* errata - read until zero */
	reread = (*sc->sc_read)(sc, HOST_RXPACKNUM_OFFSET) & 0x3f;
	while (reread != 0) {
		rxpcknum = reread;
		reread = (*sc->sc_read)(sc, HOST_RXPACKNUM_OFFSET) & 0x3f;
	}

	if (rxpcknum == 0) {
		printk(KERN_ERR "camelot: receive packet number = %d\n",
		       rxpcknum);
		return;
	}

	while (rxpcknum) {
		rendp = (*sc->sc_read)(sc, HOST_RXCPURDEND_OFFSET) & 0x03ff;
		stp   = (rendp + 1) % RXBUFSIZE;

		rx_stat =
			(*sc->sc_read)(sc, CAME_BANK1+(stp*4)) & 0xffffffff;

		blen = rx_stat & 0x000007ff;
		rx_port = (rx_stat >> 16) & 0x00ff;
		if (0) printk("rx_port %x, blen %x\n", rx_port, blen);

		if (rx_port & INTF_LAN0) {
			camelot_read(cm_unit(1), stp, blen, 0); 
		} else if (rx_port & INTF_LAN1) {
			camelot_read(cm_unit(1), stp, blen, 1); 
		} else if (rx_port & (INTF_DMZ | INTF_DMZ_H)) {
			if (came_dmz_mode) {
				camelot_read(cm_unit(2), stp, blen, 2);
			} else {
				camelot_read(cm_unit(1), stp, blen, 2);
			}
		} else if (rx_port & (INTF_WAN | INTF_WAN_H)){
			camelot_read(cm_unit(0), stp, blen, 3); 
		} else {
			printk(KERN_ERR "camelot: RX error, bad port %08x\n",
			       rx_port); 
#if 0
			print_rcv_pkt(sc, stp, blen);
#endif
			return;
		}

		if (0) printk( "RX_Packet %d %x\n", blen, rx_port);

		if (blen % 4)
			stp = stp + blen/4 + 1 ;
		else
			stp = stp + blen/4 ;

		stp = stp % RXBUFSIZE;

		(*sc->sc_write)(sc, HOST_RXCPURDEND_OFFSET, stp);
		rxpcknum--;

		if (0) printk("rdp %x %d %x %x\n",
			      rendp, blen, rx_port, stp);
	}
}

void
camelot_tint(struct camelot_softc *sc)
{
	(*sc->sc_write)(sc, HOST_TXSTEN_OFFSET, 0x0);
}

static u_int ttl0_log[ENTRY_NUM_TTL0LOG];
static u_int flt_log_out[ENTRY_NUM_LOG][16];
static u_int flt_log_in[ENTRY_NUM_LOG][16];
static u_int rstfin_log[ENTRY_NUM_RSTFINLOG];

static void
camelot_intr_ttl0(struct cm_softc *cm)
{
	struct camelot_softc *sc = &cm->camelot;
	int i;

	printk(KERN_ERR "camelot: TTL0 error in FEF\n" );

	read_TTL0LOG(sc, ttl0_log);
	for (i = 0; i < ENTRY_NUM_TTL0LOG; i++) {
		printk(KERN_DEBUG "ttl0_log[%x] = %x\n", i, ttl0_log[i]);
	}

	cm->stats.rx_errors++;
	/*camelot_reset(sc);*/
}

static void
camelot_intr_log_oflo(struct cm_softc *cm)
{
	struct camelot_softc *sc = &cm->camelot;
	int i, j, data, status;

	printk(KERN_ERR "camelot: filter log overflow\n");

	read_FLTLOG(sc, flt_log_in, flt_log_out, &status);

	printk(KERN_DEBUG " Status %08x\n", status);

	printk(KERN_DEBUG  " Out-Side Filter  \n");
	for(i=0; i < ENTRY_NUM_LOG ; i++){
		printk(KERN_DEBUG  " ENTRY NUMBER = %x \n", i);
		for(j=0; j < 16 ; j++){
			data = flt_log_out[i][j] ;
			printk(KERN_DEBUG  " ID = %x   %x \n", j, data);
		}
	}
	printk(KERN_DEBUG  " In-Side Filter  \n");
	for(i=0; i < ENTRY_NUM_LOG ; i++){
		printk(KERN_DEBUG  " ENTRY NUMBER = %x \n", i);
		for(j=0; j < 16 ; j++){
			data = flt_log_in[i][j] ;
			printk(KERN_DEBUG  " ID = %x   %x \n", j, data);
		}
	}
}

static void
camelot_intr_rst_fin(struct cm_softc *cm)
{
	struct camelot_softc *sc = &cm->camelot;
#if 0
	u_long ip_src, ip_dst, ip_nat;
	u_short port_src, port_dst, port_nat;
	int i, j, data, id;
#endif

	/*
	 * currently we ignore the rst/fin interrupt and instead
	 * rely on the kernel connection tracking code to time out
	 * entries
	 */
	if (0) printk("camelot: RST/FIN interrupt\n");
	read_RFLOG(sc, rstfin_log);

#if 0
	for (i = 0; i < ENTRY_NUM_RSTFINLOG ; i++) {

		data = rstfin_log[i];
		if (data == 0x0)
			continue;

		for (j = 0; j < 32; j++) {
			if ((data & (1 << j)) == 0)
				continue;

			id = j + 32*i;
			ip_src = (u_long)read_NATIPFTBL(sc, id, 0);
			ip_dst = (u_long)read_NATIPFTBL(sc, id, 1);
			ip_nat = (u_long)read_NATIPFTBL(sc, id, 2);
			port_src = (u_short)read_NATIPFTBL(sc, id, 12);
			port_dst = (u_short)read_NATIPFTBL(sc, id, 13);
			port_nat = (u_short)read_NATIPFTBL(sc, id, 14);
			del_flt_cm(sc, ip_nat, ip_dst, port_nat, port_dst);
		}
	}
#endif
}

void
camelot_intr(int irq, void *dev_id, struct pt_regs *regs)
{
	struct cm_softc *cm = (struct cm_softc *)dev_id;
	struct camelot_softc *sc = &cm->camelot;
	int  status[4];
	int  i, data;
	int  mask, hstatus;

	if (0) printk("camelot_intr(irq=%d,dev_id=%p)\n", irq, dev_id);

	(*sc->sc_write)(sc, HOST_INTEN_OFFSET, 0x00000000);

	hstatus = (*sc->sc_read)(sc, HOST_STATUS_OFFSET);
	if (0) printk("hstatus %x\n", hstatus);
 
	/* TTL0 Error */
	if (hstatus & 0x10000) {
		camelot_intr_ttl0(cm);
	}

	/* SMI Interrupt Status */
	if (hstatus & 0x8000) {
		data = read_SMIST(sc);
		if (0) printk("camelot: SMI Interrupt Status = %x\n", data);

		if (data & 0x00008000)
			came_link[3] = 1;
		if (data & 0x00000800)
			came_link[2] = 1;
		if (data & 0x00000080)
			came_link[1] = 1;
		if (data & 0x00000008)
			came_link[0] = 1;

		if (data & 0x00008888){
			if (0) printk("SMI_OK!\n");
		}
	}

	/* Out-Side Filter Counter Over Flow Interrupt Status */
	if (hstatus & 0x4000) {
		printk(KERN_ERR "camelot: Out Filter Counter = 255 :");
		status[0] = read_FLCNTST(sc, 1, 1);
		status[1] = read_FLCNTST(sc, 1, 2);
               
		/* Part 1 Read */
		mask = 0x01;
		for(i=0; i < 32; i++){
			if (status[0] & mask) {
				data = read_FLCNTV(sc, 1, i);
				printk(KERN_DEBUG "Entry %d %x \n", i, data);
			}
			mask <<= 1;
		}
               
		/* Part 2 Read */
		mask = 0x01;
		for(i=0; i < 32; i++){
			if (status[1] & mask) {
				data = read_FLCNTV(sc, 1, (i+32));
				printk(KERN_DEBUG "Entry %d %x \n",
				       (i+32), data);
			}
			mask <<= 1;
		}
	}

	/* In-Side Filter Counter Over Flow Interrupt Status */
	if (hstatus & 0x2000) {
		printk(KERN_ERR "camelot: In Filter Counter = 255:");
		status[0] = read_FLCNTST(sc, 0, 1);
		status[1] = read_FLCNTST(sc, 0, 2);
               
		/* Part 1 Read */
		mask = 0x01 ;
		for(i=0; i < 32; i++){
			if (status[0] & mask) {
				data = read_FLCNTV(sc, 0, i);
				printk(KERN_DEBUG "Entry %d %x \n", i, data);
			}
			mask <<= 1;
		}
               
		/* Part 2 Read */
		mask = 0x01;
		for(i=0; i < 32; i++){
			if (status[1] & mask) {
				data = read_FLCNTV(sc, 0, (i+32));
				printk(KERN_DEBUG "Entry %d %x \n",
				       (i+32), data);
			}
			mask <<= 1;
		}
	}

	/* MAC Port Status   */
	if (hstatus & 0x1000) {
		printk(KERN_INFO "camelot: MAC Port 3 Interrupt Status %x\n",
		       read_MACST(sc, 3));
		/*printk( " SMI STatus %x \n", read_SMIST(sc));*/
	}

	if (hstatus & 0x800) {
		printk(KERN_INFO "camelot: MAC Port 2 Interrupt Status %x\n",
		       read_MACST(sc, 2));
		/*printk( " SMI STatus %x \n", read_SMIST(sc));*/
	}

	if (hstatus & 0x400) {
		printk(KERN_INFO "camelot: MAC Port 1 Interrupt Status %x\n",
		       read_MACST(sc, 1));
		/*printk( " SMI STatus %x \n", read_SMIST(sc));*/
	}

	if (hstatus & 0x200) {
		printk(KERN_INFO "camelot: MAC Port 0 Interrupt Status %x\n",
		       read_MACST(sc, 0));
		/*printk( " SMI STatus %x \n", read_SMIST(sc));*/
	}

	/* Filter LOG Over Flow   */
	if (hstatus & 0x100) {
		camelot_intr_log_oflo(cm);
	}

	/*  FIN/RST Catch  */
	if (hstatus & 0x80) {
		camelot_intr_rst_fin(cm);
	}

	/* Switch Block Flow Control   */
	if (hstatus & 0x40) {
		printk(KERN_ERR
		       "camelot: Switch Block Flow Control interrupt\n");
	}

	/* Switch Block PRAM Full   */
	if (hstatus & 0x20) {
		printk(KERN_ERR "camelot: Switch Block PRAM Full Interrupt\n");
	}

	/* Host Deadlock   */
	if (hstatus & 0x10) {
		printk(KERN_ERR "camelot: Host Deadlock Interrupt!!\n");
		cm->stats.rx_errors++;
		camelot_reset(sc);
	}

	/* TX/RX Overrun */
	if (hstatus & 0x8) {
		printk(KERN_ERR "camelot: TX overrun!!\n");
		cm->stats.rx_errors++;
		sc->sc_havecarrier = 0;
		camelot_reset(sc);
	}

	if (hstatus & 0x4) {
		printk(KERN_ERR "camelot: RX Underrun!!\n");
		cm->stats.rx_errors++;
		camelot_reset(sc);
	}

	/* RX/TX */
	if (hstatus & 0x2) {
		camelot_rint(sc); 
	}

	if (hstatus & 0x1) {
		camelot_tint(sc); 
	}

	(*sc->sc_write)(sc, HOST_INTEN_OFFSET, HOST_INTEN);
}

void
camelot_config(struct camelot_softc *sc)
{
	init_camelot(sc);

#ifdef SHOW_CONFIG_AT_INIT
	read_dump_L2ALL(sc);
	read_dump_NTALL(sc);
	read_dump_FLTALL(sc, 0);
	read_dump_FLTALL(sc, 2);
	read_dump_QTALL(sc, 0);
	read_dump_QTALL(sc, 1);
#endif
}

