/*
 * linux/drivers/net/mb86977/camelot_ioctl.c
 *
 * ioctl interface to camelot network driver
 * 
 * Author: Brad Parker <brad@heeltoe.com>
 *
 * 2003 (c) MontaVista Software, Inc. This file is licensed under
 * the terms of the GNU General Public License version 2. This program
 * is licensed "as is" without any warranty of any kind, whether express
 * or implied.
 */

#include <linux/config.h>
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/skbuff.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/in.h>
#include <linux/ip.h>

#include <asm/uaccess.h>

#include "camelot_var.h"
#include "camelot_func.h"
#include "camelot_ioctl.h"
#include "camelot_defs.h"
#include "camelot_reg.h"
#include "camelot_bits.h"

extern struct common_softc came_var;
extern int came_dmz_mode;
static unsigned long filter_control_value;

/*
 * NOTE:
 * if we kept shadow copies of the nat/forwarding table and the filter table
 * we could speed up the delete quite a bit.
 */

static u_int32_t global_match_log[ENTRY_NUM_NATMTHLOG];

/* reset global match log bit */
void
cm_reset_match_log(int index)
{
	int i, j;

	i = index / 32;
	j = index % 32;

	global_match_log[i] &= ~(1 << j);
}

/* keep copy of match log in ram, because h/w version is reset-on-read */
int
cm_get_match_log(struct camelot_softc *sc, u_int32_t *match_log)
{
	int i;
	u_int32_t ml[ENTRY_NUM_NATMTHLOG];

	/*
	 * read current h/w match log
	 * (which resets the bits in h/w)
	 */
	read_NMTLOG(sc, ml);

	/* merge with our persistant copy */
	for (i = 0; i < ENTRY_NUM_NATMTHLOG; i++)
		global_match_log[i] |= ml[i];

	/* return merged version */
	for (i = 0; i < ENTRY_NUM_NATMTHLOG; i++)
		match_log[i] = global_match_log[i];

	return 0;
}

#define NEED_FIX

#ifdef NEED_FIX
/* fix for chip errata */
#define	NATIPF_TABLE_SIZE	(ENTRY_NUM_NATIPF/2)

int
map_natipf_index(int index)
{
	int zone, offset;

	zone = index / 8;
	offset = index % 8;

	return zone*16 + offset;
}

int
map_natipf_pair_index(int index)
{
	return 15 - map_natipf_index(index);
}
#else
/* no fix */
#define	NATIPF_TABLE_SIZE		(ENTRY_NUM_NATIPF/2)
#define map_natipf_index(index)		(index)
#define map_natipf_pair_index(index)	(0)
#endif

int
cm_iotcl_add_nat_entry(struct camelot_softc *sc, NAT_IPF_TBL *nit)
{
	int i, index;

	if (0) printk("cm_ioctl_add_nat_entry()\n");

	for (i = 0; i < NATIPF_TABLE_SIZE; i++) {
		/* map index into real index */
		index = map_natipf_index(i);

		/* read item 22, looking for zero entries */
		if (read_NATIPFTBL(sc, index, 22) == 0x0)
			break;
	}

	if (i == NATIPF_TABLE_SIZE) {
		printk(KERN_ERR "camelot: Nat ipf entry overflow!!\n");
		return -EBUSY;
	}

	cm_reset_match_log(i);

	set_NATIPFTBL(sc, index, nit);

#if 1
	read_dump_NTIPFTBL_entry(sc, index);
#endif

	/* handle chip errata */
	if ((index = map_natipf_pair_index(i))) {
		set_empty_NATIPFTBL(sc, index);
	}

	return 0;
}

/*
 * find an entry in the NAT/IP forwarding table and delete it
 */
int
cm_ioctl_del_nat_entry(struct camelot_softc *sc, NAT_IPF_TBL *nit)
{
	int i, index;

	if (0) printk("cm_ioctl_del_nat_entry()\n");

	for (i = 0; i < NATIPF_TABLE_SIZE; i++) {
		u_int32_t int_ip, ext_ip;
		u_short int_port, ext_port;

		/* map index into real index */
		index = map_natipf_index(i);

		/* look for active entries */
		if (read_NATIPFTBL(sc, index, 22) == 0x0)
			continue;

		/* read entry */
		int_ip = read_NATIPFTBL(sc, index, 0);
		ext_ip = read_NATIPFTBL(sc, index, 1);
		int_port = read_NATIPFTBL(sc, index, 12);
		ext_port = read_NATIPFTBL(sc, index, 13);

		if (int_ip == nit->ipv4addr_inter &&
		    ext_ip == nit->ipv4addr_exter &&
		    int_port == nit->portnum_inter &&
		    ext_port == nit->portnum_exter)
		{
#if 0
			printk("delete #%d\n", i);
			read_dump_NTIPFTBL_entry(sc, index);
#endif

			setnat_root(sc, index, 22, 0x00000000);
			came_var.tbl_match[i] = 0;
			cm_reset_match_log(i);

			/* handle chip errata */
			if ((index = map_natipf_pair_index(i))) {
				setnat_root(sc, index, 22, 0x00000000);
			}

			break;
		}
	} 

	if (i == ENTRY_NUM_NATIPF) {
		printk("camelot: can't find Nat ipf entry to delete!!\n");
		return -EINVAL;
	}

	return 0;
}

int
cm_ioctl_flush_ipf(struct camelot_softc *sc)
{
	int i, index;

	if (0) printk("cm_iotcl_flush_ipf()\n");

	for (i = 0; i < NATIPF_TABLE_SIZE; i++) {

		/* map index into real index */
		index = map_natipf_index(i);

		/* look for active entries */
		if (read_NATIPFTBL(sc, index, 22) == 0x0)
			continue;

		setnat_root(sc, index, 22, 0x0);
		came_var.tbl_match[i] = 0;
		cm_reset_match_log(i);

		/* handle chip errata */
		if ((index = map_natipf_pair_index(i))) {
			setnat_root(sc, index, 22, 0x00000000);
		}

	} 

	return 0;
}


int
cm_iotcl_add_filter_entry(struct camelot_softc *sc, int table, FLT_TBL *ft)
{
	int i, j;

	if (0) printk("cm_ioctl_add_filter_entry()\n");

	for (i = 0; i < ENTRY_NUM_L34; i++) {

		/* scan table back to front */
		j = ENTRY_NUM_L34 - i - 1;
		
		if (!(read_L3_4_filter(sc, j, table, 10) & 0x00000001))
			break;
	}

	if (i == ENTRY_NUM_L34) {
		printk(KERN_ERR "camelot: L3/L4 filter entry overflow!!\n");
		return -EBUSY;
	}

	set_L3_4(sc, j, table, ft);

#if 0
	printk("Add");
	read_dump_FLT(sc, table, j);
#endif

	return 0;
}

/*
 * find an entry in the filter table and delete it
 */
int
cm_ioctl_del_filter_entry(struct camelot_softc *sc,
			  int table,
			  struct tbl_fef *fd)
{
	int i, j;

	if (0) printk("cm_ioctl_del_filter_entry()\n");

	for (i = 0; i < ENTRY_NUM_L34; i++) {
		u_int32_t t_dst, t_src, t_port;
		u_short t_dport, t_sport;

		/* search from back to front */
		j = ENTRY_NUM_L34 - i - 1;

		/* look for active entries */
		if (read_L3_4_filter(sc, j, table, 10) == 0x0)
			continue;

		/* read entry */
		t_dst = read_L3_4_filter(sc, j, table, 4);
		t_src = read_L3_4_filter(sc, j, table, 0);

		t_port = read_L3_4_filter(sc, j, table, 8);
		t_dport = t_port >> 16;
		t_sport = t_port & 0x0000ffff;

		if (t_dst == fd->ip_nat &&
		    t_src == fd->ip_dst &&
		    t_dport == fd->port_nat &&
		    t_sport == fd->port_dst)
		{
#if 0
			printk("Del");
			read_dump_FLT(sc, table, j);
#endif
			setflt_root(sc, j, 10, table, 0x0);
			break;
		}
	}


	if (i == ENTRY_NUM_L34) {
		printk(KERN_ERR 
		       "camelot: can't find filter entry to delete!!\n");
		return -EINVAL;
	}

	return 0;
}

int
cm_ioctl_flush_filters(struct camelot_softc *sc, int table)
{
	int i;

	if (0) printk("cm_ioctl_flush_filters()\n");

	for (i = 0; i < ENTRY_NUM_L34; i++) {
		/* look for active entries */
		if (read_L3_4_filter(sc, i, table, 10) == 0x0)
			continue;

		setflt_root(sc, i, 10, table, 0x0);
	}

	return 0;
}

int
cm_ioctl_del_unmatched_entry(struct camelot_softc *sc,
			     u_int32_t ctl22, int tindex)
{
	int index;
	u_long ip_src, ip_dst, ip_nat;
	u_short port_src, port_dst, port_nat;

	
	if (0) printk("cm_ioctl_del_unmatched_entry()\n");

	/* map index into real index */
	index = map_natipf_index(tindex);

	ip_src = (u_long)read_NATIPFTBL(sc, index, 0);
	ip_dst = (u_long)read_NATIPFTBL(sc, index, 1);
	ip_nat = (u_long)read_NATIPFTBL(sc, index, 2);
	port_src = (u_short)read_NATIPFTBL(sc, index, 12);
	port_dst = (u_short)read_NATIPFTBL(sc, index, 13);
	port_nat = (u_short)read_NATIPFTBL(sc, index, 14);

#if 0
	printk("Del");
	read_dump_NTIPFTBL_entry(sc, index);
#endif

	setnat_root(sc, index, 22, 0x00000000);
	came_var.tbl_match[tindex] = 0;
	cm_reset_match_log(tindex);

	/* handle chip errata */
	if ((index = map_natipf_pair_index(index))) {
		setnat_root(sc, index, 22, 0x00000000);
	}

	return 0;
}

int
cm_ioctl_del_unmatched(struct camelot_softc *sc)

{
	int i, j, tindex, index;
	int thresh;
	u_int32_t ctl22;
	u_int32_t match_log[ENTRY_NUM_NATMTHLOG];

	if (0) printk("cm_ioctl_del_unmatched()\n");

	cm_get_match_log(sc, match_log);

	for (i = 0; i < ENTRY_NUM_NATMTHLOG; i++) {
		for (j = 0; j < 32; j++){

			tindex = 32*i + j;

			/* map index into real index */
			index = map_natipf_index(tindex);
			ctl22 = read_NATIPFTBL(sc, index, 22);

			if (ctl22 & 0x100000) {
				if ((match_log[i] >> j) & 0x00000001) {
					/* matched */
					came_var.tbl_match[tindex] = 0;
				} else {
					/* unmatched */
					came_var.tbl_match[tindex]++;

					thresh = (ctl22 & 0x002000) ?
						THRESH_DEL_UNMATCH_UDP :
						THRESH_DEL_UNMATCH_TCP;

					if (came_var.tbl_match[tindex]> thresh)
					{
						cm_ioctl_del_unmatched_entry(
							sc, ctl22, tindex);
					}
				}
			}
		}
	}

	return 0;
}

static int
set_dmz_mode(struct camelot_softc *sc, int v)
{
	came_dmz_mode = v;
	return 0;
}

static int
set_filter_control_value(struct camelot_softc *sc, int v)
{
	filter_control_value = v;
	if (0) printk("camelot: filter control %08x\n", v);
	set_FLCNT(sc, v);
}

static int
get_filter_control_value(struct camelot_softc *sc)
{
	return filter_control_value;
}


int
cm_enable_pppoe(struct camelot_softc *sc)
{
	int fc_val;

	fc_val = get_filter_control_value(sc) & ~fc_pppoe_bits;

	fc_val |=
		fc_pppoe_sess_ipv4_pass_output |
		fc_pppoe_sess_ipv4_pass_input |
		/*fc_pppoe_sess_ipv6_drop_output | */
		fc_pppoe_sess_noip_output |
		fc_pppoe_sess_ipv6_drop_input |
		/*fc_pppoe_sess_noip_input | */
		0;

	set_filter_control_value(sc, fc_val);

	return 0;
}

int
cm_disable_pppoe(struct camelot_softc *sc)
{
	int fc_val;

	fc_val = get_filter_control_value(sc) & ~fc_pppoe_bits;

	fc_val |=
		fc_pppoe_sess_ipv4_drop_output |
		fc_pppoe_sess_ipv4_drop_input |
		fc_pppoe_sess_ipv6_drop_output |
		/*fc_pppoe_sess_noip_output | */
		fc_pppoe_sess_ipv6_drop_input |
		/*fc_pppoe_sess_noip_input | */
		0; 

	set_filter_control_value(sc, fc_val);

	return 0;
}

int
cm_disable_filtering(struct camelot_softc *sc)
{
	int fc_val;

	fc_val = get_filter_control_value(sc) & ~fc_filter_bits;

	set_filter_control_value(sc, fc_val);

	return 0;
}

int
cm_ioctl_set_pppoe(struct camelot_softc *sc)
{
	int i;

	if (0) printk("cm_ioctl_set_pppoe()\n");

	for (i = 0; i < LEN_PPPOE; i++) {
		(*sc->sc_write)(sc, PPPOE_HEADER_0_OFFSET + 4*i,
				came_var.pppoe_info[i]);
	}

	if (cm_enable_pppoe(sc))
		return -1;

	return 0;
}

int
cm_ioctl_flush_pppoe(struct camelot_softc *sc)
{
	int i;

	if (0) printk("cm_ioctl_flush_pppoe()\n");

	for (i = 0; i < LEN_PPPOE; i++) {
		came_var.pppoe_info[i] = 0;
		(*sc->sc_write)(sc, PPPOE_HEADER_0_OFFSET + 4*i, 0x0);
	}

	if (cm_disable_pppoe(sc))
		return -1;

	return 0;
}

int
cm_ioctl_set_tunnel(struct camelot_softc *sc)
{
	int i;

	if (0) printk("cm_ioctl_set_tunnel()\n");

	for (i = 0; i < LEN_TUNNEL; i++){
		(*sc->sc_write)(sc, V4_HEADER_0_0_OFFSET + 4*i,
				came_var.tunnel_info[i]);
	}

	return 0;
}

int
cm_ioctl_flush_tunnels(struct camelot_softc *sc)
{
	int i;

	if (0) printk("cm_ioctl_flush_tunnels()\n");

	came_var.tunnel_info[3] = 0;
	came_var.tunnel_info[4] = 0;

	for (i = 0; i < LEN_TUNNEL; i++){
		(*sc->sc_write)(sc, V4_HEADER_0_0_OFFSET + 4*i,
				came_var.tunnel_info[i]);
	}

	return 0;
}

/* debug read routines */

static int
debug_read_ipfnat(struct camelot_softc *sc, struct camelot_debug_read *pdr)
{
	int i;
	unsigned long v[23];

	if (0) printk("debug_read_ipfnat() index %d\n", pdr->index);

	for (i = 0; i < 23; i++){
		if (i == 3 || i == 15)
			continue;

		v[i] = read_NATIPFTBL(sc, pdr->index, i);
	}

	if (copy_to_user(pdr->data, (char *)v, sizeof(v)))
	{
		return -EINVAL;
	}

	return 0;
}

static int
debug_read_filter(struct camelot_softc *sc, struct camelot_debug_read *pdr)
{
	int i, table, id, size;
	unsigned long v[12];

	table = pdr->index >> 8;
	id = pdr->index & 0xff;

	if (0) printk("debug_read_filter() table %d, index %d\n", table, id);

	switch (table) {
	case 0:
	case 2:
		for (i = 0; i < 12; i++) {
			v[i] = read_L3_4_filter(sc, id, table, i);
		}
		size = sizeof(v);
		break;

	case 1:
	case 3:
		v[0] = read_PRTTB(sc, id, table);
		size = sizeof(v[0]);
		break;
	default:
		return -1;
	}

	if (copy_to_user(pdr->data, (char *)v, size))
	{
		return -EINVAL;
	}

	return 0;
}

static int
debug_read_filter_count(struct camelot_softc *sc,
			struct camelot_debug_read *pdr)
{
	int i, table;
	unsigned long v[ENTRY_NUM_L34];

	table = pdr->index;
	if (0) printk("debug_read_filter_count() table %d\n", table);

	for (i = 0; i < ENTRY_NUM_L34; i++) {
		v[i] = read_FLCNTV(sc, table, i);
	}

	if (copy_to_user(pdr->data, (char *)v, sizeof(v)))
	{
		return -EINVAL;
	}

	return 0;
}

static u_int flt_log_out[ENTRY_NUM_LOG][16];
static u_int flt_log_in[ENTRY_NUM_LOG][16];

static int
debug_read_filter_log(struct camelot_softc *sc,
		      struct camelot_debug_read *pdr)
{
	int status;

	if (0) printk("debug_read_filter_log()\n");

	status = 0;
	memset((char *)flt_log_in, 0, sizeof(flt_log_in));
	memset((char *)flt_log_out, 0, sizeof(flt_log_out));

	read_FLTLOG(sc, flt_log_in, flt_log_out, &status);

	/* return status */
	if (copy_to_user(pdr->data, (char *)&status, sizeof(status)))
	{
		return -EINVAL;
	}

	/* input filter log */
	pdr->data += sizeof(int);
	if (copy_to_user(pdr->data, (char *)flt_log_in, sizeof(flt_log_in))) {
		return -EINVAL;
	}

	/* output filter log */
	pdr->data += sizeof(flt_log_in);
	if (copy_to_user(pdr->data, (char *)flt_log_out, sizeof(flt_log_out)))
	{
		return -EINVAL;
	}

	return 0;
}

static int
debug_read_tunnel(struct camelot_softc *sc, struct camelot_debug_read *pdr)
{
	int i, offset;
	unsigned long v[4];

	switch (pdr->index) {
	case 0:
		offset = V4_HEADER_0_0_OFFSET;
		break;
	case 1:
		offset = V4_HEADER_1_0_OFFSET;
		break;
	case 2:
		offset = V4_HEADER_2_0_OFFSET;
		break;
	case 3:
		offset = V4_HEADER_3_0_OFFSET;
		break;
	default:
		return -1;
	}

	for (i = 0; i < 4; i++){
		v[i] = (*sc->sc_read)(sc, offset + 4*i);
	}

	if (copy_to_user(pdr->data, (char *)v, sizeof(v)))
	{
		return -EINVAL;
	}

	return 0;
}

static int
debug_read_pppoe(struct camelot_softc *sc, struct camelot_debug_read *pdr)
{
	int i;
	unsigned long v[LEN_PPPOE];

	for (i = 0; i < LEN_PPPOE; i++) {
		v[i] = (*sc->sc_read)(sc, PPPOE_HEADER_0_OFFSET + 4*i);
	}

	if (copy_to_user(pdr->data, (char *)v, sizeof(v)))
	{
		return -EINVAL;
	}

	return 0;
}

/*
 * Process an ioctl request.
 */
int
camelot_ioctl(struct net_device *dev, struct ifreq *rq, int cmd)
{
	struct cm_softc *cm = (struct cm_softc *)dev->priv;
	struct camelot_softc *sc = &cm->camelot;
	char *data = (char *)rq->ifr_data;
	NAT_IPF_TBL nit;
	FLT_TBL ft;
	struct tbl_fef fd;
	struct camelot_debug_read dr;
	int error = 0;
	unsigned long v[2];

	if (0) printk("camelot_ioctl(cmd=0x%x)\n", cmd);

	switch (cmd) {
	case CM_SET_NAT:
	case CM_SET_IPF:
		if (copy_from_user((char *)&nit, data, sizeof(NAT_IPF_TBL)))
			return -EINVAL;

		error = cm_iotcl_add_nat_entry(sc, &nit);
		break;

	case CM_SET_FLT_OUT:
		if (copy_from_user((char *)&ft, data, sizeof(FLT_TBL)))
			return -EINVAL;

		error = cm_iotcl_add_filter_entry(sc, 2, &ft);
		break;

	case CM_SET_FLT_IN:
		if (copy_from_user((char *)&ft, data, sizeof(FLT_TBL)))
			return -EINVAL;

		error = cm_iotcl_add_filter_entry(sc, 0, &ft);
		break;

	case CM_DEL_UNMATCH:
		error = cm_ioctl_del_unmatched(sc);
		break;

	case CM_DEL_NAT:
	case CM_DEL_IPF:
		if (copy_from_user((char *)&nit, data, sizeof(NAT_IPF_TBL)))
			return -EINVAL;

		error = cm_ioctl_del_nat_entry(sc, &nit);
		break;

	case CM_DEL_FLT_OUT:
		if (copy_from_user((char *)&fd, data, sizeof(struct tbl_fef)))
			return -EINVAL;

		error = cm_ioctl_del_filter_entry(sc, 2, &fd);
		break;

	case CM_DEL_FLT_IN:
		if (copy_from_user((char *)&fd, data, sizeof(struct tbl_fef)))
			return -EINVAL;

		error = cm_ioctl_del_filter_entry(sc, 0, &fd);
		break;

	case CM_FLUSH:
		switch (rq->ifr_metric) {
		case FLUSH_FLT:
			error = cm_ioctl_flush_filters(sc, 2);
			error = cm_ioctl_flush_filters(sc, 0);

			/* turn off filtering */
			cm_disable_filtering(sc);
			break;
		case FLUSH_IPF:
		case FLUSH_NAT:
			error = cm_ioctl_flush_ipf(sc);
			break;
		case FLUSH_PPPOE:
			error = cm_ioctl_flush_pppoe(sc);
			break;
		case FLUSH_TUNNEL:
			error = cm_ioctl_flush_tunnels(sc);
			break;
		default:
			return -EINVAL;
		}
		break;

	case CM_GET_MATCH:
		if (data == 0) {
			return -EINVAL;
		} else {
			u_int32_t match_log[ENTRY_NUM_NATMTHLOG];

			cm_get_match_log(sc, match_log);

			if (copy_to_user(data, (char *)&match_log,
					 sizeof(match_log)))
			{
				return -EINVAL;
			}
		}
		break;

	case CM_PPPOE:
		if (copy_from_user((char *)&came_var.pppoe_info[0],
				   data,
				   LEN_PPPOE*sizeof(u_int32_t)))
		{
			return -EINVAL;
		}

		printk(KERN_DEBUG
		       "CM_PPPOE %08x %08x %08x %08x\n",
		       came_var.pppoe_info[0], came_var.pppoe_info[1],
		       came_var.pppoe_info[2], came_var.pppoe_info[3]);

		error = cm_ioctl_set_pppoe(sc);
		break;

	case CM_TUNNEL:
		if (copy_from_user((char *)&came_var.tunnel_info[3],
				   data, 2*sizeof(u_int32_t)))
		{
			return -EINVAL;
		}

		came_var.tunnel_info[0] =
			((IPVERSION << 4) & 0x000000f0) |
			(0x5 & 0x0000000f);

		came_var.tunnel_info[1] = 0x00000000;

		came_var.tunnel_info[2] =
			((IPPROTO_IPV6 << 8) & 0x0000ff00) |
			(0x1e & 0x000000ff);

		printk(KERN_DEBUG
		       "CM_TUNNEL %08x %08x %08x %08x %08x\n",
		       came_var.tunnel_info[0], came_var.tunnel_info[1],
		       came_var.tunnel_info[2], came_var.tunnel_info[3],
		       came_var.tunnel_info[4]);

		error = cm_ioctl_set_tunnel(sc);
		break;

	case CM_DEBUG_READ:
		if (copy_from_user((char *)&dr, data, sizeof(dr)))
			return -EINVAL;

		switch (dr.what) {
		case DEBUG_FILTER:
			debug_read_filter(sc, &dr);
			break;
		case DEBUG_FILTER_COUNT:
			debug_read_filter_count(sc, &dr);
			break;
		case DEBUG_FILTER_LOG:
			debug_read_filter_log(sc, &dr);
			break;
		case DEBUG_IPFNAT:
			debug_read_ipfnat(sc, &dr);
			break;
		case DEBUG_PPPOE:
			debug_read_pppoe(sc, &dr);
			break;
		case DEBUG_TUNNEL:
			debug_read_tunnel(sc, &dr);
			break;
		}

		break;

	case CM_SET_FLT_MODE:
		if (copy_from_user((char *)&v, data, sizeof(v)))
			return -EINVAL;

		printk(KERN_DEBUG
		       "camelot: set filter mode %08lx, mask %08lx\n",
		       v[0], v[1]);

		set_filter_control_value(sc, v[0]);
		set_SUBNM(sc, v[1]);
		break;

	case CM_SET_DMZ:
		if (copy_from_user((char *)&v[0], data, sizeof(v[0])))
			return -EINVAL;

		printk(KERN_DEBUG
		       "camelot: set dmz mode %08lx\n", v[0]);

		set_dmz_mode(sc, v[0]);
		break;

	default:
		error = -EOPNOTSUPP;
		break;
	}

	return error;
}

