/*
=========================================================================
 r8169.c: A RealTek RTL8169s/8110s Gigabit Ethernet driver for Linux kernel 2.4.x.
 --------------------------------------------------------------------

 History:
 Feb  4 2002	- created initially by ShuChen <shuchen@realtek.com.tw>.
 May 20 2002	- Add link status force-mode and TBI mode support.
=========================================================================

RTL8169_VERSION "1.1"	<2002/10/4>

	The bit4:0 of MII register 4 is called "selector field", and have to be
	00001b to indicate support of IEEE std 802.3 during NWay process of
	exchanging Link Code Word (FLP).

RTL8169_VERSION "1.2"	<2003/6/17>
	Update driver module name.
	Modify ISR.
        Add chip mac_version.

RTL8169_VERSION "1.3"	<2003/6/20>
        Add chip phy_version.
	Add priv->phy_timer_t, rtl8169_phy_timer_t_handler()
	Add rtl8169_hw_PHY_config()
	Add rtl8169_hw_PHY_reset()

RTL8169_VERSION "1.4"	<2003/7/14>
	Add tx_bytes, rx_bytes.

RTL8169_VERSION "1.5"	<2003/7/18>
	Set 0x0000 to PHY at offset 0x0b.
	Modify chip mac_version, phy_version
	Force media for multiple card.
RTL8169_VERSION "1.6"	<2003/8/25>
	Modify receive data buffer.

RTL8169_VERSION "1.7"	<2003/9/18>
	Add Jumbo Frame support.

RTL8169_VERSION "1.8"	<2003/10/21>
	Performance and CPU Utilizaion Enhancement.

RTL8169_VERSION "1.8n4"	<2003/11/15>
	Performance Enhancement for MPC8241.
	Add proc fs code.
	Add eeprom read/write code.

RTL8169_VERSION "1.8n4b"	<2004/3/16>
	Add support 93C56/66
	Enable JumboFrame functions

RTL8169_VERSION "1.8n5"	<2004/11/10>
	Changed EEPROM data for TeraStation

*/

#include <linux/module.h>
#include <linux/pci.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/delay.h>

#include <linux/timer.h>
#include <linux/init.h>

// BUFFALO add
#include <linux/proc_fs.h>
#include "buffalo/ls_eth.h"
#include <linux/config.h>

#define RTL8169_VERSION "1.8n5 <2004-11-10>"
#define MODULENAME "RTL8169s/8110s"
#ifdef RTL8169_JUMBO_FRAME_SUPPORT
 #define RTL8169_DRIVER_NAME   MODULENAME " Gigabit Ethernet driver supports Jambo Frame " RTL8169_VERSION
#else
 #define RTL8169_DRIVER_NAME   MODULENAME " Gigabit Ethernet driver " RTL8169_VERSION
#endif
#define PFX MODULENAME ": "

#undef RTL8169_IOCTL_SUPPORT
#undef RTL8169_DEBUG

#ifdef RTL8169_DEBUG
	#define assert(expr) \
        	if(!(expr)) { printk( "Assertion failed! %s,%s,%s,line=%d\n", #expr,__FILE__,__FUNCTION__,__LINE__); }
	#define DBG_PRINT( fmt, args...)   printk("r8169: " fmt, ## args);
#else
	#define assert(expr) do {} while (0)
	#define DBG_PRINT( fmt, args...)   ;
#endif	// end of #ifdef RTL8169_DEBUG


/* media options */
#define MAX_UNITS 8
static int media[MAX_UNITS] = {-1, -1, -1, -1, -1, -1, -1, -1};

/* Maximum events (Rx packets, etc.) to handle at each interrupt. */
static int max_interrupt_work = 20;

/* Maximum number of multicast addresses to filter (vs. Rx-all-multicast).
   The RTL chips use a 64 element hash table based on the Ethernet CRC.  */
static int multicast_filter_limit = 32;

/* MAC address length*/
#define MAC_ADDR_LEN	6

#define TX_FIFO_THRESH 256		/* In bytes */

#define RX_FIFO_THRESH	7		/* 7 means NO threshold, Rx buffer level before first PCI xfer.  */
#define RX_DMA_BURST	7		/* Maximum PCI burst, '6' is 1024 */
#define TX_DMA_BURST	7		/* Maximum PCI burst, '6' is 1024 */
#define ETTh 	0x3F		/* 0x3F means NO threshold */

#define ETH_HDR_LEN	14
#define DEFAULT_MTU	1500
#define DEFAULT_RX_BUF_LEN	(DEFAULT_MTU + ETH_HDR_LEN + 2)	/* Rx Buffer size */

//#define DEFAULT_RX_BUF_LEN	1536

#ifdef RTL8169_JUMBO_FRAME_SUPPORT
#define MAX_JUMBO_FRAME_MTU	( 7600 )
//#define MAX_RX_SKBDATA_SIZE	( MAX_JUMBO_FRAME_MTU + ETH_HDR_LEN )
#define MAX_RX_SKBDATA_SIZE	( 7700 )
#else
#define MAX_RX_SKBDATA_SIZE	1600
#endif //end #ifdef RTL8169_JUMBO_FRAME_SUPPORT


#define InterFrameGap	0x03		/* 3 means InterFrameGap = the shortest one */

#define NUM_TX_DESC	64		/* Number of Tx descriptor registers*/
#define NUM_RX_DESC	64		/* Number of Rx descriptor registers*/


#define RTL_MIN_IO_SIZE 0x80
#define TX_TIMEOUT  (6*HZ)

/* write/read MMIO register */
#define RTL_W8(reg, val8)	writeb ((val8), ioaddr + (reg))
#define RTL_W16(reg, val16)	writew ((val16), ioaddr + (reg))
#define RTL_W32(reg, val32)	writel ((val32), ioaddr + (reg))
#define RTL_R8(reg)		readb (ioaddr + (reg))
#define RTL_R16(reg)		readw (ioaddr + (reg))
#define RTL_R32(reg)		((unsigned long) readl (ioaddr + (reg)))


#define RTL_GIGA_MAC_VER_B  0x00    //MAC version 0000
//#define RTL_GIGA_MAC_VER_C  0x03
#define RTL_GIGA_MAC_VER_D  0x01    //MAC version 0001
#define RTL_GIGA_MAC_VER_E  0x02    //MAC version 0002

#define RTL_GET_MAC_VERSION(mac_version)     \
{\
	mac_version = RTL_GIGA_MAC_VER_B;\
	if( (RTL_R32(TxConfig)&0x7c800000) & (0x1<<26) ){ \
		mac_version = RTL_GIGA_MAC_VER_E;\
	}\
	else if( (RTL_R32(TxConfig)&0x7c800000) & (0x1<<23) ){\
		mac_version = RTL_GIGA_MAC_VER_D;\
	}\
	else if( (RTL_R32(TxConfig)&0x7c800000) == 0x00000000 ){\
		mac_version = RTL_GIGA_MAC_VER_B;\
	}\
}

#define RTL_PRINT_MAC_VERSION(mac_version) \
{\
	switch(mac_version) \
	{ \
		case RTL_GIGA_MAC_VER_E: \
			DBG_PRINT("mac_version == RTL_GIGA_MAC_VER_E (0002)\n"); \
			break; \
		case RTL_GIGA_MAC_VER_D: \
			DBG_PRINT("mac_version == RTL_GIGA_MAC_VER_D (0001)\n"); \
			break; \
		case RTL_GIGA_MAC_VER_B: \
			DBG_PRINT("mac_version == RTL_GIGA_MAC_VER_B (0000)\n"); \
			break; \
		default: \
			DBG_PRINT("mac_version == Unknown\n"); \
			break; \
	} \
}

#define RTL_GIGA_PHY_VER_C    0x03	//PHY Reg 0x03 bit0-3 == 0x0000
#define RTL_GIGA_PHY_VER_D    0x04	//PHY Reg 0x03 bit0-3 == 0x0000
#define RTL_GIGA_PHY_VER_E    0x05	//PHY Reg 0x03 bit0-3 == 0x0000
#define RTL_GIGA_PHY_VER_F    0x06	//PHY Reg 0x03 bit0-3 == 0x0001
#define RTL_GIGA_PHY_VER_G    0x07	//PHY Reg 0x03 bit0-3 == 0x0002

#define RTL_GET_PHY_VERSION(phy_version)     \
{\
	phy_version = RTL_GIGA_PHY_VER_D;\
	if( (RTL8169_READ_GMII_REG(ioaddr,3)&0x000f) == 0x0002 ){ \
		phy_version = RTL_GIGA_PHY_VER_G;\
	} \
	else if( (RTL8169_READ_GMII_REG(ioaddr,3)&0x000f) == 0x0001 ){ \
		phy_version = RTL_GIGA_PHY_VER_F;\
	} \
	else if( (RTL8169_READ_GMII_REG(ioaddr,3)&0x000f) == 0x0000 ){ \
		phy_version = RTL_GIGA_PHY_VER_E;\
	} \
}

#define RTL_PRINT_PHY_VERSION(phy_version)  \
{\
	switch(phy_version)\
	{ \
		case RTL_GIGA_PHY_VER_G: \
			DBG_PRINT("phy_version == RTL_GIGA_PHY_VER_G (0002)\n"); \
			break; \
		case RTL_GIGA_PHY_VER_F: \
			DBG_PRINT("phy_version == RTL_GIGA_PHY_VER_F (0001)\n"); \
			break; \
		case RTL_GIGA_PHY_VER_E: \
			DBG_PRINT("phy_version == RTL_GIGA_PHY_VER_E (0000)\n"); \
			break; \
		case RTL_GIGA_PHY_VER_D: \
			DBG_PRINT("phy_version == RTL_GIGA_PHY_VER_D (0000)\n"); \
			break; \
		default: \
			DBG_PRINT("phy_version == Unknown\n"); \
			break; \
	} \
}


const static struct {
	const char *name;
	u8 mac_version;		 /* depend on RTL8169 docs */
	u32 RxConfigMask; 	/* should clear the bits supported by this chip */
} rtl_chip_info[] = {
	{ "RTL8169",  RTL_GIGA_MAC_VER_B,  0xff7e1880 },
	{ "RTL8169s/8110s",  RTL_GIGA_MAC_VER_D,  0xff7e1880 },
	{ "RTL8169s/8110s",  RTL_GIGA_MAC_VER_E,  0xff7e1880 },
};




static struct pci_device_id rtl8169_pci_tbl[] __devinitdata = {
	{ 0x10ec, 0x8169, PCI_ANY_ID, PCI_ANY_ID, 0, 0, 0 },
	{ 0x10ec, 0x8129, PCI_ANY_ID, PCI_ANY_ID, 0, 0, 0 },    /* BUFFALO add */
	{ 0xec10, 0x6981, PCI_ANY_ID, PCI_ANY_ID, 0, 0, 0 },    /* BUFFALO add */
	{0,},
};



MODULE_DEVICE_TABLE (pci, rtl8169_pci_tbl);

enum RTL8169_registers {
	MAC0 = 0x0,		/* Ethernet hardware address. */
	MAR0 = 0x8,		/* Multicast filter. */
	TxDescStartAddr	= 0x20,
	TxHDescStartAddr= 0x28,
	FLASH	= 0x30,
	ERSR	= 0x36,
	ChipCmd	= 0x37,
	TxPoll	= 0x38,
	IntrMask = 0x3C,
	IntrStatus = 0x3E,
	TxConfig = 0x40,
	RxConfig = 0x44,
	Timer = 0x48,	// BUFFALO add
	RxMissed = 0x4C,
	Cfg9346 = 0x50,
	Config0	= 0x51,
	Config1	= 0x52,
	Config2	= 0x53,
	Config3	= 0x54,
	Config4	= 0x55,
	Config5	= 0x56,
	MultiIntr = 0x5C,
	PHYAR	= 0x60,
	TBICSR	= 0x64,
	TBI_ANAR = 0x68,
	TBI_LPAR = 0x6A,
	PHYstatus = 0x6C,
	RxMaxSize = 0xDA,
	CPlusCmd = 0xE0,
	RxDescStartAddr	= 0xE4,
	ETThReg	= 0xEC,
	FuncEvent	= 0xF0,
	FuncEventMask	= 0xF4,
	FuncPresetState	= 0xF8,
	FuncForceEvent	= 0xFC,		
};

enum RTL8169_register_content {
	/*InterruptStatusBits*/
	SYSErr 		= 0x8000,
	PCSTimeout	= 0x4000,
	SWInt		= 0x0100,
	TxDescUnavail	= 0x80,
	RxFIFOOver 	= 0x40,
	LinkChg 	= 0x20,
	RxOverflow 	= 0x10,
	TxErr 	= 0x08,
	TxOK 	= 0x04,
	RxErr 	= 0x02,
	RxOK 	= 0x01,

	/*RxStatusDesc*/
	RxRES = 0x00200000,
	RxCRC = 0x00080000,
	RxRUNT= 0x00100000,
	RxRWT = 0x00400000,

	/*ChipCmdBits*/
	CmdReset = 0x10,
	CmdRxEnb = 0x08,
	CmdTxEnb = 0x04,
	RxBufEmpty = 0x01,

	/*Cfg9346Bits*/
	Cfg9346_Lock = 0x00,
	Cfg9346_Unlock = 0xC0,

	/*rx_mode_bits*/
	AcceptErr = 0x20,
	AcceptRunt = 0x10,
	AcceptBroadcast = 0x08,
	AcceptMulticast = 0x04,
	AcceptMyPhys = 0x02,
	AcceptAllPhys = 0x01,

	/*RxConfigBits*/
	RxCfgFIFOShift = 13,
	RxCfgDMAShift = 8,

	/*TxConfigBits*/
	TxInterFrameGapShift = 24,
	TxDMAShift = 8,		/* DMA burst value (0-7) is shift this many bits */

	/*rtl8169_PHYstatus*/
	TBI_Enable	= 0x80,
	TxFlowCtrl	= 0x40,
	RxFlowCtrl	= 0x20,
	_1000bpsF	= 0x10,
	_100bps		= 0x08,
	_10bps		= 0x04,
	LinkStatus	= 0x02,
	FullDup		= 0x01,

	/*GIGABIT_PHY_registers*/
	PHY_CTRL_REG = 0,
	PHY_STAT_REG = 1,
	PHY_AUTO_NEGO_REG = 4,
	PHY_1000_CTRL_REG = 9,

	/*GIGABIT_PHY_REG_BIT*/
	PHY_Restart_Auto_Nego	= 0x0200,
	PHY_Enable_Auto_Nego	= 0x1000,

	//PHY_STAT_REG = 1;
	PHY_Auto_Neco_Comp	= 0x0020,

	//PHY_AUTO_NEGO_REG = 4;
	PHY_Cap_10_Half		= 0x0020,
	PHY_Cap_10_Full		= 0x0040,
	PHY_Cap_100_Half	= 0x0080,
	PHY_Cap_100_Full	= 0x0100,

	//PHY_1000_CTRL_REG = 9;
	PHY_Cap_1000_Full	= 0x0200,

	PHY_Cap_Null		= 0x0,


	/*_MediaType*/
	_10_Half	= 0x01,
	_10_Full	= 0x02,
	_100_Half	= 0x04,
	_100_Full	= 0x08,
	_1000_Full	= 0x10,

	/*_TBICSRBit*/
	TBILinkOK 	= 0x02000000,
};



enum _DescStatusBit {
	OWNbit	= 0x80000000,
	EORbit	= 0x40000000,
	FSbit	= 0x20000000,
	LSbit	= 0x10000000,
};


struct TxDesc {
	u32		status;
	u32		vlan_tag;
	u32		buf_addr;
	u32		buf_Haddr;
};

struct RxDesc {
	u32		status;
	u32		vlan_tag;
	u32		buf_addr;
	u32		buf_Haddr;
};

//#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,18)
//typedef struct timer_list timer_t;
//#endif //#ifndef timer_t

struct rtl8169_private {
	void *mmio_addr;				/* memory map physical address*/
	struct pci_dev *pci_dev;			/* Index of PCI device  */
	struct net_device_stats stats;			/* statistics of net device */
	spinlock_t lock;				/* spin lock flag */
	int chipset;
	int mac_version;
	int phy_version;
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,18)
	struct timer_list phy_timer_t;
#else
	timer_t phy_timer_t;
#endif //#ifndef timer_t
	unsigned long phy_link_down_cnt;
	unsigned long cur_rx;				/* Index into the Rx descriptor buffer of next Rx pkt. */
	unsigned long cur_tx;				/* Index into the Tx descriptor buffer of next Rx pkt. */
	unsigned long dirty_tx;
	unsigned char	*TxDescArrays;			/* Index of Tx Descriptor buffer */
	unsigned char	*RxDescArrays;			/* Index of Rx Descriptor buffer */
	struct	TxDesc	*TxDescArray;			/* Index of 256-alignment Tx Descriptor buffer */
	struct	RxDesc	*RxDescArray;			/* Index of 256-alignment Rx Descriptor buffer */
	struct	sk_buff	*Tx_skbuff[NUM_TX_DESC];	/* Index of Transmit data buffer */
	struct	sk_buff	*Rx_skbuff[NUM_RX_DESC];	/* Receive data buffer */
	unsigned char   drvinit_fail;
	
	int curr_mtu_size;
	int tx_pkt_len;
	int rx_pkt_len;

	int hw_rx_pkt_len;
};



MODULE_AUTHOR ("Realtek");
MODULE_DESCRIPTION ("RealTek RTL-8169 Gigabit Ethernet driver");
MODULE_PARM (media, "1-" __MODULE_STRING(MAX_UNITS) "i");
MODULE_LICENSE("GPL");



static int rtl8169_open (struct net_device *dev);
static int rtl8169_start_xmit (struct sk_buff *skb, struct net_device *dev);
static void rtl8169_interrupt (int irq, void *dev_instance, struct pt_regs *regs);
static void rtl8169_init_ring (struct net_device *dev);
static void rtl8169_hw_start (struct net_device *dev);
static int rtl8169_close (struct net_device *dev);
static inline u32 ether_crc (int length, unsigned char *data);
static void rtl8169_set_rx_mode (struct net_device *dev);
static void rtl8169_tx_timeout (struct net_device *dev);
static struct net_device_stats *rtl8169_get_stats(struct net_device *netdev);

#ifdef RTL8169_JUMBO_FRAME_SUPPORT
static int rtl8169_change_mtu(struct net_device *dev, int new_mtu);
#endif //end #ifdef RTL8169_JUMBO_FRAME_SUPPORT

static void rtl8169_hw_PHY_config (struct net_device *dev);
static void rtl8169_hw_PHY_reset(struct net_device *dev);
static const u16 rtl8169_intr_mask = LinkChg | RxOverflow | RxFIFOOver | TxErr | TxOK | RxErr | RxOK ;
static const unsigned int rtl8169_rx_config = (RX_FIFO_THRESH << RxCfgFIFOShift) | (RX_DMA_BURST << RxCfgDMAShift);

static int rtl8169_proc_status(char *buffer, char **buffer_location, off_t offset,int buffer_length,int *eof,void *data);
static void rtl8169_init_proc(struct net_device *dev);
static void rtl8169_remove_proc(struct net_device *dev);
struct proc_dir_entry *rtl8169_Proc_root;
struct proc_dir_entry *rtl8169_Proc;

static int rtl8169_read_eeprom (void *ioaddr, int location, int addr_len);
static int rtl8169_write_eeprom(long ioaddr, int location, int addr_len, short data);
static int rtl8169_write_eeprom_en(long ioaddr, int addr_len);
static int rtl8169_write_eeprom_ds(long ioaddr, int addr_len);

int rtl8169_ioctl(struct net_device *dev, struct ifreq *ifr, int cmd);

#define RTL8169_WRITE_GMII_REG_BIT( ioaddr, reg, bitnum, bitval )\
{ \
	int val; \
	if( bitval == 1 ){ val = ( RTL8169_READ_GMII_REG( ioaddr, reg ) | (bitval<<bitnum) ) & 0xffff ; } \
	else{ val = ( RTL8169_READ_GMII_REG( ioaddr, reg ) & (~(0x0001<<bitnum)) ) & 0xffff ; } \
	RTL8169_WRITE_GMII_REG( ioaddr, reg, val ); \
}



#ifdef RTL8169_DEBUG
unsigned alloc_rxskb_cnt = 0;
#define RTL8169_ALLOC_RXSKB(bufsize) 	dev_alloc_skb(bufsize); alloc_rxskb_cnt ++ ;
#define RTL8169_FREE_RXSKB(skb) 	kfree_skb(skb); alloc_rxskb_cnt -- ;
#define RTL8169_NETIF_RX(skb) 		netif_rx(skb); alloc_rxskb_cnt -- ;
#else
#define RTL8169_ALLOC_RXSKB(bufsize) 	dev_alloc_skb(bufsize);
#define RTL8169_FREE_RXSKB(skb) 	kfree_skb(skb);

#ifndef RTL8169_IOCTL_DIAG_SUPPORT
#define RTL8169_NETIF_RX(skb) 		netif_rx(skb);
#else
#define RTL8169_NETIF_RX(skb) 		rtl8169_diag_rx(skb);
#endif //end #ifdef RTL8169_IOCTL_DIAG_SUPPORT

#endif //end #ifdef RTL8169_DEBUG






//=================================================================
//	PHYAR
//	bit		Symbol
//	31		Flag
//	30-21	reserved
//	20-16	5-bit GMII/MII register address
//	15-0	16-bit GMII/MII register data
//=================================================================
void RTL8169_WRITE_GMII_REG( void *ioaddr, int RegAddr, int value )
{
	int	i;

	RTL_W32 ( PHYAR, 0x80000000 | (RegAddr&0xFF)<<16 | value);
	udelay(1000);

	for( i = 2000; i > 0 ; i -- ){
		// Check if the RTL8169 has completed writing to the specified MII register
		if( ! (RTL_R32(PHYAR)&0x80000000) ){
			break;
		}
		else{
			udelay(100);
		}// end of if( ! (RTL_R32(PHYAR)&0x80000000) )
	}// end of for() loop
}

//=================================================================
int RTL8169_READ_GMII_REG( void *ioaddr, int RegAddr )
{
	int i, value = -1;

	RTL_W32 ( PHYAR, 0x0 | (RegAddr&0xFF)<<16 );
	udelay(1000);

	for( i = 2000; i > 0 ; i -- ){
		// Check if the RTL8169 has completed retrieving data from the specified MII register
		if( RTL_R32(PHYAR) & 0x80000000 ){
			value = (int)( RTL_R32(PHYAR)&0xFFFF );
			break;
		}
		else{
			udelay(100);
		}// end of if( RTL_R32(PHYAR) & 0x80000000 )
	}// end of for() loop
	return value;
}

#ifdef RTL8169_IOCTL_SUPPORT
#include "r8169_ioctl.c"
#endif //end #ifdef RTL8169_IOCTL_SUPPORT

#define rtl8169_request_timer( timer, timer_expires, timer_func, timer_data ) \
{ \
	init_timer(timer); \
	timer->expires = (unsigned long)(jiffies + timer_expires); \
	timer->data = (unsigned long)(timer_data); \
	timer->function = (void *)(timer_func); \
	add_timer(timer); \
	DBG_PRINT("request_timer at 0x%08lx\n", (unsigned long)timer); \
}

#define rtl8169_delete_timer( del_timer_t ) \
{ \
	del_timer(del_timer_t); \
	DBG_PRINT("delete_timer at 0x%08lx\n", (unsigned long)del_timer_t); \
}

#define rtl8169_mod_timer( timer, timer_expires ) \
{ \
	mod_timer( timer, jiffies + timer_expires ); \
}


//====================================================================================
void rtl8169_phy_timer_t_handler( void	*timer_data )
{
	struct net_device *dev = (struct net_device *)timer_data;
	struct rtl8169_private *priv = (struct rtl8169_private *) (dev->priv);
	void *ioaddr = priv->mmio_addr;

	assert( priv->mac_version > RTL_GIGA_MAC_VER_B );
	assert( priv->phy_version < RTL_GIGA_PHY_VER_G );

	if( RTL_R8(PHYstatus) & LinkStatus ){
		priv->phy_link_down_cnt = 0 ;
	}
	else{
		priv->phy_link_down_cnt ++ ;
		if( priv->phy_link_down_cnt >= 12 ){
			// If link on 1000, perform phy reset.
			if( RTL8169_READ_GMII_REG( ioaddr, PHY_1000_CTRL_REG ) & PHY_Cap_1000_Full )
			{
				rtl8169_hw_PHY_reset( dev );
			}

			priv->phy_link_down_cnt = 0 ;
		}
	}

	//---------------------------------------------------------------------------
	//mod_timer is a more efficient way to update the expire field of an active timer.
	//---------------------------------------------------------------------------
	rtl8169_mod_timer( (&priv->phy_timer_t), 100 );
}




//=======================================================================================
static int __devinit rtl8169_init_board ( struct pci_dev *pdev, struct net_device **dev_out, void **ioaddr_out)
{
	void *ioaddr = NULL;
	struct net_device *dev;
	struct rtl8169_private *priv;
	int rc, i;
	unsigned long mmio_start, mmio_end, mmio_flags, mmio_len;


	assert (pdev != NULL);
	assert (ioaddr_out != NULL);

	*ioaddr_out = NULL;
	*dev_out = NULL;

	// dev zeroed in init_etherdev 
	dev = init_etherdev (NULL, sizeof (*priv));
	if (dev == NULL) {
		printk (KERN_ERR PFX "unable to alloc new ethernet\n");
		return -ENOMEM;
	}

	SET_MODULE_OWNER(dev);
	priv = dev->priv;

	// enable device (incl. PCI PM wakeup and hotplug setup)
	rc = pci_enable_device (pdev);
	if (rc)
		goto err_out;

	mmio_start = pci_resource_start (pdev, 1);
	mmio_end = pci_resource_end (pdev, 1);
	mmio_flags = pci_resource_flags (pdev, 1);
	mmio_len = pci_resource_len (pdev, 1);

	// make sure PCI base addr 1 is MMIO
	if (!(mmio_flags & IORESOURCE_MEM)) {
		printk (KERN_ERR PFX "region #1 not an MMIO resource, aborting\n");
		rc = -ENODEV;
		goto err_out;
	}

	// check for weird/broken PCI region reporting
	if ( mmio_len < RTL_MIN_IO_SIZE ) {
		printk (KERN_ERR PFX "Invalid PCI region size(s), aborting\n");
		rc = -ENODEV;
		goto err_out;
	}


	rc = pci_request_regions (pdev, dev->name);
	if (rc)
		goto err_out;

	// enable PCI bus-mastering
	pci_set_master (pdev);


	// ioremap MMIO region
	ioaddr = ioremap (mmio_start, mmio_len);
	if (ioaddr == NULL) {
		printk (KERN_ERR PFX "cannot remap MMIO, aborting\n");
		rc = -EIO;
		goto err_out_free_res;
	}


	// Soft reset the chip.
	RTL_W8 ( ChipCmd, CmdReset);

	// Check that the chip has finished the reset.
	for (i = 1000; i > 0; i--){
		if ( (RTL_R8(ChipCmd) & CmdReset) == 0){
			break;
		}
		else{
			udelay (10);
		}
	}

	// identify chip attached to board
	RTL_GET_MAC_VERSION(priv->mac_version);
	RTL_GET_PHY_VERSION(priv->phy_version);

	RTL_PRINT_MAC_VERSION(priv->mac_version);
	RTL_PRINT_PHY_VERSION(priv->phy_version);



	for (i = ARRAY_SIZE (rtl_chip_info) - 1; i >= 0; i--){
		if (priv->mac_version == rtl_chip_info[i].mac_version) {
			priv->chipset = i;
			goto match;
		}
	}

	//if unknown chip, assume array element #0, original RTL-8169 in this case
	printk (KERN_DEBUG PFX "PCI device %s: unknown chip version, assuming RTL-8169\n", pdev->slot_name);
	priv->chipset = 0;

match:

	*ioaddr_out = ioaddr;
	*dev_out = dev;
	return 0;

//err_out_iounmap:
//	assert (ioaddr > 0);
//	iounmap (ioaddr);

err_out_free_res:
	pci_release_regions (pdev);

err_out:
	unregister_netdev (dev);
	kfree (dev);
	return rc;
}

// BAFFFALO added
//======================================================================================
static int rtl8169_proc_status(char *buffer,
		  	char **buffer_location,
		  	off_t offset,
		  	int buffer_length,
		  	int *eof,
		  	void *data)
{
	char *buf = buffer;
	struct net_device *dev = (struct net_device *)data;
	char *strlink, enumlink[3][9] = {"10Mbps", "100Mbps", "1000Mbps"};
	char *strduplex, enumduplex[2][5] = {"half", "full"};
	int option, i;
	unsigned short eeprom_head = 0x0000;
	unsigned int regs, ee_addr_size;

	RTL8169_READ_GMII_REG(dev->base_addr, PHY_STAT_REG);

	option = readb(dev->base_addr + PHYstatus);
	if( option & _1000bpsF ){
		strlink = enumlink[2];
		strduplex = enumduplex[1];
	}
	else{
		strlink = (option & _100bps) ? enumlink[1] : enumlink[0];
		strduplex = (option & FullDup) ? enumduplex[1] : enumduplex[0];
	}
	
	buf += sprintf( buf, "MAC=%2.2x:%2.2x:%2.2x:%2.2x:%2.2x:%2.2x\n",
				dev->dev_addr[0], dev->dev_addr[1], dev->dev_addr[2], dev->dev_addr[3], dev->dev_addr[4], dev->dev_addr[5] );

	buf += sprintf( buf, "ioaddr=0x%8.8lx\n",  (unsigned long)dev->base_addr );
	buf += sprintf( buf, "irq=%d\n",  dev->irq );
	buf += sprintf( buf, "link=%s\n", strlink );
	buf += sprintf( buf, "duplex=%s\n", strduplex );
#ifdef RTL8169_JUMBO_FRAME_SUPPORT
	buf += sprintf( buf, "jumboframe=enable\n" );
	buf += sprintf( buf, "mtu=%d\n", dev->mtu );
#else
	buf += sprintf( buf, "jumboframe=disable\n" );
	buf += sprintf( buf, "mtu=%d\n", dev->mtu );
#endif

	/* read MAC address from EEPROM */
	regs = dev->base_addr;
// BUFFALO 2004.3.5 changed for support 93C56/66
	ee_addr_size = rtl8169_read_eeprom(regs, 0xff, 8) ? 8 : 6;


	eeprom_head = (u16 *)cpu_to_le16(rtl8169_read_eeprom (regs, 0, ee_addr_size));
	if (eeprom_head == 0x2981)
		buf += sprintf( buf, "eeprom=ready\n" );
	else
		buf += sprintf( buf, "eeprom=null\n" );

	buf += sprintf( buf, "eeprom_type=%s\n", (ee_addr_size == 8) ? "93C56/66" : "93C46" );
	for (i = 0; i < 64; i+=8)
		buf += sprintf( buf, "%4x %4x %4x %4x %4x %4x %4x %4x\n",
		             (u16 *)cpu_to_le16(rtl8169_read_eeprom (regs, i, ee_addr_size)),
		             (u16 *)cpu_to_le16(rtl8169_read_eeprom (regs, i+1, ee_addr_size)),
		             (u16 *)cpu_to_le16(rtl8169_read_eeprom (regs, i+2, ee_addr_size)),
		             (u16 *)cpu_to_le16(rtl8169_read_eeprom (regs, i+3, ee_addr_size)),
		             (u16 *)cpu_to_le16(rtl8169_read_eeprom (regs, i+4, ee_addr_size)),
		             (u16 *)cpu_to_le16(rtl8169_read_eeprom (regs, i+5, ee_addr_size)),
		             (u16 *)cpu_to_le16(rtl8169_read_eeprom (regs, i+6, ee_addr_size)),
		             (u16 *)cpu_to_le16(rtl8169_read_eeprom (regs, i+7, ee_addr_size)));

	return (buf - buffer);
}

static void rtl8169_init_proc(struct net_device *dev)
{
	if(rtl8169_Proc_root == NULL)
		rtl8169_Proc_root = proc_mkdir("rtl8169", NULL);
	if(rtl8169_Proc_root != NULL){
		rtl8169_Proc = create_proc_read_entry(
				"status",0644, rtl8169_Proc_root, rtl8169_proc_status, (void *)dev);
	}
}

static void rtl8169_remove_proc(struct net_device *dev)
{
	if(rtl8169_Proc_root != NULL){
		remove_proc_entry("status", rtl8169_Proc_root );
		remove_proc_entry("rtl8169", NULL );
	}
}
//======================================================================================

/* Serial EEPROM section. */
/*  EEPROM_Ctrl bits. */
#define EE_SHIFT_CLK	0x04	/* EEPROM shift clock. */
#define EE_CS			0x08	/* EEPROM chip select. */
#define EE_DATA_WRITE	0x02	/* EEPROM chip data in. */
#define EE_WRITE_0		0x00
#define EE_WRITE_1		0x02
#define EE_DATA_READ	0x01	/* EEPROM chip data out. */
#define EE_ENB			(0x80 | EE_CS)

/* Delay between EEPROM clock transitions.
   No extra delay is needed with 33Mhz PCI, but 66Mhz may change this.
 */

#define eeprom_delay()	readl(ee_addr)

/* The EEPROM commands include the alway-set leading bit. */
#define EE_WRITE_CMD	(5)
#define EE_READ_CMD		(6)
#define EE_ERASE_CMD	(7)
#define	EE_WRITEEN_CMD	0x13
#define	EE_WRITEDS_CMD	0x10

//================================================================================
static int rtl8169_read_eeprom (void *ioaddr, int location, int addr_len)
{
	int i;
	unsigned retval = 0;
	void *ee_addr = ioaddr + Cfg9346;
	int read_cmd = location | (EE_READ_CMD << addr_len);

	writeb (EE_ENB & ~EE_CS, ee_addr);
	writeb (EE_ENB, ee_addr);
	eeprom_delay ();

	/* Shift the read command bits out. */
//	for (i = 4 + addr_len; i >= 0; i--) {
	for (i = 2 + addr_len; i >= 0; i--) {
		int dataval = (read_cmd & (1 << i)) ? EE_DATA_WRITE : 0;
		writeb (EE_ENB | dataval, ee_addr);
		eeprom_delay ();
		writeb (EE_ENB | dataval | EE_SHIFT_CLK, ee_addr);
		eeprom_delay ();
	}
	writeb (EE_ENB, ee_addr);
	eeprom_delay ();

	for (i = 16; i > 0; i--) {
		writeb (EE_ENB | EE_SHIFT_CLK, ee_addr);
		eeprom_delay ();
		retval =
		    (retval << 1) | ((readb (ee_addr) & EE_DATA_READ) ? 1 :
				     0);
		writeb (EE_ENB, ee_addr);
		eeprom_delay ();
	}

	/* Terminate the EEPROM access. */
	writeb (~EE_CS, ee_addr);
	eeprom_delay ();

	return retval;
}

//================================================================================
static int rtl8169_write_eeprom(long ioaddr, int location, int addr_len, short data)
{
	int i;
	unsigned retval = 0;
	long ee_addr = ioaddr + Cfg9346;
	int cmd = location | (EE_WRITE_CMD << addr_len);

	rtl8169_write_eeprom_en(ioaddr, addr_len);
	writeb(EE_ENB & ~EE_CS, ee_addr);
	writeb(EE_ENB, ee_addr);

	/* Shift the read command bits out. */
	for (i = 4 + addr_len; i >= 0; i--) {
//	for (i = 2 + addr_len; i >= 0; i--) {
		short dataval = (cmd & (1 << i)) ? EE_DATA_WRITE : 0;
		writeb(EE_ENB | dataval, ee_addr);
		eeprom_delay();
		writeb(EE_ENB | dataval | EE_SHIFT_CLK, ee_addr);
		eeprom_delay();
		retval = (retval << 1) | ((inl(ee_addr) & EE_DATA_READ) ? 1 : 0);
	}

	for (i = 15; i >= 0; i--) {
		short dataval = (data & (1 << i)) ? EE_DATA_WRITE : 0;
		writeb(EE_ENB | dataval, ee_addr);
		eeprom_delay();
		writeb(EE_ENB | dataval | EE_SHIFT_CLK, ee_addr);
		eeprom_delay();
		retval = (retval << 1) | ((inl(ee_addr) & EE_DATA_READ) ? 1 : 0);
	}
	
	/* Terminate the EEPROM access. */
	writeb(EE_ENB & ~EE_CS, ee_addr);
	//eeprom_delay();
	writeb(EE_ENB, ee_addr);
	// wait ready
	for (i = 10000; i > 0; i--) {
		udelay(1);
		if (inl(ee_addr) & EE_DATA_READ) {
			break;
		}
	}
	writeb(EE_ENB & ~EE_CS, ee_addr);
	if (i == 0) {
		return 0;
	}
	
	rtl8169_write_eeprom_ds(ioaddr, addr_len);
	return retval;
}

//================================================================================
static int rtl8169_write_eeprom_en(long ioaddr, int addr_len)
{
	int i;
	unsigned retval = 0;
	long ee_addr = ioaddr + Cfg9346;
	int cmd = EE_WRITEEN_CMD << (addr_len - 2);

	//	Write Enable
	writeb(EE_ENB & ~EE_CS, ee_addr);
	writeb(EE_ENB, ee_addr);

	/* Shift the read command bits out. */
	for (i = 2 + addr_len; i >= 0; i--) {
		short dataval = (cmd & (1 << i)) ? EE_DATA_WRITE : 0;
		writeb(EE_ENB | dataval, ee_addr);
		eeprom_delay();
		writeb(EE_ENB | dataval | EE_SHIFT_CLK, ee_addr);
		eeprom_delay();
		retval = (retval << 1) | ((inl(ee_addr) & EE_DATA_READ) ? 1 : 0);
	}
	writeb(EE_ENB & ~EE_CS, ee_addr);
	return retval;
}

//================================================================================
static int rtl8169_write_eeprom_ds(long ioaddr, int addr_len)
{
	int i;
	unsigned retval = 0;
	long ee_addr = ioaddr + Cfg9346;
	int cmd = EE_WRITEDS_CMD << (addr_len - 2);

	//	Write Enable
	writeb(EE_ENB & ~EE_CS, ee_addr);
	writeb(EE_ENB, ee_addr);

	/* Shift the read command bits out. */
	for (i = 2 + addr_len; i >= 0; i--) {
		short dataval = (cmd & (1 << i)) ? EE_WRITEDS_CMD : 0;
		writeb(EE_ENB | dataval, ee_addr);
		eeprom_delay();
		writeb(EE_ENB | dataval | EE_SHIFT_CLK, ee_addr);
		eeprom_delay();
		retval = (retval << 1) | ((inl(ee_addr) & EE_DATA_READ) ? 1 : 0);
	}
	writeb(EE_ENB & ~EE_CS, ee_addr);
	return retval;
}

//======================================================================================================
int rtl8169_ioctl(struct net_device *dev, struct ifreq *ifr, int cmd)
{

	struct rtl8169_private *priv = dev->priv;
	long ioaddr = dev->base_addr;
	long flags;

	switch (cmd) {
	case SIOCETHTOOL:
		//return netdev_ethtool_ioctl(dev, (void *) rq->ifr_data);
	case SIOCGMIIPHY:		/* Get address of MII PHY in use. */
	case SIOCGMIIREG:		/* Read MII PHY register. */
	case SIOCSMIIREG:		/* Write MII PHY register. */
// EEPROM ֘A
	case	MELCO_ETH_IOCTL:
	{
		struct melco_eth_req *mreq = (struct melco_eth_req *)ifr;
		int rval, i;
		int ee_addr_size = rtl8169_read_eeprom(ioaddr, 0xff, 8) ? 8 : 6;

		switch (mreq->command) {
		case	MELCO_ETH_IOCTL_SET_PHY_MAC:
			break;
		case	MELCO_ETH_IOCTL_GET_PHY_MAC:
			break;
		case	MELCO_ETH_IOCTL_SET_MAC:
			break;
		case	MELCO_ETH_IOCTL_GET_MAC:
			break;
		case	MELCO_ETH_IOCTL_INIT_EEPROM:
			{
				//int rval = verify_area(VERIFY_READ, mreq->u.mac.mac, mreq->u.mac.len);
				short *mac;
				unsigned short data;
				int nomac = 0, rval;
				unsigned long write_sum = 0;

				//printk("\nMELCO_ETH_IOCTL_INIT_EEPROM : enter\n");
				if (mreq->u.mac.mac == NULL) {
					nomac = 1;
					mac = NULL;
				} else
					mac = (short*)mreq->u.mac.mac;

				for (i = 0; i < 64; i++) {
					data = 0x0000;
					rval = 0;
					switch (i) {
						case 0:	// EEPROM code
							data = 0x2981;
							break;
						case 1:	// Vender ID
							data = 0xEC10;
							break;
						case 2:	// Device ID
							data = 0x6981;
							break;
						case 5: // Min GNT, Max LAT
							data = 0x2040;
							break;	
						case 6: // CONFIG
							data = 0x0101;
							break;	
						case 7:
							if (!nomac) {
								data = mac[0];
							}
							break;
						case 8:
							if (!nomac) {
								data = mac[1];
							}
							break;
						case 9:
							if (!nomac) {
								data = mac[2];
							}
							break;
#if defined(CONFIG_HTGL) // 1.8n5
						case 10:
							data = 0x004C;
							break;
#else
						case 10:
							data = 0x00CC;
							break;
#endif
						case 15:
							data = 0x0001;
							break;
					}
					if (i == 7 || i == 8 || i == 9) {
						if (nomac) continue;
					}
					//printk("write eeprpm %d: %4x\n", i, data);
					write_sum += data;
					udelay(10000);	// 10ms҂
					spin_lock_irqsave(&priv->lock, flags);
					rval = rtl8169_write_eeprom(ioaddr, i, ee_addr_size, cpu_to_le16(data));
					spin_unlock_irqrestore(&priv->lock, flags);
					if (i == 0) {	// 񂾂2񏑂
						udelay(1000);	// 1ms҂
						spin_lock_irqsave(&priv->lock, flags);
						rval = rtl8169_write_eeprom(ioaddr, i, ee_addr_size, cpu_to_le16(data));
						spin_unlock_irqrestore(&priv->lock, flags);
					}
					if (rval==0) {
						;//printk("RTL8169: write eeprom count %d: timeout\n", i);
					}
				}
				//printk("write_sum: %8x\n", write_sum);
			}
			return 0;
			break;
		case MELCO_ETH_IOCTL_ERASE_EEPROM:
			{
			// BUFFALO changed 2004.3.5 for support 93C56/66
			//	int ee_addr_size = 6;// 93C46 16bitp
				int ee_addr_size = rtl8169_read_eeprom(ioaddr, 0xff, 8) ? 8 : 6;
				short data = 0xFFFF;
				int i;
				
				for (i = 0; i < 64; i++) {
					udelay(10000);
					spin_lock_irqsave(&priv->lock, flags);
					rval = rtl8169_write_eeprom(ioaddr, i, ee_addr_size, le16_to_cpu(data));
					spin_unlock_irqrestore(&priv->lock, flags);
					if (rval==0) {
						printk("RTL8169: erase eeprom count %d: timeout\n", i);
					}
				}
			}
			return 0;
			break;
		}
	}
		break;
	default:
		return -EOPNOTSUPP;
	}

	return -EOPNOTSUPP;
}


//=======================================================================================
static int __devinit rtl8169_init_one (struct pci_dev *pdev, const struct pci_device_id *ent)
{
	struct net_device *dev = NULL;
	struct rtl8169_private *priv = NULL;
	void *ioaddr = NULL;
	static int board_idx = -1;
	int i;
	int option = -1, Cap10_100 = 0, Cap1000 = 0;


	assert (pdev != NULL);
	assert (ent != NULL);

	board_idx++;


	i = rtl8169_init_board (pdev, &dev, &ioaddr);
	if (i < 0) {
		return i;
	}

	priv = dev->priv;

	assert (ioaddr != NULL);
	assert (dev != NULL);
	assert (priv != NULL);

	// Get MAC address //
	for (i = 0; i < MAC_ADDR_LEN ; i++){
		dev->dev_addr[i] = RTL_R8( MAC0 + i );
	}

	// BUFFALO added
	rtl8169_init_proc(dev);

	dev->open		= rtl8169_open;
	dev->hard_start_xmit 	= rtl8169_start_xmit;
	dev->get_stats	= rtl8169_get_stats;
	dev->stop 		= rtl8169_close;
	dev->tx_timeout	= rtl8169_tx_timeout;
	dev->set_multicast_list = rtl8169_set_rx_mode;
	dev->watchdog_timeo 	= TX_TIMEOUT;
	dev->irq 		= pdev->irq;
	dev->base_addr 	= (unsigned long) ioaddr;

#ifdef RTL8169_JUMBO_FRAME_SUPPORT
	dev->change_mtu	= rtl8169_change_mtu;
#endif //end #ifdef RTL8169_JUMBO_FRAME_SUPPORT

	dev->do_ioctl 	= rtl8169_ioctl;

	priv 			= dev->priv;	// private data //
	priv->pci_dev 	= pdev;
	priv->mmio_addr = ioaddr;

	priv->curr_mtu_size = dev->mtu;
	priv->tx_pkt_len = dev->mtu + ETH_HDR_LEN;
	priv->rx_pkt_len = dev->mtu + ETH_HDR_LEN;
	priv->hw_rx_pkt_len = priv->rx_pkt_len + 8;

	DBG_PRINT("-------------------------- \n");
	DBG_PRINT("dev->mtu = %d \n", dev->mtu);
	DBG_PRINT("priv->curr_mtu_size = %d \n", priv->curr_mtu_size);
	DBG_PRINT("priv->tx_pkt_len = %d \n", priv->tx_pkt_len);
	DBG_PRINT("priv->rx_pkt_len = %d \n", priv->rx_pkt_len);
	DBG_PRINT("priv->hw_rx_pkt_len = %d \n", priv->hw_rx_pkt_len);
	DBG_PRINT("-------------------------- \n");

	spin_lock_init (&priv->lock);

	pdev->driver_data = dev;

	printk (KERN_DEBUG "%s: Identified chip type is '%s'.\n",dev->name,rtl_chip_info[priv->chipset].name);
	printk (KERN_INFO "%s: %s at 0x%lx, "
				"%2.2x:%2.2x:%2.2x:%2.2x:%2.2x:%2.2x, "
				"IRQ %d\n",
				dev->name,
				RTL8169_DRIVER_NAME,
				dev->base_addr,
				dev->dev_addr[0], dev->dev_addr[1],
				dev->dev_addr[2], dev->dev_addr[3],
				dev->dev_addr[4], dev->dev_addr[5],
				dev->irq);

	
	// Config PHY
	rtl8169_hw_PHY_config(dev);

	DBG_PRINT("Set MAC Reg C+CR Offset 0x82h = 0x01h\n");
	RTL_W8( 0x82, 0x01 );


	if( priv->mac_version < RTL_GIGA_MAC_VER_E ){
		DBG_PRINT("Set PCI Latency=0x40\n");
		pci_write_config_byte(pdev, PCI_LATENCY_TIMER, 0x40);
	}


	if( priv->mac_version == RTL_GIGA_MAC_VER_D ){
		DBG_PRINT("Set MAC Reg C+CR Offset 0x82h = 0x01h\n");
		RTL_W8( 0x82, 0x01 );
		DBG_PRINT("Set PHY Reg 0x0bh = 0x00h\n");
		RTL8169_WRITE_GMII_REG( ioaddr, 0x0b, 0x0000 );	//w 0x0b 15 0 0
	}

	// if TBI is not endbled
	if( !(RTL_R8(PHYstatus) & TBI_Enable) ){
		int	val = RTL8169_READ_GMII_REG( ioaddr, PHY_AUTO_NEGO_REG );

		option = (board_idx >= MAX_UNITS) ? 0 : media[board_idx];
		// Force RTL8169 in 10/100/1000 Full/Half mode.
		if( option > 0 ){
			printk(KERN_INFO "%s: Force-mode Enabled. \n", dev->name);
			Cap10_100 = 0;
			Cap1000 = 0;
			switch( option ){
				case _10_Half:
						Cap10_100 = PHY_Cap_10_Half;
						Cap1000 = PHY_Cap_Null;
						break;
				case _10_Full:
						Cap10_100 = PHY_Cap_10_Full | PHY_Cap_10_Half;
						Cap1000 = PHY_Cap_Null;
						break;
				case _100_Half:
						Cap10_100 = PHY_Cap_100_Half | PHY_Cap_10_Full | PHY_Cap_10_Half;
						Cap1000 = PHY_Cap_Null;
						break;
				case _100_Full:
						Cap10_100 = PHY_Cap_100_Full | PHY_Cap_100_Half | PHY_Cap_10_Full | PHY_Cap_10_Half;
						Cap1000 = PHY_Cap_Null;
						break;
				case _1000_Full:
						Cap10_100 = PHY_Cap_100_Full | PHY_Cap_100_Half | PHY_Cap_10_Full | PHY_Cap_10_Half;
						Cap1000 = PHY_Cap_1000_Full;
						break;
				default:
						break;
			}
			RTL8169_WRITE_GMII_REG( ioaddr, PHY_AUTO_NEGO_REG, Cap10_100 | ( val&0x1F ) );	//leave PHY_AUTO_NEGO_REG bit4:0 unchanged
			RTL8169_WRITE_GMII_REG( ioaddr, PHY_1000_CTRL_REG, Cap1000 );
		}
		else{
			printk(KERN_INFO "%s: Auto-negotiation Enabled.\n", dev->name);

			// enable 10/100 Full/Half Mode, leave PHY_AUTO_NEGO_REG bit4:0 unchanged
			RTL8169_WRITE_GMII_REG( ioaddr, PHY_AUTO_NEGO_REG,
				PHY_Cap_10_Half | PHY_Cap_10_Full | PHY_Cap_100_Half | PHY_Cap_100_Full | ( val&0x1F ) );

			// enable 1000 Full Mode
			RTL8169_WRITE_GMII_REG( ioaddr, PHY_1000_CTRL_REG, PHY_Cap_1000_Full );

		}// end of if( option > 0 )

		// Enable auto-negotiation and restart auto-nigotiation
		RTL8169_WRITE_GMII_REG( ioaddr, PHY_CTRL_REG, PHY_Enable_Auto_Nego | PHY_Restart_Auto_Nego );
		udelay(100);

		// wait for auto-negotiation process
		for( i = 10000; i > 0; i-- ){
			//check if auto-negotiation complete
			if( RTL8169_READ_GMII_REG(ioaddr, PHY_STAT_REG) & PHY_Auto_Neco_Comp ){
				udelay(100);
				option = RTL_R8(PHYstatus);
#if defined(CONFIG_HTGL) // 1.8n5
				if( option & _1000bpsF ){
					printk(KERN_INFO "%s: 1000Mbps Full-duplex operation.\n", dev->name);
					mdelay(10);
					writeb(0x93, 0x80004500); // 1000Mbps up
					mdelay(10);
					writeb(0x93, 0x80004500); // 1000Mbps up
					mdelay(10);
					writeb(0x93, 0x80004500); // 1000Mbps up
					mdelay(10);
					writeb(0x93, 0x80004500); // 1000Mbps up
				} else if( option & _100bps ) {
					printk(KERN_INFO "%s: 100Mbps %s-duplex operation.\n", dev->name,
							(option & FullDup) ? "Full" : "Half" );
					mdelay(10);
					writeb(0x91, 0x80004500); // 100Mbps up
					mdelay(10);
					writeb(0x91, 0x80004500); // 100Mbps up
					mdelay(10);
					writeb(0x91, 0x80004500); // 100Mbps up
					mdelay(10);
					writeb(0x91, 0x80004500); // 100Mbps up
				} else if( option & _10bps ) {
					printk(KERN_INFO "%s: 10Mbps %s-duplex operation.\n", dev->name,
							(option & FullDup) ? "Full" : "Half" );
					mdelay(10);
					writeb(0x8F, 0x80004500); // 10Mbps up
					mdelay(10);
					writeb(0x8F, 0x80004500); // 10Mbps up
					mdelay(10);
					writeb(0x8F, 0x80004500); // 10Mbps up
					mdelay(10);
					writeb(0x8F, 0x80004500); // 10Mbps up
				}

				writeb(0x5F, 0x80004500); // _
				writeb(0x5F, 0x80004500); // _
				mdelay(100);
#else
				if( option & _1000bpsF ){
					printk(KERN_INFO "%s: 1000Mbps Full-duplex operation.\n", dev->name);
					out_8(0x80004500, 0x5D);	// FAN]
					out_8(0x80004500, 0x5D);	// FAN]
					out_8(0x80004500, 0x5D);	// FAN]
					out_8(0x80004500, 0x5D);	// FAN]
				}
				else{
					printk(KERN_INFO "%s: %sMbps %s-duplex operation.\n", dev->name,
							(option & _100bps) ? "100" : "10", (option & FullDup) ? "Full" : "Half" );
					out_8(0x80004500, 0x5C);	// FANᑬ]
					out_8(0x80004500, 0x5C);	// FANᑬ]
					out_8(0x80004500, 0x5C);	// FANᑬ]
					out_8(0x80004500, 0x5c);	// FANᑬ]
				}
#endif
				break;
			}
			else{
				udelay(100);
			}// end of if( RTL8169_READ_GMII_REG(ioaddr, 1) & 0x20 )
		}// end for-loop to wait for auto-negotiation process

	}// end of TBI is not enabled
	else{
		udelay(100);
		printk(KERN_INFO "%s: 1000Mbps Full-duplex operation, TBI Link %s!\n",
							dev->name, (RTL_R32(TBICSR) & TBILinkOK) ? "OK" : "Failed" );

	}// end of TBI is not enabled

	return 0;
}







//======================================================================================================
static void __devexit rtl8169_remove_one (struct pci_dev *pdev)
{
	struct net_device *dev = pdev->driver_data;
	struct rtl8169_private *priv = (struct rtl8169_private *) (dev->priv);

	assert (dev != NULL);
	assert (priv != NULL);

	unregister_netdev (dev);
	iounmap (priv->mmio_addr);
	pci_release_regions (pdev);

	// poison memory before freeing
	memset (dev, 0xBC, sizeof (struct net_device) +	sizeof (struct rtl8169_private));

	// BUFFALO added
	rtl8169_init_proc(dev);

	kfree (dev);
	pdev->driver_data = NULL;
}







//======================================================================================================
static int rtl8169_open (struct net_device *dev)
{
	struct rtl8169_private *priv = dev->priv;
	int retval;
	u8 diff;
	u32 TxPhyAddr, RxPhyAddr;


	if( priv->drvinit_fail == 1 ){
		printk("%s: Gigabit driver open failed.\n", dev->name );
		return -ENOMEM;
	}

	retval = request_irq (dev->irq, rtl8169_interrupt, SA_SHIRQ, dev->name, dev);
	if (retval) {
		return retval;
	}


	//////////////////////////////////////////////////////////////////////////////
	priv->TxDescArrays = kmalloc( NUM_TX_DESC * sizeof(struct TxDesc)+256 , GFP_KERNEL );
	// Tx Desscriptor needs 256 bytes alignment;
	TxPhyAddr = virt_to_bus(priv->TxDescArrays);
	diff = 256 - (TxPhyAddr-((TxPhyAddr >> 8)<< 8));
	TxPhyAddr += diff;
	priv->TxDescArray = (struct TxDesc *)(priv->TxDescArrays + diff);

	priv->RxDescArrays = kmalloc( NUM_RX_DESC * sizeof(struct RxDesc)+256 , GFP_KERNEL );
	// Rx Desscriptor needs 256 bytes alignment;
	RxPhyAddr = virt_to_bus(priv->RxDescArrays);
	diff = 256 - (RxPhyAddr-((RxPhyAddr >> 8)<< 8));
	RxPhyAddr += diff;
	priv->RxDescArray = (struct RxDesc *)(priv->RxDescArrays + diff);

	if ( priv->TxDescArrays == NULL || priv->RxDescArrays == NULL ) {
		printk(KERN_INFO"Allocate RxDescArray or TxDescArray failed\n");
		free_irq(dev->irq, dev);
		if (priv->TxDescArrays) kfree(priv->TxDescArrays);
		if (priv->RxDescArrays) kfree(priv->RxDescArrays);
		return -ENOMEM;
	}

	{
        int i;
		struct sk_buff *skb = NULL;

		for(i=0;i<NUM_RX_DESC;i++){
			skb = RTL8169_ALLOC_RXSKB(MAX_RX_SKBDATA_SIZE);
			if( skb != NULL ) {
				skb_reserve (skb, 2);	// 16 byte align the IP fields. //
				priv->Rx_skbuff[i] = skb;
			}
			else{
				printk("%s: Gigabit driver failed to allocate skbuff.\n", dev->name);
				priv->drvinit_fail = 1;
			}
		}
	}


	//////////////////////////////////////////////////////////////////////////////
	rtl8169_init_ring (dev);
	rtl8169_hw_start (dev);

	// ------------------------------------------------------
	if( (priv->mac_version > RTL_GIGA_MAC_VER_B) && (priv->phy_version < RTL_GIGA_PHY_VER_G) ){
		DBG_PRINT("FIX PCS -> rtl8169_request_timer\n");
		rtl8169_request_timer( (&priv->phy_timer_t), 100, rtl8169_phy_timer_t_handler, ((void *)dev) );  //in open()
		priv->phy_link_down_cnt = 0;
	}

	DBG_PRINT("%s: %s() alloc_rxskb_cnt = %d\n", dev->name, __FUNCTION__, alloc_rxskb_cnt );

	return 0;

}//end of rtl8169_open (struct net_device *dev)

//=======================================================================================
static void rtl8169_hw_PHY_reset(struct net_device *dev)
{
	int val, phy_reset_expiretime = 50;
	struct rtl8169_private *priv = dev->priv;
	void *ioaddr = priv->mmio_addr;

	DBG_PRINT("%s: Reset RTL8169s PHY\n", dev->name);

	val = ( RTL8169_READ_GMII_REG( ioaddr, 0 ) | 0x8000 ) & 0xffff;
	RTL8169_WRITE_GMII_REG( ioaddr, 0, val );

	do //waiting for phy reset
	{
		if( RTL8169_READ_GMII_REG( ioaddr, 0 ) & 0x8000 ){
			phy_reset_expiretime --;
			udelay(100);
		}
		else{
			break;
		}
	}while( phy_reset_expiretime >= 0 );

	assert( phy_reset_expiretime > 0 );
}




//=======================================================================================
static void rtl8169_hw_PHY_config (struct net_device *dev)
{
	struct rtl8169_private *priv = dev->priv;
	void *ioaddr = priv->mmio_addr;
	int val;

	RTL_PRINT_MAC_VERSION(priv->mac_version);
	RTL_PRINT_PHY_VERSION(priv->phy_version);

	if( (priv->mac_version > RTL_GIGA_MAC_VER_B) && ( priv->phy_version < RTL_GIGA_PHY_VER_F ) )
	{
			DBG_PRINT("MAC version!=0 && PHY version == 0 or 1\n");
			DBG_PRINT("Do final_reg2.cfg\n");
			// phy config for RTL8169s mac_version C chip
			RTL8169_WRITE_GMII_REG( ioaddr, 31, 0x0001 );	//w 31 2 0 1
			RTL8169_WRITE_GMII_REG( ioaddr, 21, 0x1000 );	//w 21 15 0 1000
			RTL8169_WRITE_GMII_REG( ioaddr, 24, 0x65c7 );	//w 24 15 0 65c7
			RTL8169_WRITE_GMII_REG_BIT( ioaddr, 4, 11, 0 );	//w 4 11 11 0
			val = RTL8169_READ_GMII_REG( ioaddr, 4 ) & 0x0fff;
			RTL8169_WRITE_GMII_REG( ioaddr, 4, val );	//w 4 15 12 0
			RTL8169_WRITE_GMII_REG( ioaddr, 3, 0x00a1 );	//w 3 15 0 00a1
			RTL8169_WRITE_GMII_REG( ioaddr, 2, 0x0008 );	//w 2 15 0 0008
			RTL8169_WRITE_GMII_REG( ioaddr, 1, 0x1020 );	//w 1 15 0 1020
			RTL8169_WRITE_GMII_REG( ioaddr, 0, 0x1000 );	//w 0 15 0 1000
			RTL8169_WRITE_GMII_REG_BIT( ioaddr, 4, 11, 1 );	//w 4 11 11 1
			RTL8169_WRITE_GMII_REG_BIT( ioaddr, 4, 11, 0 );	//w 4 11 11 0
			val = ( RTL8169_READ_GMII_REG( ioaddr, 4 ) & 0x0fff ) | 0x7000;
			RTL8169_WRITE_GMII_REG( ioaddr, 4, val );	//w 4 15 12 7
			RTL8169_WRITE_GMII_REG( ioaddr, 3, 0xff41 );	//w 3 15 0 ff41
			RTL8169_WRITE_GMII_REG( ioaddr, 2, 0xde60 );	//w 2 15 0 de60
			RTL8169_WRITE_GMII_REG( ioaddr, 1, 0x0140 );	//w 1 15 0 0140
			RTL8169_WRITE_GMII_REG( ioaddr, 0, 0x0077 );	//w 0 15 0 0077
			RTL8169_WRITE_GMII_REG_BIT( ioaddr, 4, 11, 1 );	//w 4 11 11 1
			RTL8169_WRITE_GMII_REG_BIT( ioaddr, 4, 11, 0 );	//w 4 11 11 0
			val = ( RTL8169_READ_GMII_REG( ioaddr, 4 ) & 0x0fff ) | 0xa000;
			RTL8169_WRITE_GMII_REG( ioaddr, 4, val );	//w 4 15 12 a
			RTL8169_WRITE_GMII_REG( ioaddr, 3, 0xdf01 );	//w 3 15 0 df01
			RTL8169_WRITE_GMII_REG( ioaddr, 2, 0xdf20 );	//w 2 15 0 df20
			RTL8169_WRITE_GMII_REG( ioaddr, 1, 0xff95 );	//w 1 15 0 ff95
			RTL8169_WRITE_GMII_REG( ioaddr, 0, 0xfa00 );	//w 0 15 0 fa00
			RTL8169_WRITE_GMII_REG_BIT( ioaddr, 4, 11, 1 );	//w 4 11 11 1
			RTL8169_WRITE_GMII_REG_BIT( ioaddr, 4, 11, 0 );	//w 4 11 11 0
			val = ( RTL8169_READ_GMII_REG( ioaddr, 4 ) & 0x0fff ) | 0xb000;
			RTL8169_WRITE_GMII_REG( ioaddr, 4, val );	//w 4 15 12 b
			RTL8169_WRITE_GMII_REG( ioaddr, 3, 0xff41 );	//w 3 15 0 ff41
			RTL8169_WRITE_GMII_REG( ioaddr, 2, 0xde20 );	//w 2 15 0 de20
			RTL8169_WRITE_GMII_REG( ioaddr, 1, 0x0140 );	//w 1 15 0 0140
			RTL8169_WRITE_GMII_REG( ioaddr, 0, 0x00bb );	//w 0 15 0 00bb
			RTL8169_WRITE_GMII_REG_BIT( ioaddr, 4, 11, 1 );	//w 4 11 11 1
			RTL8169_WRITE_GMII_REG_BIT( ioaddr, 4, 11, 0 );	//w 4 11 11 0
			val = ( RTL8169_READ_GMII_REG( ioaddr, 4 ) & 0x0fff ) | 0xf000;
			RTL8169_WRITE_GMII_REG( ioaddr, 4, val );	//w 4 15 12 f
			RTL8169_WRITE_GMII_REG( ioaddr, 3, 0xdf01 );	//w 3 15 0 df01
			RTL8169_WRITE_GMII_REG( ioaddr, 2, 0xdf20 );	//w 2 15 0 df20
			RTL8169_WRITE_GMII_REG( ioaddr, 1, 0xff95 );	//w 1 15 0 ff95
			RTL8169_WRITE_GMII_REG( ioaddr, 0, 0xbf00 );	//w 0 15 0 bf00
			RTL8169_WRITE_GMII_REG_BIT( ioaddr, 4, 11, 1 );	//w 4 11 11 1
			RTL8169_WRITE_GMII_REG_BIT( ioaddr, 4, 11, 0 );	//w 4 11 11 0
			RTL8169_WRITE_GMII_REG( ioaddr, 31, 0x0000 );	//w 31 2 0 0
	}
}

//=======================================================================================
static void rtl8169_hw_start (struct net_device *dev)
{
	struct rtl8169_private *priv = dev->priv;
	void *ioaddr = priv->mmio_addr;
	u32 i;


	/* Soft reset the chip. */
	RTL_W8 ( ChipCmd, CmdReset);

	/* Check that the chip has finished the reset. */
	for (i = 1000; i > 0; i--){
		if ((RTL_R8( ChipCmd ) & CmdReset) == 0) break;
		else udelay (10);
	}

	RTL_W8 ( Cfg9346, Cfg9346_Unlock);
	RTL_W8 ( ChipCmd, CmdTxEnb | CmdRxEnb);
	RTL_W8 ( ETThReg, ETTh);

	// For gigabit rtl8169
	RTL_W16	( RxMaxSize, (unsigned short)priv->hw_rx_pkt_len );
	DBG_PRINT("RTL_W16( RxMaxSize, %d )\n", priv->hw_rx_pkt_len);

//	RTL_W16	( RxMaxSize, 0x0800 );
//	printk("RTL_W16( RxMaxSize, 0x%04x )\n", RTL_R16 ( RxMaxSize ) );
	
	// Set Rx Config register
	i = rtl8169_rx_config | ( RTL_R32( RxConfig ) & rtl_chip_info[priv->chipset].RxConfigMask);
	RTL_W32 ( RxConfig, i);


	/* Set DMA burst size and Interframe Gap Time */
	RTL_W32 ( TxConfig, (TX_DMA_BURST << TxDMAShift) | (InterFrameGap << TxInterFrameGapShift) );



	RTL_W16( CPlusCmd, 0x0000 );
//	RTL_W16( CPlusCmd, RTL_R16(CPlusCmd) );

	//2003-07-18
	if( priv->mac_version == RTL_GIGA_MAC_VER_D ){
		DBG_PRINT("Set MAC Reg C+CR Offset 0xE0: bit-3 and bit-14 MUST be 1\n");
		RTL_W16( CPlusCmd, (RTL_R16(CPlusCmd)|(1<<14)|(1<<3)) );
	}

	{
		//RTL_W16(0xE2, 0x1517);
		
		RTL_W16(0xE2, 0x0000);
		DBG_PRINT("RTL_R16(0xE2) = 0x%4x\n", RTL_R16(0xE2));
		//printk("RTL_R16(0xE2) = 0x%4x\n", RTL_R16(0xE2));
	}

	priv->cur_rx = 0;

	RTL_W32 ( TxDescStartAddr, virt_to_bus(priv->TxDescArray));
	RTL_W32 ( TxDescStartAddr+4, 0);
	RTL_W32 ( RxDescStartAddr, virt_to_bus(priv->RxDescArray));
	RTL_W32 ( RxDescStartAddr+4, 0);
	RTL_W8 ( Cfg9346, Cfg9346_Lock );
	udelay (10);

	RTL_W32 ( RxMissed, 0 );

	rtl8169_set_rx_mode (dev);

	/* no early-rx interrupts */
	RTL_W16 ( MultiIntr, RTL_R16(MultiIntr) & 0xF000);
                       
	/* Enable all known interrupts by setting the interrupt mask. */
	RTL_W16 ( IntrMask, rtl8169_intr_mask);

	netif_start_queue (dev);

}//end of rtl8169_hw_start (struct net_device *dev)


//=======================================================================================
static void rtl8169_init_ring (struct net_device *dev)
{
	struct rtl8169_private *priv = dev->priv;
	int i;

	priv->cur_rx = 0;
	priv->cur_tx = 0;
	priv->dirty_tx = 0;
	memset(priv->TxDescArray, 0x0, NUM_TX_DESC*sizeof(struct TxDesc));
	memset(priv->RxDescArray, 0x0, NUM_RX_DESC*sizeof(struct RxDesc));

	for (i=0 ; i<NUM_TX_DESC ; i++){
		priv->Tx_skbuff[i]=NULL;
	}
	for (i=0; i <NUM_RX_DESC; i++) {

//Jumbo frame
		if(i==(NUM_RX_DESC-1)){
			//priv->RxDescArray[i].status = (OWNbit | EORbit) | (unsigned short)priv->hw_rx_pkt_len;
			priv->RxDescArray[i].status = cpu_to_le32((OWNbit | EORbit) | (unsigned long)priv->hw_rx_pkt_len);//BUFFALO changed
		}
		else{
			//priv->RxDescArray[i].status = OWNbit | (unsigned short)priv->hw_rx_pkt_len;
			priv->RxDescArray[i].status = cpu_to_le32(OWNbit | (unsigned long)priv->hw_rx_pkt_len);//BUFFALO changed
		}

		{//-----------------------------------------------------------------------
			struct sk_buff *skb = priv->Rx_skbuff[i];

			if( skb != NULL ){
				//priv->RxDescArray[i].buf_addr = virt_to_bus( skb->data );
				priv->RxDescArray[i].buf_addr = cpu_to_le32(virt_to_bus( skb->data ));//BUFFALO changed
				priv->RxDescArray[i].buf_Haddr=0;       //BUFFALO add
			}
			else{
				DBG_PRINT("%s: %s() Rx_skbuff == NULL\n", dev->name, __FUNCTION__);
				priv->drvinit_fail = 1;
			}
		}//-----------------------------------------------------------------------

	}
}


//=======================================================================================
static void rtl8169_tx_clear (struct rtl8169_private *priv)
{
	int i;

	priv->cur_tx = 0;
	for ( i = 0 ; i < NUM_TX_DESC ; i++ ){
		if ( priv->Tx_skbuff[i] != NULL ) {
			dev_kfree_skb ( priv->Tx_skbuff[i] );
			priv->Tx_skbuff[i] = NULL;
			priv->stats.tx_dropped++;
		}
	}
}


//=======================================================================================
static void rtl8169_tx_timeout (struct net_device *dev)
{
	struct rtl8169_private *priv = dev->priv;
	void *ioaddr = priv->mmio_addr;
	u8 tmp8;

	/* disable Tx, if not already */
	tmp8 = RTL_R8( ChipCmd );
	if (tmp8 & CmdTxEnb){
		RTL_W8 ( ChipCmd, tmp8 & ~CmdTxEnb);
	}

	/* Disable interrupts by clearing the interrupt mask. */
	RTL_W16 ( IntrMask, 0x0000);

	/* Stop a shared interrupt from scavenging while we are. */
	spin_lock_irq (&priv->lock);
	rtl8169_tx_clear (priv);
	spin_unlock_irq (&priv->lock);

	/* ...and finally, reset everything */
	rtl8169_hw_start (dev);

	netif_wake_queue (dev);
}

//=======================================================================================
static int rtl8169_start_xmit (struct sk_buff *skb, struct net_device *dev)
{
	struct rtl8169_private *priv = dev->priv;
	void *ioaddr = priv->mmio_addr;
	int entry = priv->cur_tx % NUM_TX_DESC;
	int buf_len = 60;

	spin_lock_irq (&priv->lock);

	DBG_PRINT("================ \n");
	DBG_PRINT("skb->len = %d \n", skb->len);
	DBG_PRINT("================ \n");


//	if( (priv->TxDescArray[entry].status & OWNbit)==0 ){
	if( (le32_to_cpu(priv->TxDescArray[entry].status) & OWNbit)==0 ){ //BUFFALO changed

		priv->Tx_skbuff[entry] = skb;
		//priv->TxDescArray[entry].buf_addr = virt_to_bus(skb->data);
		priv->TxDescArray[entry].buf_addr = cpu_to_le32(virt_to_bus(skb->data)); //BUFFALO changed
//Jumbo frame
		if( skb->len <= priv->tx_pkt_len ){
			buf_len = skb->len;
		}
		else{
			printk("%s: Error -- Tx packet size(%d) > mtu(%d)+14\n", dev->name, skb->len, dev->mtu);
			buf_len = priv->tx_pkt_len;
		}

		if( entry != (NUM_TX_DESC-1) ){
			//priv->TxDescArray[entry].status = (OWNbit | FSbit | LSbit) | buf_len;
			priv->TxDescArray[entry].status = cpu_to_le32((OWNbit | FSbit | LSbit) | buf_len);
		}
		else{
		 	//priv->TxDescArray[entry].status = (OWNbit | EORbit | FSbit | LSbit) | buf_len;
			priv->TxDescArray[entry].status = cpu_to_le32((OWNbit | EORbit | FSbit | LSbit) | buf_len);
		}

		RTL_W8 ( TxPoll, 0x40);		//set polling bit

		dev->trans_start = jiffies;

		priv->stats.tx_bytes += ( (skb->len > ETH_ZLEN) ? skb->len : ETH_ZLEN);
		priv->cur_tx++;
	}//end of if( (priv->TxDescArray[entry].status & 0x80000000)==0 )

	spin_unlock_irq (&priv->lock);

	if ( (priv->cur_tx - NUM_TX_DESC) == priv->dirty_tx ){
		netif_stop_queue (dev);
	}
	else{
		if (netif_queue_stopped (dev)){
			netif_wake_queue (dev);
		}
	}

	return 0;
}

//=======================================================================================
static void rtl8169_tx_interrupt (struct net_device *dev, struct rtl8169_private *priv, void *ioaddr)
{
	unsigned long dirty_tx, tx_left=0;
	int entry = priv->cur_tx % NUM_TX_DESC;
    	int txloop_cnt = 0;

	assert (dev != NULL);
	assert (priv != NULL);
	assert (ioaddr != NULL);


	dirty_tx = priv->dirty_tx;
	tx_left = priv->cur_tx - dirty_tx;

	while( (tx_left > 0) && (txloop_cnt < 30) ){
		if( (le32_to_cpu(priv->TxDescArray[entry].status) & OWNbit) == 0 ){ //BUFFALO changed
			dev_kfree_skb_irq( priv->Tx_skbuff[dirty_tx % NUM_TX_DESC] );
			priv->Tx_skbuff[dirty_tx % NUM_TX_DESC] = NULL;
			priv->stats.tx_packets++;
			dirty_tx++;
			tx_left--;
			entry++;
		}
		txloop_cnt ++;
	}

	if (priv->dirty_tx != dirty_tx) {
		priv->dirty_tx = dirty_tx;
		if (netif_queue_stopped (dev))
			netif_wake_queue (dev);
	}
}

//=======================================================================================
static void rtl8169_rx_interrupt (struct net_device *dev, struct rtl8169_private *priv, void *ioaddr)
{
	int cur_rx;
	int pkt_size = 0 ;
	int rxdesc_cnt = 0;
	int ret;
	struct sk_buff *n_skb = NULL;
	struct sk_buff *cur_skb;
	struct sk_buff *rx_skb;
	struct	RxDesc	*rxdesc;

	assert (dev != NULL);
	assert (priv != NULL);
	assert (ioaddr != NULL);


	cur_rx = priv->cur_rx;

	rxdesc = &priv->RxDescArray[cur_rx];

	while ( ((le32_to_cpu(rxdesc->status) & OWNbit)== 0) && (rxdesc_cnt < 20) ){ //BUFFALO changed

	    rxdesc_cnt++;

	    //if( rxdesc->status & RxRES ){
	    if( le32_to_cpu(rxdesc->status) & RxRES ){
			printk(KERN_INFO "%s: Rx ERROR!!!\n", dev->name);
			priv->stats.rx_errors++;
			//if (rxdesc->status & (RxRWT|RxRUNT) )
			if ( le32_to_cpu(rxdesc->status) & (RxRWT|RxRUNT) )
				priv->stats.rx_length_errors++;
			//if (rxdesc->status & RxCRC)
			if ( le32_to_cpu(rxdesc->status) & RxCRC)
				priv->stats.rx_crc_errors++;
	    }
	    else{
			//pkt_size=(int)(rxdesc->status & 0x00001FFF)-4;
			pkt_size=(int)(le32_to_cpu(rxdesc->status) & 0x00001FFF)-4;

//Jumbo frame
#ifdef RTL8169_JUMBO_FRAME_SUPPORT
			if( pkt_size > priv->rx_pkt_len ){
				printk("%s: Error -- Rx packet size(%d) > mtu(%d)+14\n", dev->name, pkt_size, dev->mtu);
				pkt_size = priv->rx_pkt_len;
			}
#endif //end #ifdef RTL8169_JUMBO_FRAME_SUPPORT

			DBG_PRINT("%s: RX pkt_size = %d\n", __FUNCTION__, pkt_size);

			{// -----------------------------------------------------
				rx_skb = priv->Rx_skbuff[cur_rx];

				//n_skb = NULL;
				n_skb = RTL8169_ALLOC_RXSKB(MAX_RX_SKBDATA_SIZE);
				if( n_skb != NULL ) {
					skb_reserve (n_skb, 2);	// 16 byte align the IP fields. //

					// Indicate rx_skb
					if( rx_skb != NULL ){
						rx_skb->dev = dev;
						skb_put ( rx_skb, pkt_size );
						rx_skb->protocol = eth_type_trans ( rx_skb, dev );
						ret = RTL8169_NETIF_RX (rx_skb);

//						dev->last_rx = jiffies;
						priv->stats.rx_bytes += pkt_size;
						priv->stats.rx_packets++;

#if 0
						switch(ret)
						{
						case NET_RX_SUCCESS:    printk("%s: NETIF_RX_SUCCESS\n", dev->name);    break;
						case NET_RX_CN_LOW:     printk("%s: NETIF_RX_CN_LOW\n", dev->name);     break;
						case NET_RX_CN_MOD:     printk("%s: NETIF_CN_MOD\n", dev->name);        break;
						case NET_RX_CN_HIGH:    printk("%s: NETIF_CN_HIGH\n", dev->name);       break;
						case NET_RX_DROP:       printk("%s: NETIF_RX_DROP\n", dev->name);       break;
						default:                printk("%s: netif_rx():Unknown\n", dev->name);  break;
						}
#endif
					}//end if( rx_skb != NULL )

					priv->Rx_skbuff[cur_rx] = n_skb;
				}
				else{
					printk("TTTTTTT\n");
					priv->Rx_skbuff[cur_rx] = rx_skb;
				}


				// Update rx descriptor
				if( cur_rx == (NUM_RX_DESC-1) ){
					//priv->RxDescArray[cur_rx].status  = (OWNbit | EORbit) + priv->rx_pkt_len;
					priv->RxDescArray[cur_rx].status  = cpu_to_le32((OWNbit | EORbit) | (unsigned long)priv->hw_rx_pkt_len);//BUFFALO changed
				}
				else{
					//priv->RxDescArray[cur_rx].status  = OWNbit + priv->rx_pkt_len;
					priv->RxDescArray[cur_rx].status  = cpu_to_le32(OWNbit | (unsigned long)priv->hw_rx_pkt_len);//BUFFALO changed
				}


				cur_skb = priv->Rx_skbuff[cur_rx];
				if( cur_skb != NULL ){
					//rxdesc->buf_addr = virt_to_bus( cur_skb->data );
					rxdesc->buf_addr = cpu_to_le32(virt_to_bus( cur_skb->data ));//BUFFALO changed
				}
				else{
					DBG_PRINT("%s: %s() cur_skb == NULL\n", dev->name, __FUNCTION__);
				}

			}//------------------------------------------------------------

	    }// end of if( priv->RxDescArray[cur_rx].status & RxRES )

	    cur_rx = (cur_rx +1) % NUM_RX_DESC;
	    rxdesc = &priv->RxDescArray[cur_rx];

	}// end of while ( (priv->RxDescArray[cur_rx].status & 0x80000000)== 0)


	if( rxdesc_cnt >= 30 ){
		printk("%s: Too much work at Rx interrupt.\n", dev->name);
	}


	priv->cur_rx = cur_rx;
}

//=======================================================================================
/* The interrupt handler does all of the Rx thread work and cleans up after the Tx thread. */
static void rtl8169_interrupt (int irq, void *dev_instance, struct pt_regs *regs)
{
	struct net_device *dev = (struct net_device *) dev_instance;
	struct rtl8169_private *priv = dev->priv;
	int boguscnt = max_interrupt_work;
	void *ioaddr = priv->mmio_addr;
	int status = 0, option;

	do {
		status = RTL_R16(IntrStatus);

		/* h/w no longer present (hotplug?) or major error, bail */
		if (status == 0xFFFF)
			break;

/*
		if (status & LinkChg)
			link_changed = RTL_R16 (CSCR) & CSCR_LinkChangeBit;
*/
		if (status & LinkChg) {
			RTL8169_READ_GMII_REG(dev->base_addr, PHY_STAT_REG);

			option = readb(dev->base_addr + PHYstatus);
#if defined(CONFIG_HTGL)
			writeb(0x92, 0x80004500); // 1000Mbps down
			writeb(0x92, 0x80004500); // 1000Mbps down
			writeb(0x92, 0x80004500); // 1000Mbps down
			writeb(0x92, 0x80004500); // 1000Mbps down
			mdelay(10);
			writeb(0x90, 0x80004500); // 100Mbps down
			writeb(0x90, 0x80004500); // 100Mbps down
			writeb(0x90, 0x80004500); // 100Mbps down
			writeb(0x90, 0x80004500); // 100Mbps down
			mdelay(10);
			writeb(0x8E, 0x80004500); // 10Mbps down
			writeb(0x8E, 0x80004500); // 10Mbps down
			writeb(0x8E, 0x80004500); // 10Mbps down
			writeb(0x8E, 0x80004500); // 10Mbps down
			mdelay(10);

			if( !(option & LinkStatus) ) { // no link
				printk(KERN_INFO "%s: No Link.\n", dev->name);
			} else if( option & _1000bpsF ) { // 1000Mbps
				printk(KERN_INFO "%s: 1Gbps Linked.\n", dev->name);
				writeb(0x93, 0x80004500); // 1000Mbps up
				mdelay(10);
				writeb(0x93, 0x80004500); // 1000Mbps up
				mdelay(10);
				writeb(0x93, 0x80004500); // 1000Mbps up
				mdelay(10);
				writeb(0x93, 0x80004500); // 1000Mbps up
				mdelay(10);
			} else if( option & _100bps ) {
				printk(KERN_INFO "%s: 100Mbps Linked.\n", dev->name);
				writeb(0x91, 0x80004500); // 100Mbps up
				mdelay(10);
				writeb(0x91, 0x80004500); // 100Mbps up
				mdelay(10);
				writeb(0x91, 0x80004500); // 100Mbps up
				mdelay(10);
				writeb(0x91, 0x80004500); // 100Mbps up
				mdelay(10);
			} else if( option & _10bps ) {
				printk(KERN_INFO "%s: 10Mbps Linked.\n", dev->name);
				writeb(0x8F, 0x80004500); // 10Mbps up
				mdelay(10);
				writeb(0x8F, 0x80004500); // 10Mbps up
				mdelay(10);
				writeb(0x8F, 0x80004500); // 10Mbps up
				mdelay(10);
				writeb(0x8F, 0x80004500); // 10Mbps up
				mdelay(10);
			} else {
				printk(KERN_INFO "%s: Link status changed.\n", dev->name);
			}
#else
			if( option & _1000bpsF ){
				printk(KERN_INFO "%s: 1Gbps Linked.\n", dev->name);
				out_8(0x80004500, 0x5D);	// FAN]
				out_8(0x80004500, 0x5D);	// FAN]
				out_8(0x80004500, 0x5D);	// FAN]
				out_8(0x80004500, 0x5D);	// FAN]
			} else{
				printk(KERN_INFO "%s: Link status changed.\n", dev->name);
				out_8(0x80004500, 0x5C);	// FANᑬ]
				out_8(0x80004500, 0x5C);	// FANᑬ]
				out_8(0x80004500, 0x5C);	// FANᑬ]
				out_8(0x80004500, 0x5C);	// FANᑬ]
			}
#endif
		}

		RTL_W16( IntrStatus, status );


		if ( (status & rtl8169_intr_mask ) == 0 )
			break;


		// Rx interrupt
//		if (status & (RxOK | RxErr /* | LinkChg | RxOverflow | RxFIFOOver*/)){
			rtl8169_rx_interrupt (dev, priv, ioaddr);
//		}

		// Tx interrupt
//		if (status & (TxOK | TxErr)) {
			spin_lock (&priv->lock);
			rtl8169_tx_interrupt (dev, priv, ioaddr);
			spin_unlock (&priv->lock);
//		}

		boguscnt--;
	} while (boguscnt > 0);

	if (boguscnt <= 0) {
		printk (KERN_WARNING "%s: Too much work at interrupt!\n", dev->name);
		/* Clear all interrupt sources. */
		RTL_W16( IntrStatus, 0xffff);
	}
}

//=======================================================================================
static int rtl8169_close (struct net_device *dev)
{
	struct rtl8169_private *priv = dev->priv;
	void *ioaddr = priv->mmio_addr;
	int i;

	//////////////////////////////////////////////////////////////////////////////
	// ------------------------------------------------------
	if( (priv->mac_version > RTL_GIGA_MAC_VER_B) && (priv->phy_version < RTL_GIGA_PHY_VER_G) ){
		DBG_PRINT("FIX PCS -> rtl8169_delete_timer\n");
		rtl8169_delete_timer( &(priv->phy_timer_t) ); //in close()
		priv->phy_link_down_cnt = 0;
	}

	netif_stop_queue (dev);

	spin_lock_irq (&priv->lock);

	/* Stop the chip's Tx and Rx DMA processes. */
	RTL_W8 ( ChipCmd, 0x00);

	/* Disable interrupts by clearing the interrupt mask. */
	RTL_W16 ( IntrMask, 0x0000);

	/* Update the error counts. */
	priv->stats.rx_missed_errors += RTL_R32(RxMissed);
	RTL_W32( RxMissed, 0);

	spin_unlock_irq (&priv->lock);

	synchronize_irq ();
	free_irq (dev->irq, dev);

	rtl8169_tx_clear (priv);
	kfree(priv->TxDescArrays);
	kfree(priv->RxDescArrays);
	priv->TxDescArrays = NULL;
	priv->RxDescArrays = NULL;
	priv->TxDescArray = NULL;
	priv->RxDescArray = NULL;

	{//-----------------------------------------------------------------------------
		for(i=0;i<NUM_RX_DESC;i++){
			if( priv->Rx_skbuff[i] != NULL ) {
				RTL8169_FREE_RXSKB ( priv->Rx_skbuff[i] );
			}
		}
	}//-----------------------------------------------------------------------------

	DBG_PRINT("%s: %s() alloc_rxskb_cnt = %d\n", dev->name, __FUNCTION__, alloc_rxskb_cnt );

	return 0;
}

//=======================================================================================
static unsigned const ethernet_polynomial = 0x04c11db7U;
static inline u32 ether_crc (int length, unsigned char *data)
{
	int crc = -1;


	while (--length >= 0) {
		unsigned char current_octet = *data++;
		int bit;
		for (bit = 0; bit < 8; bit++, current_octet >>= 1)
			crc = (crc << 1) ^ ((crc < 0) ^ (current_octet & 1) ? ethernet_polynomial : 0);
	}

	return crc;
}

//=======================================================================================
static void rtl8169_set_rx_mode (struct net_device *dev)
{
	struct rtl8169_private *priv = dev->priv;
	void *ioaddr = priv->mmio_addr;
	unsigned long flags;
	u32 mc_filter[2];	/* Multicast hash filter */
	int i, rx_mode;
	u32 tmp=0;
	

	if (dev->flags & IFF_PROMISC) {
		/* Unconditionally log net taps. */
		printk (KERN_NOTICE "%s: Promiscuous mode enabled.\n", dev->name);
		rx_mode = AcceptBroadcast | AcceptMulticast | AcceptMyPhys | AcceptAllPhys;
		mc_filter[1] = mc_filter[0] = 0xffffffff;
	} else if ((dev->mc_count > multicast_filter_limit) || (dev->flags & IFF_ALLMULTI)) {
		/* Too many to filter perfectly -- accept all multicasts. */
		rx_mode = AcceptBroadcast | AcceptMulticast | AcceptMyPhys;
		mc_filter[1] = mc_filter[0] = 0xffffffff;
	} else {
		struct dev_mc_list *mclist;
		rx_mode = AcceptBroadcast | AcceptMulticast | AcceptMyPhys;
		mc_filter[1] = mc_filter[0] = 0;
		for (i = 0, mclist = dev->mc_list; mclist && i < dev->mc_count; i++, mclist = mclist->next)
			set_bit (ether_crc (ETH_ALEN, mclist->dmi_addr) >> 26, mc_filter);
	}

	spin_lock_irqsave (&priv->lock, flags);

	tmp = rtl8169_rx_config | rx_mode | (RTL_R32(RxConfig) & rtl_chip_info[priv->chipset].RxConfigMask);

	RTL_W32 ( RxConfig, tmp);
	RTL_W32 ( MAR0 + 0, mc_filter[0]);
	RTL_W32 ( MAR0 + 4, mc_filter[1]);

	spin_unlock_irqrestore (&priv->lock, flags);

}//end of rtl8169_set_rx_mode (struct net_device *dev)

//================================================================================
struct net_device_stats *rtl8169_get_stats(struct net_device *dev)

{
	struct rtl8169_private *priv = dev->priv;

    return &priv->stats;
}

//================================================================================
static struct pci_driver rtl8169_pci_driver = {
	name:		MODULENAME,
	id_table:	rtl8169_pci_tbl,
	probe:		rtl8169_init_one,
	remove:		rtl8169_remove_one,
	suspend:	NULL,
	resume:		NULL,
};

//=======================================================================================
static int __init rtl8169_init_module (void)
{
	return pci_module_init (&rtl8169_pci_driver);	// pci_register_driver (drv)
}

//=======================================================================================
static void __exit rtl8169_cleanup_module (void)
{
	pci_unregister_driver (&rtl8169_pci_driver);
}


#ifdef RTL8169_JUMBO_FRAME_SUPPORT
static int rtl8169_change_mtu(struct net_device *dev, int new_mtu)
{
	struct rtl8169_private *priv = dev->priv;
	void *ioaddr = priv->mmio_addr;

	if( new_mtu > MAX_JUMBO_FRAME_MTU ){
		printk(KERN_WARNING "%s: wrong MTU value. New MTU(%d) > MAX_JUMBO_FRAME_MTU(%d).\n", dev->name, new_mtu, MAX_JUMBO_FRAME_MTU);
		return -EINVAL;
	}
	if( new_mtu < 256 ){
		printk(KERN_WARNING "%s: wrong MTU value. New MTU(%d) < 256.\n", dev->name, new_mtu);
		return -EINVAL;
	}

	printk(KERN_INFO "%s: changed MTU %d -> %d.\n", dev->name, dev->mtu, new_mtu);
	dev->mtu = new_mtu;

//	priv->curr_mtu_size = new_mtu;
	priv->curr_mtu_size = new_mtu + 64;
	priv->tx_pkt_len = new_mtu + ETH_HDR_LEN;
	priv->rx_pkt_len = new_mtu + ETH_HDR_LEN;
	priv->hw_rx_pkt_len = priv->rx_pkt_len + 8;

	RTL_W8 ( Cfg9346, Cfg9346_Unlock);
	RTL_W16	( RxMaxSize, (unsigned short)priv->rx_pkt_len );
	RTL_W8 ( Cfg9346, Cfg9346_Lock);

	DBG_PRINT("-------------------------- \n");
	DBG_PRINT("dev->mtu = %d \n", dev->mtu);
	DBG_PRINT("priv->curr_mtu_size = %d \n", priv->curr_mtu_size);
	DBG_PRINT("priv->rx_pkt_len = %d \n", priv->rx_pkt_len);
	DBG_PRINT("priv->tx_pkt_len = %d \n", priv->tx_pkt_len);
	DBG_PRINT("RTL_W16( RxMaxSize, %d )\n", priv->rx_pkt_len);
	DBG_PRINT("-------------------------- \n");

//	rtl8169_close (dev);
//	rtl8169_open (dev);

	return 0;
}
#endif //end #ifdef RTL8169_JUMBO_FRAME_SUPPORT

//=======================================================================================
module_init(rtl8169_init_module);
module_exit(rtl8169_cleanup_module);
