/******************************************************************************

	mx1ads-dac3550a.c
	driver for Motorola MX1ADS on-board sound chip DAC3550a
	features:
	  - volume control
	  - OSS Sound Support
	  - volume is turned up after initialization to remove clicks
	  - supports only 16-bit stereo mode due to HW limitations.

	Author: MontaVista Software, Inc. <source@mvista.com>
	Copyright (c) 2003 MontaVista Software, Inc.

	Original code: ssiDAC3550a.c from Motorola MX1ADS BSP 0.3.4
	Author: Li Qin
	Copyright (c) 2001 Motorola Semiconductors HK Ltd

	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
	
	Modifications:
	Nov 2003 - ver 1.1, MontaVista Software, Inc: added i2c search for dev-fs

********************************************************************************/

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/config.h>
#include <linux/init.h>

#include <linux/sched.h>
#include <linux/delay.h>
#include <linux/errno.h>
#include <linux/i2c.h>
#ifdef CONFIG_PM
#include <linux/pm.h>
#endif

#include <asm/uaccess.h>
#include <asm/arch/platform.h>
#include <asm/irq.h>
#include <asm/arch/mx1ads-gpio.h>
#include <asm/io.h>
#include <asm/dma.h>
#include <asm/arch/dma.h>
#include <asm/mach/dma.h>

#include "sound_config.h"

//#define DEBUG_DAC3550A
#ifdef DEBUG_DAC3550A
#define debugprintk(fmt, args...) printk("%s: " fmt, __FUNCTION__ , ## args)
#else
#define debugprintk(fmt, args...)
#endif

/*register access macros*/
#define mx1_ssi_reg_out(r,x) outl(x,IO_ADDRESS((SSI_BASE + r)))
#define mx1_ssi_reg_in(r) inl(IO_ADDRESS((SSI_BASE + r)))
#define mx1_pll_reg_out(r,x) outl(x,IO_ADDRESS((PLL_BASE + r)))
#define mx1_pll_reg_in(r) inl(IO_ADDRESS((PLL_BASE + r)))
#define mx1_dma_chan_reg_out(r,c,x) outl(x,IO_ADDRESS((DMA_CH0_BASE + r + c * DMA_REG_SET_OFS * sizeof(VU32))))

/*DAC3550A I2C address */
#define DAC3550A_I2C_ADDR (0x9A>>1)

/*DAC3550A I2C registers */
#define DAC3550A_REG_SR 1
#define DAC3550A_REG_AVOL 2
#define DAC3550A_REG_GCFG 3

/*I2C access structures*/
static struct i2c_client *dac3550_client;
static struct file *dac3550_i2c;

/*ioctl data*/
static unsigned int dac3550a_bits = AFMT_S16_LE;
static signed short dac3550a_channels = 2;
static int      dac3550a_speed = 44100;

/*mixer data*/
static int      dac3550a_vol_right = 70;
static int      dac3550a_vol_left = 70;

extern void     mx1ads_request_dma_intr(dmach_t channel,
					callback_t dma_callback,
					err_callback_t dma_err_callback);
extern void     mx1ads_free_dma_intr(dmach_t channel);

static dmach_t  dac3550a_dma = -1;
static int      dac3550a_busy = 0;
static int      dac3550a_dev = -1;

#ifdef CONFIG_PM
static struct pm_dev *dac3550a_pmdev;
#endif
static int      dac3550a_mixerdev = -1;
static char     dac3550a_triggered = 0;

static int      dac3550a_initstate_gpio;
static int      dac3550a_initstate_ssi;
static int      dac3550a_initstate_thread;
static int      dac3550a_initstate_i2c;

/*the DAC3550a registers are write-only,
so here global variables are defined to store values written to the DAC*/
static char     dac3550a_avol_r, dac3550a_avol_l, dac3550a_sr, dac3550a_gcfg;

extern void     DMAbuf_outputintr(int dev, int notify_only);
extern int      DMAbuf_start_dma(int dev, unsigned long physaddr,
				 int count, int dma_mode);

static void     dac3550a_start_input(int dev, unsigned long buf, int count,
				     int intrflag);
static int      dac3550a_prepare_for_input(int dev, int bufsize, int nbufs);
static int      dac3550a_prepare_for_output(int dev, int bufsize, int nbufs);
static void     dac3550a_trigger(int dev, int state);
static void     dac3550a_output_block(int dev, unsigned long buf,
				      int count, int intrflag);
static void     dac3550a_close(int dev);
static int      dac3550a_open(int dev, int mode);
static void     dac3550a_halt_io(int dev);
static int      dac3550a_ioctl(int dev, unsigned int cmd, caddr_t arg);
static int      dac3550a_set_speed(int dev, int speed);
static signed short dac3550a_set_channels(int dev, signed short channels);
static unsigned int dac3550a_set_bits(int dev, unsigned int bits);

#ifdef CONFIG_PM
static int      dac3550a_pm_callback(struct pm_dev *pmdev,
				     pm_request_t rqst, void *data);
#endif
static void     dac3550a_dma_int_handler(void);
static void     dac3550a_dma_err_handler(int errno);

static int      dac3550a_mixer_ioctl(int dev, unsigned int cmd, caddr_t arg);
static int      dac3550a_reg_write(char dac_reg);
static int      dac3550a_clear_all_regs(void);
static int      dac3550a_i2c_thread(void *data);

int __init      mx1ads_dac3550a_init(void);
void __init     mx1ads_dac3550a_exit(void);

static int      dac3550a_i2c_init(void);
static void     dac3550a_i2c_exit(void);

static          DECLARE_COMPLETION(dac3550a_th_exit);
static          DECLARE_MUTEX_LOCKED(dac3550a_i2c_go);
static int      dac3550a_thread_terminating;

static struct audio_driver dac3550a_driver = {
	.owner = THIS_MODULE,
	.open = dac3550a_open,
	.close = dac3550a_close,
	.output_block = dac3550a_output_block,
	.start_input = dac3550a_start_input,
	.ioctl = dac3550a_ioctl,
	.prepare_for_input = dac3550a_prepare_for_input,
	.prepare_for_output = dac3550a_prepare_for_output,
	.halt_io = dac3550a_halt_io,
	.local_qlen = 0,	   /*optional */
	.copy_user = 0,		   /*optional */
	.halt_input = 0,	   /*optional */
	.halt_output = 0,	   /*optional */
	.trigger = dac3550a_trigger,
	.set_speed = dac3550a_set_speed,
	.set_bits = dac3550a_set_bits,
	.set_channels = dac3550a_set_channels,
	.postprocess_write = 0,	   /*optional */
	.preprocess_read = 0,	   /*optional */
	.mmap = 0,		   /*optional */
};

static struct mixer_operations dac3550a_mixer_operations = {
	.owner = THIS_MODULE,
	.id = "DAC3550A",
	.name = "DAC3550a Volume Ctl",
	.ioctl = dac3550a_mixer_ioctl,
};

static int
dac3550a_i2c_thread(void *data)
{
	daemonize();
	reparent_to_init();
	strcpy(current->comm, "dac3550vol");

	for (;;)
	  {
		  down(&dac3550a_i2c_go);
		  if (dac3550a_thread_terminating == 1)
			  break;
		  dac3550a_avol_r = dac3550a_vol_right * 56 / 100;
		  dac3550a_avol_l = dac3550a_vol_left * 56 / 100;
		  dac3550a_reg_write(DAC3550A_REG_AVOL);
	  }

	complete_and_exit(&dac3550a_th_exit, 0);
	return 0;
}

static int
dac3550a_mixer_ioctl(int dev, unsigned int cmd, caddr_t arg)
{
	int             val, left, right;

	if (((cmd >> 8) & 0xff) != 'M')
		return -EINVAL;

	if (_SIOC_DIR(cmd) & _SIOC_WRITE)
	  {
		  if (get_user(val, (int *) arg))
			  return -EFAULT;

		  switch (cmd & 0xff)
		    {
		    case SOUND_MIXER_VOLUME:
			    left = val & 0x00ff;
			    right = (val >> 8) & 0x00ff;

			    if (left < 0)
				    left = 0;
			    if (left > 100)
				    left = 100;
			    if (right < 0)
				    right = 0;
			    if (right > 100)
				    right = 100;

			    dac3550a_vol_right = right;
			    dac3550a_vol_left = left;
			    if (dac3550a_busy == 1)
			      {
				      dac3550a_avol_r = right * 56 / 100;
				      dac3550a_avol_l = left * 56 / 100;
				      dac3550a_reg_write(DAC3550A_REG_AVOL);
			      }
			    break;
		    default:
			    return -EINVAL;
		    }
		  return put_user(val, (int *) arg);
	  }
	switch (cmd & 0xff)
	  {
	  case SOUND_MIXER_VOLUME:
		  val =
		      dac3550a_vol_left + ((dac3550a_vol_right << 8) & 0xFF00);
		  break;
	  case SOUND_MIXER_DEVMASK:
		  val = SOUND_MASK_VOLUME;
		  break;
	  default:
		  return -EINVAL;
	  }
	return put_user(val, (int *) arg);
}

/*write to the DAC control registers via I2C, uses dedicated global variables*/
static int
dac3550a_reg_write(char dac_reg)
{
	char            msgbuf[3];
	int             msglen;
	int             tmp;

	if (dac3550a_i2c_init() < 0)
		return -EPERM;

	/*I2C slave address for the DAC is set in init in this module */
	switch (dac_reg)
	  {
	  case DAC3550A_REG_AVOL:
		  msgbuf[1] = dac3550a_avol_r;
		  msgbuf[2] = dac3550a_avol_l;
		  msglen = 3;
		  break;
	  case DAC3550A_REG_SR:
		  msgbuf[1] = dac3550a_sr;
		  msglen = 2;
		  break;
	  case DAC3550A_REG_GCFG:
		  msgbuf[1] = dac3550a_gcfg;
		  msglen = 2;
		  break;
	  default:
		  printk(KERN_ERR
			 "DAC3550A ERROR: writing to a wrong DAC register %d\n",
			 dac_reg);
		  return -EINVAL;
	  }

	msgbuf[0] = dac_reg;

	if ((tmp =
	     i2c_master_send(dac3550_client, &msgbuf[0], msglen)) != msglen)
	  {
		  printk(KERN_ERR
			 "DAC3550A ERROR: cannot write to I2C: sent %d, confirmed %d\n",
			 msglen, tmp);
		  return -1;
	  }

	return 0;
}

/*set prescale modulus*/
static void
dac3550a_set_pm(int number)
{
	unsigned long   tmp;
	unsigned long   num;

	num = (unsigned long) number;
	tmp = mx1_ssi_reg_in(SSI_STCCR);
	tmp &= 0xff00;
	tmp |= num;
	mx1_ssi_reg_out(SSI_STCCR, tmp);
}

/*set peripheral clock divider register*/
static void
dac3550a_set_pclk3(void)
{
	unsigned int    div = 1;
	unsigned int    tmp;

	tmp = mx1_pll_reg_in(PLL_PCDR);
	tmp &= ~PCDR_PCLKDIV3_MASK;	/* clear it */
	mx1_pll_reg_out(PLL_PCDR, tmp);
	tmp |= (((div - 1) << PCDR_PCLKDIV3_BIT) & PCDR_PCLKDIV3_MASK);
	mx1_pll_reg_out(PLL_PCDR, tmp);
}

/*restore necessary DAC I2C register values*/
static int
dac3550a_restore_all_regs(void)
{
	dac3550a_clear_all_regs();

	dac3550a_set_pclk3();

	mx1_ssi_reg_out(SSI_SCSR, (1 << 8));	/*enable SSI */
	mx1_ssi_reg_out(SSI_SOR, 0);

	/* SCSR
	   bit 0-7 - read only status bits
	   bit 8 - SSI enable
	   bit 9 -  transmit enable
	   bit 10 - receive enable
	   bits 11, 12 ignored in i2s mode
	   bits 14:13:
	   I2S master - 01
	   I2S slave - 10
	   SSI mode - 00 or 11
	   bit 15 - display perclk3 at SSI_RXCLK pin

	   1010 0001 0000 0000 0xA100 - init
	   1010 0011 0000 0000 0xA300 - trigger output */

	mx1_ssi_reg_out(SSI_SCSR, 0xa100);

	/* STCR
	   bits 0-6 ignored in i2s mode
	   bit 7 - enable FIFO
	   bit 8 - enable Int (don't enable if using DMA)
	   bit 9 - enable DMA
	   0000 0000 0000 0000 - init
	   0000 0010 1000 0000 0x0280 - trigger output */

	/*mx1_ssi_reg_out(SSI_STCR, 0); */

	/* SFCSR
	   bits 0-3 - tx FIFO empty watermark
	   min 1 - if 1 empty slot ...
	   max 8 - if 8 empty slots
	   bits 4-7 - rcv FIFO full watermark
	   min 1 - if 1 word rcvd ...
	   max 8 - if 8 words rcvd */

	mx1_ssi_reg_out(SSI_SFCSR, 0x84);

	dac3550a_set_bits(0, dac3550a_bits);
	dac3550a_set_channels(0, dac3550a_channels);
	dac3550a_set_speed(0, dac3550a_speed);

	return 0;
}

/*clear DAC I2C registers*/
/*disable SSI to prevent output to DAC*/
static int
dac3550a_clear_all_regs(void)
{
	dac3550a_triggered = 0;
	dac3550a_gcfg = 0;
	dac3550a_sr = 0;
	dac3550a_avol_r = 0;
	dac3550a_avol_l = 0;
	dac3550a_reg_write(DAC3550A_REG_AVOL);
	mx1_ssi_reg_out(SSI_SCSR, (1 << 8));	/*keep SSI enbaled */
	mx1_ssi_reg_out(SSI_STCR, 0);
	mx1_ssi_reg_out(SSI_STCCR, 0);
	mx1_ssi_reg_out(SSI_SFCSR, 0);
	mx1_ssi_reg_out(SSI_SOR, (1 << 4));	/*flush tx fifo */
	mx1_ssi_reg_out(SSI_SOR, 0);
	mx1_ssi_reg_out(SSI_SOR, (1 << 6));	/*turn off clocks when SSI is disabled */
	mx1_ssi_reg_out(SSI_SCSR, 0);	/*disable SSI */
	dac3550a_reg_write(DAC3550A_REG_GCFG);
	dac3550a_reg_write(DAC3550A_REG_SR);
}

/* set 8 or 16 bits*/
static unsigned int
dac3550a_set_bits(int dev, unsigned int bits)
{
	unsigned long   tmp;

	tmp = mx1_ssi_reg_in(SSI_STCCR);
	switch (bits)
	  {
	  case 0:
		  return dac3550a_bits;
	  case AFMT_S16_LE:
		  dac3550a_bits = 16;
		  tmp &= 0x00ff;
		  tmp |= 0x6100;
		  mx1_ssi_reg_out(SSI_STCCR, tmp);
		  break;
	  default:
		  return -EINVAL;
	  }
	debugprintk("*******set bits %d *****\n", bits);
	return dac3550a_bits;
}

/* set MONO or STEREO*/
static signed short
dac3550a_set_channels(int dev, signed short channels)
{
	switch (channels)
	  {
	  case 0:
		  return dac3550a_channels;
	  case 2:
		  dac3550a_channels = 2;
		  dac3550a_gcfg = 0x04;
		  if (dac3550a_reg_write(DAC3550A_REG_GCFG) < 0)
			  return -1;
		  break;
	  default:
		  return -EINVAL;
	  }
	debugprintk("*******set channels %d *****\n", channels);
	return dac3550a_channels;
}

/* set DAC sampling rate*/
/*
pclk3=1...pm=15...samplerate=46875.000000
pclk3=1...pm=16...samplerate=44117.000000
pclk3=1...pm=22...samplerate=32608.000000
pclk3=1...pm=33...samplerate=22058.000000
pclk3=1...pm=46...samplerate=15957.000000
pclk3=1...pm=67...samplerate=11029.000000
pclk3=1...pm=93...samplerate=7978.000000
*/
static int
dac3550a_set_speed(int dev, int speed)
{
	switch (speed)
	  {
	  case 0:
		  return dac3550a_speed;
	  case 8000:
		  dac3550a_speed = 8000;
		  dac3550a_set_pm(93);
		  dac3550a_sr = 0x0D;
		  break;
	  case 11025:
		  dac3550a_speed = 11025;
		  dac3550a_set_pm(67);
		  dac3550a_sr = 0x0C;
		  break;
	  case 16000:
		  dac3550a_speed = 16000;
		  dac3550a_set_pm(46);
		  dac3550a_sr = 0x0B;
		  break;
	  case 22050:
		  dac3550a_speed = 22050;
		  dac3550a_set_pm(33);
		  dac3550a_sr = 0x0A;
		  break;
	  case 32000:
		  dac3550a_speed = 32000;
		  dac3550a_set_pm(22);
		  dac3550a_sr = 0x09;
		  break;
	  case 44100:
		  dac3550a_speed = 44100;
		  dac3550a_set_pm(16);
		  dac3550a_sr = 0x08;
		  break;
	  case 48000:
		  dac3550a_speed = 48000;
		  dac3550a_set_pm(15);
		  dac3550a_sr = 0x08;
		  break;

	  default:
		  return -EINVAL;
	  }

	if (dac3550a_reg_write(DAC3550A_REG_SR) < 0)
		return -1;

	debugprintk("*****speed %d*****\n", speed);

	return dac3550a_speed;
}

static int
dac3550a_ioctl(int dev, unsigned int cmd, caddr_t arg)
{
	int             val, ret = 0;

	switch (cmd)
	  {
	  case SOUND_PCM_WRITE_RATE:
		  if (get_user(val, (int *) arg))
			  return -EFAULT;
		  ret = dac3550a_set_speed(dev, val);
		  break;
	  case SOUND_PCM_READ_RATE:
		  ret = dac3550a_speed;
		  break;
	  case SNDCTL_DSP_STEREO:
		  if (get_user(val, (int *) arg))
			  return -EFAULT;
		  ret = dac3550a_set_channels(dev, (val + 1)) - 1;
		  break;
	  case SOUND_PCM_WRITE_CHANNELS:
		  if (get_user(val, (int *) arg))
			  return -EFAULT;
		  ret = dac3550a_set_channels(dev, val);
		  break;
	  case SOUND_PCM_READ_CHANNELS:
		  ret = dac3550a_channels;
		  break;
	  case SNDCTL_DSP_SETFMT:
		  if (get_user(val, (int *) arg))
			  return -EFAULT;
		  ret = dac3550a_set_bits(dev, val);
		  break;
	  case SOUND_PCM_READ_BITS:
		  ret = dac3550a_bits;
		  break;
	  default:
		  return -EINVAL;
	  }
	return put_user(ret, (int *) arg);
}

#ifdef CONFIG_PM
/*power management event handling*/
static int
dac3550a_pm_callback(struct pm_dev *pmdev, pm_request_t rqst, void *data)
{
	int             tmp_trig = 0;

	switch (rqst)
	  {
	  case PM_SUSPEND:	/*put the DAC to stand by mode */
		  if (dac3550a_busy)
		    {
			    tmp_trig = dac3550a_triggered;
			    disable_dma(dac3550a_dma);
			    dac3550a_clear_all_regs();
		    }
		  dac3550a_gcfg = 0x20;	/*bit 5 - low power mode */
		  dac3550a_reg_write(DAC3550A_REG_SR);
		  break;
	  case PM_RESUME:	/*wake the DAC up */
		  /*restore settings */
		  if (dac3550a_busy)
		    {
			    dac3550a_restore_all_regs();
			    enable_dma(dac3550a_dma);
		    }
		  break;
	  }
	return 0;
}
#endif

static void
dac3550a_dma_int_handler(void)
{
	DMAbuf_outputintr(dac3550a_dev, 1);
}

static void
dac3550a_dma_err_handler(int errno)
{
	printk(KERN_ERR "DAC3550A ERROR: dma_err_handler %d called!\n", errno);
}

static int
dac3550a_i2c_init(void)
{
	char            filename[20];
	int             tmp;

	if (dac3550a_initstate_i2c)
		return 0;
	/*find the I2C driver we need */
	for (tmp = 0; tmp < I2C_ADAP_MAX; tmp++)
	  {
#ifdef CONFIG_DEVFS_FS
		  sprintf(filename, "/dev/i2c/%d", tmp);
#else
		  sprintf(filename, "/dev/i2c-%d", tmp);
#endif

		  if (!IS_ERR(dac3550_i2c = filp_open(filename, O_RDWR, 0)))
		    {
			    /*found some driver */
			    dac3550_client =
				(struct i2c_client *) dac3550_i2c->private_data;
			    if (strlen(dac3550_client->adapter->name) >= 9)
			      {
				      if (!memcmp
					  (dac3550_client->adapter->name,
					   "DBMX1 I2C", 9))
					      break;	/*we found our driver! */
			      }
			    filp_close(dac3550_i2c, NULL);
		    }
	  }

	if (tmp == I2C_ADAP_MAX)	/*no matching I2C driver found */
	  {
		  printk(KERN_ERR
			 "DAC3550A ERROR: cannot find DBMX1 I2C driver\n");
		  return -EPERM;
	  }

	dac3550_client->addr = DAC3550A_I2C_ADDR;
	dac3550a_initstate_i2c = 1;
	return 0;
}

static void
dac3550a_i2c_exit(void)
{
	if (dac3550a_initstate_i2c)
		filp_close(dac3550_i2c, NULL);
	dac3550a_initstate_i2c = 0;
}

/*start the device*/
static int
dac3550a_open(int dev, int mode)
{
	if (dac3550a_busy)
		return -EBUSY;
	if (mode != (int) OPEN_WRITE)
		return -1;	/*no input from this device */
	if (dac3550a_i2c_init() < 0)
		return -EPERM;
	dac3550a_restore_all_regs();
	dac3550a_busy = 1;
	return (0);
}

/*restart the device*/
static void
dac3550a_halt_io(int dev)
{
	dac3550a_restore_all_regs();
}

/*stop the device*/
static void
dac3550a_close(int dev)
{
	disable_dma(dac3550a_dma);
	dac3550a_clear_all_regs();
	dac3550a_busy = 0;
}

/*this function is a part of dma buffer processing*/
static int
dac3550a_prepare_for_output(int dev, int bufsize, int nbufs)
{
	struct dma_buffparms *dmap = audio_devs[dac3550a_dev]->dmap_out;

	dmap->flags |= DMA_NODMA;
	return 0;
}

/*this function is a part of dma buffer processing*/
static void
dac3550a_output_block(int dev, unsigned long buf, int count, int intrflag)
{
	/*ideally this function should be called within dmabuf.c,
	   but it is not because  DMA_NODMA flag turned on */
	DMAbuf_start_dma(dev, buf, count, intrflag);
}

/*this function is a part of dma buffer processing*/
static void
dac3550a_trigger(int dev, int state)
{
	if (!dac3550a_triggered)
	  {
		  /*enable transfer and DMA */
		  mx1_ssi_reg_out(SSI_STCR, 0x280);	/*enable FIFO and DMA */
		  mx1_ssi_reg_out(SSI_STX, 0);	/*swap channels to correct sides */
		  mx1_ssi_reg_out(SSI_SCSR, 0xA300);	/*enable transmitter */

		  up(&dac3550a_i2c_go);	/*put the volume up */
		  dac3550a_triggered = 1;
	  }
}

/*this function must not be called - there is no input from the device*/
static void
dac3550a_start_input(int dev, unsigned long buf, int count, int intrflag)
{
	printk(KERN_ERR "DAC3550A ERROR: start_input called!\n");
}

/*this function must not be called - there is no input from the device*/
static int
dac3550a_prepare_for_input(int dev, int bufsize, int nbufs)
{
	printk(KERN_ERR "DAC3550A ERROR: prepare_for_input called!\n");
	return 0;
}

/*initialize the device*/
int             __init
mx1ads_dac3550a_init(void)
{
	int             tmp;

	printk(KERN_INFO "MX1ADS DAC3550A Driver Ver. 1.1\n");
	debugprintk("%s %s\n", __TIME__, __DATE__);
	dac3550a_dma = -1;
	dac3550a_dev = -1;

	dac3550a_initstate_ssi = 0;
	dac3550a_initstate_gpio = 0;
	dac3550a_initstate_thread = 0;
	dac3550a_initstate_i2c = 0;

	tmp =
	    (int) request_region(IO_ADDRESS(SSI_BASE), 0x30, "mx1ads_dac3550a");
	if (!tmp)
	  {
		  printk(KERN_ERR "DAC3550A ERROR: SSI is already in use\n");

		  mx1ads_dac3550a_exit();
		  return -1;

	  }
	dac3550a_initstate_ssi = 1;

	tmp = mx1_register_gpios(PORT_C, 0x1C0, PRIMARY | TRISTATE | OUTPUT);
	if (tmp < 0)
	  {
		  printk(KERN_ERR
			 "DAC3550A ERROR: PORTC mask 0x1C is already in use\n");

		  mx1ads_dac3550a_exit();
		  return tmp;
	  }
	dac3550a_initstate_gpio = 1;

/*configure DMA support*/
	for (dac3550a_dma = 0; dac3550a_dma < MAX_DMA_CHANNELS; dac3550a_dma++)
	  {
		  if (!sound_alloc_dma(dac3550a_dma, "DAC3550A"))
			  break;
	  }
	if (dac3550a_dma == MAX_DMA_CHANNELS)
	  {
		  dac3550a_dma = -1;
		  printk(KERN_ERR "DAC3550A ERROR: can't get DMA channel!\n");
		  mx1ads_dac3550a_exit();
		  return -1;
	  }

	dac3550a_thread_terminating = 0;
	tmp =
	    kernel_thread(&dac3550a_i2c_thread, NULL,
			  CLONE_FS | CLONE_FILES | CLONE_SIGHAND);
	if (tmp < 0)
	  {

		  printk(KERN_ERR "DAC3550A ERROR: could not start thread\n");
		  mx1ads_dac3550a_exit();
		  return tmp;
	  }
	dac3550a_initstate_thread = 1;

	if ((dac3550a_dev = sound_install_audiodrv(AUDIO_DRIVER_VERSION,
						   "mx1ads dac3550a",
						   &dac3550a_driver,
						   sizeof (struct
							   audio_driver), 0,
						   AFMT_S16_LE, NULL,
						   dac3550a_dma,
						   dac3550a_dma)) < 0)
	  {
		  printk(KERN_ERR
			 "DAC3550A ERROR: too many audio devices in the system!\n");
		  mx1ads_dac3550a_exit();
		  return dac3550a_dev;
	  }

	if ((dac3550a_mixerdev = sound_alloc_mixerdev()) >= 0)
	  {
		  mixer_devs[dac3550a_mixerdev] = &dac3550a_mixer_operations;
	} else
		printk(KERN_WARNING
		       "DAC3550A WARNING: failed to load mixer device\n");
#ifdef CONFIG_PM
	dac3550a_pmdev =
	    pm_register(PM_UNKNOWN_DEV, PM_SYS_UNKNOWN, dac3550a_pm_callback);

	if (!dac3550a_pmdev)
		printk(KERN_WARNING
		       "DAC3550A WARNING: failed to init power management\n");
#endif

	mx1_dma_chan_reg_out(DMA_DAR, dac3550a_dma, 0x218000);	/*destination address */
	/*target--fifo; source--linermem; des--16bit;sour--32bit;ren-- 1 */
	mx1_dma_chan_reg_out(DMA_CCR, dac3550a_dma, 0x2088);
	mx1_dma_chan_reg_out(DMA_RSSR, dac3550a_dma, 0x10);	/*NO.16 request */
	mx1_dma_chan_reg_out(DMA_BLR, dac3550a_dma, 0x8);	/*8 bytes per burst */
	mx1_dma_chan_reg_out(DMA_BUCR, dac3550a_dma, 0);

	mx1ads_request_dma_intr(dac3550a_dma,
				(callback_t) dac3550a_dma_int_handler,
				(err_callback_t) dac3550a_dma_err_handler);

	return (0);
}

/*deinitialize the device*/
void            __init
mx1ads_dac3550a_exit(void)
{

	if (dac3550a_dev >= 0)
	  {
		  mx1ads_free_dma_intr(dac3550a_dma);
#ifdef CONFIG_PM
		  pm_unregister(dac3550a_pmdev);
#endif
		  dac3550a_i2c_exit();
		  if (dac3550a_mixerdev >= 0)
			  sound_unload_mixerdev(dac3550a_mixerdev);
		  sound_unload_audiodev(dac3550a_dev);
	  }

	if (dac3550a_initstate_thread)
	  {
		  dac3550a_thread_terminating = 1;
		  up(&dac3550a_i2c_go);
		  wait_for_completion(&dac3550a_th_exit);
	  }

	if (dac3550a_dma != -1)
		sound_free_dma(dac3550a_dma);
	if (dac3550a_initstate_gpio)
		mx1_unregister_gpios(PORT_C, 0x1C0);
	if (dac3550a_initstate_ssi)
		release_region(IO_ADDRESS(SSI_BASE), 0x30);
}

MODULE_AUTHOR("MontaVista Software, Inc. <source@mvista.com>");
MODULE_DESCRIPTION("MX1ADS DAC3550A Driver");
MODULE_LICENSE("GPL");

module_init(mx1ads_dac3550a_init);
module_exit(mx1ads_dac3550a_exit);
