/*
 *	IPv6-IPv6 tunneling module
 *
 *	Authors:
 *	Sami Kivisaari		<skivisaa@cc.hut.fi>
 *	Ville Nuorvala          <vnuorval@tml.hut.fi>
 *
 *	$Id: s.tunnel.c 1.31 02/10/16 00:31:21+03:00 vnuorval@eric.hut.mediapoli.com $
 *
 *	This program is free software; you can redistribute it and/or
 *      modify it under the terms of the GNU General Public License
 *      as published by the Free Software Foundation; either version
 *      2 of the License, or (at your option) any later version.
 *
 */

#include <linux/net.h>
#include <linux/skbuff.h>
#include <linux/ipv6.h>
#include <linux/net.h>
#include <linux/netdevice.h>
#include <linux/init.h>
#include <linux/route.h>
#include <linux/ipv6_route.h>

#ifdef CONFIG_SYSCTL
#include <linux/sysctl.h>
#endif /* CONFIG_SYSCTL */

#include <net/protocol.h>
#include <net/ipv6.h>
#include <net/ip6_route.h>
#include <net/dst.h>
#include <net/addrconf.h>
#include <net/ipv6_tunnel.h>
#include "tunnel.h"
#include "debug.h"
#include "util.h"
#include "stats.h"

#define MIPV6_TNL_MAX IPV6_TNL_MAX
#define MIPV6_TNL_MIN 1

int mipv6_max_tnls = 3;
int mipv6_min_tnls = 1;

DECLARE_MUTEX(tnl_sem);

int mipv6_max_tnls_sysctl(ctl_table *ctl, int write, struct file *filp,
			  void *buffer, size_t *lenp)
{
	int err;
	
	DEBUG_FUNC();

	down(&tnl_sem);
	if (write) {
		int diff;
		int old_max_tnls = mipv6_max_tnls;
		err = proc_dointvec(ctl, write, filp, buffer, lenp);
		if (err < 0) 
			goto out;
		if (mipv6_max_tnls < mipv6_min_tnls || 
		    mipv6_max_tnls > MIPV6_TNL_MAX) {
			mipv6_max_tnls = old_max_tnls;
			goto out;
		}
		if (mipv6_max_tnls < old_max_tnls) {
			diff = old_max_tnls - mipv6_max_tnls;
			ipv6_ipv6_tnl_dec_max_kdev_count(diff);
		} else if (mipv6_max_tnls > old_max_tnls) {
			diff = mipv6_max_tnls - old_max_tnls;
			ipv6_ipv6_tnl_inc_max_kdev_count(diff);
		}
	} else {
		err = proc_dointvec(ctl, write, filp, buffer, lenp);
	}
out:
	up(&tnl_sem);
	return err;
}

int mipv6_min_tnls_sysctl(ctl_table *ctl, int write, struct file *filp,
			  void *buffer, size_t *lenp)
{
	int err;

	DEBUG_FUNC();

	down(&tnl_sem);
	if (write) {
		int diff;
		int old_min_tnls = mipv6_min_tnls;
		err = proc_dointvec(ctl, write, filp, buffer, lenp);
		if (err < 0) 
			goto out;
		if (mipv6_min_tnls > mipv6_max_tnls || 
		    mipv6_min_tnls < MIPV6_TNL_MIN) {
			mipv6_min_tnls = old_min_tnls;
			goto out;
		}
		if (mipv6_min_tnls < old_min_tnls) {
			diff = old_min_tnls - mipv6_min_tnls;
			ipv6_ipv6_tnl_dec_min_kdev_count(diff);
		} else if (mipv6_min_tnls > old_min_tnls) {
			diff = mipv6_min_tnls - old_min_tnls;
			ipv6_ipv6_tnl_inc_min_kdev_count(diff);
		}
	} else {
		err = proc_dointvec(ctl, write, filp, buffer, lenp);
	}
out:
	up(&tnl_sem);
	return err;
}

__init void mipv6_initialize_tunnel(void)
{
	down(&tnl_sem);
	ipv6_ipv6_tnl_inc_max_kdev_count(mipv6_max_tnls);
	ipv6_ipv6_tnl_inc_min_kdev_count(mipv6_min_tnls);
	up(&tnl_sem);
}

__exit void mipv6_shutdown_tunnel(void)
{
	down(&tnl_sem);
	ipv6_ipv6_tnl_dec_min_kdev_count(mipv6_min_tnls);
	ipv6_ipv6_tnl_dec_max_kdev_count(mipv6_max_tnls);
	up(&tnl_sem);
}

static int tnl_add(struct in6_addr *remote, 
		   struct in6_addr *local, 
		   int local_origin) 
{
	struct ipv6_tnl_parm p;
	int ret;

	DEBUG_FUNC();

 	memset(&p, 0, sizeof(p));
	p.proto = IPPROTO_IPV6;
	ipv6_addr_copy(&p.laddr, local);
	ipv6_addr_copy(&p.raddr, remote);
	p.hop_limit = -1;
	p.flags = (IPV6_TNL_F_KERNEL_DEV | IPV6_TNL_F_MIPV6_DEV |
		   IPV6_TNL_F_IGN_ENCAP_LIMIT | 
		   (local_origin ? IPV6_TNL_F_ALLOW_LOCAL : 0));

	ret = ipv6_ipv6_kernel_tnl_add(&p);
	if (ret > 0) {
		DEBUG(DBG_INFO, "added tunnel from: "
		      "%x:%x:%x:%x:%x:%x:%x:%x to: %x:%x:%x:%x:%x:%x:%x:%x", 
		      NIPV6ADDR(local), NIPV6ADDR(remote));
	} else {
		DEBUG(DBG_WARNING, "unable to add tunnel from: "
		      "%x:%x:%x:%x:%x:%x:%x:%x to: %x:%x:%x:%x:%x:%x:%x:%x", 
		      NIPV6ADDR(local), NIPV6ADDR(remote));		
	}
	return ret;
}

static int tnl_del(struct in6_addr *remote, 
		   struct in6_addr *local) 
{
	struct ipv6_tnl *t = ipv6_ipv6_tnl_lookup(remote, local);
	
	DEBUG_FUNC();
	
	if (t != NULL && (t->parms.flags & IPV6_TNL_F_MIPV6_DEV)) {
		DEBUG(DBG_INFO, "deleting tunnel from: "
		      "%x:%x:%x:%x:%x:%x:%x:%x to: %x:%x:%x:%x:%x:%x:%x:%x", 
		      NIPV6ADDR(local), NIPV6ADDR(remote));

		return ipv6_ipv6_kernel_tnl_del(t);
	}
	return 0;
}

static __inline__ int add_route_to_mn(struct in6_addr *coa, 
				      struct in6_addr *ha_addr, 
				      struct in6_addr *home_addr) 
{
	struct in6_rtmsg rtmsg;
	int err;
	struct ipv6_tnl *t = ipv6_ipv6_tnl_lookup(coa, ha_addr);
	
	if (!is_mipv6_tnl(t)) {
		DEBUG(DBG_CRITICAL,"Tunnel missing");
		return -ENODEV;
	}
	
	DEBUG(DBG_INFO, "adding route to: %x:%x:%x:%x:%x:%x:%x:%x via "
	      "tunnel device", NIPV6ADDR(home_addr));

	memset(&rtmsg, 0, sizeof(rtmsg));
	ipv6_addr_copy(&rtmsg.rtmsg_dst, home_addr);
	rtmsg.rtmsg_dst_len = 128;
	rtmsg.rtmsg_type = RTMSG_NEWROUTE;
	rtmsg.rtmsg_flags = RTF_UP | RTF_NONEXTHOP | RTF_HOST | RTF_MOBILENODE;
	rtmsg.rtmsg_ifindex = t->dev->ifindex;
	rtmsg.rtmsg_metric = IP6_RT_PRIO_MIPV6;
	if ((err = ip6_route_add(&rtmsg)) == -EEXIST) {
		err = 0;
	}
	return err;
}


extern int ip6_route_del(struct in6_rtmsg *rtmsg);

static __inline__ void del_route_to_mn(struct in6_addr *coa, 
				       struct in6_addr *ha_addr, 
				       struct in6_addr *home_addr) 
{
	struct ipv6_tnl *t = ipv6_ipv6_tnl_lookup(coa, ha_addr);

	DEBUG_FUNC();

	if (is_mipv6_tnl(t)) {
		struct in6_rtmsg rtmsg;

		DEBUG(DBG_INFO, "deleting route to: %x:%x:%x:%x:%x:%x:%x:%x "
		      " via tunnel device", NIPV6ADDR(home_addr));

		memset(&rtmsg, 0, sizeof(rtmsg));
		ipv6_addr_copy(&rtmsg.rtmsg_dst, home_addr);
		rtmsg.rtmsg_dst_len = 128;
		rtmsg.rtmsg_ifindex = t->dev->ifindex;
		rtmsg.rtmsg_metric = IP6_RT_PRIO_MIPV6;
		ip6_route_del(&rtmsg);
	}
}


int mipv6_add_tnl_to_mn(struct in6_addr *coa, 
			struct in6_addr *ha_addr,
			struct in6_addr *home_addr)
{
	int ret;

	DEBUG_FUNC();

	ret = tnl_add(coa, ha_addr, 1);

	if (ret > 0) {
		int err = add_route_to_mn(coa, ha_addr, home_addr);
		if (err) {
			if (err != -ENODEV) {
				tnl_del(coa, ha_addr);
			}
			return err;
		}
	}
	return ret;
} 

int mipv6_del_tnl_to_mn(struct in6_addr *coa, 
			struct in6_addr *ha_addr,
			struct in6_addr *home_addr)
{
	DEBUG_FUNC();
	del_route_to_mn(coa, ha_addr, home_addr);
	return tnl_del(coa, ha_addr);
} 

static __inline__ int add_reverse_route(struct in6_addr *ha_addr,
					struct in6_addr *coa,
					struct in6_addr *home_addr) 
{
	struct in6_rtmsg rtmsg;
	int ret;
	struct ipv6_tnl *t;

	DEBUG_FUNC();

	t = ipv6_ipv6_tnl_lookup(ha_addr, coa);	
	
	if (!is_mipv6_tnl(t)) {
		DEBUG(DBG_CRITICAL, "Tunnel missing");
		return -ENODEV;
	}
	
	DEBUG(DBG_INFO, "adding reverse route via tunnel device");
	
	memset(&rtmsg, 0, sizeof(rtmsg));
	rtmsg.rtmsg_type = RTMSG_NEWROUTE;
	ipv6_addr_copy(&rtmsg.rtmsg_src, home_addr);
	rtmsg.rtmsg_src_len = 128;
	rtmsg.rtmsg_flags = RTF_UP | RTF_DEFAULT;
	rtmsg.rtmsg_ifindex = t->dev->ifindex;
	rtmsg.rtmsg_metric = IP6_RT_PRIO_MIPV6;
	if ((ret = ip6_route_add(&rtmsg)) == -EEXIST) {
		DEBUG(DBG_INFO, "add_reverse_route: route exists");
		return 0;
	}
	return ret;	
}

static __inline__ void del_reverse_route(struct in6_addr *ha_addr, 
					 struct in6_addr *coa, 
					 struct in6_addr *home_addr) 
{
	struct ipv6_tnl *t = ipv6_ipv6_tnl_lookup(ha_addr, coa);

	DEBUG_FUNC();

	if (is_mipv6_tnl(t)) {
		struct in6_rtmsg rtmsg;

		DEBUG(DBG_INFO, "removing reverse route via tunnel device");
	
		memset(&rtmsg, 0, sizeof(rtmsg));
		ipv6_addr_copy(&rtmsg.rtmsg_src, home_addr);
		rtmsg.rtmsg_src_len = 128;
		rtmsg.rtmsg_ifindex = t->dev->ifindex;
		rtmsg.rtmsg_metric = IP6_RT_PRIO_MIPV6;
		ip6_route_del(&rtmsg);
	}
}

int mipv6_add_tnl_to_ha(struct in6_addr *ha_addr, 
			struct in6_addr *coa,
			struct in6_addr *home_addr)
{
	int ret;
	
	DEBUG_FUNC();
	
	if (!ipv6_addr_cmp(home_addr, coa)) {
		return 1;
	}
	
	ret = tnl_add(ha_addr, coa, 1);

	if (ret >= 1) {
		int err = add_reverse_route(ha_addr, coa, home_addr);
		
		if (err) { 
			if (err != -ENODEV) {
				tnl_del(ha_addr, coa);
			}
			ret = err;
		}
	}
	return ret;
} 

int mipv6_del_tnl_to_ha(struct in6_addr *ha_addr, 
			struct in6_addr *coa,
			struct in6_addr *home_addr)
{
	DEBUG_FUNC();

	if (!ipv6_addr_cmp(home_addr, coa)) {
		return 0;
	}
	
	del_reverse_route(ha_addr, coa, home_addr);

	return tnl_del(ha_addr, coa);
} 
