/*
 *
 *  Embedded Planets RPX-Lite/HIOX Touch Panel Driver
 *  Copyright (c) 2000  Embedded Planet
 *
 *  8/00  -  CPM SPI works, so just use that
 *  Copyright (c) 2000  Montavista Software, Inc <source@mvista.com>
 *  based on the following
 *
 *  VR41xx Touch Panel Driver
 *  Copyright (c) 1999,2000 Michael Klar <wyldfier@iname.com>
 *
 *  This file is subject to the terms and conditions of the GNU General
 *  Public License.  See the file "COPYING" in the main directory of 
 *  this archive for more details
 */
//#define TPANEL_DEBUG	/* define this to compile in debug code */
#define TPANEL_FILTER
#include <linux/module.h>

#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/signal.h>
#include <linux/errno.h>
#include <linux/mm.h>
#include <linux/poll.h>
#include <linux/miscdevice.h>
#include <linux/random.h>
#include <linux/init.h>
#include <linux/interrupt.h>

#include <asm/uaccess.h>
#include <asm/system.h>
#include <asm/irq.h>
#include <asm/mpc8xx.h>
#include <asm/io.h>
#include <asm/delay.h>

#include <asm/commproc.h>

/*
 *  The BB ADS7843 and logic in the HIOX card generate pendown interrupts
 *  every 1, 10, or 100 ms.  10 ms interrupts seem reasonable.  The penup
 *  event is generate by timer interrupt set to timeout at 15 ms after the 
 *  last pendown.  The first packet read after a interrupt is garbage, so 
 *  read two packets per interrupt and throw the first away.
 *  The data to/from the 7843 looks like this if set for 12 bits
 *		zeros	 cmd wrd    data
 *		xxxxxxxx xxxxxxxx x1234567 89ABCxxx
 *
 *  UPDATE:  The pendown interrupt period has been changed from 10ms to 
 *  1ms because the coordinate filtering in the Microwindows touchpanel 
 *  driver that layers on top of this one seems to work better with the 
 *  faster updates.  The pen up timer interrupt period has been shortened 
 *  from 15ms to 4ms. Also, the first sample read after an interrupt is 
 *  good.  The bad sample was due to a bug in a previous version of this 
 *  driver.
 */

#define SQUARE(x)       ((x)*(x))

/* 
 * Touch panel commands.
 *  for BB ADS7843
 */

#define TP_CMD_START            ((u_char)0x80)
#define TP_CMD_NOPOWERDOWN      ((u_char)0x03)
#define TP_CMD_8BIT             ((u_char)0x08)
#define TP_CMD_XP               ((u_char)0x10)
#define TP_CMD_YP               ((u_char)0x50)

#define TP_MEASURE_X (TP_CMD_START|TP_CMD_XP)
#define TP_MEASURE_Y (TP_CMD_START|TP_CMD_YP)

#define X_CMD	0
#define X_DATA	1
#define Y_CMD	3
#define Y_DATA	4
static unsigned char TP_sample_cmd[6];

#define SAMPLES_PER_INTR 1
#define SAMPLES_PER_LOOP 1
#define SAMPLE_DELAY 30
#define SETUP_DELAY  50
#define DATA_CHANGE_LIMIT 15
/*
 * Timer commands
 */
#define TIMER_TGCR_EN1  0x0001
#define TIMER_TGCR_STP1 0x0002
#define TIMER_TGCR_EN2  0x0010
#define TIMER_TGCR_STP2 0x0020

/*
 *  minor number
 */
#define TPANEL_MINOR 11

// BUFFSIZE can be increased, but it must be a power of 2

#define BUFFSIZE 256 

struct tpanel_data {
	unsigned short header;
	unsigned short x;
	unsigned short y;
};

struct tpanel_status {
	struct tpanel_data buffer[BUFFSIZE];
	unsigned char	head;
	unsigned char	tail;
	unsigned char	datavalid;
	unsigned char	lastcontact;
	unsigned short	save_x[3];
	unsigned short	save_y[3];
	wait_queue_head_t wait_list;
	struct fasync_struct *fasyncptr;
	int		active;
	cbd_t		*rbdf;
	cbd_t		*tbdf;
};

void cpm_spi_init(void);
ssize_t cpm_spi_io(void *, size_t);

static struct tpanel_status tpanel;
static spinlock_t tpanel_lock = SPIN_LOCK_UNLOCKED;
unsigned int saved_points = 0;
		
#define XMAX 310
#define YMAX 240
#define XMIN 3200
#define YMIN 3000

#define LCDX 640
#define LCDY 480

#define TPANEL_CAL
int xoffset = -20;
int yoffset = -20;

void tpanel_add_data(ushort x, ushort y)
{
	unsigned long flags;

	spin_lock_irqsave(&tpanel_lock, flags);

#ifdef TPANEL_DEBUG
	printk("tpanel_add: x = %4d, y = %4d, valid = %1d, contact = %1d\n", 
               x, y, tpanel.datavalid, tpanel.lastcontact);
#endif
	tpanel.buffer[tpanel.head].header = (tpanel.datavalid << 15)|
				              (tpanel.lastcontact << 14);
#ifdef TPANEL_CAL
	tpanel.buffer[tpanel.head].x = (((x - XMIN) * LCDX)/ (XMAX - XMIN)) + xoffset;
	tpanel.buffer[tpanel.head++].y = (((y - YMIN) * LCDY)/ (YMAX - YMIN)) + yoffset;
#else
	tpanel.buffer[tpanel.head].x = x;
	tpanel.buffer[tpanel.head++].y = y;
#endif
	tpanel.head &= (BUFFSIZE - 1);
	
	if (tpanel.head == tpanel.tail) {
		tpanel.tail++;
		tpanel.tail &= (BUFFSIZE - 1);
	}

	spin_unlock_irqrestore(&tpanel_lock, flags);

	wake_up_interruptible(&tpanel.wait_list);

	if (tpanel.fasyncptr)
		kill_fasync(&tpanel.fasyncptr, SIGIO, POLL_IN);

}

static void tpanel_pendn_intr(int irq, void *dev_id, struct pt_regs *regs)
{
	volatile uint *hcsr;
	volatile immap_t	*immap;
	volatile cpmtimer8xx_t 	*timers;

	immap = (immap_t *)IMAP_ADDR;
	timers = (cpmtimer8xx_t *)&(immap->im_cpmtimer);
	hcsr = (volatile uint *)HIOX_CSR0_ADDR; 

	// TIMER - stop and clear counter
	timers->cpmt_tgcr |= (TIMER_TGCR_STP2);
	timers->cpmt_tcn2 = 0;
		
/* 
 *  acorrding to HIOX FW manual it is possible to get
 *  a interrupt with time-out still 0, so 
 *  check for interrupt
 */
	if (*hcsr & HIOX_CSR0_TIRQSTAT) {

		tpanel.lastcontact = 1;

#ifdef TPANEL_DEBUG
		printk("pendn intr\n");
#endif

		cpm_spi_io(&TP_sample_cmd[0],sizeof(TP_sample_cmd));
	}
}
#ifdef TPANEL_FILTER
static void tpanel_filter(unsigned long ignored)
{
	uint x_out,y_out;
	unsigned long flags;

	spin_lock_irqsave(&tpanel_lock, flags);
	if (saved_points == 2 && 
		(abs(tpanel.save_x[0] - tpanel.save_x[1]) < DATA_CHANGE_LIMIT) && 
		(abs(tpanel.save_y[0] -tpanel.save_y[1]) < DATA_CHANGE_LIMIT))
	{
#ifdef TPANEL_DEBUG
		printk("x0 %d y0 %d\n",tpanel.save_x[0],tpanel.save_y[0]);
		printk("x1 %d y1 %d\n",tpanel.save_x[1],tpanel.save_y[1]);
#endif
	
		tpanel.datavalid = 1;
		tpanel_add_data((tpanel.save_x[0] + tpanel.save_x[1]) >> 1,
			(tpanel.save_y[0] + tpanel.save_y[1]) >> 1);
	}

        if (saved_points == 3) {

		tpanel.datavalid = 1;

                if ((abs(tpanel.save_x[1] - tpanel.save_x[2]) < DATA_CHANGE_LIMIT && 
			abs(tpanel.save_y[1] - tpanel.save_y[2]) < DATA_CHANGE_LIMIT) &&
                    (abs(tpanel.save_x[0] - tpanel.save_x[1]) < DATA_CHANGE_LIMIT && 
			abs(tpanel.save_y[0] - tpanel.save_y[1]) < DATA_CHANGE_LIMIT))
                {
#ifdef TPANEL_DEBUG
			printk("x0 %d y0 %d\n",tpanel.save_x[0],tpanel.save_y[0]);
			printk("x1 %d y1 %d\n",tpanel.save_x[1],tpanel.save_y[1]);
			printk("x2 %d y2 %d\n",tpanel.save_x[2],tpanel.save_y[2]);
#endif
                        x_out = (tpanel.save_x[0] + tpanel.save_x[1] + tpanel.save_x[2]) / 3;
                        y_out = (tpanel.save_y[0] + tpanel.save_y[1] + tpanel.save_y[2]) / 3;
                        tpanel_add_data(x_out, y_out);
                }
                else if (abs(tpanel.save_x[1] - tpanel.save_x[2]) < DATA_CHANGE_LIMIT && 
		       	 abs(tpanel.save_y[1] - tpanel.save_y[2]) < DATA_CHANGE_LIMIT)
                {
#ifdef TPANEL_DEBUG
			printk("x1 %d y1 %d\n",tpanel.save_x[1],tpanel.save_y[1]);
			printk("x2 %d y2 %d\n",tpanel.save_x[2],tpanel.save_y[2]);
#endif
                        x_out = (tpanel.save_x[1] + tpanel.save_x[2]) >> 1;
                        y_out = (tpanel.save_y[1] + tpanel.save_y[2]) >> 1;
                        tpanel_add_data(x_out, y_out);
                }
                else
                {
#ifdef TPANEL_DEBUG
			printk("x0 %d y0 %d\n",tpanel.save_x[0],tpanel.save_y[0]);
			printk("x1 %d y1 %d\n",tpanel.save_x[1],tpanel.save_y[1]);
			printk("x2 %d y2 %d\n",tpanel.save_x[2],tpanel.save_y[2]);
#endif

                        tpanel.save_x[0] = tpanel.save_x[1];
                        tpanel.save_x[1] = tpanel.save_x[2];
                        tpanel.save_y[0] = tpanel.save_y[1];
                        tpanel.save_y[1] = tpanel.save_y[2];
                }
        }
#ifdef TPANEL_DEBUG
printk("saved points %d\n",saved_points);
#endif
	spin_unlock_irqrestore(&tpanel_lock, flags);
}
static void tpanel_up_event(unsigned long ignored)
{
	if(tpanel.lastcontact)  {
		saved_points = 0;
		tpanel.lastcontact = 0;
		tpanel.datavalid = 0;
		tpanel_add_data(0,0);
	}
}
DECLARE_TASKLET_DISABLED(tpanel_dn_tasklet, tpanel_filter, 0);
DECLARE_TASKLET_DISABLED(tpanel_up_tasklet, tpanel_up_event, 0);
#endif

static void tpanel_penup_intr(void *dev_id, struct pt_regs *regs)
{
	volatile immap_t	*immap;
	volatile cpmtimer8xx_t 	*timers;

#ifdef TPANEL_DEBUG
	printk("penup intr\n");
#endif

	immap = (immap_t *)IMAP_ADDR;
	timers = (cpmtimer8xx_t *)&(immap->im_cpmtimer);


	// pendown timeout
	// TIMER - stop
	timers->cpmt_tgcr |= (TIMER_TGCR_STP2);
	timers->cpmt_ter2 = 0xffff;
	timers->cpmt_tcn2 = 0;

	if(!tpanel.datavalid) {
		return;
	}

	tasklet_schedule(&tpanel_up_tasklet);
//	tpanel_up_event(0);

}


static int fasync_tpanel(int fd, struct file *filp, int on)
{
	int retval;

	retval = fasync_helper(fd, filp, on, &tpanel.fasyncptr);
	if (retval < 0)
		return retval;
	return 0;
}

static int close_tpanel(struct inode * inode, struct file * file)
{
	volatile uint *hcsr;

	hcsr = (volatile uint *)HIOX_CSR0_ADDR;

	*hcsr |= HIOX_CSR0_TIRQSTAT;
	*hcsr &= ~HIOX_CSR0_ENTOUCH;

	fasync_tpanel(-1, file, 0);
	if (--tpanel.active)
		return 0;
		// set for standby
	MOD_DEC_USE_COUNT;
	return 0;
}

static int open_tpanel(struct inode * inode, struct file * file)
{
	volatile uint *hcsr;

	hcsr = (volatile uint *)HIOX_CSR0_ADDR;

	if (tpanel.active++)
		return 0;
	tpanel.tail = tpanel.head;
	tpanel.lastcontact = 0;
	tpanel.datavalid = 0;

	TP_sample_cmd[0] = TP_MEASURE_X;
	TP_sample_cmd[1] = 0;
	TP_sample_cmd[2] = 0;
	TP_sample_cmd[3] = TP_MEASURE_Y;
	TP_sample_cmd[4] = 0;
	TP_sample_cmd[5] = 0;

/*
 *  clear interrupt, turn ADC chip on
 */
	*hcsr |= HIOX_CSR0_TIRQSTAT;
	*hcsr |= HIOX_CSR0_ENTOUCH;

/*
 *   set pen down timer/enable hiox interrupts
 */
	*hcsr |= (HIOX_CSR0_PDOWN1);

	MOD_INC_USE_COUNT;
	return 0;
}

/****************************************************************************
 *  Data format:
 *  unsigned short status: bit 15 = xyz data valid (0 means contact state only)
 *                         bit 14 = pen contact state (1 means contact)
 *                         bit 13 = soft data lost flag: if this is 1, this point
 *                                  is valid (if bit 15 is 1), but data was lost
 *                                  between this point and the previous point
 *                                  due to a buffer overrun in the driver
 *                         bit 12 = hard data lost flag: if this is 1, this point
 *                                  is valid (if bit 15 is 1), but data was lost
 *                                  between this point and the previous point
 *                         bits 11-0 = reserved
 *  unsigned short x+ raw data (if status:15 = 1)
 *  unsigned short y+ raw data (if status:15 = 1)
 */

static ssize_t read_tpanel(struct file * file,
	char * buffer, size_t count, loff_t *ppos)
{
	DECLARE_WAITQUEUE(wait, current);
	int r = 0;
	struct tpanel_data data;
	ulong flags;

	/* blocking call */
	if (tpanel.head == tpanel.tail) {
		if(file->f_flags & O_NONBLOCK) {
			return -EAGAIN;
		}


		add_wait_queue(&tpanel.wait_list, &wait);
repeat:
                set_current_state(TASK_INTERRUPTIBLE);
                if ((tpanel.head == tpanel.tail) && !signal_pending(current)) {
                        schedule();
                        goto repeat;
                }
                current->state = TASK_RUNNING;
                remove_wait_queue(&tpanel.wait_list, &wait);
	}

	while (tpanel.head != tpanel.tail) {

		spin_lock_irqsave(&tpanel_lock, flags);
		data = tpanel.buffer[tpanel.tail++];
		tpanel.tail &= BUFFSIZE - 1;
		spin_unlock_irqrestore(&tpanel_lock, flags);

		if (copy_to_user(buffer + r, (void *)&data, sizeof(struct tpanel_data)))
			return -EFAULT;

#ifdef TPANEL_DEBUG
		printk("read_tpanel: x = %4d, y = %4d, header = %d, ",
		       data.x, data.y, data.header);
#endif
		r += sizeof(struct tpanel_data);
		if(r > (count -sizeof(struct tpanel_data)))
		  return r;
	}	
	return r;
}

static unsigned int poll_tpanel(struct file *file, poll_table * wait)
{
  int flags,comp;

	poll_wait(file, &tpanel.wait_list, wait);

	save_flags(flags); cli();
	comp = (tpanel.head != tpanel.tail);
	restore_flags(flags);

	if (comp)
		return POLLIN | POLLRDNORM;
	return 0;
}


/*
 *  This is currently ignored but it does apply to us
 *  We have several parameters that the user should
 *  be allowed to set, # of bits, interrupt period
 */

static int ioctl_tpanel(struct inode * inode, struct file * file,
	unsigned int cmd, unsigned long arg)
{
	struct scanparam *sparm;

	sparm = (struct scanparam *)arg;

	switch (cmd) {
	case 1:
		xoffset = arg;
		break;

	case 2:
		yoffset = arg;
		break;
	case 3:
		xoffset = -arg;
		break;

	case 4:
		yoffset = -arg;
		break;
	default:
		return -EINVAL;
	}
	return 0;
}


void
cpm_spi_init()
{
	uint	mem_addr, dp_addr, reloc;
	volatile cpm8xx_t	*cp;
	volatile spi_t		*spp;
	volatile immap_t	*immap;

	cp = cpmp;	/* Get pointer to Communication Processor */
	immap = (immap_t *)IMAP_ADDR;	/* and to internal registers */

//    immap->im_siu_conf.sc_sdcr &= ~(1 << (31 - 25)); /* LAM = 0 */

	spp = (spi_t *)&cp->cp_dparam[PROFF_SPI];

	/* Check for and use a microcode relocation patch.
	*/
	if ((reloc = spp->spi_rpbase))
		spp = (spi_t *)&cp->cp_dpmem[spp->spi_rpbase];
		

	/* Initialize Port B SPI pins.
	*/
	cp->cp_pbpar |= 0x0000000E;
	cp->cp_pbdir |= 0x0000000E;
	cp->cp_pbdat &= ~(0x00000002);
	cp->cp_pbodr &= ~0x0000000E;

	/* Initialize the parameter ram.
	 * We need to make sure many things are initialized to zero,
	 * especially in the case of a microcode patch.
	 */
	spp->spi_rstate = 0;
	spp->spi_rdp = 0;
	spp->spi_rbptr = 0;
	spp->spi_rbc = 0;
	spp->spi_rxtmp = 0;
	spp->spi_tstate = 0;
	spp->spi_tdp = 0;
	spp->spi_tbptr = 0;
	spp->spi_tbc = 0;
	spp->spi_txtmp = 0;


	/* Allocate space for one transmit and one receive buffer
	 * descriptor in the DP ram.
	 */
	dp_addr = m8xx_cpm_dpalloc(sizeof(cbd_t) * 2);

	/* Allocate space for receive buffer in the host memory.
	 * 
	 */

	/* Set up the SPI parameters in the parameter ram.
	*/
	spp->spi_tbase = dp_addr;
	spp->spi_rbase = dp_addr + sizeof(cbd_t);

	/*
	 *  manually init if using microcode patch
	 */
	if(reloc)
	{
		spp->spi_rbptr = spp->spi_rbase;
		spp->spi_tbptr = spp->spi_tbase;
	}

	spp->spi_tfcr = 0x18;//SMC_EB;
	spp->spi_rfcr = 0x18;//SMC_EB;

	/* Set maximum receive size.
	*/
	spp->spi_mrblr = 16;

	/* Initialize Tx/Rx parameters.
	*/
	if (reloc == 0) {
		cp->cp_cpcr =
			mk_cr_cmd(CPM_CR_CH_SPI, CPM_CR_INIT_TRX) | CPM_CR_FLG;
		while (cp->cp_cpcr & CPM_CR_FLG);
	}
#if 0	/* The notes on the SPI relocation patch indicate the the init rx and 
	   tx command doesn't work.  It isn't clear if this also includes 
	   the individual rx and tx init commands, but at any rate it seems 
	   to work fine if we just skip these commands.
	*/
	else {
		cp->cp_cpcr =
			mk_cr_cmd(CPM_CR_CH_SPI, CPM_CR_INIT_TX) | CPM_CR_FLG;
		while (cp->cp_cpcr & CPM_CR_FLG);

		cp->cp_cpcr =
			mk_cr_cmd(CPM_CR_CH_SPI, CPM_CR_INIT_RX) | CPM_CR_FLG;
		while (cp->cp_cpcr & CPM_CR_FLG);
	}
#endif

	/* Set the buffer address.
	*/
	mem_addr = m8xx_cpm_hostalloc(16);
	tpanel.rbdf = (cbd_t *)&cp->cp_dpmem[spp->spi_rbase];
	tpanel.rbdf->cbd_bufaddr = __pa(mem_addr);
	tpanel.rbdf->cbd_sc = BD_SC_WRAP;

	mem_addr = m8xx_cpm_hostalloc(16);
	tpanel.tbdf = (cbd_t *)&cp->cp_dpmem[spp->spi_tbase];
	tpanel.tbdf->cbd_bufaddr = __pa(mem_addr);
	tpanel.tbdf->cbd_sc = BD_SC_WRAP|BD_SC_LAST;

	/* Clear event
	*/
	cp->cp_spie = ~0;

	/* No events 
	*/
	cp->cp_spim = 0;


/*
 *  NOTE:  for a real spi driver these need to be set for 
 *	   each peripheral. These are the Touch Panel settings
 */


    /*
     *  62.5KHz clock for the A/D
     *  8 bit 
     *  The datasheet specs for the ADS7843 indicate that it should work 
     *  with a clock rate of up to 2MHz, but we don't seem to be able to 
     *  get reliable touchpanel samples with a clock much faster than 
     *  62.5KHz.  This translates to a sample time of 768 microseconds 
     *  to sample both the X and Y coordinates.
     */

    cp->cp_spmode = 	SPMODE_REV | SPMODE_MSTR | 
			SPMODE_DIV16 | ((8-1) << 4) | 0xF | SPMODE_EN;


}

ssize_t
cpm_spi_io(void *buf, size_t count)
{
	volatile cpm8xx_t	*cp;
	volatile uint *hcsr;

	cp = cpmp;

	hcsr = (volatile uint *)HIOX_CSR0_ADDR;

       	*hcsr |= HIOX_CSR0_TSELSPI;

	memcpy((void *) __va(tpanel.tbdf->cbd_bufaddr), buf, count);

	tpanel.tbdf->cbd_datlen = count;
	tpanel.tbdf->cbd_sc |= BD_SC_READY;

	tpanel.rbdf->cbd_datlen = count;
	tpanel.rbdf->cbd_sc |= BD_SC_EMPTY|BD_SC_INTRPT;

	cp->cp_spim = 0x01;

	cp->cp_spcom = 0x80;	/* Start Transmit */

	return count;
}

static void
cpm_spi_interrupt(void *dev_id, struct pt_regs *regs)
{
	volatile uint *hcsr;
	unsigned char *buf;
	ushort x,y;
	volatile immap_t *immap;
	volatile cpmtimer8xx_t *timers;

	hcsr = (volatile uint *)HIOX_CSR0_ADDR;
	immap = (immap_t *)IMAP_ADDR;
	timers = (cpmtimer8xx_t *)&(immap->im_cpmtimer);

#ifdef TPANEL_DEBUG
	printk("SPI intr: spie = 0x%04x, BD_SC_EMPTY = %1d\n", 
	cpmp->cp_spie, ((tpanel.rbdf->cbd_sc & BD_SC_EMPTY) != 0));
#endif
	/* Clear interrupt.
	*/
	cpmp->cp_spie = ~0;

	if(!(tpanel.rbdf->cbd_sc & BD_SC_EMPTY))
	{ 

		buf = (unsigned char *) __va(tpanel.rbdf->cbd_bufaddr);

		x = ((*((unsigned short *)(buf + X_DATA))) >> 3) & 0x0fff;
		y = ((*((unsigned short *)(buf + Y_DATA))) >> 3) & 0x0fff;
		
#ifdef TPANEL_DEBUG
printk("buf %02x%02x%02x%02x%02x%02x\n",*buf,*buf+1,*buf+2,*buf+3,*buf+4,*buf+5);
printk("x %d y %d\n",x,y);
#endif
#ifdef TPANEL_FILTER
		if(saved_points != 3)
			saved_points++;
		tpanel.save_x[saved_points-1] = x;
		tpanel.save_y[saved_points-1] = y;
		tasklet_schedule(&tpanel_dn_tasklet);
		//tpanel_filter(0);
#else
		tpanel.datavalid = 1;
                tpanel_add_data(x, y);
#endif

		/* Turn off the touchpanel SPI interface and get ready 
		   for the next pen down interrupt. */
	       	*hcsr &= ~HIOX_CSR0_TSELSPI;
		*hcsr |=  HIOX_CSR0_TIRQSTAT;
		*hcsr |=  HIOX_CSR0_PDOWN1;

		/* restart penup timer */
		timers->cpmt_tgcr &= ~(TIMER_TGCR_STP2);
	}
  	
}

static struct file_operations tpanel_fops = {
	read: read_tpanel,
	poll: poll_tpanel,
	ioctl: ioctl_tpanel,
	open: open_tpanel,
	release: close_tpanel,
	fasync: fasync_tpanel,
};

static struct miscdevice tpanel_device = {
	TPANEL_MINOR, "tpanel", &tpanel_fops
};

int __init tpanel_init(void)
{
	volatile uint *hcsr4;
	volatile uint *bcsr;
	volatile immap_t	*immap;
	volatile cpmtimer8xx_t 	*timers;

	hcsr4 = (volatile uint *)HIOX_CSR4_ADDR;
	bcsr = (volatile uint *)RPX_CSR_ADDR;
	immap = (immap_t *)IMAP_ADDR;
	timers = (cpmtimer8xx_t *)&(immap->im_cpmtimer);

/*
 *  Naming is a little confusing here
 *  HIOX IRQ2 is SIU IRQ4
 *  HIOX IRQ3 is SIU IRQ1
 */
	*bcsr &= ~(0x00000100);
	*hcsr4 &= ~(HIOX_CSR4_ENTIRQ2 | HIOX_CSR4_ENTIRQ3);
	*hcsr4 |= HIOX_CSR4_ENTIRQ2;

/*
 *  init TIMER2 for 4ms timeout
 */
	timers->cpmt_tgcr = 0x0000;  // DOH, fix so doesn't clear other timers
	timers->cpmt_tmr2 = 0xff14;  // divide by 16, ps by 256
	timers->cpmt_tcn2 = 0;
	timers->cpmt_trr2 = 62;
	timers->cpmt_ter2 = 0xffff;
	timers->cpmt_tgcr |= TIMER_TGCR_EN2;
	timers->cpmt_tgcr |= TIMER_TGCR_STP2;
	timers->cpmt_tcn2 = 0;

        cpm_spi_init();

	tpanel.active = 0;
	tpanel.head = tpanel.tail = 0;
	tpanel.fasyncptr = NULL;
	init_waitqueue_head(&tpanel.wait_list);
/*  
 *  Make IRQ4 edge sensitive.
 */
        ((immap_t *)IMAP_ADDR)->im_siu_conf.sc_siel |= 0x00800000;

	if(request_8xxirq(SIU_IRQ4, tpanel_pendn_intr,0, "tpanel",NULL) != 0)
		printk("Could not allocate tpanel IRQ\n");

/* 
 * Install cpm interrupt handlers.
 */
	cpm_install_handler(CPMVEC_TIMER2, tpanel_penup_intr,NULL);
	cpm_install_handler(CPMVEC_SPI, cpm_spi_interrupt, NULL);

	if(misc_register(&tpanel_device))
		printk("unable to register TPANEL\n");

#ifdef TPANEL_FILTER
	tasklet_enable(&tpanel_dn_tasklet);
	tasklet_enable(&tpanel_up_tasklet);
#endif

	printk("TPANEL v1.90 driver installed\n");

	return 0;
}

#ifdef MODULE
MODULE_DESCRIPTION("LinuxPlanet touch panel driver");

int init_module(void)
{
	printk(KERN_INFO "installing LinuxPlanet touch panel\n");
	return tpanel_init();
}

void cleanup_module(void)
{
	free_irq(SIU_IRQ4, NULL);
	cpm_free_handler(CPMVEC_TIMER2);
	cpm_free_handler(CPMVEC_SPI);
	misc_deregister(&tpanel_device);
	printk(KERN_INFO "LinuxPlanet touch panel uninstalled\n");
}
#endif


