/*
 * MPC8xx Function (client) Controller Interface driver for USB.
 *
 * designed for the EmbeddedPlanet RPX lite board
 * http://www.embeddedplanet.com
 *
 * The MPC850/832 processors don't provide either the OHCI or UHCI
 * interface.  They have, however, all the basics needed to be a function
 * controller.
 *
 * this is really just an example, with minor code to field the various
 * requests.
 *
 * brad@parker.boston.ma.us 02/23/2000
 */

#include <linux/config.h>
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/delay.h>
#include <linux/sched.h>
#include <linux/slab.h>
#include <linux/errno.h>
#include <linux/init.h>
#include <linux/spinlock.h>

#include <asm/io.h>
#include <asm/8xx_immap.h>
#include <asm/pgtable.h>
#include <asm/mpc8xx.h>

#include <asm/commproc.h>
#include <linux/usb.h>

#define BD_USB_TC	((ushort)0x0400)	/* transmit crc after last */
#define BD_USB_CNF	((ushort)0x0200)	/* wait for handshake */
#define BD_USB_LSP	((ushort)0x0100)	/* low speed */
#define BD_USB_DATA0	((ushort)0x0080)	/* send data0 pid */
#define BD_USB_DATA1	((ushort)0x00c0)	/* send data1 pid */
#define BD_USB_RX_PID	((ushort)0x00c0)	/* rx pid type bits */
#define BD_USB_RX_DATA0	((ushort)0x0000)	/* rx data0 pid */
#define BD_USB_RX_DATA1	((ushort)0x0040)	/* rx data1 pid */
#define BD_USB_RX_SETUP	((ushort)0x0080)	/* rx setup pid */
#define BD_USB_NAK	((ushort)0x0010)	/* NAK received */
#define BD_USB_STAL	((ushort)0x0008)	/* STALL received */
#define BD_USB_TO	((ushort)0x0004)	/* timeout */
#define BD_USB_UN	((ushort)0x0002)	/* usb underrun */

/* FCR bits */
#define FCR_LE	0x08	/* little endian */
#define FCR_BE	0x18	/* big endian */

/* USEPx bits */
#define USEP_TM_CONTROL		0x0000
#define USEP_TM_INTERRUPT	0x0100
#define USEP_TM_BULK		0x0200
#define USEP_TM_ISOCHRONOUS	0x0300
#define USEP_MF_ENABLED		0x0020
#define USEP_RTE_ENABLED	0x0010
#define USEP_THS_NORMAL		0x0000
#define USEP_THS_IGNORE		0x0004
#define USEP_RHS_NORMAL		0x0000
#define USEP_RHS_IGNORE		0x0001
		
/* USMOD bits */
#define USMOD_LSS	0x80
#define USMOD_RESUME	0x40
#define USMOD_TEST	0x04
#define USMOD_HOST	0x02
#define USMOD_EN	0x01

/* USBER bits */	
#define BER_RESET	0x0200
#define BER_IDLE	0x0100
#define BER_TXE3	0x0080
#define BER_TXE2	0x0040
#define BER_TXE1	0x0020
#define BER_TXE0	0x0010
#define BER_SOF		0x0008
#define BER_BSY		0x0004
#define BER_TXB		0x0002
#define BER_RXB		0x0001

/* USB tokens */
#define SOF	0xa5
#define OUT	0xe1
#define IN	0x69
#define SETUP	0x2d
#define DATA0	0xc3
#define DATA1	0x4b
#define ACK	0xd2

/* Rx & Tx ring sizes */
#define CPM_USB_RX_PAGES	1
#define CPM_USB_RX_FRSIZE	256
#define CPM_USB_RX_FRPPG	(PAGE_SIZE / CPM_USB_RX_FRSIZE)
#if 1
#define RX_RING_SIZE		(CPM_USB_RX_FRPPG * CPM_USB_RX_PAGES)
#else
#define RX_RING_SIZE		4
#endif
#define TX_RING_SIZE		4

#define NUM_EPS			2

/* MPC850 USB parameter RAM */
typedef struct usbpr {
	ushort	usb_epbptr[4];
	uint	usb_rstate;
	uint	usb_rptr;
	ushort	usb_frame_n;
	ushort	usb_rbcnt;
	uint	usb_rtemp;
} usbpr_t;

/* USB endpoint parameter block */
typedef struct epb {
	ushort	epb_rbase;
	ushort	epb_tbase;
	u_char	epb_rfcr;
	u_char	epb_tfcr;
	ushort	epb_mrblr;
	ushort	epb_rbptr;
	ushort	epb_tbptr;
	uint	epb_tstate;
	uint	epb_tptr;
	ushort	epb_tcrc;
	ushort	epb_tbcnt;
} epb_t;

/* MPC850 USB registers - mapped onto SCC1 address space */
typedef struct usbregs {
	u_char	usb_usmod;
	u_char	usb_usadr;
	u_char	usb_uscom;
	char	res0;
	ushort	usb_usep[4];
	char	res1[4];
	ushort	usb_usber;
	ushort	res2;
	ushort	usb_usbmr;
	u_char	res3;
	u_char	usb_usbs;
	u_char	res4[8];
} usbregs_t;

/* bits in parallel i/o port registers that have to be cleared to
 * configure the pins for SCC1 USB use.
 */
#define PA_USB_RXD	((ushort)0x0001)
#define PA_USB_OE	((ushort)0x0002)
#define PC_USB_RXP	((ushort)0x0010)
#define PC_USB_RXN	((ushort)0x0020)
#define PC_USB_TXP	((ushort)0x0100)
#define PC_USB_TXN	((ushort)0x0200)

#define CPMVEC_USB	CPMVEC_SCC1

#include <linux/list.h>

struct m8xxusb_private {
	epb_t *epbptr[4];	/* epb ptr */
	cbd_t *rbase[4];	/* rx ring bd ptr */
	cbd_t *tbase[4];	/* tx ring bd ptr */

	int rxnext[4];		/* index of next rx to be filled */
	int txlast[4];		/* index of last tx bd fill */
	int txnext[4];		/* index of next available tx bd */
	int txfree[4];		/* count of free tx bds */

	int xmit_state[4];
#define XS_IDLE		0

	int driver_state;
#define DS_INIT		0
#define DS_ENUM		1
#define DS_READY	2

	int sof_frame;
	u_char	setup_resp[128];

	/* stats */
	struct {
		ulong interrupts;
		ulong rxb;
		ulong txb;
		ulong bsy;
		ulong sof;
		ulong txe[4];
		ulong idle;
		ulong reset;
		ulong tx_nak;
		ulong tx_stal;
		ulong tx_to;
		ulong tx_un;
	} stats;
};

static volatile struct m8xxusb_private *m8xxusb_ptr;
int usb_func_debug = 0;
#define DEBUG	if (usb_func_debug) printk
static int debug_setup = 1;
static int usb_flag;
static int set_usb_address;
static int reset_count;

#define RX_ONLY

void m8xxusb_kick_xmit(int tx_ep);
void m8xxusb_flush_xmit(int tx_ep);
void m8xxusb_flush_recv(int tx_ep);
void m8xxusb_stop(void);
static void m8xxusb_interrupt(void *, struct pt_regs *regs);

static inline cbd_t *next_bd(volatile struct m8xxusb_private *hp, int ep);
static void process_done_rxbds(int ep);
static void process_done_txbds(int ep);

#if 0
/* debug */
static void dump_tx_state(void);
static void dump_tx_bds(char *str, int ep);
static void dump_rx_state(void);
static void dump_rx_bds(char *str, int ep);
static void dump_state(int stats, int rx, int tx);
#endif


/* ---- */

/* get ppc time base register (64 bits) */
static long long
_get_TBR(void)
{
    long long t;

    __asm__ __volatile__ ("\n\
1:  mftbu   %0\n\
    mftb    %0+1\n\
    mftbu   5\n\
    cmpw    5,%0\n\
    bne     1b\n\
    isync"
    : "=r" (t)
    : /* no inputs */
    : "cc", "5" );

    return t;
}

#define MAX_EVENTS	1000
static int e_count;
static struct {
	long long e_time;
	char *e_str;
	int e_num;
} events[MAX_EVENTS];

#if 0
static void
dump_events(void)
{
	int i;
	u_long t1, t2;
	for (i = 0; i < e_count; i++) {
		t1 = events[i].e_time >> 32;
		t2 = events[i].e_time;
		printk("%08x:%08x %s %d (0x%x)\n",
		       (int)t1, (int)t2, events[i].e_str,
		       events[i].e_num, events[i].e_num);
	}
	e_count = 0;
}
#endif

static void
log_event(char *s, int n)
{
	if (e_count < MAX_EVENTS) {
		events[e_count].e_time = _get_TBR();
		events[e_count].e_str = s;
		events[e_count].e_num = n;
		e_count++;
	}
#if 0
	else dump_events();
#endif
}

/* ---- */

/*
Universal Serial Bus Specification Revision 1.1 158

8.3.5 Cyclic Redundancy Checks

Cyclic redundancy checks (CRCs) are used to protect all non-PID fields
in token and data packets. In this context, these fields are
considered to be protected fields. The PID is not included in the CRC
check of a packet containing a CRC. All CRCs are generated over their
respective fields in the transmitter before bit stuffing is
performed. Similarly, CRCs are decoded in the receiver after stuffed
bits have been removed. Token and data packet CRCs provide 100%
coverage for all single- and double-bit errors. A failed CRC is
considered to indicate that one or more of the protected fields is
corrupted and causes the receiver to ignore those fields, and, in most
cases, the entire packet.

For CRC generation and checking, the shift registers in the generator
and checker are seeded with an all-ones pattern. For each data bit
sent or received, the high order bit of the current remainder is XORed
with the data bit and then the remainder is shifted left one bit and
the low-order bit set to zero. If the result of that XOR is one, then
the remainder is XORed with the generator polynomial.

When the last bit of the checked field is sent, the CRC in the
generator is inverted and sent to the checker MSb first. When the last
bit of the CRC is received by the checker and no errors have occurred,
the remainder will be equal to the polynomial residual.

A CRC error exists if the computed checksum remainder at the end of a
packet reception does not match the residual.

Bit stuffing requirements must
be met for the CRC, and this includes the need to insert a zero at the
end of a CRC if the preceding six bits were all ones.

8.3.5.1 Token CRCs

A five-bit CRC field is provided for tokens and covers the ADDR and
ENDP fields of IN, SETUP, and OUT tokens or the time stamp field of an
SOF token. The generator polynomial is:

	G(X) = X 5 + X 2 + 1

The binary bit pattern that represents this polynomial is 00101B. If
all token bits are received without error, the five-bit residual at
the receiver will be 01100B.

*/

#if 0
static unsigned int polynomial = 0x0014;

static int
do_crc(int in, int bits)
{
	unsigned char temp;
	unsigned int crc;

	crc = 0x1f;			/* initial CRC */

	while (bits-- > 0) {
		temp = in ^ crc; 	/* do next bit */
		crc /= 2;		/* update CRC */
		if (temp & 0x01) 	/* if LSB XOR == 1 */
			crc ^= polynomial; /* then XOR polynomial with CRC */
		in /= 2;		/* next bit */
	}

	return crc;
}
#endif

#if 0
static int
calc_crc5(int addr, int endpoint)
{
	int bytes, final, crc;

	bytes = (endpoint << 7) | addr;

	crc = (~do_crc(bytes, 11)) & 0x1f;

	final = (crc << 11) | bytes;
	/*printk("crc 0x%x, final %08x\n", crc, final);*/

	return final;
}
#endif

#if 0
static spinlock_t queue_lock = SPIN_LOCK_UNLOCKED;
static spinlock_t txbd_list_lock = SPIN_LOCK_UNLOCKED;
static int txbd_list_busy;
static int queues_busy;
#endif

#if 0
static void
lock_tx_ring(struct m8xxusb_private *hp)
{
	unsigned long flags;
	spin_lock_irqsave(&txbd_list_lock, flags);
	txbd_list_busy++;
	spin_unlock_irqrestore(&txbd_list_lock, flags);
}
#endif

#if 0
static void
unlock_tx_ring(struct m8xxusb_private *hp)
{
	unsigned long flags;
	spin_lock_irqsave(&txbd_list_lock, flags);
	txbd_list_busy--;
	spin_unlock_irqrestore(&txbd_list_lock, flags);
}
#endif

#if 0
static void
lock_queues(struct m8xxusb_private *hp)
{
	unsigned long flags;
	spin_lock_irqsave(&queue_lock, flags);
	queues_busy++;
	spin_unlock_irqrestore(&queue_lock, flags);
}
#endif

#if 0
static void
unlock_queues(struct m8xxusb_private *hp)
{
	unsigned long flags;
	spin_lock_irqsave(&queue_lock, flags);
	queues_busy--;
	spin_unlock_irqrestore(&queue_lock, flags);
}
#endif

#if 0
static void
dump_tx_bds(char *str, int ep)
{
	int i;
	volatile struct	m8xxusb_private	*hp = m8xxusb_ptr;
	u_char *p;
	printk("%s\n", str);
	for (i = 0; i < TX_RING_SIZE; i++) {
		printk("%p %08x/%08x ",
		       (uint *)(hp->tbase[ep]+i),
		       ((uint *)(hp->tbase[ep]+i))[0],
		       ((uint *)(hp->tbase[ep]+i))[1]);
		p = (u_char *)((uint *)(hp->tbase[ep]+i))[1];
		if (p) {
			p = (u_char *)__va(p);
			printk("%02x %02x %02x %02x",
			       p[0], p[1], p[2], p[3]);
		}
		printk("\n");
	}
}
#endif

#if 0
static void
dump_tx_state(void)
{
	volatile struct	m8xxusb_private	*hp = m8xxusb_ptr;
	volatile epb_t *epb = hp->epbptr[0];
	printk("ep0: tstate %x, tbptr %x tptr %x\n",
	       epb->epb_tstate, epb->epb_tbptr, epb->epb_tptr);
}
#endif

#if 0
static void
dump_rx_bds(char *str, int ep)
{
	int i;
	volatile struct	m8xxusb_private	*hp = m8xxusb_ptr;
	printk("%s\n", str);
	printk("rxnext[%d] %d\n", ep, hp->rxnext[ep]);
	for (i = 0; i < RX_RING_SIZE; i++) {
		int len;
		u_char *p;

		printk("%p %08x/%08x\n",
		       (uint *)(hp->rbase[ep]+i),
		       ((uint *)(hp->rbase[ep]+i))[0],
		       ((uint *)(hp->rbase[ep]+i))[1]);
		
		len = ((uint *)(hp->rbase[ep]+i))[0];
		len &= 0x0fff;
		if (len > 16) len = 16;
		p = (u_char *)((uint *)(hp->rbase[ep]+i))[1];
		if (len > 0 && p) {
			p = (u_char *)__va(p);
			while (len > 0) {
				printk("    %02x %02x %02x %02x %02x %02x %02x %02x\n",
				       p[0], p[1], p[2], p[3],
				       p[4], p[5], p[6], p[7]);
				p += 8;
				len -= 8;
			}
		}
	}
}
#endif

#if 0
static void
dump_rx_state(void)
{
	volatile usbpr_t	*usbprmap =
		(usbpr_t *)((immap_t *)IMAP_ADDR)->im_cpm.cp_dparam;

	printk("rstate 0x%x, rptr %08x, rbcnt 0x%08x\n",
	       usbprmap->usb_rstate, 
	       usbprmap->usb_rptr, 
	       usbprmap->usb_rbcnt);

	if (0) {
		volatile immap_t *immap = (immap_t *)IMAP_ADDR;
		printk("padat 0x%04x (masked 0x%04x)\n",
		       immap->im_ioport.iop_padat,
		       immap->im_ioport.iop_padat & (PA_USB_RXD | PA_USB_OE));
		printk("pcdat 0x%04x (masked 0x%04x)\n",
		       immap->im_ioport.iop_pcdat,
		       immap->im_ioport.iop_pcdat & (PC_USB_RXP | PC_USB_RXN));
	}
}
#endif

#if 0
static void
dump_state(int stats, int rx, int tx)
{
	struct m8xxusb_private *hp = (struct m8xxusb_private *)m8xxusb_ptr;
	volatile immap_t *immap = (immap_t *)IMAP_ADDR;
	volatile usbregs_t *usbregs = (usbregs_t *)&immap->im_cpm.cp_scc[0];
	ushort ber;

	if (0) printk("kick, txbd_list_busy %d\n", txbd_list_busy);

	ber = usbregs->usb_usber;

	if (stats) {
		printk("ber 0x%x\n", ber);
		printk("int %lu: idle %lu, rst %lu, bsy %lu, rxb %lu, txb %lu\n",
		       hp->stats.interrupts,
		       hp->stats.idle,
		       hp->stats.reset,
		       hp->stats.bsy,
		       hp->stats.rxb,
		       hp->stats.txb);

		printk("txe0 %lu, nak %lu, stal %lu, to %lu, un %lu\n",
		       hp->stats.txe[0], hp->stats.tx_nak, hp->stats.tx_stal,
		       hp->stats.tx_to, hp->stats.tx_un);
	}

	if (rx) {
		dump_rx_state();
		dump_rx_bds("rx0 bds: ", 0);
	}

	if (tx) {
		dump_tx_state();
		lock_tx_ring(hp);
		dump_tx_bds("tx0 bds: ", 0);
		unlock_tx_ring(hp);
	}
}
#endif

/* alloc cpm memory on a 32 byte boundary */
static int
cpm_32b_dpalloc(int size)
{
	int index, new_index;
	index = m8xx_cpm_dpalloc(size + 32);
	new_index = (index + 31) & ~0x1f;
	/*printk("index old 0x%x new 0x%x\n", index, new_index);*/
	return new_index;
}

static int
cpm_8b_dpalloc(int size)
{
	int index, new_index;
	index = m8xx_cpm_dpalloc(size + 8);
	new_index = (index + 7) & ~0x7;
	/*printk("index old 0x%x new 0x%x\n", index, new_index);*/
	return new_index;
}

static inline cbd_t *
next_bd(volatile struct m8xxusb_private *hp, int ep)
{
	cbd_t *bdp;
	int index;

	index = hp->txnext[ep];

	bdp = hp->tbase[ep] + hp->txnext[ep]++;
	if (bdp->cbd_sc & BD_SC_WRAP)
		hp->txnext[ep] = 0;

	bdp->cbd_sc &= BD_SC_WRAP;

	return bdp;
}

static inline int
free_bds(struct m8xxusb_private *hp, int ep)
{
	return hp->txfree[ep];
}

static void
set_address(int addr)
{
	volatile immap_t *immap = (immap_t *)IMAP_ADDR;
	volatile usbregs_t *usbregs = (usbregs_t *)&immap->im_cpm.cp_scc[0];

	usbregs->usb_usadr = addr;
}

static int
make_string(u_char *b, char *s)
{
	struct usb_string_descriptor sd;
	int i, length;
	char *p;

	length = 2 + (strlen(s) * 2);

	sd.bLength = length;
	sd.bDescriptorType = USB_DT_STRING;
	memcpy(b, (char *)&sd, 2);

	/* make it unicode */
	for (i = 0, p = &b[2]; i < strlen(s); i++) {
		*p++ = s[i];
		*p++ = 0;
	}

	return length;
}

static int
build_string_desc(char *buf, int value, int index)
{
	struct usb_string_descriptor sd;
	int length;

	length = 2;
	sd.bLength = 2;
	sd.bDescriptorType = USB_DT_STRING;

	memcpy(buf, (char *)&sd, 2);

	switch (value) {
	case 0:
		switch (index) {
		case 0:
			length = make_string(buf, "  ");
			buf[2] = 9;
			buf[3] = 4;
			break;
		case 1:
			length = make_string(buf, "Manufacturer");
			break;
		case 2:
			length = make_string(buf, "Product");
			break;
		case 3:
			length = make_string(buf, "Config name");
			break;
		case 4:
			length = make_string(buf, "Intf name");
			break;
		}
		break;
	case 1:
		switch (index) {
		case 25:
			break;
		}
		break;
	}

	return length;
}

static int
build_device_desc(char *buf)
{
	struct usb_device_descriptor dd;
	int length;

	length = 18;

	dd.bLength = 18;
	dd.bDescriptorType = USB_DT_DEVICE;
	dd.bcdUSB = cpu_to_le16(0x100);
	dd.bDeviceClass = 0xff; /* vendor specific */
	dd.bDeviceSubClass = 0;
	dd.bDeviceProtocol = 0;
	dd.bMaxPacketSize0 = 32;
	dd.idVendor = cpu_to_le16(1);
	dd.idProduct = cpu_to_le16(1);
	dd.bcdDevice = cpu_to_le16(1);
	dd.iManufacturer = 1;
	dd.iProduct = 1;
	dd.iSerialNumber = 1;
	dd.bNumConfigurations = 1;

	memcpy(buf, (char *)&dd, sizeof(dd));

	return length;
}

static int
build_config_desc(char *buf)
{
	struct usb_config_descriptor cd;
	struct usb_interface_descriptor id;
	struct usb_endpoint_descriptor ed;
	int length;

	length = 9 + 9 + 7;

	cd.bLength = 9;
	cd.bDescriptorType = USB_DT_CONFIG;
	cd.wTotalLength = cpu_to_le16(length);
	cd.bNumInterfaces = 1;
	cd.bConfigurationValue = 1;
	cd.iConfiguration = 0;
	cd.bmAttributes = 0x40;
	/* 7: bus-pwrd, 6: self-pwrd, 5 Rmt-wkup, 4..0:resvd */
	cd.MaxPower = 0;

	memcpy(buf, (char *)&cd, 9);

	id.bLength = 9;
	id.bDescriptorType = USB_DT_INTERFACE;
	id.bInterfaceNumber = 0;
	id.bAlternateSetting = 0;
	id.bNumEndpoints = 1;
	id.bInterfaceClass = 10; /* USB_CLASS_DATA */
	id.bInterfaceSubClass = 0;
	id.bInterfaceProtocol = 0;
	id.iInterface = 0;

	memcpy(buf + 9, (char *)&id, 9);

	ed.bLength = 7;
	ed.bDescriptorType = USB_DT_ENDPOINT;
	ed.bEndpointAddress = 0x81; /* IN 1 */
	ed.bmAttributes = 0x03; /* interrupt */
	ed.wMaxPacketSize = cpu_to_le16(64);
//	ed.bInterval = 0xff; /* 255ms */
ed.bInterval = 0x04;
	
	memcpy(buf + 9 + 9, (char *)&ed, 7);

	return length;
}

/* ---- */
static void
process_setup(u_char *bp, int bl)
{
	volatile struct m8xxusb_private *hp = m8xxusb_ptr;
	volatile cbd_t *bdp;
	u_char requesttype, request;
	int value, index, length;
	int send, dtype, resp_len;
	
	log_event("process_setup", bl);
	
	if (debug_setup) {
		printk("setup[%2d] %02x %02x %02x %02x %02x %02x %02x %02x\n",
		       bl,
		       bp[0], bp[1], bp[2], bp[3], bp[4], bp[5], bp[6], bp[7]);
	}
	
	send = 0;
	
	requesttype = bp[0];
	request = bp[1];
	value = (bp[3] << 8) | bp[2];
	index = (bp[5] << 8) | bp[4];
	length = (bp[7] << 8) | bp[6];
	
	if (debug_setup) printk("request %d, value %d, index %d, length %d\n",
				request, value, index, length);
	
	switch (request) {
	case USB_REQ_SET_ADDRESS: /* set address */
		if (debug_setup) printk("set address %d\n", value);
		log_event("set address", value);
		hp->driver_state = DS_ENUM;
		send = 1;
		set_usb_address = value;
		length = 0;
		/* don't set address here as we don't have time */
		break;
		
	case USB_REQ_GET_DESCRIPTOR: /* get descriptor */
		if (debug_setup) printk("get descriptor 0x%x\n", value);
		log_event("get descriptor", value);
		
		send = 1;
		dtype = value >> 8;
		value &= 0xff;
		
		switch (dtype) {
		case USB_DT_DEVICE: /* device */
			resp_len = build_device_desc((char *)hp->setup_resp);
			
			if (length > resp_len)
				length = resp_len;
			
			break;
			
		case USB_DT_CONFIG: /* config */
			resp_len = build_config_desc((char *)hp->setup_resp);
			
			if (length > resp_len)
				length = resp_len;
			
			break;
			
		case USB_DT_STRING:
			length = build_string_desc((char *)hp->setup_resp,
						   value, index);
			break;
			
		default:
			printk("unknown descriptor type %d\n", dtype);
			printk("request %d, value %d, index %d, length %d\n",
			       request, value, index, length);
			
		}
		break;

	default:
		printk("unknown request %d, value %d, index %d, length %d\n",
		       request, value, index, length);
		break;
	}

#ifdef RX_ONLY
	send = 0;
#endif
	
	if (send) {
		int flags;
		
		if (debug_setup) printk("sending response to %d\n", request);
		log_event("sending response to", request);
		
		flags = BD_SC_READY | BD_SC_LAST | BD_USB_DATA1 |
			BD_USB_TC | BD_SC_INTRPT;
		
		bdp = next_bd(hp, 0);
		bdp->cbd_datlen = length;
		bdp->cbd_bufaddr = __pa(hp->setup_resp);
		
		if (length > 0) {
			flush_dcache_range((unsigned long) hp->setup_resp,
					   (unsigned long) (hp->setup_resp + length));
		}
		
		bdp->cbd_sc |= flags;
		
		m8xxusb_kick_xmit(0);
		
	}
	
	if (set_usb_address) {
		udelay(50); /* 50us */
		log_event("wait,set address", set_usb_address);
		set_address(set_usb_address);
		set_usb_address = 0;
		usb_flag = 1;
	}
}

static void
process_data(int d01, u_char *bp, int bl)
{
	if (d01)
		log_event("process_data1 len", bl);
	else
		log_event("process_data0 len", bl);

	if (debug_setup) {
	  if (bl == 0)
	    printk("data%d[%2d]\n", d01, bl);
	  else
		printk("data%d[%2d] %02x %02x %02x %02x %02x %02x %02x %02x\n",
		       d01, bl,
		       bp[0], bp[1], bp[2], bp[3], bp[4], bp[5], bp[6], bp[7]);
	}

}

static void
process_sof(void)
{
	struct m8xxusb_private *hp = (struct m8xxusb_private *)m8xxusb_ptr;
	volatile usbpr_t *usbprmap =
		(usbpr_t *)((immap_t *)IMAP_ADDR)->im_cpm.cp_dparam;
	
	
	hp->sof_frame = usbprmap->usb_frame_n & 0x7ff;
	
	usbprmap->usb_frame_n &= 0x7fff;
	
	log_event("process_sof", hp->sof_frame);
}

static void
process_reset(void)
{
	int i;
	
	log_event("process_reset", 0);
	
	set_address(0);
	
	for (i = 0; i < NUM_EPS; i++) {
		m8xxusb_flush_xmit(i);
		m8xxusb_flush_recv(i);
	}
	
	printk("usb reset\n");
}

/* ---- */

static void
advance_rx_bd(int ep)
{
	struct m8xxusb_private *hp = (struct m8xxusb_private *)m8xxusb_ptr;
	cbd_t *bdp;
	
	bdp = hp->rbase[ep] + hp->rxnext[ep];
	
	hp->rxnext[ep]++;
	if (bdp->cbd_sc & BD_SC_WRAP)
		hp->rxnext[ep] = 0;
	
	bdp->cbd_datlen = 0;
	bdp->cbd_sc &= BD_SC_WRAP;
	bdp->cbd_sc |= BD_SC_EMPTY | BD_SC_INTRPT;
}

/* run through completed rx bd's, matching them up with irq's */
static void
process_done_rxbds(int ep)
{
	struct m8xxusb_private *hp = (struct m8xxusb_private *)m8xxusb_ptr;
	cbd_t *bdp;
	u_char *bp;
	int bl, status;
	
	log_event("process_done_rxbds", ep);
	
	while (1) {
		bdp = hp->rbase[ep] + hp->rxnext[ep];
		log_event("process rx_next", hp->rxnext[ep]);
		
		status = bdp->cbd_sc;
		bp = (u_char *) __va(bdp->cbd_bufaddr);
		bl = bdp->cbd_datlen - 2;
		
		if ((status & BD_SC_EMPTY))
			break;
		
		switch (status & BD_USB_RX_PID) {
		case BD_USB_RX_SETUP:
			process_setup(bp, bl);
			break;
		case BD_USB_RX_DATA0:
			process_data(0, bp, bl);
			break;
		case BD_USB_RX_DATA1:
			process_data(1, bp, bl);
			break;
		}
		
		advance_rx_bd(ep);
	}
	
	log_event("process rx_next done", hp->rxnext[ep]);
	
#if 0
	/* just some paranoia code for debugging... */
{
	int i;
	bdp = hp->rbase[ep];
	for (i = 0; i < RX_RING_SIZE; i++) {
		if ((bdp->cbd_sc & BD_SC_EMPTY) == 0) {
			printk("woa! %d ready, next %d\n", i, hp->rxnext[ep]);
			log_event("woa! ready", i);
			
			status = bdp->cbd_sc;
			bp = __va(bdp->cbd_bufaddr);
			bl = bdp->cbd_datlen - 2;
			
			switch (status & BD_USB_RX_PID) {
			case BD_USB_RX_SETUP:
				process_setup(bp, bl);
				break;
			case BD_USB_RX_DATA0:
				process_data(0, bp, bl);
				break;
			case BD_USB_RX_DATA1:
				process_data(1, bp, bl);
				break;
			}
		}
		bdp++;
	}
}
#endif

}

/* reset a bd and advance the txlast ptr */
static inline void
advance_tx_bd(int ep)
{
	struct m8xxusb_private *hp = (struct m8xxusb_private *)m8xxusb_ptr;
	cbd_t *bdp;
	
	bdp = hp->tbase[ep] + hp->txlast[ep];
	
	hp->txlast[ep]++;
	if (bdp->cbd_sc & BD_SC_WRAP)
		hp->txlast[ep] = 0;
	
	/* I took this out so I could see what had been sent */
#if 0
	bdp->cbd_sc &= BD_SC_WRAP;
	bdp->cbd_datlen = 0;
	bdp->cbd_bufaddr = 0;
#endif
}

/* run through completed tx bd's, matching them up with qe's */
static void
process_done_txbds(int ep)
{
	struct m8xxusb_private *hp = (struct m8xxusb_private *)m8xxusb_ptr;
	cbd_t *bdp;
	
	log_event("process_done_txbds", ep);
	
	while (hp->txlast[ep] != hp->txnext[ep]) {
		bdp = hp->tbase[ep] + hp->txlast[ep];
		
		log_event("process txlast", hp->txlast[ep]);
		log_event("process datlen", bdp->cbd_datlen);
		if (0) printk("txlast %d, txnext %d, sc %04x\n",
			      hp->txlast[ep], hp->txnext[ep], bdp->cbd_sc);
		
		if (bdp->cbd_sc & BD_SC_READY) 
			break;
		
		if (bdp->cbd_sc & BD_USB_NAK)
			hp->stats.tx_nak++;
		if (bdp->cbd_sc & BD_USB_STAL)
			hp->stats.tx_stal++;
		if (bdp->cbd_sc & BD_USB_TO)
			hp->stats.tx_to++;
		if (bdp->cbd_sc & BD_USB_UN)
			hp->stats.tx_un++;
		advance_tx_bd(ep);
	}
	
}

static void
reset_tx_ring(int ep)
{
	volatile	struct m8xxusb_private *hp = m8xxusb_ptr;
	volatile	cbd_t		*bdp;
	int i;
	
	/* reset tx bd ring entries */
	bdp = hp->tbase[ep];
	for (i = 0; i < TX_RING_SIZE; i++) {
		bdp->cbd_sc = 0;
		bdp->cbd_bufaddr = 0;
		bdp++;
	}
	
	/* set the last buffer to wrap */
	bdp--;
	bdp->cbd_sc |= BD_SC_WRAP;
	
	hp->txnext[ep] = 0;
	hp->txlast[ep] = 0;
	hp->txfree[ep] = TX_RING_SIZE;
}

static void
reset_rx_ring(int ep)
{
	volatile	struct m8xxusb_private *hp = m8xxusb_ptr;
	volatile	cbd_t		*bdp;
	int i;
	
	bdp = hp->rbase[ep];
	for (i = 0; i < RX_RING_SIZE; i++) {
		bdp->cbd_sc = BD_SC_EMPTY | BD_SC_INTRPT;
		bdp->cbd_datlen = 0;
		bdp++;
	}
	
	/* set the last buffer to wrap */
	bdp--;
	bdp->cbd_sc |= BD_SC_WRAP;
	
	hp->rxnext[ep] = 0;
}		

void
m8xxusb_flush_recv(int rx_ep)
{
	struct m8xxusb_private *hp = (struct m8xxusb_private *)m8xxusb_ptr;
	volatile epb_t *epb;
	
	epb = hp->epbptr[rx_ep];
	epb->epb_rbptr = epb->epb_rbase;
	
	reset_rx_ring(rx_ep);
}

void
m8xxusb_flush_xmit(int tx_ep)
{
	struct m8xxusb_private *hp = (struct m8xxusb_private *)m8xxusb_ptr;
	volatile cpm8xx_t *cp = cpmp;
	volatile immap_t *immap = (immap_t *)IMAP_ADDR;
	volatile usbregs_t *usbregs = (usbregs_t *)&immap->im_cpm.cp_scc[0];
	volatile epb_t *epb;
	
	/* stop tx endpoint */
	cp->cp_cpcr = 0x1f01 | (tx_ep << 2);
	mb();
	while (cp->cp_cpcr & 0x1);
	
	/* flush fifo */
	eieio();
	usbregs->usb_uscom = 0x40 | tx_ep;
	mb();
	
	/* reset ring */
	epb = hp->epbptr[tx_ep];
	epb->epb_tbptr = epb->epb_tbase;
	
	reset_tx_ring(tx_ep);
	
	/* restart tx endpoint */
	cp->cp_cpcr = 0x2f01 | (tx_ep << 2);
	mb();
	while (cp->cp_cpcr & 0x1);
}

void
m8xxusb_kick_xmit(int tx_ep)
{
	volatile immap_t	*immap = (immap_t *)IMAP_ADDR;
	volatile usbregs_t	*usbregs;
	
	usbregs = (usbregs_t *)&immap->im_cpm.cp_scc[0];
	
	eieio();
	usbregs->usb_uscom = 0x80 | tx_ep;
	mb();
}

/* ---- */

static void
m8xxusb_tx_err(int ep, int ber)
{
	volatile cpm8xx_t *cp = cpmp;
	struct m8xxusb_private *hp = (struct m8xxusb_private *)m8xxusb_ptr;
	
	if ((ber & BER_TXB) == 0) {
		process_done_txbds(ep);
	}
	
	hp->stats.txe[ep]++;
	
	/* restart tx endpoint */
	cp->cp_cpcr = 0x2f01 | (ep << 2);
	mb();
	
	while (cp->cp_cpcr & 0x1);
	
	m8xxusb_kick_xmit(ep);
}

static void
m8xxusb_interrupt(void *p, struct pt_regs *regs)
{
	volatile struct	m8xxusb_private	*hp = (struct m8xxusb_private *)p;
	volatile	immap_t		*immap;
	volatile	usbpr_t		*usbprmap;
	volatile	usbregs_t	*usbregs;
	ushort ber;
	int i;
	
	
	hp->stats.interrupts++;
	
	/* get ptr to 8xx internal registers */
	immap = (immap_t *)IMAP_ADDR;
	
	/* usb param ram */
	usbprmap = (usbpr_t *)immap->im_cpm.cp_dparam;
	
	/* usb control registers */
	usbregs = (usbregs_t *)&immap->im_cpm.cp_scc[0];
	
	/* sample and reset the ber */
	ber = usbregs->usb_usber;
	usbregs->usb_usber = ber;
	
	log_event("interrupt, ber", ber);
#if 0
	printk("INTERRUPT hp %p, usbregs %p, ber 0x%x\n",
	       hp, usbregs, ber);
#endif
	
	if (ber & BER_RXB) {
		hp->stats.rxb++;
		for (i = 0; i < NUM_EPS; i++)
			process_done_rxbds(i);
	}
	
	if (ber & BER_TXB) {
		hp->stats.txb++;
		for (i = 0; i < NUM_EPS; i++)
			process_done_txbds(i);
	}
	
	if (ber & BER_BSY) {
		hp->stats.bsy++;
	}
	
	if (ber & BER_SOF) {
		hp->stats.sof++;
		process_sof();
	}
	
	if (ber & BER_TXE0) {
		m8xxusb_tx_err(0, ber);
	}
	
	if (ber & BER_TXE1) {
		m8xxusb_tx_err(1, ber);
	}
	
	if (ber & BER_TXE2) {
		m8xxusb_tx_err(2, ber);
	}
	
	if (ber & BER_TXE3) {
		m8xxusb_tx_err(3, ber);
	}
	
	/* we should time time in idle and suspend... */
	if (ber & BER_IDLE) {
		hp->stats.idle++;
		
		if (usbregs->usb_usbmr & BER_IDLE) {
			usbregs->usb_usbmr &= ~BER_IDLE;
			if (0) printk("usbmr off 0x%x\n", usbregs->usb_usbmr);
		}
	}
	
	/* ignore resets unless we see two in a row */
	if (ber & BER_RESET) {
		hp->stats.reset++;
		if (++reset_count > 1) {
			reset_count = 0;
			process_reset();
		}
	} else
		reset_count = 0;
}

int
m8xxusb_setup_usb_clock(void)
{
	volatile	cpm8xx_t	*cp;
	volatile	immap_t		*immap;
	
	/* get ptr to 8xx internal registers */
	immap = (immap_t *)IMAP_ADDR;
	
	/* Get pointer to Communication Processor */
	cp = cpmp;
	
#define USE_PA5_CLK3
	//#define USE_PA7_CLK1
		//#define USE_BRG3
			//#define USE_BRG4
				
#ifdef USE_PA7_CLK1
#define PA_DR7	((ushort)0x0100)
				printk("USING CLK1!\n");
	
	/* we assume a 48Mhz system clock connected to CLK1 via PA7 */
	immap->im_ioport.iop_padir &= ~PA_DR7;
	immap->im_ioport.iop_papar |= PA_DR7;
	
	/* control bits in SICR to route CLK1 to USB (R1CS) */
#define SICR_USB_MASK	((uint)0x000000ff)
#define SICR_USB_CLKRT	((uint)0x00000020) /* CLK1 */
	
	/* configure Serial Interface clock routing */
	cp->cp_sicr &= ~SICR_USB_MASK;
	cp->cp_sicr |= SICR_USB_CLKRT;
#endif
	
#ifdef USE_PA5_CLK3
#define PA_DR5	((ushort)0x0400)
	printk("USING CLK3!\n");
	
	/* we assume a 48Mhz system clock connected to CLK3 via PA5 */
	immap->im_ioport.iop_padir &= ~PA_DR5;
	immap->im_ioport.iop_papar |= PA_DR5;
	
	/* control bits in SICR to route CLK3 to USB (R1CS) */
#define SICR_USB_MASK	((uint)0x000000ff)
#define SICR_USB_CLKRT	((uint)0x00000030)
	
	/* configure Serial Interface clock routing */
#if 0
	cp->cp_sicr &= ~SICR_USB_MASK;
	cp->cp_sicr |= SICR_USB_CLKRT;
#else
	cp->cp_sicr = (cp->cp_sicr & ~SICR_USB_MASK) | SICR_USB_CLKRT;
#endif
#endif
	
#ifdef USE_BRG3
	printk("USING BRG3!\n");
	
	/* we assume a 48Mhz system clock */
	cp->cp_brgc3 = 0x00010000;
	
#define PB_DR28 0x0008
	immap->im_cpm.cp_pbdir &= ~PB_DR28;
	immap->im_cpm.cp_pbpar |= PB_DR28;
	
	/* control bits in SICR to route BRG3 to USB (R1CS) */
#define SICR_USB_MASK	((uint)0x000000ff)
#define SICR_USB_CLKRT	((uint)0x00000010) /* brg3 */
	
	/* configure Serial Interface clock routing */
	cp->cp_sicr &= ~SICR_USB_MASK;
	cp->cp_sicr |= SICR_USB_CLKRT;
#endif
	
#ifdef USE_BRG4
	printk("USING BRG4!\n");
	
	/* we assume a 48Mhz system clock */
	printk("cp_brgc4 0x%x before\n", cp->cp_brgc4);
	cp->cp_brgc4 = 0x00010000;
	if (0) printk("cp_brgc4 0x%x\n", cp->cp_brgc4);
	
	/* control bits in SICR to route BRG4 to USB (R1CS) */
#define SICR_USB_MASK	((uint)0x000000ff)
#define SICR_USB_CLKRT	((uint)0x00000018)
	
	/* configure Serial Interface clock routing */
	cp->cp_sicr &= ~SICR_USB_MASK;
	cp->cp_sicr |= SICR_USB_CLKRT;
#endif
	
#if 0
	printk("SCCR %08x, PLPRCR %08x\n",
	       immap->im_clkrst.car_sccr, immap->im_clkrst.car_plprcr);
#endif
	
	return 0;
}

int
m8xxusb_setup_usb_pins(void)
{
	volatile	immap_t		*immap;
	
	/* get ptr to 8xx internal registers */
	immap = (immap_t *)IMAP_ADDR;
	
	/* select USBRXD & USBOE* */
	immap->im_ioport.iop_padir &= ~(PA_USB_RXD | PA_USB_OE);
	immap->im_ioport.iop_papar |= (PA_USB_RXD | PA_USB_OE);
	immap->im_ioport.iop_paodr &= ~PA_USB_OE;
	immap->im_ioport.iop_padat = 0;
	
	/* select USBRXP & USBRXN */
	immap->im_ioport.iop_pcdir &= ~(PC_USB_RXP | PC_USB_RXN);
	immap->im_ioport.iop_pcpar &= ~(PC_USB_RXP | PC_USB_RXN);
	immap->im_ioport.iop_pcso |= (PC_USB_RXP | PC_USB_RXN);
	
	/* select USBTXP and USBTXN */
#ifndef RX_ONLY
	immap->im_ioport.iop_pcdir |= (PC_USB_TXP | PC_USB_TXN);
	immap->im_ioport.iop_pcpar |= (PC_USB_TXP | PC_USB_TXN);
	immap->im_ioport.iop_pcdat = 0;
#else
	/* disable send side */
	immap->im_ioport.iop_padir |= PA_USB_OE;
	immap->im_ioport.iop_papar &= ~PA_USB_OE;
	immap->im_ioport.iop_padat |= PA_USB_OE;

	immap->im_ioport.iop_pcdir &= ~(PC_USB_TXP | PC_USB_TXN);
	immap->im_ioport.iop_pcpar &= ~(PC_USB_TXP | PC_USB_TXN);
	immap->im_ioport.iop_pcdat = 0;
#endif
	
	return 0;
}

int
m8xxusb_setup_board_specific(void)
{
#ifdef CONFIG_RPXLITE_AW
#endif
	
#define CONFIG_RPXLITE_CW
#ifdef CONFIG_RPXLITE_CW
	/* CSR bits moved on rev CW boards */
#undef BCSR0_USBDISABLE
#undef BCSR0_USBHISPEED
#undef BCSR0_USBPWREN
#define BCSR0_USBDISABLE	((uint)0x00008000)
#define BCSR0_USBHISPEED	((uint)0x00004000)
#define BCSR0_USBPWREN		((uint)0x00002000)
#define BCSR0_ENUSBCLK		((uint)0x00001000)
#define BCSR0_ENPA5HDR		((uint)0x00000800)
#endif
	
#if defined(CONFIG_RPXLITE)
	/* set the configuration to enable USB */
	*((volatile uint *)RPX_CSR_ADDR) |=
		BCSR0_USBHISPEED
#ifdef CONFIG_RPXLITE_CW
			| (BCSR0_ENUSBCLK | BCSR0_ENPA5HDR)
#endif
				;
	
	*((volatile uint *)RPX_CSR_ADDR) &=
		~(BCSR0_USBDISABLE | BCSR0_USBPWREN);
	
	if (0) printk("RPX_CSR %08x\n", *((volatile uint *)RPX_CSR_ADDR));
#endif
	
	return 0;
}

void
m8xxusb_stop_controller(void)
{
	volatile immap_t	*immap;
	volatile usbregs_t	*usbregs;
	
	printk("m8xxusb_stop()\n");
	
	/* get ptr to 8xx internal registers */
	immap = (immap_t *)IMAP_ADDR;
	
	/* usb control registers */
	usbregs = (usbregs_t *)&immap->im_cpm.cp_scc[0];
	
	usbregs->usb_usmod = USMOD_HOST | USMOD_TEST;
}

int
m8xxusb_start_controller(void)
{
	volatile	struct m8xxusb_private *hp = m8xxusb_ptr;
	volatile	cpm8xx_t	*cp;
	volatile	immap_t		*immap;
	volatile	usbpr_t		*usbprmap;
	volatile	usbregs_t	*usbregs;
	volatile	cbd_t		*bdp;
	volatile	epb_t		*epb;
	unsigned long	mem_addr;
	pte_t		*pte;
	int		i, j, k, index, count;
	
	printk("m8xxusb_start_controller()\n");
	
	/* get ptr to 8xx internal registers */
	immap = (immap_t *)IMAP_ADDR;
	
	/* usb param ram */
	usbprmap = (usbpr_t *)immap->im_cpm.cp_dparam;
	
	/* usb control registers */
	usbregs = (usbregs_t *)&immap->im_cpm.cp_scc[0];
	
	/* get pointer to Communication Processor */
	cp = cpmp;
	
	/* set up USB section of chip */
	m8xxusb_setup_usb_clock();
	m8xxusb_setup_usb_pins();
	
	printk("ring sizes: rx %d, tx %d\n", (int) RX_RING_SIZE,
						(int) TX_RING_SIZE);
	
	/* set up EPxPTR's */
	for (i = 0; i < NUM_EPS; i++) {
		/* XXX these addresses need to be a on 32 byte boundary */
		index = cpm_32b_dpalloc(sizeof(epb_t));
		usbprmap->usb_epbptr[i] = index;
		hp->epbptr[i] = (epb_t *)&cp->cp_dpmem[index];
		epb = hp->epbptr[i];
		
		if (0) printk("endpoint [%d] 0x%x, epb %p\n", i, index, epb);
		
		/* alloc rx bd ring */
		index = cpm_8b_dpalloc(sizeof(cbd_t) * RX_RING_SIZE);
		epb->epb_rbase = index;
		hp->rbase[i] = (cbd_t *)&cp->cp_dpmem[index];
		
		/* alloc tx bd ring */
		index = cpm_8b_dpalloc(sizeof(cbd_t) * TX_RING_SIZE);
		epb->epb_tbase = index;
		hp->tbase[i] = (cbd_t *)&cp->cp_dpmem[index];
		
		if (0) printk("set up tx ring @ %p\n", bdp);
		/* reset tx bd ring entries */
		reset_tx_ring(i);

		/* set rx bd ring entries */
		bdp = hp->rbase[i];
		count = 0;
		if (0) printk("set up rx ring @ %p\n", bdp);
		for (j = 0; j < CPM_USB_RX_PAGES; j++) {
			extern pte_t *va_to_pte(unsigned long address);

			/* allocate a page */
			mem_addr = __get_free_page(GFP_KERNEL);
			
			/* make it uncached */
			pte = va_to_pte(mem_addr);
			pte_val(*pte) |= _PAGE_NO_CACHE;
			flush_tlb_page(current->mm->mmap, mem_addr);
			
			/* initialize the BD for every fragment in the page */
			for (k = 0; k < CPM_USB_RX_FRPPG; k++) {
				bdp->cbd_sc = BD_SC_EMPTY | BD_SC_INTRPT;
				bdp->cbd_datlen = 0;
				bdp->cbd_bufaddr = __pa(mem_addr);
				mem_addr += CPM_USB_RX_FRSIZE;
				bdp++;
				/* allow for small ring (and wasted space) */
				if (++count >= RX_RING_SIZE)
					goto done;
			}
		}
		
		/* set the last buffer to wrap */
	done:
		bdp--;
		bdp->cbd_sc |= BD_SC_WRAP;
		
		epb->epb_rfcr = FCR_BE;
		epb->epb_tfcr = FCR_BE;
		
#define MAX_RBE	(CPM_USB_RX_FRSIZE-4)	/* max receive buffer size (bytes) */
		
		epb->epb_mrblr = MAX_RBE;
		
		epb->epb_rbptr = epb->epb_rbase;
		epb->epb_tbptr = epb->epb_tbase;
		if (0) printk("tbptr %08x\n", epb->epb_tbptr);
		epb->epb_tstate = 0;
		
		if (0) printk("usep%d @ %p\n", i, &usbregs->usb_usep[i]);
		
		switch (i) {
		case 0:
			usbregs->usb_usep[i] =
#ifdef RX_ONLY
				USEP_THS_IGNORE | USEP_RHS_IGNORE |
0xf | 
#endif
				USEP_TM_CONTROL | USEP_MF_ENABLED;
			break;
		case 1:
			usbregs->usb_usep[i] = ((i) << 12) | USEP_TM_INTERRUPT;
			break;
		case 2:
			usbregs->usb_usep[i] = ((i) << 12) | USEP_TM_BULK;
			break;
		case 3:
			usbregs->usb_usep[i] = ((i) << 12) | USEP_TM_ISOCHRONOUS;
			break;
		}
	}
	
	usbprmap->usb_rstate = 0;
	usbprmap->usb_frame_n = 0;
	
	/* set 12Mbps function mode & disable usb */
	usbregs->usb_usmod = USMOD_TEST;
	
	/* set address to zero initially */
	usbregs->usb_usadr = 0;
	
	/* clear USCOM */
	usbregs->usb_uscom = 0;
	
	/* reset event register & interrupt mask */
	usbregs->usb_usber = 0xffff;
	usbregs->usb_usbmr = 0xffff;
	
	/* install our interrupt handler */
	cpm_install_handler(CPMVEC_USB, m8xxusb_interrupt, (void *)hp);
	
	/* turn on board specific bits */
	m8xxusb_setup_board_specific();
	
	/* enable USB controller */
	printk("m8xxusb_init() enable USB controller\n");
#ifdef RX_ONLY
	printk("m8xxusb_init() read-only mode\n");
#endif
	usbregs->usb_usmod &= ~USMOD_TEST;
	usbregs->usb_usmod |= USMOD_EN;
	
#if 0
	printk("usber 0x%x\n", usbregs->usb_usber);
	printk("usbs 0x%x\n", usbregs->usb_usbs);
#endif
	
	return 0;
}


static int __init m8xxusb_init_driver(void)
{
	volatile struct	m8xxusb_private *hp;
	int i;
	
	printk("m8xxusb_init_driver()\n");

	/* allocate controller private storage */
	hp = (struct m8xxusb_private *)kmalloc(sizeof(*hp), GFP_KERNEL);
	m8xxusb_ptr = hp;
	
	for (i = 0; i < 4; i++) {
		hp->xmit_state[i] = XS_IDLE;
	}
	
	hp->driver_state = DS_INIT;

	return 0;
}

int __init m8xxusb_init(void)
{
	/* hack - look at config switches to turn on driver */
	if ((*((volatile uint *)RPX_CSR_ADDR) & 0x10) != 0) {
		printk("m8xxusb_init() disabled via dip switches\n");
		return 0;
	}
	
	printk("m8xxusb_init()\n");

	/* init driver state */
	m8xxusb_init_driver();
	
	/* start controller */
	m8xxusb_start_controller();
	
	printk("m8xxusb_init() done\n");
	
	return 0;
}

module_init(m8xxusb_init);

