/*
 * arch/ppc/common/misc-simple.c
 *
 * Misc. bootloader code for many machines.  This assumes you have are using
 * a 6xx/7xx/74xx CPU in your machine.  This assumes the chunk of memory
 * below 8MB is free.  Finally, it assumes you have a NS16550-style uart for
 * your serial console.  If a machine meets these requirements, it can quite
 * likely use this code during boot.
 *
 * Author: Matt Porter <mporter@mvista.com>
 * Derived from arch/ppc/boot/prep/misc.c
 *
 * Copyright 2001 MontaVista Software Inc.
 *
 * This program is free software; you can redistribute  it and/or modify it
 * under  the terms of  the GNU General  Public License as published by the
 * Free Software Foundation;  either version 2 of the  License, or (at your
 * option) any later version.
 */

#include <linux/types.h>
#include <linux/elf.h>
#include <linux/config.h>

#include <asm/page.h>
#include <asm/processor.h>
#include <asm/mmu.h>
#include <asm/bootinfo.h>

#include "nonstdio.h"
#include "zlib.h"

#include <melco/melco_hwctl.h>

#ifdef FLASH_BOOT_IMAGE

#include <melco/firminfo.h>

#define KERNEL_IMAGE_ADDRESS  0xffc00000
#define KERNEL_IMG_SIZE_MAX   0x00400000
struct flash_dev_func* flash = NULL;

#define FLAG_ADDR 0xfff70000

void dbg_mode(void);
#endif

unsigned long com_port;

char *avail_ram;
char *end_avail;

/* The linker tells us where the image is. */
extern char __image_begin, __image_end;
extern char __ramdisk_begin, __ramdisk_end;
extern char _end[];

#ifdef CONFIG_CMDLINE
#define CMDLINE CONFIG_CMDLINE
#else
#define CMDLINE ""
#endif
char cmd_preset[] = CMDLINE;
char cmd_buf[256];
char *cmd_line = cmd_buf;

void (*timer_proc)(void) = NULL;
unsigned long initrd_start = 0, initrd_end = 0;

unsigned long initrd_size = 0;
char *zimage_start;
int zimage_size;

#define CONSOLE_PORT	0
#define AVR_PORT		1
extern unsigned long serial_init(int chan, int isConsole);
extern unsigned long in_ledw(volatile unsigned long *addr);
extern unsigned long in_bedw(volatile unsigned long *addr);
extern void out_ledw(volatile unsigned long *addr, unsigned long val);
extern void out_bedw(volatile unsigned long *addr, unsigned long val);
extern unsigned long start;
extern void gunzip(void *, int, unsigned char *, int *);
extern void setup_legacy(void);

char* input_cmd(char* cmd,int max)
{
	char ch,*cp;
	int    i = 0;
	cp = cmd;
	while ((ch = getc()) != '\n' && ch != '\r') {
		if (ch == '\b' || ch == '\177') {
			if (cp != cmd) {
				cp--;i--;
				puts("\b \b");
			}
		/* Test for ^x/^u (and wipe the line) */
		} else if (ch == '\030' || ch == '\025') {
			while (cp != cmd) {
				cp--;i--;
				puts("\b \b");
			}
		} else {
			if(i+1 >= max) {
				cp--;i--;
				puts("\b \b");
				}
			*cp++ = ch;i++;
			putc(ch);
		}
	}
	*cp = 0;
	return cp;
}

#ifdef FLASH_BOOT_IMAGE
void alart_mode(void);
struct firminfo* image_addr = (struct firminfo*)KERNEL_IMAGE_ADDRESS;

int timer      = 0;
int next_timer = 0;
int led_count  = 0;
int warn_count = 0;

void warning_led(void) {
	timer++;
	if(timer>next_timer) {
		if(led_count>0) {
			if(led_count & 1)
				HWCtl_Led_Down(LED_DIAG);
			else
				HWCtl_Led_Up(LED_DIAG);
			led_count--;
			if(led_count == 0)
				next_timer = 6000;
			else
				next_timer = 1000;
		}
		if(led_count==0) led_count = warn_count*2;
		timer = 0;
	}
	return;
}

unsigned long checksum_check(unsigned char* addr,unsigned long size)
{
	long* laddr = (long*)addr;
	unsigned long sum = 0,remain = 0;
	int i;
	while(size>=4) {
		sum += *laddr;
		laddr++;
		size -= 4;
	}
	addr = (unsigned char*)laddr;
	for(i=0;i<4;++i) {
		remain = remain << 8;
		if(size>i) remain += *addr;
		addr++;
		}
	sum += remain;
	return sum;
}
#endif

void
decompress_kernel(unsigned long load_addr, int num_words, unsigned long cksum)
{

	int timer = 0;
	int i;
	extern unsigned long start;
	char *cp, ch;
	struct bi_record *rec, *birecs;

#ifdef FLASH_BOOT_IMAGE
	struct firminfo* info;

#endif

	serial_init(0,AVR_PORT);
	com_port = serial_init(1,CONSOLE_PORT);

#ifdef FLASH_BOOT_IMAGE

	if ((check_pci_device(11, VID_ADMTEK, DID_AN983B, 0) != 0) ||
	    (check_pci_device(12, VID_SiI, DID_SiI680, 0) != 0) ||
	    (check_pci_device(14, VID_NEC, DID_720101_U1, 0) != 0) ||
	    (check_pci_device(14, VID_NEC, DID_720101_U2, 2) != 0))
		while(1) outb(0x80004500, 0x6F);

#endif

	/* assume the chunk below 8M is free */
	avail_ram = (char *)0x00400000;
	end_avail = (char *)0x00800000;

	/*
	 * Reveal where we were loaded at and where we
	 * were relocated to.
	 */
#ifdef FLASH_BOOT_IMAGE
	{
	int              i = 0;
	info = image_addr;
	_printk("\n******* Product Information *******\n");
	_printk("----------------------------------\n");
	_printk("Product Name: ");
	while(i<32 && info->firmname[i]!=0) {putc(info->firmname[i]);i++;}
	_printk("\n          VER: %d.%02d\n"  ,info->ver_major,info->ver_minor);
	_printk("         Date: %d/%d/%d %d:%d:%d\n",info->year+1900,info->mon,info->day,
												info->hour,info->min,info->sec);
	_printk("----------------------------------\n");
	
	_printk("Firmware check:");
	if (info->size < sizeof(struct firminfo*) 
			|| info->size > KERNEL_IMG_SIZE_MAX
			||  info->kernel_size < 1
			|| info->kernel_size>KERNEL_IMG_SIZE_MAX
			||  info->kernel_offset < sizeof(struct firminfo*)
			||  info->kernel_offset > KERNEL_IMG_SIZE_MAX
			||  (info->kernel_offset + info->kernel_size) > KERNEL_IMG_SIZE_MAX) {
		_printk("Fail!:invalid Firmware size\n");
		warn_count = 2;
		timer_proc = warning_led;
		while(1) outb(0x80004500, 0x6B);
	}
	
	if (info->initrd_size < 1
			|| info->initrd_size > KERNEL_IMG_SIZE_MAX
			|| info->initrd_offset < (info->kernel_offset + info->kernel_size)
			|| info->initrd_offset > KERNEL_IMG_SIZE_MAX
			|| (info->initrd_offset + info->initrd_size) > KERNEL_IMG_SIZE_MAX) {
		_printk("Warning:invalid data size\n");
	}

	if (checksum_check((unsigned char*)info,info->size) != 0) {
		_printk("Fail!:checksum error %08X\n",checksum_check((unsigned char*)info,info->size));
		warn_count = 2;
		timer_proc = warning_led;
		while(1) outb(0x80004500, 0x6B);
	}
	_printk("done.\n");
	zimage_start = (char*)info + info->kernel_offset;
	zimage_size  = (int)info->kernel_size;
	if (info->initrd_size > 0) {
		initrd_start = (unsigned long)((char*)info + info->initrd_offset);
		initrd_end   = initrd_start + info->initrd_size;
		
		if(initrd_start > 0xffc00000 && initrd_end<0xffefffff) {
			avail_ram = (char *)PAGE_ALIGN((unsigned long)_end);
			memcpy ((void *)avail_ram, (void *)initrd_start, info->initrd_size);
			initrd_start = (unsigned long)avail_ram;
			initrd_end = initrd_start + info->initrd_size;
			}
		}
	else {
		initrd_start = 0;
		initrd_end   = 0;
		}
	}
#else
	/* we have to subtract 0x10000 here to correct for objdump including
	   the size of the elf header which we strip -- Cort */
	_printk("\n******* Kernel Information *******\n");
	_printk("Hardware ID %02X\n", HWCtl_GetHardWare_ID());
	zimage_start = (char *)(load_addr - 0x10000 + ZIMAGE_OFFSET);
	zimage_size = ZIMAGE_SIZE;

	if ( INITRD_OFFSET )
		initrd_start = load_addr - 0x10000 + INITRD_OFFSET;
	else
		initrd_start = 0;
	initrd_end = INITRD_SIZE + initrd_start;

	/*
	 * Find a place to stick the zimage and initrd and
	 * relocate them if we have to. -- Cort
	 */
	avail_ram = (char *)PAGE_ALIGN((unsigned long)_end);
	puts("zimage at:     "); puthex((unsigned long)zimage_start);
	puts(" "); puthex((unsigned long)(zimage_size+zimage_start)); puts("\n");
	if ( (unsigned long)zimage_start <= 0x00800000 )
	{
		memcpy( (void *)avail_ram, (void *)zimage_start, zimage_size );
		zimage_start = (char *)avail_ram;
		puts("relocated to:  "); puthex((unsigned long)zimage_start);
		puts(" ");
		puthex((unsigned long)zimage_size+(unsigned long)zimage_start);
		puts("\n");

		/* relocate initrd */
		if ( initrd_start )
		{
			puts("initrd at:     "); puthex(initrd_start);
			puts(" "); puthex(initrd_end); puts("\n");
			avail_ram = (char *)PAGE_ALIGN(
				(unsigned long)zimage_size+(unsigned long)zimage_start);
			memcpy ((void *)avail_ram, (void *)initrd_start, INITRD_SIZE );
			initrd_start = (unsigned long)avail_ram;
			initrd_end = initrd_start + INITRD_SIZE;
			puts("relocated to:  "); puthex(initrd_start);
			puts(" "); puthex(initrd_end); puts("\n");
		}
	} else if ( initrd_start ) {
		if ((unsigned long)zimage_start >= 0xffc00000) {
			memcpy( (void *)avail_ram, (void *)zimage_start, zimage_size );
			zimage_start = (char *)avail_ram;
			puts("relocated to:  "); puthex((unsigned long)zimage_start);
			puts(" ");
			puthex((unsigned long)zimage_size+(unsigned long)zimage_start);
			puts("\n");

			puts("initrd at:     "); puthex(initrd_start);
			puts(" "); puthex(initrd_end); puts("\n");

			avail_ram = (char *)PAGE_ALIGN(
				(unsigned long)zimage_size+(unsigned long)zimage_start);
			{
				int i;
				unsigned long image_size = INITRD_SIZE;
				unsigned long copy_size;
				unsigned long addr = initrd_start;
				#define	DATA_SIZE 0x100000
				
				for (image_size = INITRD_SIZE; image_size > 0;) {
					copy_size = image_size > DATA_SIZE?DATA_SIZE:image_size;
					memcpy ((void *)avail_ram, (void *)addr, copy_size);
					image_size -= copy_size;
					addr += copy_size;
					avail_ram += copy_size;
				}
				avail_ram = (char *)PAGE_ALIGN(
					(unsigned long)zimage_size+(unsigned long)zimage_start);
			}

			initrd_start = (unsigned long)avail_ram;
			initrd_end = initrd_start + INITRD_SIZE;
			puts("relocated to:  "); puthex(initrd_start);
			puts(" "); puthex(initrd_end); puts("\n");
		} else {
			puts("initrd at:     "); puthex(initrd_start);
			puts(" "); puthex(initrd_end); puts("\n");
		}
	}
	
#endif
	
	avail_ram = (char *)0x00400000;
	end_avail = (char *)0x00800000;
	/* Display standard Linux/PPC boot prompt for kernel args */
	puts("\n>>");
	cp = cmd_line;
	*cp = 0;
	memcpy (cmd_line, cmd_preset, sizeof(cmd_preset));
	while ( *cp ) putc(*cp++);
	timer = 0;
	while (timer++ < 1*1000) {
		if (tstc()) {
			cp = input_cmd(cp,256);
			break;
		}
		udelay(10000);
	}
	*cp = 0;
	puts("\n");
	
#ifdef FLASH_BOOT_IMAGE
	i=0;while(i<5 && cmd_line[i] == "debug"[i]) ++i;
	if(i>4) {dbg_mode();}
#endif
	
	/* mappings on early boot can only handle 16M */
	puts("Now Loading...");
	gunzip(0, 0x400000, zimage_start, &zimage_size);
	puts("done.\n");

	outb(0x80004500, 0x49);
	outb(0x80004500, 0x49);
	outb(0x80004500, 0x49);
	outb(0x80004500, 0x49);

	puts("Now Booting\n");

}

#ifdef FLASH_BOOT_IMAGE
#define b2llong(a) \
	((a & 0xff000000)>>24) + ((a & 0x00ff0000)>>8)  + \
	((a & 0x0000ff00)<<8 ) + ((a & 0x000000ff)<<24) 
#define bit(a) 1<<a

#define DBG_OK 					0
#define DBG_NG_NOFLASH			1
#define DBG_NG_FAIL_ERASE		2
#define DBG_NG_OUT_OF_AREA		3
#define DBG_NG_FEW_PARAM		4
#define DBG_NG_SUM_ERROR		5
#define DBG_NG_CANT_WRITE		6

char* str_trim(char** pstr) {
	char* str = *pstr;
	char* trm;
	while(*str==' ') str++;
	trm = str;
	while(*str!=' ' && *str!='\0') str++;
	if(*str==' ') {*str=0;str++;}
	*pstr = str;
	return trm;
	}

unsigned long hextolong(char* str) {
	int           i   = 0;
	unsigned long ret = 0;
	while(i<8 && *str!=0) {
		if(*str>='0' && *str<='9')      ret = (ret<<4)+(*str-'0');
		else if(*str>='a' && *str<='f') ret = (ret<<4)+(*str - 'a'+10);
		else if(*str>='A' && *str<='f') ret = (ret<<4)+(*str - 'A'+10);
		str++;
		i++;
		}
	return ret;
	}

void dbg_chgrom(char* param)
{
	char* arg;
	volatile long* cpu_conf_addr = (long*)0xfec00000;
	volatile long* cpu_conf_data = (long*)0xfee00000;
	long           data;
	*cpu_conf_addr = b2llong((long)0x800000a8);
	data           = *cpu_conf_data;
	data           = b2llong(data);
	str_trim(&param);
	arg = str_trim(&param);
	if(*arg != '\0') {
		puts("Change current rom\n");
		if(arg[0]=='p' && arg[1]=='c' && arg[2]=='i')
			data &= ~(bit(20));
		else
			data |= bit(20);
			}
	data = b2llong(data);
	*cpu_conf_data = data;
	puts("Now current rom:");
	if(data & bit(20)) puts("on bord\n"); else puts("mini-pci\n");
}


unsigned long jiff_count = 0;
unsigned long dummy_jiffies(void) {
	udelay(1000);
	jiff_count++;
	return jiff_count;
	}
#define jiffies  dummy_jiffies()


#define printk   _printk
#include <melco/flashd.h>

struct flash_dev_func* flash_list[] = {
	&flash_toshiba,
	&flash_fujitsu,
	NULL
	};


int flash_name(char *param)
{
	if(flash==NULL) _printk("NOFLASH\n");
	else _printk("%s\n",flash->flash_name);
	return DBG_OK;
}

int erase_kernel(char *param)
{
	unsigned long addr = KERNEL_IMAGE_ADDRESS;
	unsigned long size = KERNEL_IMG_SIZE_MAX;
	int           tgl  = 0;
	if(flash == NULL) {
		return DBG_NG_NOFLASH;
		}
	while(size>0) {
		if(tgl) {HWCtl_Led_Down(LED_BLINK1);HWCtl_Led_Up(LED_BLINK2);}
		else    {HWCtl_Led_Down(LED_BLINK2);HWCtl_Led_Up(LED_BLINK1);}
		tgl = (tgl==0);
		jiff_count = 0;
		if(flash->blk_erase(addr,1)) {
			return DBG_NG_FAIL_ERASE;
			}
		addr += flash->blocksizes;
		size -= flash->blocksizes;
		}
	return DBG_OK;
}

int write_buff(char *param)
	{
	char* arg1;
	char* arg2;
	unsigned long  adr;
	char* buff = avail_ram;
	char  c;
	unsigned long *datap,data;
	int   i;
	static int tgl = 0;

	str_trim(&param);
	arg1 = str_trim(&param);
	arg2 = str_trim(&param);

	if(*arg1 != '\0' && arg2 != '\0') {
		adr = hextolong(arg1);
		if(adr >= 0x10000 || adr & 0x0f) return DBG_NG_OUT_OF_AREA;
		datap = (unsigned long*)(buff+adr);
		i=0;data=0;
		while((c=*arg2) != '\0') {
			data <<= 4;
			if     (c>='0' && c<='9') data += (c-'0');
			else if(c>='a' && c<='f') data += (c-'a'+10);
			else if(c>='A' && c<='f') data += (c-'A'+10);
			i++;
			if(i==8) {
				*datap  = data;
				datap++;
				i=0;data=0;
				}
			arg2++;
			}
		}
	else return DBG_NG_FEW_PARAM;

	if(tgl) {HWCtl_Led_Down(LED_BLINK1);HWCtl_Led_Up(LED_BLINK2);}
	else    {HWCtl_Led_Down(LED_BLINK2);HWCtl_Led_Up(LED_BLINK1);}
	tgl = (tgl==0);
	return DBG_OK;
	}

int write_flash(char *param)
	{
	char*         arg1;
	char*         arg2;
	char*         buff = avail_ram;
	unsigned long adr;
	unsigned long sum;
	if(flash == NULL) {
		return DBG_NG_NOFLASH;
		}
	str_trim(&param);
	arg1 = str_trim(&param);
	arg2 = str_trim(&param);
	if(*arg1 != '\0' && arg2 != '\0') {
		adr  = hextolong(arg1);
		sum  = hextolong(arg2);
		if(sum != checksum_check(buff,0x10000)) {
			_printk("checksum fail:%08X %08X\n",sum,checksum_check(buff,0x10000));
			return DBG_NG_SUM_ERROR;
			}
		if(adr<KERNEL_IMAGE_ADDRESS
		|| adr>KERNEL_IMAGE_ADDRESS+KERNEL_IMG_SIZE_MAX
		|| adr+0x10000>KERNEL_IMAGE_ADDRESS+KERNEL_IMG_SIZE_MAX
		|| (adr & 0x0000ffff) > 0) {
			return DBG_NG_OUT_OF_AREA;
			}

		{HWCtl_Led_Down(LED_BLINK1);HWCtl_Led_Up(LED_BLINK2);}
		if(flash->blk_erase(adr,1)) {
			return DBG_NG_FAIL_ERASE;
			}
		{HWCtl_Led_Down(LED_BLINK2);HWCtl_Led_Up(LED_BLINK1);}
		if(flash->blk_write((unsigned long)adr,buff,0x10000)) {
			return DBG_NG_CANT_WRITE;
			}
		}
	else return DBG_NG_FEW_PARAM;
	return DBG_OK;
	}


int memory_modify_byte(char *param)
	{
	char*         arg1;
	char*         arg2;
	unsigned long  adr;
	unsigned char  data;
	unsigned char* tgt;
	str_trim(&param);
	arg1 = str_trim(&param);
	arg2 = str_trim(&param);
	if(*arg1 != '\0' && arg2 != '\0') {
		adr   = hextolong(arg1);
		data  = (unsigned char)(hextolong(arg2) & 0xff);
		_printk("mmb [%08X]<%02X\n",adr,data);
		tgt = (unsigned char*)adr;
		*tgt = data;
		}
	else return DBG_NG_FEW_PARAM;
	return DBG_OK;
	}

int memory_display(char *param)
	{
	char*         arg1;
	char*         arg2;
	static unsigned long  adr = 0;
	unsigned char* tgt;
	int i,j;
	str_trim(&param);
	arg1 = str_trim(&param);
	if(*arg1 != '\0') {
		adr   = hextolong(arg1);
		}
	for(j=0;j<8;++j) {
		_printk("%08X :",adr);
		tgt = (unsigned char*)adr;
		for(i=0;i<16;++i) {
			if(i%4==0) _printk(" ");
			_printk("%02X",*tgt);
			tgt++;
		}
		_printk("\n");
		adr += 16;
	}
	return DBG_OK;
	}

int dbg_ledstop(char* p)
{
	timer_proc = NULL;
	return 0;
}

struct dbg_cmdlist {
		const char* cmd;
		int (*func)(char *param);
};
int dbg_help(char* p);

struct dbg_cmdlist dbg_commands[] = {
	{"ledstop",		dbg_ledstop},
	{"flash",		flash_name},
	{"erasek",		erase_kernel},
	{"writeb",		write_buff},
	{"writef",		write_flash},
	{"mmb",			memory_modify_byte},
	{"md",			memory_display},
	{"help",		dbg_help},
	{"quit",		NULL},
	{NULL,			NULL}
	};

int dbg_help(char* p)
{
	struct dbg_cmdlist* list = dbg_commands;
	int j,i;
	_printk("commands:\n");
	i=0;
	while(list->cmd != NULL) {
		j=0;
		if(i>3) {_printk("\n");i=0;}
		while(list->cmd[j]>31 && j<16) {_printk("%c",list->cmd[j]);j++;}
		while(j<17) {_printk(" ");j++;}
		list++;
		i++;
		}
	_printk("\n");
	return 0;
}

void dbg_mode(void)
{
	int  i;
	char cmd[256];
	struct dbg_cmdlist* list;
	_printk("[Alart mode]\n");
	i=0;
	while((flash = flash_list[i])!=NULL) {
		if(flash->probe()) break;
		++i;
		}
	while(1) {
		puts("#");input_cmd(cmd,256);
		putc('\n');
		if(*cmd==0) continue;
		list = dbg_commands;
		while(list->cmd != NULL) {
			i=0;
			while(list->cmd[i] != 0 && list->cmd[i] == cmd[i]) i++;
			if(list->cmd[i]=='\0') {
				if(list->func == NULL) return;
				printk(">RET:%d\n",list->func(cmd));
				break;
			}
			list++;
		}
	if(list->cmd == NULL) _printk("Invalid cmd:%s\n",cmd);
	}
}

void print_pci_config_reg(unsigned long idsel, unsigned long fn) {

	unsigned int offset;
	unsigned long reg, in_reg;

	reg = (0x80 << 24) + (idsel << 11) + (fn << 8);

	_printk("PCI IDSEL%d device configuration registers\n", idsel);
	for (offset = 0x00; offset < 0x40; offset += 4) {
		out_ledw(0xFEC00000, (reg | (offset << 2)) );
		_printk("%02X: %08X\n", offset, in_ledw(0xFEE00000));
	}

}
#endif // #ifdef FLASH_BOOT_IMAGE


