/*
 * arch/ppc/kernel/gt64260_pic.c
 * 
 * Interrupt controller support for Galileo's GT64260.
 *
 * Author: Chris Zankel <chris@mvista.com>
 *
 * Based on sources from Rabeeh Khoury / Galileo Technology
 *
 * Copyright 2001 MontaVista Software Inc.
 *
 * This program is free software; you can redistribute  it and/or modify it
 * under  the terms of  the GNU General  Public License as published by the
 * Free Software Foundation;  either version 2 of the  License, or (at your
 * option) any later version.
 */

/*
 * This file contains the specific functions to support the GT64260
 * interrupt controller.
 *
 * The GT64260 has two main interrupt registers (high and low) that
 * summarizes the interrupts generated by the units of the GT64260.
 * Each bit is assigned to an interrupt number, where the low register
 * are assigned from IRQ0 to IRQ31 and the high cause register
 * from IRQ32 to IRQ63 
 * The GPP (General Purpose Port) interrupts are assigned from IRQ64 (GPP0) 
 * to IRQ95 (GPP31). 
 * get_irq() returns the lowest interrupt number that is currently asserted.
 *
 * Note: 
 *  - This driver does not initialize the GPP when used as an interrupt 
 *    input.
 */

#include <linux/stddef.h>
#include <linux/init.h>
#include <linux/sched.h>
#include <linux/signal.h>
#include <linux/stddef.h>
#include <linux/delay.h>
#include <linux/irq.h>

#include <asm/io.h>
#include <asm/processor.h>
#include <asm/system.h>
#include <asm/irq.h>
#include <asm/gt64260.h>


/* ========================== forward declaration ========================== */

static void gt64260_unmask_irq(unsigned int);
static void gt64260_mask_irq(unsigned int);
static void gt64260_mask_and_ack_irq(unsigned int);
static void gt64260_end_irq(unsigned int irq);

/* ========================== local declarations =========================== */

struct hw_interrupt_type gt64260_pic = {
	" GT64260_PIC ",		/* typename */
	NULL,				/* startup */
	NULL,				/* shutdown */
	gt64260_unmask_irq,		/* enable */
	gt64260_mask_irq,		/* disable */
	gt64260_mask_and_ack_irq,	/* ack */
	gt64260_end_irq,		/* end */
	NULL				/* set_affinity */
};

/* gt64260_init_irq()
 *
 *  This function initializes the interrupt controller. It assigns
 *  all interrupts from IRQ0 to IRQ95 to the gt64260 interrupt controller.
 *
 * Input Variable(s):
 *  None.
 *
 * Outpu. Variable(s):
 *  None.
 *
 * Returns:
 *  void
 *
 * Note:
 *  We register all GPP inputs as interrupt source, but disable them.
 */

__init void
gt64260_init_irq(void)
{
	int i;

	if ( ppc_md.progress ) ppc_md.progress("gt64260_init_irq: enter", 0x0);

	/* cached interrupt masks for high and low interrupt mask register */

	ppc_cached_irq_mask[0] = 0;
	/* we don't want CPU0 to take interrupts for GPP[31:24] */
	/* ppc_cached_irq_mask[1] = 0x0f000000; */
	/* use this mask instead - ted peters */
	ppc_cached_irq_mask[1] = 0x0f000000;
	ppc_cached_irq_mask[2] = 0;

	/* disable all interrupts and clear current interrupts */

	GT64260_REG_WRITE(GT64260_GPP_INTR_MASK, 0);/* disable and ... */
	GT64260_REG_WRITE(GT64260_GPP_INTR_CAUSE,0);/* clear all gpp intrs */
	GT64260_REG_WRITE(GT64260_IC_CPU_INTR_MASK_LO, 0);/* disable intrs */

	/* we don't want CPU0 to take interrupts for GPP[31:24] */
	/* GT64260_REG_WRITE(GT_PIC_HIMR, 0x0f000000); */
	/* use this mask instead - ted peters */
	GT64260_REG_WRITE(GT64260_IC_CPU_INTR_MASK_HI, 0x0f000000); /*enable GPP (secondary mask)*/
	GT64260_REG_WRITE(GT64260_IC_MAIN_CAUSE_LO, 0);   //This is really a read-only register.
	GT64260_REG_WRITE(GT64260_IC_MAIN_CAUSE_HI, 0);   //This is really a read-only register.

	/* use the gt64260 for all (possible) interrupt sources */

	for ( i = 0 ; i < 96 ; i++ )
		irq_desc[i].handler = &gt64260_pic;

	if ( ppc_md.progress ) ppc_md.progress("gt64260_init_irq: exit", 0x0);
	return;
}


/* gt64260_get_irq()
 *
 *  This function returns the lowest interrupt number of all interrupts that
 *  are currently asserted. 
 *
 * Input Variable(s):
 *  struct pt_regs*	not used
 *
 * Output Variable(s):
 *  None.
 *
 * Returns:
 *  int	<interrupt number> or -2 (bogus interrupt)
 * 
 */


int
gt64260_get_irq(struct pt_regs *regs)
{
	int irq;
	int irq_gpp;

	/* __ilog2 returns the msb or -1 if all bits are 0 */
	
	irq = GT64260_REG_READ(GT64260_IC_MAIN_CAUSE_LO);
	irq = __ilog2(irq & 0x3ffffffe /* XXXX */);

	if (irq == -1) {
		irq = GT64260_REG_READ(GT64260_IC_MAIN_CAUSE_HI);
		irq = __ilog2(irq & 0x0f000db7 /* XXXX */);

		if (irq == -1) {
			irq = -2;   /* bogus interrupt, should never happen */
		} else {
			if (irq >= 24) {
				irq_gpp = GT64260_REG_READ(GT64260_GPP_INTR_CAUSE);
				irq_gpp = __ilog2(irq_gpp &
						   ppc_cached_irq_mask[2]);

				if (irq_gpp == -1) {
					irq = -2;
				} else {
					irq = irq_gpp + 64;
					GT64260_REG_WRITE(GT64260_GPP_INTR_CAUSE, ~(1<<(irq-64)));
				}
			}
			else {
				irq += 32;
			}
		}
	}

	return irq;
}

/* gt64260_unmask_irq()
 *
 *  This function enables an interrupt.
 *
 * Input Variable(s):
 *  unsigned int	interrupt number (IRQ0...IRQ95).
 *
 * Output Variable(s):
 *  None.
 *
 * Returns:
 *  void
 */

static void
gt64260_unmask_irq(unsigned int irq)
{
	if (irq > 31) {
		if (irq > 63) {
			/* unmask GPP irq */
			GT64260_REG_WRITE(GT64260_GPP_INTR_MASK,  
				     ppc_cached_irq_mask[2] |= (1<<(irq-64)));
		} else {
			/* mask high interrupt register */
			GT64260_REG_WRITE(GT64260_IC_CPU_INTR_MASK_HI, 
				     ppc_cached_irq_mask[1] |= (1<<(irq-32)));
		}
	} else {
		/* mask low interrupt register */
		GT64260_REG_WRITE(GT64260_IC_CPU_INTR_MASK_LO, 
			     ppc_cached_irq_mask[0] |= (1<<irq));
	}
}


/* gt64260_mask_irq()
 *
 *  This funktion disables the requested interrupt.
 *
 * Input Variable(s):
 *  unsigned int	interrupt number (IRQ0...IRQ95).
 *
 * Output Variable(s):
 *  None.
 *
 * Returns:
 *  void
 */

static void
gt64260_mask_irq(unsigned int irq)
{
	if (irq > 31) {
		if (irq > 63) {
			/* mask GPP irq */
			GT64260_REG_WRITE(GT64260_GPP_INTR_MASK,  
				     ppc_cached_irq_mask[2] &= ~(1<<(irq-64)));
		} else {
			/* mask high interrupt register */
			GT64260_REG_WRITE(GT64260_IC_CPU_INTR_MASK_HI, 
				     ppc_cached_irq_mask[1] &= ~(1<<(irq-32)));
		}
	} else {
		/* mask low interrupt register */
		GT64260_REG_WRITE(GT64260_IC_CPU_INTR_MASK_LO, 
			     ppc_cached_irq_mask[0] &= ~(1<<irq));
	}
}


/* gt64260_mask_irq()
 *
 *  This funktion disables and acknowledges the requested interrupt.
 *
 * Input Variable(s):
 *  unsigned int	interrupt number (IRQ0...IRQ95).
 *
 * Output Variable(s):
 *  None.
 *
 * Returns:
 *  void
 *
 * Note:
 *  - Only the GPP interrupts can be acknowledged. 
 */

static void
gt64260_mask_and_ack_irq(unsigned int irq)
{
	gt64260_mask_irq(irq);
#if 0	/* XXXX */
	gt64260_end_irq(irq);
#endif
}


#if 0	/* XXXX */
/* gt64260_post_irq()
 *
 *  This routine makes sure that we didn't miss one interrupt because
 *  the GPP interrupts are edge triggered.
 *
 * Input Variable(s):
 *  struct pt_regs*	register,
 *  int			interrupt number.
 *
 * Output Variable(s):
 *  None.
 *
 * Returns:
 *  void
 *
 */

static void
gt64260_post_irq(struct pt_regs *regs, int irq) 
{
	if (irq >= 64) {
again:	
		GT64260_REG_WRITE(GT_GPP_ICR, ~(1<<(irq-64)));
		if ((GTREGREAD(GPP_VALUE)
		     & ppc_cached_irq_mask[2] 
		     & (1<<(irq-64))) != 0) {
			ppc_irq_dispatch_handler(regs, irq);
			goto again;
		}
	}
}
#endif


static void
gt64260_end_irq(unsigned int irq)
{
	u32	val;

        if (!(irq_desc[irq].status & (IRQ_DISABLED|IRQ_INPROGRESS))) {
                gt64260_unmask_irq(irq);
	}

	if (irq > 63) {
again:
		val = GT64260_REG_READ(GT64260_GPP_VALUE);

		if ((val & ppc_cached_irq_mask[2] & (1<<(irq-64))) != 0) {
			GT64260_REG_WRITE(GT64260_GPP_INTR_CAUSE,
					  ~(1<<(irq-64)));
			ppc_irq_dispatch_handler(NULL, irq);
			goto again;
		}
	}

	return;
} 
