/*
 *  LinkStation HW Control Driver
 *
 *  Copyright (C) 2001-2004  BUFFALO INC.
 *
 *  This software may be used and distributed according to the terms of
 *  the GNU General Public License (GPL), incorporated herein by reference.
 *  Drivers based on or derived from this code fall under the GPL and must
 *  retain the authorship, copyright and license notice.  This file is not
 *  a complete program and may only be used when the entire operating
 *  system is licensed under the GPL.
 *
 */

#include <melco/melco_hwctl.h>

enum tHardwareCtlType  HW_Type = HWTYPE_UNKNOWN;

#define	MELCO_PLD_BASE		0xff000000
#define	MELCO_PLD_SYS		0xff000000
 #define	FLASH_WRITE_E	0x7f
 #define	FLASH_WRITE_D	0x80
#define	MELCO_PLD_LED		0xff000001
 #define LED_WIRELESS1_E	0x40
 #define LED_WIRELESS1_D	0xbF
 #define LED_WIRELESS2_E	0x20
 #define LED_WIRELESS2_D	0xdf
 #define LED_DIAG_E			0x10
 #define LED_DIAG_D			0xef
#define	MELCO_PLD_HWID		0xff000002
#define MELCO_GET_HWID		((*(volatile char*)0xff000002 & 0xf8) | ((*(volatile char*)0xff000003 & 0xE0) >> 5))
#define	MELCO_PLD_STATUS	0xff000003
 #define	INIT_PUSH		0x10
#define	MELCO_PLD_WDT		0xff000006
#define NONE_PLD_LED_WIRELESS   (1<<7)
#define NONE_PLD_LED_DIAG_FRONT (1<<6)
#define NONE_PLD_LED_DIAG_SIDE  (1<<5)
#define NONE_PLD_WRITE_ENABLE   (1<<4)

unsigned char Status_None_PLD;

#define NONE_PLD_CTL   ((volatile unsigned char*)0xFF200000)
#define NONE_PLD_WDT   ((volatile unsigned char*)0xFF100000)
#define NONE_PLD_INIT  ((volatile unsigned char*)0xFF000000)
#define INIT_SWITCH_MASK  0x01

#define PCI_CONFIG_ADDR		0xFEC00000
#define PCI_CONFIG_DATA		0xFEE00000

#define PCI_IOSPACE_BASE	0xFE000000
#define PCI_MEMSPACE_BASE	0xBFFFF000

#define UART1				0x80004500

enum tHardwareCtlType HWCtl_Init(void)
{
	*(volatile unsigned char*)MELCO_PLD_SYS = FLASH_WRITE_E;
	*(volatile unsigned char*)MELCO_PLD_LED = 0x00;

	if((*(volatile unsigned char*)MELCO_PLD_SYS & 0xf0) == (FLASH_WRITE_E & 0xf0))
		HW_Type = HWTYPE_WITH_PLD_MODEL;
	else 
		HW_Type = HWTYPE_NONE_PLD_MODEL;

	switch(HW_Type) {
		case HWTYPE_WITH_PLD_MODEL:
			*((volatile unsigned char*)MELCO_PLD_SYS) = FLASH_WRITE_D;
			*((volatile unsigned char*)MELCO_PLD_LED) = 0x0;
			break;
		case HWTYPE_NONE_PLD_MODEL:
			Status_None_PLD = NONE_PLD_LED_WIRELESS   |
			                  NONE_PLD_LED_DIAG_FRONT |
			                  NONE_PLD_LED_DIAG_SIDE;
			*NONE_PLD_CTL   = Status_None_PLD;
			break;
		default:
			Status_None_PLD = NONE_PLD_LED_WIRELESS   |
			                  NONE_PLD_LED_DIAG_FRONT |
			                  NONE_PLD_LED_DIAG_SIDE;
			*NONE_PLD_CTL   = Status_None_PLD;
			break;
	}
	return HW_Type;
}

enum tHardwareCtlType HWCtl_GetType(void)
{
	return HW_Type;
}

int HWCtl_Led_Up(enum tLedList led)
{
	switch(HW_Type) {
		case HWTYPE_WITH_PLD_MODEL: 
			{
			volatile unsigned char pld_led = *((volatile unsigned char*)MELCO_PLD_LED);
			if(led==LED_DIAG)        pld_led |= LED_DIAG_E;
			if(led==LED_WIRELESS)    pld_led |= LED_WIRELESS1_E;
			if(led==LED_WIRELESS_EX) pld_led |= LED_WIRELESS2_E;
			if(led==LED_BLINK1)      pld_led |= LED_DIAG_E;
			if(led==LED_BLINK2)      pld_led |= LED_WIRELESS1_E;
			if(led==LED_ALL)         pld_led |= LED_WIRELESS1_E | LED_DIAG_E | LED_WIRELESS2_E;
			*((volatile unsigned char*)MELCO_PLD_LED) = pld_led;
			}
			return 0;
		case HWTYPE_NONE_PLD_MODEL:
			if(led==LED_DIAG)       Status_None_PLD &= ~(NONE_PLD_LED_DIAG_FRONT | NONE_PLD_LED_DIAG_SIDE);
			if(led==LED_WIRELESS)   Status_None_PLD &= ~(NONE_PLD_LED_WIRELESS);
			if(led==LED_BLINK1)     Status_None_PLD &= ~(NONE_PLD_LED_DIAG_FRONT);
			if(led==LED_BLINK2)     Status_None_PLD &= ~(NONE_PLD_LED_DIAG_SIDE);
			if(led==LED_ALL)        Status_None_PLD &= ~(NONE_PLD_LED_DIAG_FRONT | NONE_PLD_LED_DIAG_SIDE | NONE_PLD_LED_WIRELESS);
			*NONE_PLD_CTL   = Status_None_PLD;
			return 0;
		default:
			if(led==LED_DIAG)       Status_None_PLD &= ~(NONE_PLD_LED_DIAG_FRONT | NONE_PLD_LED_DIAG_SIDE);
			if(led==LED_WIRELESS)   Status_None_PLD &= ~(NONE_PLD_LED_WIRELESS);
			if(led==LED_BLINK1)     Status_None_PLD &= ~(NONE_PLD_LED_DIAG_FRONT);
			if(led==LED_BLINK2)     Status_None_PLD &= ~(NONE_PLD_LED_DIAG_SIDE);
			if(led==LED_ALL)        Status_None_PLD &= ~(NONE_PLD_LED_DIAG_FRONT | NONE_PLD_LED_DIAG_SIDE | NONE_PLD_LED_WIRELESS);
			*NONE_PLD_CTL   = Status_None_PLD;
			return 0;
	}
	return -1;
}

int HWCtl_Led_Down(enum tLedList led)
{
	switch(HW_Type) {
		case HWTYPE_WITH_PLD_MODEL: 
			{
			volatile unsigned char pld_led = *((volatile unsigned char*)MELCO_PLD_LED);
			if(led==LED_DIAG)        pld_led &= ~LED_DIAG_E;
			if(led==LED_WIRELESS)    pld_led &= ~LED_WIRELESS1_E;
			if(led==LED_WIRELESS_EX) pld_led &= ~LED_WIRELESS2_E;
			if(led==LED_BLINK1)      pld_led &= ~LED_DIAG_E;
			if(led==LED_BLINK2)      pld_led &= ~LED_WIRELESS1_E;
			if(led==LED_ALL)         pld_led &= ~(LED_WIRELESS1_E | LED_DIAG_E | LED_WIRELESS2_E);
			*((volatile char*)MELCO_PLD_LED) = pld_led;
			}
			return 0;
		case HWTYPE_NONE_PLD_MODEL:
			if(led==LED_DIAG)       Status_None_PLD |= (NONE_PLD_LED_DIAG_FRONT | NONE_PLD_LED_DIAG_SIDE);
			if(led==LED_WIRELESS)   Status_None_PLD |= (NONE_PLD_LED_WIRELESS);
			if(led==LED_BLINK1)     Status_None_PLD |= (NONE_PLD_LED_DIAG_FRONT);
			if(led==LED_BLINK2)     Status_None_PLD |= (NONE_PLD_LED_DIAG_SIDE);
			if(led==LED_ALL)        Status_None_PLD |= (NONE_PLD_LED_DIAG_FRONT | NONE_PLD_LED_DIAG_SIDE | NONE_PLD_LED_WIRELESS);
			*NONE_PLD_CTL   = Status_None_PLD;
			return 0;
		default:
			if(led==LED_DIAG)       Status_None_PLD |= (NONE_PLD_LED_DIAG_FRONT | NONE_PLD_LED_DIAG_SIDE);
			if(led==LED_WIRELESS)   Status_None_PLD |= (NONE_PLD_LED_WIRELESS);
			if(led==LED_BLINK1)     Status_None_PLD |= (NONE_PLD_LED_DIAG_FRONT);
			if(led==LED_BLINK2)     Status_None_PLD |= (NONE_PLD_LED_DIAG_SIDE);
			if(led==LED_ALL)        Status_None_PLD |= (NONE_PLD_LED_DIAG_FRONT | NONE_PLD_LED_DIAG_SIDE | NONE_PLD_LED_WIRELESS);
			*NONE_PLD_CTL   = Status_None_PLD;
			return 0;
	}
	return -1;
}

int HWCtl_Rom_Write_Enable(void)
{
	switch(HW_Type) {
		case HWTYPE_WITH_PLD_MODEL: 
			*((volatile unsigned char*)MELCO_PLD_SYS) = FLASH_WRITE_E;
			return 0;
		case HWTYPE_NONE_PLD_MODEL:
			Status_None_PLD |= NONE_PLD_WRITE_ENABLE;
			*NONE_PLD_CTL   = Status_None_PLD;
			return 0;
		default:
			Status_None_PLD |= NONE_PLD_WRITE_ENABLE;
			*NONE_PLD_CTL   = Status_None_PLD;
			return 0;
	}
	return -1;
}

int HWCtl_Rom_Write_Disable(void)
{
	switch(HW_Type) {
		case HWTYPE_WITH_PLD_MODEL: 
			*((volatile unsigned char*)MELCO_PLD_SYS) = FLASH_WRITE_D;
			return 0;
		case HWTYPE_NONE_PLD_MODEL:
			Status_None_PLD &= ~NONE_PLD_WRITE_ENABLE;
			*NONE_PLD_CTL   = Status_None_PLD;
			return 0;
		default:
			Status_None_PLD &= ~NONE_PLD_WRITE_ENABLE;
			*NONE_PLD_CTL   = Status_None_PLD;
			return 0;
	}
	return -1;
}

unsigned char HWCtl_GetHardWare_ID(void)
{
	unsigned char ret = 0x00;
	switch(HW_Type) {
		case HWTYPE_WITH_PLD_MODEL: 
			ret = MELCO_GET_HWID;
			break;
		default:
			break;
	}
	return ret;
}

int HWCtl_GetInitSwitch(void) 
{
	switch(HW_Type) {
		case HWTYPE_WITH_PLD_MODEL: 
			return ((*(unsigned char*)MELCO_PLD_STATUS & INIT_PUSH) == 0);
		case HWTYPE_NONE_PLD_MODEL:
			return ((*NONE_PLD_INIT & INIT_SWITCH_MASK)==0);
		default:
			return ((*NONE_PLD_INIT & INIT_SWITCH_MASK)==0);
	}
	return 0;
}

int HWCtl_Watchdog_Knock(void)
{
	int ret = 0;
	switch(HW_Type) {
		case HWTYPE_WITH_PLD_MODEL: 
			*((volatile unsigned char*)MELCO_PLD_WDT) = 0xff;
			break;
		case HWTYPE_NONE_PLD_MODEL:
			{
			*NONE_PLD_WDT = 0xf0;
			}
			break;
		default:
			{
			*NONE_PLD_WDT = 0xf0;
			}
			break;
	}
	return ret;
}

int check_pci_device(int idsel, unsigned short vid, unsigned short did, int fn) 
{
	unsigned int offset;
	unsigned long base, in_reg;

	base = (0x80 << 24) + (idsel << 11) + (fn << 8);

	out_ledw((unsigned long *)PCI_CONFIG_ADDR, base);
	in_reg = in_ledw((unsigned long *)PCI_CONFIG_DATA);

	if ((did == (unsigned short)(in_reg >> 16)) && 
		(vid == (unsigned short)(in_reg & 0x0000ffff)))
		return 0;
	else
		return -1;
}

int boot_init_ide_controller(int idsel, unsigned short device)
{
	int offset = 0;
	unsigned int base;

	if (device != DID_SiI680)
		return -1;

	switch (idsel) {
		case 11:
			base = 0x80005800UL;
			break;
		case 12:
			base = 0x80006000UL;
			break;
		case 13:
			base = 0x80006800UL;
			break;
		default:
			base = 0x80006000UL;
			break;
	}

	offset = 0x04;
	out_ledw((unsigned long *)PCI_CONFIG_ADDR, (base | (offset << 2)));
	out_ledw((unsigned long *)PCI_CONFIG_DATA, 0x00000003);

	offset = 0x08;
	out_ledw((unsigned long *)PCI_CONFIG_ADDR, (base | (offset << 2)));
	out_ledw((unsigned long *)PCI_CONFIG_DATA, 0x01018501);

	if (device == DID_SiI680)
	{
		offset = 0x10;
		out_ledw((unsigned long *)PCI_CONFIG_ADDR, (base | (offset << 2)));
		out_ledw((unsigned long *)PCI_CONFIG_DATA, 0x00BFFEF8);
		
		offset = 0x14;
		out_ledw((unsigned long *)PCI_CONFIG_ADDR, (base | (offset << 2)));
		out_ledw((unsigned long *)PCI_CONFIG_DATA, 0x00BFFEF4);

		offset = 0x18;
		out_ledw((unsigned long *)PCI_CONFIG_ADDR, (base | (offset << 2)));
		out_ledw((unsigned long *)PCI_CONFIG_DATA, 0x00BFFEE8);

		offset = 0x1C;
		out_ledw((unsigned long *)PCI_CONFIG_ADDR, (base | (offset << 2)));
		out_ledw((unsigned long *)PCI_CONFIG_DATA, 0x00BFFEE4);

		offset = 0x20;
		out_ledw((unsigned long *)PCI_CONFIG_ADDR, (base | (offset << 2)));
		out_ledw((unsigned long *)PCI_CONFIG_DATA, 0x00BFFED0);

		offset = 0x24;
		out_ledw((unsigned long *)PCI_CONFIG_ADDR, (base | (offset << 2)));
		out_ledw((unsigned long *)PCI_CONFIG_DATA, (PCI_MEMSPACE_BASE + 0xB00));
	}

	return 0;

}

void blink_led(unsigned char state)
{
	out_b(UART1,state);
	out_b(UART1,state);
	out_b(UART1,state);
	out_b(UART1,state);
}

inline void out_b(volatile unsigned char *addr, int val)
{
	__asm__ __volatile__("stb%U0%X0 %1,%0; eieio" : "=m" (*addr) : "r" (val));
}

inline int in_b(volatile unsigned char *addr)
{
	int ret;

	__asm__ __volatile__("lbz%U1%X1 %0,%1; eieio" : "=r" (ret) : "m" (*addr));
	return ret;
}

inline unsigned long in_ledw(volatile unsigned long *addr)
{ 
	unsigned long ret;

	__asm__ __volatile__("lwbrx %0,0,%1; eieio" : "=r" (ret) :
			     "r" (addr), "m" (*addr));
	return ret;
}

inline unsigned long in_bedw(volatile unsigned long *addr)
{ 
	unsigned long ret;

	__asm__ __volatile__("lwz%U1%X1 %0,%1; eieio" : "=r" (ret) : "m" (*addr));
	return ret;
}

inline void out_ledw(volatile unsigned long *addr, unsigned long val)
{ 
	__asm__ __volatile__("stwbrx %1,0,%2; eieio" : "=m" (*addr) :
			     "r" (val), "r" (addr));
}

inline void out_bedw(volatile unsigned long *addr, unsigned long val)
{ 
	__asm__ __volatile__("stw%U0%X0 %1,%0; eieio" : "=m" (*addr) : "r" (val));
}

inline int in_lew(volatile unsigned short *addr)
{
	int ret;

	__asm__ __volatile__("lhbrx %0,0,%1; eieio" : "=r" (ret) :
			      "r" (addr), "m" (*addr));
	return ret;
}

inline int in_bew(volatile unsigned short *addr)
{
	int ret;

	__asm__ __volatile__("lhz%U1%X1 %0,%1; eieio" : "=r" (ret) : "m" (*addr));
	return ret;
}

inline void out_lew(volatile unsigned short *addr, int val)
{
	__asm__ __volatile__("sthbrx %1,0,%2; eieio" : "=m" (*addr) :
			      "r" (val), "r" (addr));
}

inline void out_bew(volatile unsigned short *addr, int val)
{
	__asm__ __volatile__("sth%U0%X0 %1,%0; eieio" : "=m" (*addr) : "r" (val));
}
