/*
 * BK Id: SCCS/s.qspan_pci.c 1.8 11/04/01 21:07:38 paulus
 */
/*
 * QSpan pci routines.
 * Most 8xx boards use the QSpan PCI bridge.  The config address register
 * is located 0x500 from the base of the bridge control/status registers.
 * The data register is located at 0x504.
 * This is a two step operation.  First, the address register is written,
 * then the data register is read/written as required.
 */

#undef QSPAN_DEBUG
#undef QSPAN_BUS0_DEVFN0	/* define this to force bus 0, devfn 0
				 * configuration space accesses to the
				 * configuration space for the QSpan
				 * bridge
				 */

#include <linux/config.h>
#include <linux/kernel.h>
#include <linux/pci.h>
#include <linux/delay.h>
#include <linux/string.h>
#include <linux/init.h>

#include <asm/io.h>
#include <asm/byteorder.h>
#include <asm/io.h>
#include <asm/irq.h>
#include <asm/uaccess.h>
#include <asm/mpc8xx.h>
#include <asm/system.h>
#include <asm/machdep.h>
#include <asm/prom.h>
#include <asm/pci-bridge.h>


/*
 * This blows......
 * When reading the configuration space, if something does not respond
 * the bus times out and we get a machine check interrupt.  So, the
 * good ol' exception tables come to mind to trap it and return some
 * value.
 *
 * On an error we just return a -1, since that is what the caller wants
 * returned if nothing is present.  I copied this from __get_user_asm,
 * with the only difference of returning -1 instead of EFAULT.
 * There is an associated hack in the machine check trap code.
 *
 * The QSPAN is also a big endian device, that is it makes the PCI
 * look big endian to us.  This presents a problem for the Linux PCI
 * functions, which assume little endian.  For example, we see the
 * first 32-bit word like this:
 *	------------------------
 *	| Device ID | Vendor ID |
 *	------------------------
 * If we read/write as a double word, that's OK.  But in our world,
 * when read as a word, device ID is at location 0, not location 2 as
 * the little endian PCI would believe.  We have to switch bits in
 * the PCI addresses given to us to get the data to/from the correct
 * byte lanes.
 *
 * The QSPAN only supports 4 bits of "slot" in the dev_fn instead of 5.
 * It always forces the MS bit to zero.  Therefore, dev_fn values
 * greater than 128 are returned as "no device found" errors.
 *
 * The QSPAN can only perform long word (32-bit) configuration cycles.
 * The "offset" must have the two LS bits set to zero.  Read operations
 * require we read the entire word and then sort out what should be
 * returned.  Write operations other than long word require that we
 * read the long word, update the proper word or byte, then write the
 * entire long word back.
 *
 * PCI Bridge hack.  We assume (correctly) that bus 0 is the primary
 * PCI bus from the QSPAN.  If we are called with a bus number other
 * than zero, we create a Type 1 configuration access that a downstream
 * PCI bridge will interpret.
 */

#define __get_qspan_pci_config(x, addr, op)		\
	__asm__ __volatile__(				\
		"1:	"op" %0,0(%1)\n"		\
		"	eieio\n"			\
		"2:\n"					\
		".section .fixup,\"ax\"\n"		\
		"3:	li %0,-1\n"			\
		"	b 2b\n"				\
                ".previous\n"                           \
		".section __ex_table,\"a\"\n"		\
		"	.align 2\n"			\
		"	.long 1b,3b\n"			\
                ".previous"                             \
		: "=r"(x) : "r"(addr))

static u32 get_qspan_pci_config(u32 *ptr_cfg_data)
{
	u32 cfg_data;
	
	__get_qspan_pci_config(cfg_data, ptr_cfg_data, "lwz");
	
	return cfg_data;
}

#define QS_CONFIG_ADDR	(PCI_CSR_ADDR + 0x500)
#define QS_CONFIG_DATA	(PCI_CSR_ADDR + 0x504)

#define mk_config_addr_type0(bus, dev, offset) \
	(((bus)<<16) | ((dev)<<8) | ((offset) & 0xfc))

#define mk_config_addr_type1(bus, dev, offset) \
	(mk_config_addr_type0(bus, dev, offset) | 1)

int qspan_read_config_byte(struct pci_dev *dev, int offset, u8 *val)
{
        struct pci_controller *hose = dev->sysdata;
	u32 cfg_data;
	u8 *cp;

#ifdef QSPAN_BUS0_DEVFN0
	if (!dev->bus->number && !dev->devfn) {
		cfg_data = in_be32((volatile unsigned *)
			(PCI_CSR_ADDR + (offset & 0xfc)));
		cp = ((u8 *)&cfg_data) + ((offset ^ 0x03) & 0x03);
		*val = *cp;

#ifdef QSPAN_DEBUG
		printk("qspan_read_config_byte:   (bus, devfn, offset, val) = "
			"(%02X, %02X, %02X, %02X)\n",
			dev->bus->number, dev->devfn, offset, *val);
#endif
	
		return PCIBIOS_SUCCESSFUL;		
	}
#endif

	if ((dev->bus->number > 7) || (dev->devfn > 127)) {
		*val = 0xff;
		return PCIBIOS_DEVICE_NOT_FOUND;
	}

	if (dev->bus->number == 0)
        	out_be32(hose->cfg_addr,
        		mk_config_addr_type0(dev->bus->number, dev->devfn, offset));
	else
        	out_be32(hose->cfg_addr,
        		mk_config_addr_type1(dev->bus->number, dev->devfn, offset));
	cfg_data = get_qspan_pci_config((u32 *) hose->cfg_data);

	cp = ((u8 *)&cfg_data) + ((offset ^ 0x03) & 0x03);
	*val = *cp;

#ifdef QSPAN_DEBUG
	printk("qspan_read_config_byte:   (bus, devfn, offset, val) = "
	       "(%02X, %02X, %02X, %02X)\n",
	       dev->bus->number, dev->devfn, offset, *val);
#endif
	
	return PCIBIOS_SUCCESSFUL;
}

int qspan_read_config_word(struct pci_dev *dev, int offset, u16 *val)
{
        struct pci_controller *hose = dev->sysdata;
	u32 cfg_data;
	u16 *sp;

#ifdef QSPAN_BUS0_DEVFN0
	if (!dev->bus->number && !dev->devfn) {
		cfg_data = in_be32((volatile unsigned *)
			(PCI_CSR_ADDR + (offset & 0xfc)));
		sp = ((u16 *)&cfg_data) + (((offset ^ 0x02) >> 1) & 1);
		*val = *sp;

#ifdef QSPAN_DEBUG
		printk("qspan_read_config_word:   (bus, devfn, offset, val) = "
			"(%02X, %02X, %02X, %04X)\n",
			dev->bus->number, dev->devfn, offset, *val);
#endif

		return PCIBIOS_SUCCESSFUL;		
	}
#endif

	if ((dev->bus->number > 7) || (dev->devfn > 127)) {
		*val = 0xffff;
		return PCIBIOS_DEVICE_NOT_FOUND;
	}

	if (dev->bus->number == 0)
        	out_be32(hose->cfg_addr,
        		mk_config_addr_type0(dev->bus->number, dev->devfn, offset));
	else
        	out_be32(hose->cfg_addr,
        		mk_config_addr_type1(dev->bus->number, dev->devfn, offset));
	cfg_data = get_qspan_pci_config((u32 *) hose->cfg_data);

	sp = ((u16 *)&cfg_data) + (((offset ^ 0x02) >> 1) & 1);
	*val = *sp;

#ifdef QSPAN_DEBUG
	printk("qspan_read_config_word:   (bus, devfn, offset, val) = "
	       "(%02X, %02X, %02X, %04X)\n",
	       dev->bus->number, dev->devfn, offset, *val);
#endif

	return PCIBIOS_SUCCESSFUL;
}

int qspan_read_config_dword(struct pci_dev *dev, int offset, u32 *val)
{
        struct pci_controller *hose = dev->sysdata;

#ifdef QSPAN_BUS0_DEVFN0
	if (!dev->bus->number && !dev->devfn) {
		*val = in_be32((volatile unsigned *)
			(PCI_CSR_ADDR + (offset & 0xfc)));

#ifdef QSPAN_DEBUG
		printk("qspan_read_config_dword:  (bus, devfn, offset, val) = "
			"(%02X, %02X, %02X, %08X)\n",
			dev->bus->number, dev->devfn, offset, *val);
#endif

		return PCIBIOS_SUCCESSFUL;		
	}
#endif

	if ((dev->bus->number > 7) || (dev->devfn > 127)) {
		*val = 0xffffffff;
		return PCIBIOS_DEVICE_NOT_FOUND;
	}

	if (dev->bus->number == 0)
        	out_be32(hose->cfg_addr,
        		mk_config_addr_type0(dev->bus->number, dev->devfn, offset));
	else
        	out_be32(hose->cfg_addr,
        		mk_config_addr_type1(dev->bus->number, dev->devfn, offset));
	*val = get_qspan_pci_config((u32 *) hose->cfg_data);

#ifdef QSPAN_DEBUG
	printk("qspan_read_config_dword:  (bus, devfn, offset, val) = "
	       "(%02X, %02X, %02X, %08X)\n",
	       dev->bus->number, dev->devfn, offset, *val);
#endif

	return PCIBIOS_SUCCESSFUL;
}

int qspan_write_config_byte(struct pci_dev *dev, int offset, u8 val)
{
        struct pci_controller *hose = dev->sysdata;
	u32 cfg_data;
	u8 *cp;

	if ((dev->bus->number > 7) || (dev->devfn > 127))
		return PCIBIOS_DEVICE_NOT_FOUND;

	qspan_read_config_dword(dev, offset, &cfg_data);

	cp = ((u8 *)&cfg_data) + ((offset ^ 0x03) & 0x03);
	*cp = val;

#ifdef QSPAN_BUS0_DEVFN0
	if (!dev->bus->number && !dev->devfn) {
		out_be32((volatile unsigned *) (PCI_CSR_ADDR + (offset & 0xfc)),
			cfg_data);

#ifdef QSPAN_DEBUG
		printk("qspan_write_config_byte:  (bus, devfn, offset, val) = "
			"(%02X, %02X, %02X, %02X)\n",
			dev->bus->number, dev->devfn, offset, val);
#endif

		return PCIBIOS_SUCCESSFUL;		
	}
#endif

	if (dev->bus->number == 0)
        	out_be32(hose->cfg_addr,
        		mk_config_addr_type0(dev->bus->number, dev->devfn, offset));
	else
        	out_be32(hose->cfg_addr,
        		mk_config_addr_type1(dev->bus->number, dev->devfn, offset));
 	out_be32((volatile unsigned *) hose->cfg_data, cfg_data);

#ifdef QSPAN_DEBUG
	printk("qspan_write_config_byte:  (bus, devfn, offset, val) = "
	       "(%02X, %02X, %02X, %02X)\n",
	       dev->bus->number, dev->devfn, offset, val);
#endif

	return PCIBIOS_SUCCESSFUL;
}

int qspan_write_config_word(struct pci_dev *dev, int offset, u16 val)
{
        struct pci_controller *hose = dev->sysdata;
	u32 cfg_data;
	u16 *sp;

	if ((dev->bus->number > 7) || (dev->devfn > 127))
		return PCIBIOS_DEVICE_NOT_FOUND;

	qspan_read_config_dword(dev, offset, &cfg_data);

	sp = ((ushort *)&cfg_data) + (((offset ^ 0x02) >> 1) & 1);
	*sp = val;

#ifdef QSPAN_BUS0_DEVFN0
	if (!dev->bus->number && !dev->devfn) {
		out_be32((volatile unsigned *) (PCI_CSR_ADDR + (offset & 0xfc)),
			cfg_data);

#ifdef QSPAN_DEBUG
		printk("qspan_write_config_word:  (bus, devfn, offset, val) = "
			"(%02X, %02X, %02X, %04X)\n",
			dev->bus->number, dev->devfn, offset, val);
#endif

		return PCIBIOS_SUCCESSFUL;		
	}
#endif

	if (dev->bus->number == 0)
        	out_be32(hose->cfg_addr,
        		mk_config_addr_type0(dev->bus->number, dev->devfn, offset));
	else
        	out_be32(hose->cfg_addr,
        		mk_config_addr_type1(dev->bus->number, dev->devfn, offset));
 	out_be32((volatile unsigned *) hose->cfg_data, cfg_data);

#ifdef QSPAN_DEBUG
	printk("qspan_write_config_word:  (bus, devfn, offset, val) = "
	       "(%02X, %02X, %02X, %04X)\n",
	       dev->bus->number, dev->devfn, offset, val);
#endif

	return PCIBIOS_SUCCESSFUL;
}

int qspan_write_config_dword(struct pci_dev *dev, int offset, u32 val)
{
        struct pci_controller *hose = dev->sysdata;

	if ((dev->bus->number > 7) || (dev->devfn > 127))
		return PCIBIOS_DEVICE_NOT_FOUND;

#ifdef QSPAN_BUS0_DEVFN0
	if (!dev->bus->number && !dev->devfn) {
		out_be32((volatile unsigned *) (PCI_CSR_ADDR + (offset & 0xfc)),
			val);

#ifdef QSPAN_DEBUG
		printk("qspan_write_config_dword: (bus, devfn, offset, val) = "
			"(%02X, %02X, %02X, %08X)\n",
			dev->bus->number, dev->devfn, offset, val);
#endif

		return PCIBIOS_SUCCESSFUL;		
	}
#endif

	if (dev->bus->number == 0)
        	out_be32(hose->cfg_addr,
        		mk_config_addr_type0(dev->bus->number, dev->devfn, offset));
	else
        	out_be32(hose->cfg_addr,
        		mk_config_addr_type1(dev->bus->number, dev->devfn, offset));
 	out_be32((volatile unsigned *) hose->cfg_data, val);

#ifdef QSPAN_DEBUG
	printk("qspan_write_config_dword: (bus, devfn, offset, val) = "
	       "(%02X, %02X, %02X, %08X)\n",
	       dev->bus->number, dev->devfn, offset, val);
#endif

	return PCIBIOS_SUCCESSFUL;
}

static struct pci_ops qspan_pci_ops =
{
        qspan_read_config_byte,
        qspan_read_config_word,
        qspan_read_config_dword,
        qspan_write_config_byte,
        qspan_write_config_word,
        qspan_write_config_dword
};

void
setup_qspan_pci(struct pci_controller* hose, u32 cfg_addr, u32 cfg_data)
{
        hose->ops = &qspan_pci_ops;
        hose->cfg_addr = (unsigned int *) cfg_addr;
        hose->cfg_data = (unsigned char *) cfg_data;
}

void
m8xx_pcibios_fixup(void)
{
   /* Nothing to do here? */
}

void
m8xx_pcibios_fixup_resources(struct pci_dev *dev)
{
   /* Nothing to do here? */
}

int
m8xx_map_irq(struct pci_dev *dev, unsigned char idsel, unsigned char pin)
{
	/*
	 *	The interrupt structure on the MBX board is as follows.
	 *	The INTA, INTB, INTC, and INTD interrupt signals
	 *	on the PCI bus are routed to the corresponding input pins
	 *	on the Winbond.  These interrupts are routed to the Winbond's
	 *	integrated 8259 interrupt controller.  The 8259 IRQ
	 *	corresponding to each PCI interrupt is software configurable
	 *	via the 16-bit Winbond PCI configuration register at offset
	 *	0x44.  The MBX firmware initializes this register with value
	 *	0xABEF, which establishes the following interrupt mapping:
	 *
	 *		PCI INT		8259 IRQ
	 *		-------		--------
	 *		INTA		IRQ10
	 *		INTB		IRQ11
	 *		INTC		IRQ14
	 *		INTD		IRQ15
	 *
	 *	The interrupt output from the Winbond's 8259 is routed to
	 *	MPC8xx SIU IRQ3.
	 *
	 *	The PCI bus interrupt handling facilities of the QSpan bridge
	 *	are not utilized.  The QSpan interrupt control register (0x604)
	 *	is initialized by the MBX firmware to 0x00000000, which
	 *	disables all interrupt generation by the QSpan.
	 *
	 *	The table defined below maps the PCI configuration space device
	 *	number (aka slot) and PCI interrupt pin (1=INTA - 4=INTD) to
	 *	an interrupt vector for the 8259 interrupt controller in the
	 *	range 0 to 15.  Handlers for these interrupts should be
	 *	installed via the routine request_irq().
	 *
	 *	The Winbond has its PCI IDSEL input connected to PCI signal
	 *	AD19, corresponding to PCI slot 3.  I can't find any
	 *	documentation stating which PCI address/data lines are
	 *	connected to the four IDSEL signals on the PC/104+ PCI bus.
	 *	The PC/104+ standard suggests (mandates?) that AD20 through
	 *	AD23 should be used for the four IDSEL signals, so for now
	 *	we will assume that is the case for the MBX.  The assumed
	 *	mapping of PCI slot numbers to PCI interrupts is thus as
	 *	follows:
	 *
	 *		slot	IDSEL	INT0	INT1	INT2	INT3
	 *		----	-----	----	----	----	----
	 *		4	AD20	INTD	INTA	INTB	INTC
	 *		5	AD21	INTC	INTD	INTA	INTB
	 *		6	AD22	INTB	INTC	INTD	INTA
	 *		7	AD23	INTA	INTB	INTC	INTD
	 *
	 *	Note that the mapping of interrupts from PC/104+ modules to
	 *	the interrupt lines on the PC/104+ PCI bus is actually
	 *	implemented on the modules themselves, or on the
	 *	PC/104+-to-PCI adapter if you are using standard PCI cards.
	 *	The interrupt mapping is often jumper selectable, so you must
	 *	ensure that your interrupt mapping matches the table below by
	 *	either changing the table or changing the jumpers on your
	 *	modules.
	 *
	 *	If you are using the MBX Probe Card, the PCI slot on it is
	 *	PCI slot 7 and the interrupt pins from the PCI slot are routed
	 *	straight through to the PC/104+ bus.  The interrupt routing
	 *	table below will work for the PCI slot on the MBX Probe Card.
	 */

#define	BA I8259_IRQ_OFFSET	
#define IA (BA + 10)
#define IB (BA + 11)
#define IC (BA + 14)
#define ID (BA + 15)
	static char pci_irq_table[][4] =
	/*
	 *      [PCI slot][INTPIN] --> INTLINE
	 *	 A	B	C	D
	 */
	{
		{BA+14,	0,	0,	0},	/* slot 3 - Winbond */
                {ID,	IA,	IB,	IC},	/* slot 4 - PC/104+ Module 1 */
                {IC,	ID,	IA,	IB},	/* slot 5 - PC/104+ Module 2 */
                {IB,	IC,	ID,	IA},	/* slot 6 - PC/104+ Module 3 */
                {IA,	IB,	IC,	ID},	/* slot 7 - PC/104+ Module 4 */
	};
        const long min_idsel = 3, max_idsel = 7, irqs_per_slot = 4;
        return PCI_IRQ_TABLE_LOOKUP;
}
#undef IA
#undef IB
#undef IC
#undef ID
#undef BA

void
m8xx_find_bridges(void)
{
	struct pci_controller *hose;

	/* Setup PCI hose */
	hose = pcibios_alloc_controller();
	if (!hose)
		return;

	hose->first_busno = 0;
	hose->last_busno = 0xff;
	hose->pci_mem_offset = PCI_ISA_MEM_ADDR;

	pci_init_resource(&hose->io_resource,
			0x00000000,
			PCI_ISA_IO_SIZE - 1,
			IORESOURCE_IO,
			"QSpan PCI host bridge");

	pci_init_resource(&hose->mem_resources[0],
			PCI_ISA_MEM_ADDR + 0x00000000,
			PCI_ISA_MEM_ADDR + PCI_ISA_MEM_SIZE - 1,
			IORESOURCE_MEM,
			"QSpan PCI host bridge");

	hose->io_space.start =  0x00000000;
	hose->io_space.end =    PCI_ISA_IO_SIZE - 1;
	hose->mem_space.start = 0x00000000;
	hose->mem_space.end =   PCI_ISA_MEM_SIZE - 1;
	hose->io_base_virt = (void *)PCI_ISA_IO_ADDR;

	setup_qspan_pci(hose, QS_CONFIG_ADDR, QS_CONFIG_DATA);

	ppc_md.pcibios_fixup = m8xx_pcibios_fixup;
	ppc_md.pcibios_fixup_resources = m8xx_pcibios_fixup_resources;
	ppc_md.pci_swizzle = common_swizzle;
	ppc_md.pci_map_irq = m8xx_map_irq;

	hose->last_busno = pciauto_bus_scan(hose, hose->first_busno);
}
