/*
 * BK Id: %F% %I% %G% %U% %#%
 *
 *  PowerPC version derived from arch/arm/mm/consistent.c
 *    Copyright (C) 2001 Dan Malek (dmalek@jlc.net)
 *
 *  linux/arch/arm/mm/consistent.c
 *
 *  Copyright (C) 2000 Russell King
 *
 * Consistent memory allocators.  Used for DMA devices that want to
 * share uncached memory with the processor core.  The function return
 * is the virtual address and 'dma_handle' is the physical address.
 * Mostly stolen from the ARM port, with some changes for PowerPC.
 *						-- Dan
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 */

#include <linux/config.h>
#include <linux/module.h>
#include <linux/signal.h>
#include <linux/sched.h>
#include <linux/kernel.h>
#include <linux/errno.h>
#include <linux/string.h>
#include <linux/types.h>
#include <linux/ptrace.h>
#include <linux/mman.h>
#include <linux/mm.h>
#include <linux/swap.h>
#include <linux/stddef.h>
#include <linux/vmalloc.h>
#include <linux/init.h>
#include <linux/delay.h>
#include <linux/bootmem.h>
#include <linux/highmem.h>
#include <linux/pci.h>

#include <asm/pgalloc.h>
#include <asm/prom.h>
#include <asm/io.h>
#include <asm/hardirq.h>
#include <asm/mmu_context.h>
#include <asm/pgtable.h>
#include <asm/mmu.h>
#include <asm/uaccess.h>
#include <asm/smp.h>
#include <asm/machdep.h>

extern int get_pteptr(struct mm_struct *mm, unsigned long addr, pte_t **ptep);

#ifdef CONFIG_4xx
/* This function will allocate the requested contiguous pages and
 * map them into the kernel's vmalloc() space.  This is done so we
 * get unique mapping for these pages, outside of the kernel's 1:1
 * virtual:physical mapping.  This is necessary so we can cover large
 * portions of the kernel with single large page TLB entries, and
 * still get unique uncached pages for consistent DMA.
 */

void *consistent_alloc(int gfp, size_t size, dma_addr_t *dma_handle)
{
	int order, err, i;
	unsigned long page, va, pa, flags;
	struct vm_struct *area;
	void     *ret;

	if (in_interrupt())
		BUG();

	/* Only allocate page size areas.
	*/
	size = PAGE_ALIGN(size);
	order = get_order(size);

	page = __get_free_pages(gfp, order);
	if (!page) {
		BUG();
		return NULL;
	}

	/*
	 * we need to ensure that there are no cachelines in use,
	 * or worse dirty in this area.
	 */
	invalidate_dcache_range(page, page + size);


	/* Allocate some common virtual space to map the new pages.
	 */
	area = get_vm_area(size, VM_ALLOC);
	if (area == 0) {
		free_pages(page, order);
		return NULL;
	}
        va = VMALLOC_VMADDR(area->addr);
	ret = (void *)va;

	*dma_handle = pa = virt_to_bus(page);

	flags = _PAGE_KERNEL | _PAGE_NO_CACHE;

	err = 0;
	for (i = 0; i < size && err == 0; i += PAGE_SIZE)
		err = map_page(va+i, pa+i, flags);
		    
	if (err) {
		vfree((void *)va);
		return NULL;
	}

	return ret;
}

void consistent_free(void *vaddr)
{
	vfree(vaddr);
}
#else

void *consistent_alloc(int gfp, size_t size, dma_addr_t *dma_handle)
{
	int order, rsize;
	unsigned long page;
	void *ret;
	pte_t	*pte;

	if (in_interrupt())
		BUG();

	/* Only allocate page size areas.
	*/
	size = PAGE_ALIGN(size);
	order = get_order(size);

	page = __get_free_pages(gfp, order);
	if (!page) {
		BUG();
		return NULL;
	}

	/*
	 * we need to ensure that there are no cachelines in use,
	 * or worse dirty in this area.
	 */
	invalidate_dcache_range(page, page + size);

	ret = (void *)page;
	*dma_handle = virt_to_bus(ret);

	/* Chase down all of the PTEs and mark them uncached.
	*/
	rsize = (int)size;
	while (rsize > 0) {
		if (get_pteptr(&init_mm, page, &pte)) {
			pte_val(*pte) |= _PAGE_NO_CACHE | _PAGE_GUARDED;
			flush_tlb_page(find_vma(&init_mm,page),page);
		}
		else {
			BUG();
			return NULL;
		}
		page += PAGE_SIZE;
		rsize -= PAGE_SIZE;
	}

	return ret;
}

/*
 * free page(s) as defined by the above mapping.
 * The caller has to tell us the size so we can free the proper number
 * of pages.  We can't vmalloc() a new space for these pages and simply
 * call vfree() like some other architectures because we could end up
 * with aliased cache lines (or at least a cache line with the wrong
 * attributes).  This can happen when the PowerPC speculative loads
 * across page boundaries.
 */
void consistent_free(void *vaddr, size_t size)
{
	int order, rsize;
	unsigned long addr;
	pte_t	*pte;

	if (in_interrupt())
		BUG();

	size = PAGE_ALIGN(size);
	order = get_order(size);

	/* Chase down all of the PTEs and mark them cached again.
	*/
	addr = (unsigned long)vaddr;
	rsize = (int)size;
	while (rsize > 0) {
		if (get_pteptr(&init_mm, addr, &pte)) {
			pte_val(*pte) &= ~(_PAGE_NO_CACHE | _PAGE_GUARDED);
			flush_tlb_page(find_vma(&init_mm,addr),addr);
		}
		else {
			BUG();
			return;
		}
		addr += PAGE_SIZE;
		rsize -= PAGE_SIZE;
	}
	free_pages((unsigned long)vaddr, order);
}
#endif /* CONFIG_4xx */

/*
 * make an area consistent.
 */
void consistent_sync(void *vaddr, size_t size, int direction)
{
	unsigned long start = (unsigned long)vaddr;
	unsigned long end   = start + size;

	switch (direction) {
	case PCI_DMA_NONE:
		BUG();
	case PCI_DMA_FROMDEVICE:	/* invalidate only */
#if (defined(CONFIG_4xx) && defined(CONFIG_USB))
		/* 
		 * USB code does DMA off the stack, so when a driver
		 * uses the stock pci DMA routines, they end up calling
		 * invalidate_dcache_range(). 
		 * The cache invalidate code in the 405 port invalidates 
		 * an entire cache line, regardless of what addresses are 
		 * passed in - it 'rounds out' to the surrounding cache line.
		 * While other ports will notice if part of the cache line
		 * is NOT part of the range being invalidated and 'do
		 * the right thing', the 405 code will invalidate the 
		 * entire line, potentially trashing the stack, causing a
		 * panic.
		 * Until all the USB drivers are fixed, we flush the data
		 * buffer, a much safer operation.
		 */
		flush_dcache_range(start, end);
#else
		invalidate_dcache_range(start, end);
#endif
		break;
	case PCI_DMA_TODEVICE:		/* writeback only */
		clean_dcache_range(start, end);
		break;
	case PCI_DMA_BIDIRECTIONAL:	/* writeback and invalidate */
		flush_dcache_range(start, end);
		break;
	}
}

/*
 * consistent_sync_page make a page are consistent. identical
 * to consistent_sync, but takes a struct page instead of a virtual address
 */

void consistent_sync_page(struct page *page, unsigned long offset,
size_t size, int direction)
{
	unsigned long start;

	start = (unsigned long)(page->virtual) + offset; 
	consistent_sync((void *)start, size, direction);
}
