/*
 * arch/ppc/platforms/ev64260_pci.c
 * 
 * PCI setup routines for the Marvell/Galileo EV-64260-BP Evaluation Board.
 *
 * Author: Mark A. Greer <mgreer@mvista.com>
 *
 * Copyright 2001 MontaVista Software Inc.
 *
 * This program is free software; you can redistribute  it and/or modify it
 * under  the terms of  the GNU General  Public License as published by the
 * Free Software Foundation;  either version 2 of the  License, or (at your
 * option) any later version.
 */
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/pci.h>
#include <linux/slab.h>

#include <asm/byteorder.h>
#include <asm/io.h>
#include <asm/irq.h>
#include <asm/uaccess.h>
#include <asm/machdep.h>
#include <asm/pci-bridge.h>
#include <asm/gt64260.h>
#include <platforms/ev64260.h>

extern u_long	ev64260_mem_size;


/*
 * Marvell/Galileo EV-64260-BP Evaluation Board PCI interrupt routing.
 */
static inline int
ev64260_map_irq(struct pci_dev *dev, unsigned char idsel, unsigned char pin)
{
	struct pci_controller	*hose = pci_bus_to_hose(dev->bus->number);

	if (hose->index == 0) {
		static char pci_irq_table[][4] =
		/*
		 *	PCI IDSEL/INTPIN->INTLINE 
		 * 	   A   B   C   D
		 */
		{
			{ 91, 0, 0, 0 },	/* IDSEL 7 - PCI bus 0 */
			{ 91, 0, 0, 0 },	/* IDSEL 8 - PCI bus 0 */
		};

		const long min_idsel = 7, max_idsel = 8, irqs_per_slot = 4;
		return PCI_IRQ_TABLE_LOOKUP;
	}
	else {
		static char pci_irq_table[][4] =
		/*
		 *	PCI IDSEL/INTPIN->INTLINE 
		 * 	   A   B   C   D
		 */
		{
			{ 93, 0, 0, 0 },	/* IDSEL 7 - PCI bus 1 */
			{ 93, 0, 0, 0 },	/* IDSEL 8 - PCI bus 1 */
		};

		const long min_idsel = 7, max_idsel = 8, irqs_per_slot = 4;
		return PCI_IRQ_TABLE_LOOKUP;
	}
}

static int
ev64260_bridge_init(struct pci_controller *hose_a,
		    struct pci_controller *hose_b,
		    u32 mem_size)
{
	gt64260_bridge_init_info_t	info;
	int				window;
	u32				u32_val;
	int				rc;


	gt64260_base = EV64260_BRIDGE_REG_BASE;

	setup_indirect_pci(hose_a,
			   EV64260_PCI_0_CONFIG_ADDR,
			   EV64260_PCI_0_CONFIG_DATA);

	setup_indirect_pci(hose_b,
			   EV64260_PCI_1_CONFIG_ADDR,
			   EV64260_PCI_1_CONFIG_DATA);

	info.hose_a = hose_a;
	info.hose_b = hose_b;
	info.mem_size = mem_size;

	info.pci_0_io_start_proc = EV64260_PCI_0_IO_START_PROC;
	info.pci_0_io_start_pci  = EV64260_PCI_0_IO_START;
	info.pci_0_io_size	 = EV64260_PCI_0_IO_SIZE;
	info.pci_0_io_swap	 = GT64260_CPU_PCI_SWAP_NONE;

	info.pci_0_mem_start_proc   = EV64260_PCI_0_MEM_START_PROC;
	info.pci_0_mem_start_pci_hi = 0x00000000;
	info.pci_0_mem_start_pci_lo = EV64260_PCI_0_MEM_START;
	info.pci_0_mem_size	    = EV64260_PCI_0_MEM_SIZE;
	info.pci_0_mem_swap	    = GT64260_CPU_PCI_SWAP_NONE;

	info.pci_1_io_start_proc = EV64260_PCI_1_IO_START_PROC;
	info.pci_1_io_start_pci  = EV64260_PCI_1_IO_START;
	info.pci_1_io_size	 = EV64260_PCI_1_IO_SIZE;
	info.pci_1_io_swap	 = GT64260_CPU_PCI_SWAP_NONE;

	info.pci_1_mem_start_proc   = EV64260_PCI_1_MEM_START_PROC;
	info.pci_1_mem_start_pci_hi = 0x00000000;
	info.pci_1_mem_start_pci_lo = EV64260_PCI_1_MEM_START;
	info.pci_1_mem_size	    = EV64260_PCI_1_MEM_SIZE;
	info.pci_1_mem_swap	    = GT64260_CPU_PCI_SWAP_NONE;

	if ((rc = gt64260_bridge_init(&info)) != 0) {
		iounmap((void *)hose_a->cfg_addr);
		iounmap((void *)hose_a->cfg_data);
		iounmap((void *)hose_b->cfg_addr);
		iounmap((void *)hose_b->cfg_data);
		return rc;
	}

	GT64260_SET_REG_BITS(GT64260_CPU_MASTER_CNTL, BIT9); /* Only 1 cpu */

	/* SCS windows not disabled above, disable all but SCS 0 */
	for (window=1; window<GT64260_CPU_SCS_DECODE_WINDOWS; window++) {
		gt64260_cpu_scs_set_window(window, 0, 0);
	}

	/* Set up windows to RTC/TODC and DUART on device module (CS 1 & 2) */
	gt64260_cpu_cs_set_window(1, EV64260_TODC_BASE, EV64260_TODC_LEN);
	gt64260_cpu_cs_set_window(2, EV64260_UART_BASE, EV64260_UART_LEN);

	gt64260_pci_exclude_bridge = FALSE;

	/* Set latency timer (to 64) & cacheline size; clear BIST */
	u32_val = ((0x04 << 8) | (L1_CACHE_LINE_SIZE / 4));

	early_write_config_dword(hose_a,
				 hose_a->first_busno,
				 PCI_DEVFN(0,0),
				 PCI_CACHE_LINE_SIZE,
				 u32_val);
	early_write_config_dword(hose_b,
				 hose_b->first_busno,
				 PCI_DEVFN(0,0),
				 PCI_CACHE_LINE_SIZE,
				 u32_val);

	gt64260_pci_exclude_bridge = TRUE;

	/*
	 * The EV-64260-BP uses several Multi-Purpose Pins (MPP) on the 64260
	 * bridge as interrupt inputs (via the General Purpose Ports (GPP)
	 * register).  Need to route the MPP inputs to the GPP and set the
	 * polarity correctly.
	 *
	 * In MPP Control 2 Register
	 *   MPP 21 -> GPP 21 (DUART channel A intr)
	 *   MPP 22 -> GPP 22 (DUART channel A intr)
	 *
	 * In MPP Control 3 Register
	 *   MPP 27 -> GPP 27 (PCI 0 INTA)
	 *   MPP 29 -> GPP 29 (PCI 1 INTA)
	 */
	GT64260_RESET_REG_BITS(GT64260_MPP_CNTL_2,
			       (BIT20 | BIT21 | BIT22 | BIT23 |
			        BIT24 | BIT25 | BIT26 | BIT27));

	GT64260_RESET_REG_BITS(GT64260_MPP_CNTL_3,
			       (BIT12 | BIT13 | BIT14 | BIT15 |
			        BIT20 | BIT21 | BIT22 | BIT23));

	GT64260_REG_WRITE(GT64260_GPP_LEVEL_CNTL, 0x000002c6);

	/* DUART & PCI interrupts are active low */
	GT64260_SET_REG_BITS(GT64260_GPP_LEVEL_CNTL,
			     BIT21 | BIT22 | BIT27 | BIT29);

	/* Clear any pending interrupts for these inputs and enable them. */
	GT64260_REG_WRITE(GT64260_GPP_INTR_CAUSE,
			  ~(BIT21 | BIT22 | BIT27 | BIT29));
	GT64260_SET_REG_BITS(GT64260_GPP_INTR_MASK,
			     (BIT21 | BIT22 | BIT27 | BIT29));
	GT64260_SET_REG_BITS(GT64260_IC_CPU_INTR_MASK_HI, (BIT26 | BIT27));

	/* Set MPSC Multiplex RMII */
	GT64260_REG_WRITE(GT64260_MPP_SERIAL_PORTS_MULTIPLEX, 0x00000102);

	return 0;
}

void __init
ev64260_find_bridges(void)
{
	struct pci_controller		*hose_a, *hose_b;


	hose_a = pcibios_alloc_controller();
	if (!hose_a)
		return;

	hose_b = pcibios_alloc_controller();
	if (!hose_b)
		return;

	if (ev64260_bridge_init(hose_a, hose_b, ev64260_mem_size) != 0) {
		return;
	}

	hose_a->first_busno = 0;
	hose_a->last_busno  = 0xff;

	hose_a->io_space.start    = EV64260_PCI_0_IO_START;
	hose_a->io_space.end      = EV64260_PCI_0_IO_END;
	hose_a->io_resource.start = EV64260_PCI_0_IO_START_PROC - isa_io_base;
	hose_a->io_resource.end   = EV64260_PCI_0_IO_END_PROC - isa_io_base;
	hose_a->io_resource.flags = IORESOURCE_IO;

	hose_a->mem_space.start        = EV64260_PCI_0_MEM_START;
	hose_a->mem_space.end          = EV64260_PCI_0_MEM_END;
	hose_a->mem_resources[0].start = EV64260_PCI_0_MEM_START_PROC;
	hose_a->mem_resources[0].end   = EV64260_PCI_0_MEM_END_PROC;
	hose_a->mem_resources[0].flags = IORESOURCE_MEM;

	hose_a->pci_mem_offset = EV64260_PCI_0_PCI_MEM_OFFSET;
	hose_a->io_base_virt   = (void *)EV64260_PCI_0_IO_START_PROC;

	hose_a->last_busno = pciauto_bus_scan(hose_a, hose_a->first_busno);


	hose_b->first_busno = hose_a->last_busno + 1;
	hose_b->bus_offset  = hose_b->first_busno;
	hose_b->last_busno  = 0xff;

	hose_b->io_space.start    = EV64260_PCI_1_IO_START;
	hose_b->io_space.end      = EV64260_PCI_1_IO_END;
	hose_b->io_resource.start = EV64260_PCI_1_IO_START_PROC - isa_io_base;
	hose_b->io_resource.end   = EV64260_PCI_1_IO_END_PROC - isa_io_base;
	hose_b->io_resource.flags = IORESOURCE_IO;

	hose_b->mem_space.start        = EV64260_PCI_1_MEM_START;
	hose_b->mem_space.end          = EV64260_PCI_1_MEM_END;
	hose_b->mem_resources[0].start = EV64260_PCI_1_MEM_START_PROC;
	hose_b->mem_resources[0].end   = EV64260_PCI_1_MEM_END_PROC;
	hose_b->mem_resources[0].flags = IORESOURCE_MEM;

	hose_b->pci_mem_offset = EV64260_PCI_1_PCI_MEM_OFFSET;
	hose_b->io_base_virt   = (void *)EV64260_PCI_0_IO_START_PROC;

	hose_b->last_busno = pciauto_bus_scan(hose_b, hose_b->first_busno);


	ppc_md.pci_exclude_device = gt64260_pci_exclude_device;
	ppc_md.pci_swizzle        = common_swizzle;
	ppc_md.pci_map_irq        = ev64260_map_irq;

	return;
}
