/*
 * arch/ppc/platforms/ev64260_setup.c
 *
 * Board setup routines for the Marvell/Galileo EV-64260-BP Evaluation Board.
 *
 * Author: Mark A. Greer <mgreer@mvista.com>
 *
 * Copyright 2001 MontaVista Software Inc.
 *
 * This program is free software; you can redistribute  it and/or modify it
 * under  the terms of  the GNU General  Public License as published by the
 * Free Software Foundation;  either version 2 of the  License, or (at your
 * option) any later version.
 */

/*
 * The EV-64260-BP port is the result of hard work from many people from
 * many companies.  In particular, employees of Marvell/Galileo, Mission
 * Critical Linux, Xyterra, and MontaVista Software were heavily involved.
 */
#include <linux/config.h>
#include <linux/stddef.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/errno.h>
#include <linux/reboot.h>
#include <linux/pci.h>
#include <linux/kdev_t.h>
#include <linux/major.h>
#include <linux/blk.h>
#include <linux/console.h>
#include <linux/delay.h>
#include <linux/irq.h>
#include <linux/ide.h>
#include <linux/seq_file.h>

#include <asm/system.h>
#include <asm/pgtable.h>
#include <asm/page.h>
#include <asm/time.h>
#include <asm/dma.h>
#include <asm/io.h>
#include <asm/machdep.h>
#include <asm/prom.h>
#include <asm/smp.h>
#include <asm/todc.h>
#include <asm/bootinfo.h>
#include <asm/gt64260.h>
#include <platforms/ev64260.h>


extern char cmd_line[];

extern void ev64260_find_bridges(void);

ulong	ev64260_mem_size = 0;



static void __init
ev64260_setup_arch(void)
{
	if ( ppc_md.progress )
		ppc_md.progress("ev64260_setup_arch: enter", 0);

	loops_per_jiffy = 50000000 / HZ;

#ifdef CONFIG_BLK_DEV_INITRD
	if (initrd_start)
		ROOT_DEV = MKDEV(RAMDISK_MAJOR, 0);
	else
#endif
#ifdef	CONFIG_ROOT_NFS
		ROOT_DEV = to_kdev_t(0x00FF);	/* /dev/nfs pseudo device */
#else
		ROOT_DEV = to_kdev_t(0x0802);	/* /dev/sda2 SCSI disk */
#endif

	if ( ppc_md.progress )
		ppc_md.progress("ev64260_setup_arch: find_bridges", 0);

	/*
	 * Set up the L2CR register.
	 * L2 cache was invalidated by bootloader.
	 */
	switch (PVR_VER(mfspr(PVR))) {
		case PVR_VER(PVR_750):
			_set_L2CR(0xfd100000);
			break;
		case PVR_VER(PVR_7400):
		case PVR_VER(PVR_7410):
			_set_L2CR(0xcd100000);
			break;
		/* case PVR_VER(PVR_7450): */
			/* XXXX WHAT VALUE?? FIXME */
			break;
	}

	/* Lookup PCI host bridges */
	ev64260_find_bridges();

	printk("Marvell/Galileo EV-64260-BP Evaluation Board\n");
	printk("EV-64260-BP port (C) 2001 MontaVista Software, Inc. (source@mvista.com)\n");

	if ( ppc_md.progress )
		ppc_md.progress("ev64260_setup_arch: exit", 0);

	return;
}

unsigned long __init
ev64260_find_end_of_memory(void)
{
	ev64260_mem_size = 32*1024*1024;	/* XXXX FIXME */
	return ev64260_mem_size;
}

static void __init
ev64260_map_io(void)
{
	io_block_mapping(EV64260_BRIDGE_REG_BASE,
			 EV64260_BRIDGE_REG_BASE,
			 EV64260_BRIDGE_REG_BASE_TO_TOP,
			 _PAGE_IO);
	isa_io_base = EV64260_PCI_0_IO_START_PROC;
}

static void
ev64260_reset_board(void)
{
	__cli();

	/* Set exception prefix high - to the firmware */
	_nmask_and_or_msr(0, MSR_IP);

#if 0	/* XXXX */
	out_8((u_char *)EV64260_BOARD_MODRST_REG, 0x01);
#else
	printk("XXXX **** trying to reset board ****\n");
#endif

	return;
}

static void
ev64260_restart(char *cmd)
{
	volatile ulong	i = 10000000;

	ev64260_reset_board();

	while (i-- > 0);
	panic("restart failed\n");
}

static void
ev64260_halt(void)
{
	__cli();
	while (1);
	/* NOTREACHED */
}

static void
ev64260_power_off(void)
{
	ev64260_halt();
	/* NOTREACHED */
}

static int
ev64260_show_cpuinfo(struct seq_file *m)
{
	uint pvid;

	pvid = mfspr(PVR);
	seq_printf(m, "vendor\t\t: Marvell/Galileo\n");
	seq_printf(m, "machine\t\t: EV-64260-BP\n");
	seq_printf(m, "PVID\t\t: 0x%x, vendor: %s\n",
			pvid, (pvid & (1<<15) ? "IBM" : "Motorola"));

	return 0;
}

/* DS1501 RTC has too much variation to use RTC for calibration */
static void __init
ev64260_calibrate_decr(void)
{
	ulong freq;

	freq = 100000000 / 4;

	printk("time_init: decrementer frequency = %lu.%.6lu MHz\n",
	       freq/1000000, freq%1000000);

	tb_ticks_per_jiffy = freq / HZ;
	tb_to_us = mulhwu_scale_factor(freq, 1000000);
#ifdef CONFIG_ILATENCY
	{
		extern unsigned ticks_per_usec;
		ticks_per_usec = freq;
	}
#endif

	return;
}

TODC_ALLOC();

void __init
platform_init(unsigned long r3, unsigned long r4, unsigned long r5,
	      unsigned long r6, unsigned long r7)
{
	parse_bootinfo(find_bootinfo());

	isa_mem_base = 0;
	pci_dram_offset = 0;

	ppc_md.setup_arch = ev64260_setup_arch;
	ppc_md.show_cpuinfo = ev64260_show_cpuinfo;
	ppc_md.irq_cannonicalize = NULL;
	ppc_md.init_IRQ = gt64260_init_irq;
	ppc_md.get_irq = gt64260_get_irq;
	ppc_md.init = NULL;

	ppc_md.restart = ev64260_restart;
	ppc_md.power_off = ev64260_power_off;
	ppc_md.halt = ev64260_halt;

	ppc_md.find_end_of_memory = ev64260_find_end_of_memory;
	ppc_md.setup_io_mappings = ev64260_map_io;

	TODC_INIT(TODC_TYPE_DS1501, 0, 0, EV64260_TODC_BASE, 8);

	ppc_md.time_init = todc_time_init;
	ppc_md.set_rtc_time = todc_set_rtc_time;
	ppc_md.get_rtc_time = todc_get_rtc_time;
	ppc_md.calibrate_decr = ev64260_calibrate_decr;

	ppc_md.nvram_read_val = todc_direct_read_val;
	ppc_md.nvram_write_val = todc_direct_write_val;

	ppc_md.heartbeat = NULL;
	ppc_md.heartbeat_reset = 0;
	ppc_md.heartbeat_count = 0;

#ifdef	CONFIG_SERIAL_TEXT_DEBUG
	ppc_md.progress = ev64260_progress;
#else	/* !CONFIG_SERIAL_TEXT_DEBUG */
	ppc_md.progress = NULL;
#endif	/* CONFIG_SERIAL_TEXT_DEBUG */

	return;
}
