/*
 *  Flash Rom Disk Driver
 *
 *  Copyright (C) 2001-2004  BUFFALO INC.
 *
 *  This software may be used and distributed according to the terms of
 *  the GNU General Public License (GPL), incorporated herein by reference.
 *  Drivers based on or derived from this code fall under the GPL and must
 *  retain the authorship, copyright and license notice.  This file is not
 *  a complete program and may only be used when the entire operating
 *  system is licensed under the GPL.
 *
 */
#include <linux/config.h>
#include <linux/sched.h>
#include <linux/minix_fs.h>
#include <linux/ext2_fs.h>
#include <linux/romfs_fs.h>
#include <linux/fs.h>
#include <linux/kernel.h>
#include <linux/hdreg.h>
#include <linux/string.h>
#include <linux/mm.h>
#include <linux/mman.h>
#include <linux/slab.h>
#include <linux/ioctl.h>
#include <linux/fd.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/devfs_fs_kernel.h>
#include <linux/smp_lock.h>
#include <linux/proc_fs.h>

#include <asm/system.h>
#include <asm/uaccess.h>
#include <asm/byteorder.h>

#define  MAJOR_NR FLASHDISK_MAJOR
#include <linux/blk.h>
#include <linux/blkpg.h>

#include <linux/delay.h>

#include <melco/melco_hwctl.h>

#include <melco/firminfo.h>
#include <melco/flashd.h>

#define	dbg_printk(args...) 

#define DEVICE_NR(device) MINOR(device)

static devfs_handle_t devfs_handle;

struct flash_dev_func* flash_dev_list[] = {
	&flash_st,
	&flash_fujitsu,
	&flash_mx,
	NULL
	};

struct flashd_list area_list[] = {
	{ 0xFFF80000,  0x80000  , ROM_ON_BORD },
	{ 0xFFC00000,  0x300000 , ROM_ON_BORD },
	{ 0xFFF00000,  0x70000  , ROM_ON_BORD },
	{ 0xFFF70000,  0x10000  , ROM_ON_BORD },
	{ 0xFFC00000,  0x400000 , ROM_ON_BORD },
	{ 0xFFF80000,  0x80000  , ROM_MINIPCI },
	{ 0xFF800000,  0x600000 , ROM_MINIPCI },
	{ 0xFFF00000,  0x80000  , ROM_MINIPCI },
	{ 0         ,  0        , 0           }
	};

int                 flashd_num;
struct flash_info*  flashd;
int                *flashd_hardsec;
int                *flashd_blocksize;
int                *flashd_kbsize;

char              dev_winbond;

int               isopened;
int               boot_from_mini_pci=0;
int               modifying_cash;

#define FLASHD_BLOCK_SIZE 1*1024

struct flashd_cache_info {
	int               modify;
	char*             addr;
	char*             end;
	unsigned long     size;
	struct flash_info *dev;
	};

struct flashd_cache_info cacheinfo;
char*                    flash_cache;

#define FLASHD_RETRY 3

void chg_rom(char mode)
{
	unsigned long           data;
	
	cli();
	*((pvlong)PPC_CFG_REG) = cpu_to_le32((unsigned long)PPC_CFG_PCIR);
	data                   = le32_to_cpu(*((pvlong)PPC_DAT_REG));
	if(mode == ROM_MINIPCI) 
		data &= ~(1<<20);
	else 
		data |= 1<<20;
	*((pvlong)PPC_DAT_REG) = cpu_to_le32(data);
	sti();
	
}

void outw_pci(unsigned char dev,unsigned char func,unsigned char reg,unsigned short val)
{
	unsigned long base;
	base = 0x80000000+((dev&0x1f)<<11)+((func&0x7)<<8)+(reg & 0xfc);
	cli();
	*((pvlong)  PPC_CFG_REG)            = cpu_to_le32(base);
	*((pvshort)(PPC_DAT_REG+(reg&0x3))) = cpu_to_le16(val);
	sti();
}

unsigned short inw_pci(unsigned char dev,unsigned char func,unsigned char reg)
{
	unsigned long base;
	unsigned short val;
	base = 0x80000000+((dev&0x1f)<<11)+((func&0x7)<<8)+(reg & 0xfc);
	cli();
	*((pvlong)  PPC_CFG_REG)            = cpu_to_le32(base);
	val = *((unsigned short*)(PPC_DAT_REG+(reg&0x3)));
	sti();
	return le16_to_cpu(val);
}

void write_enable_winbond(char dev)
{
	unsigned short data;
	data = inw_pci(dev,0x0,WINBOND_CSC_REG);
	data = data & ~(1<<5);
	outw_pci(dev,0x0,WINBOND_CSC_REG,data);
}

void write_disable_winbond(char dev)
{
	unsigned short data;
	data = inw_pci(dev,0x0,WINBOND_CSC_REG);
	data = data | (1<<5);
	outw_pci(dev,0x0,WINBOND_CSC_REG,data);
}



static int flashd_read_write_cache(struct flash_info *dev,char* src)
{
	unsigned long block;
	block = (unsigned long)src / cacheinfo.size;
	src   = (char*)(block * cacheinfo.size);
	memcpy(flash_cache,src,cacheinfo.size);
	cacheinfo.addr   = src;
	cacheinfo.end    = src + cacheinfo.size -1;
	cacheinfo.modify = 1;
	cacheinfo.dev    = dev;
	dbg_printk("FLASHDISK:cache[%8p-%8p(%lX)]\n",cacheinfo.addr,cacheinfo.end,block);
	return 0;
}

static int flashd_write_write_cache(void)
{
	struct flash_dev_func *dev;
	int    is_blink = 0;

	if(cacheinfo.modify == 0) return 0;
	if(cacheinfo.dev == NULL) return -1;
	dev = cacheinfo.dev->device;
	dbg_printk("FLASHDISK:cache->write [%8p-%8p]\n",cacheinfo.addr,cacheinfo.end);
	if((unsigned long)cacheinfo.addr < 0xFFF80000)  is_blink = 1;

	if(dev->blk_erase == NULL || dev->blk_write==NULL) {
		printk("FLASHDISK:This Flash don't have Write func!\n");
		return -1;
		}

	if(is_blink == 1)  blink_led(FLASH_UPDATE_START);

	if(cacheinfo.dev->rom_type == ROM_MINIPCI) write_enable_winbond(dev_winbond);

	if(dev->blk_erase((unsigned long)cacheinfo.addr,1) != 0) {
		printk("FLASHDISK:Fatal! Can't Erase Flash!\n");
		if(is_blink == 1)  blink_led(FLASH_UPDATE_END);
		return -1;
		}

	if(dev->blk_write((unsigned long)cacheinfo.addr,flash_cache,cacheinfo.size) != 0) {
		printk("FLASHDISK:Fatal! Can't Write Flash!\n");
		if(is_blink == 1)  blink_led(FLASH_UPDATE_END);
		return -1;
		}
	
	if(cacheinfo.dev->rom_type == ROM_MINIPCI) write_disable_winbond(dev_winbond);

	if(is_blink == 1)  blink_led(FLASH_UPDATE_END);
	
	cacheinfo.modify = 0;
	return 0;
}

static int flashd_write_data_to_cache(struct flash_info *dev,char* dst,char* src,unsigned long size)
{
	int           len;
	unsigned long dstb,cchb;
	
	if(cacheinfo.modify == 0) flashd_read_write_cache(dev,dst);
	while(size > 0) {
		dstb = (unsigned long)dst            / cacheinfo.size;
		cchb = (unsigned long)cacheinfo.addr / cacheinfo.size;
		
		dbg_printk("FLASHDISK:wstc[%8p:%lX] cache[%8p:%lX] ",dst,dstb,cacheinfo.addr,cchb);
		if(dstb != cchb) {
			flashd_write_write_cache();
			flashd_read_write_cache(dev,dst);
			}
		if((dst+size) > cacheinfo.end) 
			len = (int)(cacheinfo.end-dst) + 1;
		else 
			len = size;
		dbg_printk("fsc %8p:%8p:%d\n",flash_cache,flash_cache + (dst-cacheinfo.addr),len);
		memcpy(flash_cache + (dst-cacheinfo.addr),src,len);
		dst += len;
		src += len;
		size-= len;
		}
	return 0;
}

static int flashd_read_data(char* dst,char* target,unsigned long size)
{
	unsigned long tstart,cstart,
	              tend  ,cend;
	char*         baddr;
	unsigned long len;
	int           hit_cache;
	while(size>0) {
		hit_cache = 0;
		tstart = (unsigned long)target;
		tend   = tstart+size-1;
		cstart = (unsigned long)cacheinfo.addr;
		cend   = (unsigned long)cacheinfo.end;
		if(cacheinfo.modify != 0) {
			if(tstart>=cstart && tstart<=cend) {
				hit_cache = 1;
				}
			if(hit_cache) {
				if(tend>cend) tend = cend;
				}
			else {
				if(tstart<cstart && tend>=cstart) tend = cstart-1;
				}
			}
		len = tend - tstart + 1;
		if(hit_cache) {
			baddr = flash_cache + (tstart - cstart);
			memcpy(dst,baddr,len);
			}
		else {
			baddr = (char*)tstart;
			memcpy(dst,baddr,len);
			}
		size   -= len;
		dst    += len;
		target += len;
		}
	return 0;
}

static int flashd_make_request(request_queue_t * q, int rw, struct buffer_head *sbh)
{
	int               retry;
	char              *data,*bdata;
	unsigned int      no;
	unsigned long     offset,len;
	struct flash_info *dev;
	
	no = MINOR(sbh->b_rdev);
	if(no >= flashd_num) goto fail;
	
	dev    = &flashd[no];
	offset = (sbh->b_rsector/(FLASHD_BLOCK_SIZE>>9)) * FLASHD_BLOCK_SIZE;
	len    = sbh->b_size;
	if((offset+len) > dev->length) goto fail;
	if(rw == READA) rw = READ;
	if ((rw != READ) && (rw != WRITE)) {
		printk(KERN_INFO "FLASHDISK: bad command: %d\n", rw);
		goto fail;
	}
	
	data = (char*)(dev->start_addr + offset);
	bdata = bh_kmap(sbh);
	switch(rw) {
		case READ:
			flashd_read_data(bdata,data,len);
			break;
		case WRITE:
			retry = 0;
			while(flashd_write_data_to_cache(dev,data,bdata,len) != 0 && retry<FLASHD_RETRY) retry++;
			if(retry>=FLASHD_RETRY) goto fail;
			break;
		}
	bh_kunmap(sbh);

	sbh->b_end_io(sbh,1);
	return 0;
fail:
	sbh->b_end_io(sbh,0);
	return 0;
} 

static int flashd_ioctl(struct inode *inode, struct file *file, unsigned int cmd, unsigned long arg)
{
	unsigned int minor;
	
	if (!inode || !inode->i_rdev) 	
		return -EINVAL;
	minor = MINOR(inode->i_rdev);
	if(minor >= flashd_num) return -EIO;
	switch (cmd) {
		case BLKFLSBUF:
			if (!capable(CAP_SYS_ADMIN))
				return -EACCES;
			if (inode->i_bdev->bd_openers > 2)
				return -EBUSY;
			destroy_buffers(inode->i_rdev);
			flashd_blocksize[minor] = 0;
			break;
		case BLKGETSIZE:
			if (!arg)  return -EINVAL;
			return put_user(flashd_kbsize[minor] << 1, (long *) arg);
		case BLKROSET:
		case BLKROGET:
		case BLKSSZGET:
			return blk_ioctl(inode->i_rdev, cmd, arg);

		default:
			return -EINVAL;
	};

	return 0;
}

static int flashd_open(struct inode * inode, struct file * filp)
{
	int               no;
	struct flash_info *dev;
	dbg_printk("FLASHDISK: Open!\n");
	if (DEVICE_NR(inode->i_rdev) >= flashd_num)
		return -ENODEV;
	
	no = MINOR(inode->i_rdev);
	dev = &flashd[no];
	if(dev->device == NULL) return -ENODEV;
	
	cli();
	if(isopened == 1) {
		sti();
		return -ENXIO;
		}
	isopened = 1;
	sti();
	
	if(dev->rom_type == ROM_MINIPCI) {
		chg_rom(ROM_MINIPCI);
		}
	else {
		chg_rom(ROM_ON_BORD);
		}
	cacheinfo.modify = 0;
	cacheinfo.addr   = 0;
	return 0;
}

static int flashd_release(struct inode * inode, struct file * filp)
{
	int               no;
	dbg_printk("FLASHDISK: release!\n");
	no = MINOR(inode->i_rdev);
	if(area_list[no].rom_type == ROM_MINIPCI) {
		write_disable_winbond(dev_winbond);
		}
	flashd_write_write_cache();
	isopened = 0;
	return 0;
}

static int flashd_read_proc(char *page, char **start, off_t off,
				 int count, int *eof, void *data)
{
	int              i;
	int              len  = 0;
	char*            buf  = page;
	struct firminfo* info;
	unsigned long    *boot_flag;

	if (boot_from_mini_pci)
		info = (struct firminfo*)0xFF800000;
	else
		info = (struct firminfo*)0xFFC00000;
	boot_flag = (unsigned long *)0xFFF70000;

	len += sprintf(buf+len,"PRODUCTNAME=");
	i=0;
	while(i<FIRMNAME_MAX+1 && info->firmname[i] != 0) {
		*(buf+len) = info->firmname[i];len++;i++;
		}
	len += sprintf(buf+len,"\n");
	len += sprintf(buf+len,"VERSION=%d.%02d\n",info->ver_major,info->ver_minor);
	len += sprintf(buf+len,"SUBVERSION=");
	
	i=0;
	while(i<FIRMNAME_MAX+1 && info->subver[i] != 0) {
		*(buf+len) = info->subver[i];len++;i++;
		}
	len += sprintf(buf+len,"\n");
	len += sprintf(buf+len,"PRODUCTID=0x%08lX\n",info->firmid);
	len += sprintf(buf+len,"BUILDDATE=%d/%d/%d %d:%d:%d\n",	info->year+1900,info->mon,info->day,
															info->hour,info->min,info->sec);
	if(boot_from_mini_pci) len += sprintf(buf+len,"MINI_PCI_BOOT\n");
	if(*boot_flag == 0x4e474e47)
		len += sprintf(buf+len,"BOOTFLAG=NG\n");
	else if(*boot_flag == 0x4f4b4f4b)
		len += sprintf(buf+len,"BOOTFLAG=OK\n");
	else
		len += sprintf(buf+len,"BOOTFLAG=OTHER\n");
	
	if (len <= off+count) *eof = 1;
	*start = page + off;
	len -= off;
	if (len>count) len = count;
	if (len<0) len = 0;
	return len;
}

static struct block_device_operations flashd_fops = {
	open:		flashd_open,
	release:	flashd_release,
	ioctl:		flashd_ioctl,
};

int __init flashd_init (void)
{
	int		               i;
	struct flash_dev_func* flash[1];
	unsigned char          dev = 0; 
	unsigned long          total_size  = 0;
	
	printk("FLASHDISK:");
	
	flash[ROM_ON_BORD] = NULL;
	flash[ROM_MINIPCI] = NULL;
	
	{
		unsigned long           data;
		cli();
		*((pvlong)PPC_CFG_REG) = cpu_to_le32((unsigned long)PPC_CFG_PCIR);
		data                   = le32_to_cpu(*((pvlong)PPC_DAT_REG));
		if(data & (1<<20)) 
			boot_from_mini_pci = 0;
		else 
			boot_from_mini_pci = 1;
		sti();
	}
	
	chg_rom(ROM_ON_BORD);
	i = 0;
	while((flash[ROM_ON_BORD] = flash_dev_list[i]) != NULL) {
		if(flash[ROM_ON_BORD]->probe()) {
			break;
			}
		++i;
		}
	
	if(flash[ROM_ON_BORD]==NULL) {
		printk("Can't find support flash rom!\n");
		return -EIO;
		}
	
	// Search Winbond of Mini-PCI Bord --
	for(i=1;i<0x1f;++i) {
		if(inw_pci(i,0,WINBOND_VENDOR_REG) == WINBOND_VENDOR_ID
		&& inw_pci(i,0,WINBOND_DEVICE_REG) == WINBOND_DEVICE_ID) {
			dev = i;
			break;
		}
	}
	if(dev) {
		chg_rom(ROM_MINIPCI);
		write_enable_winbond(dev);
		i=0;
		while((flash[ROM_MINIPCI] = flash_dev_list[i]) != NULL) {
			if(flash[ROM_MINIPCI]->probe()) {
				break;
			}
			++i;
		}
		if(flash[ROM_MINIPCI]!=NULL) dev_winbond = dev;
		write_disable_winbond(dev);
		chg_rom(ROM_ON_BORD);
	}
	
	flash_cache = kmalloc(flash[ROM_ON_BORD]->blocksizes,GFP_KERNEL);
	if(flash_cache == NULL) {
		printk("Can't kmalloc flash_cash.\n");
		return -EIO;
		}
	cacheinfo.size = flash[ROM_ON_BORD]->blocksizes;
	
	flashd_num = 0;
	while(area_list[flashd_num].length != 0) flashd_num++;
	if(flashd_num == 0) return -EIO;
	flashd           = kmalloc(flashd_num * sizeof(struct flash_info),GFP_KERNEL);
	flashd_hardsec   = kmalloc(flashd_num * sizeof(int),              GFP_KERNEL);
	flashd_blocksize = kmalloc(flashd_num * sizeof(int),              GFP_KERNEL);
	flashd_kbsize    = kmalloc(flashd_num * sizeof(int),              GFP_KERNEL);
	if(flashd==NULL           || 
	   flashd_hardsec==NULL   || 
	   flashd_blocksize==NULL || 
	   flashd_kbsize==NULL) {
		kfree(flash_cache);
		printk("Out of Memory..\n");
		return -EIO;
		}
	for(i=0;i<flashd_num;++i) {
		flashd[i].device     = flash[area_list[i].rom_type];
		flashd[i].rom_type   = area_list[i].rom_type;
		flashd[i].start_addr = area_list[i].start_addr;
		flashd[i].length     = area_list[i].length;
		flashd_hardsec[i]    = FLASHD_BLOCK_SIZE;
		flashd_blocksize[i]  = FLASHD_BLOCK_SIZE;
		flashd_kbsize[i]     = area_list[i].length>>10;
		total_size += area_list[i].length;
		}
	
	if (register_blkdev(MAJOR_NR, "flashd", &flashd_fops)) {
		printk("Could not get major %d", MAJOR_NR);
		return -EIO;
		}
	blk_queue_make_request(BLK_DEFAULT_QUEUE(MAJOR_NR), &flashd_make_request);
	
	devfs_handle = devfs_mk_dir (NULL, "fl", NULL);
	devfs_register_series (devfs_handle, "%u", flashd_num,
			       DEVFS_FL_DEFAULT, MAJOR_NR, 0,
			       S_IFBLK | S_IRUSR | S_IWUSR,
			       &flashd_fops, NULL);
	
	for (i = 0; i < flashd_num; i++)
		register_disk(NULL, MKDEV(MAJOR_NR,i), 1, &flashd_fops, flashd_kbsize[i]<<1);
	
	hardsect_size[MAJOR_NR] = flashd_hardsec;
	blksize_size[MAJOR_NR]  = flashd_blocksize;
	blk_size[MAJOR_NR]      = flashd_kbsize;
	
	isopened = 0;

	create_proc_read_entry("asconf_info", 0, NULL, flashd_read_proc, NULL);
	
	printk("Initialized ");
	for(i=0;i<2;++i) if(flash[i]) printk("[%s] ",flash[i]->flash_name);
	printk("\n");
	return 0;
}

void flashd_exit(void)
{
	devfs_unregister(devfs_handle);
	if (devfs_unregister_blkdev(MAJOR_NR, "flashd"))
		printk(KERN_WARNING "flashd: cannot unregister blkdev\n");
	if(flash_cache)      kfree(flash_cache);
	if(flashd)           kfree(flashd);
	if(flashd_hardsec)   kfree(flashd_hardsec);
	if(flashd_blocksize) kfree(flashd_blocksize);
	if(flashd_kbsize)    kfree(flashd_kbsize);
	flash_cache      = NULL;
	flashd           = NULL;
	flashd_hardsec   = NULL;
	flashd_blocksize = NULL;
	flashd_kbsize    = NULL;
	remove_proc_entry("asconf_info", NULL);
}

module_init(flashd_init);
module_exit(flashd_exit);

