/*
 *  Flash Rom Disk Driver
 *
 *  Copyright (C) 2001-2004  BUFFALO INC.
 *
 *  This software may be used and distributed according to the terms of
 *  the GNU General Public License (GPL), incorporated herein by reference.
 *  Drivers based on or derived from this code fall under the GPL and must
 *  retain the authorship, copyright and license notice.  This file is not
 *  a complete program and may only be used when the entire operating
 *  system is licensed under the GPL.
 *
 */
#ifndef _H_FLASHD_HEAD_
#define _H_FLASHD_HEAD_

#include <melco/melco_hwctl.h>

struct flash_dev_func {
	char *flash_name;
	int  blocksizes;
	int (*probe)(void);
	int (*blk_erase)(unsigned long addr ,int num);
	int (*blk_write)(unsigned long addr ,unsigned char* buf,int size);
	int (*blk_read) (unsigned long addr ,unsigned char* buf,int size);
	};

struct flashd_list {
	unsigned long start_addr;
	unsigned long length;
	unsigned char rom_type;
	};

struct flash_info {
	struct flash_dev_func* device;
	unsigned long          start_addr;
	unsigned long          length;
	struct inode           *inode;
	unsigned char          rom_type;
	};

#define ATTACH_FLASH_ADDRESS_1 0xFF000000
#define ATTACH_FLASH_ADDRESS_2 0xFF800000
#define ATTACH_FLASH_ADDRESS 0xFFC00000

#define PPC_CFG_REG  0xFEC00000
#define PPC_DAT_REG  0xFEE00000

#define PPC_CFG_PCIR 0x800000A8

#define ROM_ON_BORD 0
#define ROM_MINIPCI 1

#define WINBOND_VENDOR_ID 0x10AD
#define WINBOND_DEVICE_ID 0x0565

#define WINBOND_VENDOR_REG   0x00
#define WINBOND_DEVICE_REG   0x02
#define WINBOND_CSC_REG      0x4D

typedef volatile unsigned long*    pvlong;
typedef volatile unsigned short*   pvshort;

static int flash_amd_compt_probe(unsigned char factory , unsigned char devnum);
static int flash_amd_compt_erase(unsigned long addr ,int num , int flag);
int flash_st_compt_erase(unsigned long addr ,int num , int flag);

static int flash_toshiba_probe(void);
static int flash_toshiba_erase(unsigned long addr ,int num);
int flash_toshiba_write(unsigned long addr ,unsigned char* buf,int size);

static int flash_fujitsu_probe(void);
static int flash_fujitsu_erase(unsigned long addr ,int num);
static int flash_fujitsu_compat_erase(unsigned long addr ,int num , int flag);
static int flash_fujitsu_write(unsigned long addr ,unsigned char* buf,int size);

static int flash_st_probe(void);
static int flash_st_erase(unsigned long addr ,int num);
static int flash_st_write(unsigned long addr ,unsigned char* buf,int size);

static int flash_mx_probe(void);
static int flash_mx_write(unsigned long addr ,unsigned char* buf,int size);

#define HEAD_BLOCK    0x01
#define BOTTOM_BLOCK  0x02

struct flash_dev_func flash_toshiba  = 
	{
	"TOSHIBA VT641FT",
	64 * 1024,
	flash_toshiba_probe,
	flash_toshiba_erase,
	flash_toshiba_write,
	NULL
	};

struct flash_dev_func flash_fujitsu  = 
	{
	"FUJITSU MBM29PL32TM",
	64 * 1024,
	flash_fujitsu_probe,
	flash_fujitsu_erase,
	flash_fujitsu_write,
	NULL
	};

struct flash_dev_func flash_st  = 
	{
	"STMICRO M29W320DT",
	64 * 1024,
	flash_st_probe,
	flash_st_erase,
	flash_st_write,
	NULL
	};

struct flash_dev_func flash_mx  = 
	{
	"MXIC MX29LV320T",
	64 * 1024,
	flash_mx_probe,
	flash_toshiba_erase,
	flash_mx_write,
	NULL
	};

#define TOSHIBA_FLASH_CMDA(x)	\
	*(volatile char*)(ATTACH_FLASH_ADDRESS + 0xaaa) = x;
#define TOSHIBA_FLASH_CMD5(x)	\
	*(volatile char*)(ATTACH_FLASH_ADDRESS + 0x555) = x;
#define TOSHIBA_FLASH_START (volatile char*)(ATTACH_FLASH_ADDRESS)

#define FLASH_COMMAND_CLEAR TOSHIBA_FLASH_CMDA(0xf0)
#define FLASH_COMMAND_READY TOSHIBA_FLASH_CMDA(0xaa);TOSHIBA_FLASH_CMD5(0x55)


static int flash_toshiba_probe(void) 
{
	return flash_amd_compt_probe(0x98,0x93);
}

static int flash_toshiba_erase(unsigned long addr ,int num) 
{
	return flash_amd_compt_erase(addr ,num , BOTTOM_BLOCK);
}

static int flash_fujitsu_probe(void) 
{
	return flash_amd_compt_probe(0x04,0x7e);
}

static int flash_fujitsu_erase(unsigned long addr ,int num)
{
	return flash_fujitsu_compt_erase(addr ,num , BOTTOM_BLOCK);
}

static int flash_st_probe(void) 
{
	return flash_amd_compt_probe(0x20,0xCA);
}

static int flash_st_erase(unsigned long addr ,int num)
{
	return flash_st_compt_erase(addr ,num , BOTTOM_BLOCK);
}
static int flash_mx_probe(void) 
{
	return flash_amd_compt_probe(0xC2,0xA7);
}

static int flash_amd_compt_probe(unsigned char factory , unsigned char devnum)
{
	unsigned long time = jiffies;
	unsigned char vendor,device;
	FLASH_COMMAND_CLEAR;
	FLASH_COMMAND_READY;
	TOSHIBA_FLASH_CMDA(0x90);
	do {
		vendor = *(TOSHIBA_FLASH_START);
		device = *(TOSHIBA_FLASH_START+2);
		if(vendor == factory || device == devnum) break;
		} while(jiffies-time<100);
	FLASH_COMMAND_CLEAR;
	if(vendor == factory || device == devnum) return 1;
	return 0;
}

#define BIT(x) 1<<x
static int flash_toshiba_state(volatile unsigned char* addr ,unsigned char target)
{
	unsigned long time = jiffies;
	unsigned char data;
	do {
		data = *addr;
		if((data & BIT(7))==(target & BIT(7))) return 0;
		if(data & BIT(5)) {
			data = *addr;
			if((data & BIT(7))==(target & BIT(7))) 
				return 0;
			else
				return -1;
		}
	} while(jiffies-time<5000);
	return -1;
}

static int flash_amd_compt_erase(unsigned long addr ,int num , int flag)
{
	volatile unsigned char* baddr;
	volatile unsigned char* caddr;
	unsigned char           c;
	long  i,times;
	long  size;
	addr  = addr & 0xffff0000;
	baddr = (char*)addr;

	while(num>0) {
		if( ((flag & HEAD_BLOCK) != 0) &&
			((unsigned long)baddr >= (unsigned long)0xffc00000 &&
			 (unsigned long)baddr < (unsigned long)0xffc10000) ) {
				times = 8;
				size  = 0x2000;
		}
		else if( ((flag & BOTTOM_BLOCK) != 0) &&
			 (unsigned long)baddr >= (unsigned long)0xffff0000 ) {
			times = 8;
			size  = 0x2000;
		}
		else {
			times = 1;
			size  = 0x10000;
		}

		for(i=0;i<times;++i) {
			FLASH_COMMAND_CLEAR;
			FLASH_COMMAND_READY;
			TOSHIBA_FLASH_CMDA(0x80);
			FLASH_COMMAND_READY;
			*(baddr+i*size) = 0x30;
			if(flash_toshiba_state(baddr+i*size,0xff)) {
				printk("Flash Erase fail?!\n");
			}

			FLASH_COMMAND_CLEAR;
		}
		caddr = baddr;
		for(i=0;i<0x10000;++i) {
			c = *caddr;
			if(c != (unsigned char)0xff) {
				printk("verify fail! %p = %02X\n",caddr,c);
				return -1;
				}
			caddr++;
		}
		num--;
		baddr += 0x10000;
	}

	return 0;
}

int flash_st_compt_erase(unsigned long addr ,int num , int flag)
{
	volatile unsigned char* baddr;
	volatile unsigned char* caddr;
	unsigned char           c;
	long  i,times;
	long  size, erase_addr;
	addr  = addr & 0xffff0000;
	baddr = (char*)addr;

	while(num>0) {
		if( ((flag & HEAD_BLOCK) != 0) &&
			((unsigned long)baddr >= (unsigned long)0xffc00000 &&
			 (unsigned long)baddr < (unsigned long)0xffc10000) ) {
				times = 4;
				size  = 0x2000;
		}
		else if( ((flag & BOTTOM_BLOCK) != 0) &&
			 (unsigned long)baddr >= (unsigned long)0xffff0000 ) {
			times = 4;
			size  = 0x2000;
		}
		else {
			times = 1;
			size  = 0x10000;
		}
		for(i=0;i<times;++i) {
			if (times > 1) {
				switch (i) {
				case 0:
					if ((flag & HEAD_BLOCK) != 0)
						size = 0x4000;
					else
						size = 0x8000;
					erase_addr = 0;
					break;
				case 1:
				case 2:
					size = 0x2000;
					if ((flag & HEAD_BLOCK) != 0)
						erase_addr = 0x4000 + (i-1)*size;
					else
						erase_addr = 0x8000 + (i-1)*size;
					break;
				case 3:
					if ((flag & HEAD_BLOCK) != 0) {
						size = 0x8000;
						erase_addr = 0x8000;
					}
					else {
						size = 0x4000;
						erase_addr = 0xc000;
					}
					break;
				default:
					size = 0x2000;
				}
			}
			else
				erase_addr = 0;

			FLASH_COMMAND_CLEAR;
			udelay(50);
			FLASH_COMMAND_READY;
			TOSHIBA_FLASH_CMDA(0x80);
			FLASH_COMMAND_READY;
			*(baddr+erase_addr) = 0x30;
			if(flash_toshiba_state(baddr+erase_addr,0xff)) {
				printk("Flash Erase fail?!\n");
			}
			FLASH_COMMAND_CLEAR;
		}

		caddr = baddr;
		for(i=0;i<0x10000;++i) {
			c = *caddr;
			if(c != (unsigned char)0xff) {
				printk("verify fail! %p = %02X\n",caddr,c);
				return -1;
				}
			caddr++;
		}
		num--;
		baddr += 0x10000;
	}

	return 0;
}

static int flash_fujitsu_compt_erase(unsigned long addr ,int num , int flag)
{
	volatile unsigned char* baddr;
	volatile unsigned char* caddr;
	unsigned char           c;
	long  i,times;
	long  size;
	addr  = addr & 0xffff0000;
	baddr = (char*)addr;

	while(num>0) {
		if( ((flag & HEAD_BLOCK) != 0) &&
			((unsigned long)baddr >= (unsigned long)0xffc00000 &&
			 (unsigned long)baddr < (unsigned long)0xffc10000) ) {
				times = 8;
				size  = 0x2000;
		}
		else if( ((flag & BOTTOM_BLOCK) != 0) &&
			 (unsigned long)baddr >= (unsigned long)0xffff0000 ) {
			times = 8;
			size  = 0x2000;
		}
		else {
			times = 1;
			size  = 0x10000;
		}

		for(i=0;i<times;++i) {
			FLASH_COMMAND_CLEAR;
			FLASH_COMMAND_READY;
			TOSHIBA_FLASH_CMDA(0x80);
			FLASH_COMMAND_READY;
			*(baddr+i*size) = 0x30;
			if(flash_toshiba_state(baddr+i*size,0xff)) {
				printk("Flash Erase fail?!\n");
			}

			FLASH_COMMAND_CLEAR;
		}

		caddr = baddr;
		for(i=0;i<0x10000;++i) {
			c = *caddr;
			if(c != (unsigned char)0xff) {
				printk("verify fail! %p = %02X\n",caddr,c);
				return -1;
				}
			caddr++;
		}
		num--;
		baddr += 0x10000;
	}

	return 0;
}

int flash_toshiba_write(unsigned long addr ,unsigned char* buf,int size)
{
	volatile unsigned char* baddr;
	long                    i;
	baddr = (char*)addr;


	FLASH_COMMAND_CLEAR;
	FLASH_COMMAND_READY;
	TOSHIBA_FLASH_CMDA(0x20);
	for(i=0;i<size;++i) {
		*baddr = 0xa0;
		*baddr = buf[i];
		if(flash_toshiba_state(baddr,buf[i])) {
			printk("Flash Write fail?!\n");
		}
		baddr++;
	}
	TOSHIBA_FLASH_CMDA(0x90);
	FLASH_COMMAND_CLEAR;
	baddr = (char*)addr;

	for(i=0;i<size;++i) {
		if(baddr[i] != buf[i]) {
			printk("Flash Write verify fail %p = %02X\n", baddr[i], buf[i]);
			return -1;
		}
	}

	return 0;
}

static int flash_st_write(unsigned long addr ,unsigned char* buf,int size)
{
	volatile unsigned char* baddr;
	long                    i;
	int                     verify_fail = 0;
	baddr = (char*)addr;

	FLASH_COMMAND_CLEAR;
	FLASH_COMMAND_READY;
	TOSHIBA_FLASH_CMDA(0x20);
	for(i=0;i<size;++i) {
		*baddr = 0xa0;
		*baddr = buf[i];
		if(flash_toshiba_state(baddr,buf[i])) {
			printk("Flash Write fail?!\n");
		}
		baddr++;
	}
	TOSHIBA_FLASH_CMDA(0x90);
	TOSHIBA_FLASH_CMDA(0x00);
	FLASH_COMMAND_CLEAR;
	baddr = (char*)addr;

	do {
		if (verify_fail > 0)
			printk("Flash Verify %d times\n", verify_fail);
		for(i=0;i<size;++i) {
			if(*baddr != buf[i]) {
				printk("Flash Write verify fail %08x: %02x\n", *baddr, buf[i]);

				FLASH_COMMAND_CLEAR;
				FLASH_COMMAND_READY;
				TOSHIBA_FLASH_CMDA(0xa0);
				*baddr = buf[i];

				if(flash_toshiba_state(baddr,buf[i])) {
					printk("Flash Write retry fail?! %08x:%02p\n", *baddr, buf[i]);
					FLASH_COMMAND_CLEAR;
				} else {
					printk("Flash Write retry success %08x:%02p\n", *baddr, buf[i]);
					FLASH_COMMAND_CLEAR;
				}
				verify_fail++;
			}
			baddr++;
		}
	} while (verify_fail >= 2);

	return 0;
}

static int flash_fujitsu_write(unsigned long addr ,unsigned char* buf,int size)
{
	volatile unsigned char  *baddr, *naddr, *saddr;
	int                     i, j, write_time, surplus;
	int                     verify_fail = 0;

	baddr = (char*)addr;
	write_time = size / 32;
	surplus = size % 32;

	for(i = 0;i < write_time; i++) {
		baddr = (char*)addr + (i * 32);
		naddr = baddr;
		if(baddr < 0xffff0000)
			saddr = baddr - ((addr+(i*32)) % 0x10000);
		else
			saddr = baddr - ((addr+(i*32)) % 0x2000);

		FLASH_COMMAND_CLEAR;
		FLASH_COMMAND_READY;
		*saddr = 0x25;
		*saddr = 31;

		for(j = 0;j < 32; ++j) {
			*naddr = buf[(i*32) + j];
			naddr++;
		}

		*saddr = 0x29;
		if(flash_toshiba_state(naddr-1,buf[(i * 32) + j - 1])) {
				printk("%8X:%8X, %d/%d, Flash Write fail?!\n", saddr, baddr, i+1, write_time);
		}
		FLASH_COMMAND_READY;
		FLASH_COMMAND_CLEAR;
	}

	if(surplus > 0) {
		baddr = (char*)addr + (write_time * 32);
		naddr = baddr;
		if(baddr < 0xffff0000)
			saddr = baddr - ((addr+(write_time*32)) % 0x10000);
		else
			saddr = baddr - ((addr+(write_time*32)) % 0x2000);

		FLASH_COMMAND_CLEAR;
		FLASH_COMMAND_READY;
		*saddr = 0x25;
		*saddr = surplus - 1;

		for(j = 0;j < surplus; ++j) {
			*naddr = buf[(write_time * 32) + j];
			naddr++;
		}

		*saddr = 0x29;
		if(flash_toshiba_state(naddr-1,buf[(write_time * 32) + j - 1])) {
				printk("%8X:%8X:%d, Flash Write fail?!\n", saddr, baddr, surplus);
		}
		FLASH_COMMAND_READY;
		FLASH_COMMAND_CLEAR;
	}

	baddr = (char*)addr;

	do {
		if (verify_fail > 0)
			printk("Flash Verify %d times\n", verify_fail);
		for(i=0;i<size;++i) {
			if(*baddr != buf[i]) {
				printk("Flash Write verify fail %08x: %02x\n", *baddr, buf[i]);

				FLASH_COMMAND_CLEAR;
				FLASH_COMMAND_READY;
				TOSHIBA_FLASH_CMDA(0xa0);
				*baddr = buf[i];

				if(flash_toshiba_state(baddr,buf[i])) {
					printk("Flash Write retry fail?! %08x:%02p\n", *baddr, buf[i]);
					FLASH_COMMAND_CLEAR;
				} else {
					printk("Flash Write retry success %08x:%02p\n", *baddr, buf[i]);
					FLASH_COMMAND_CLEAR;
				}
				verify_fail++;
			}
			baddr++;
		}
	} while (verify_fail >= 2);

	return 0;
}

static int flash_mx_write(unsigned long addr ,unsigned char* buf,int size)
{
	volatile unsigned char* baddr;
	long                    i;
	baddr = (char*)addr;
	FLASH_COMMAND_CLEAR;
	for(i=0;i<size;++i) {
		FLASH_COMMAND_READY;
		TOSHIBA_FLASH_CMDA(0xa0);
		*baddr = buf[i];
		if(flash_toshiba_state(baddr,buf[i])) {
			printk("Flash Write fail?!\n");
		}
		baddr++;
	}
	FLASH_COMMAND_CLEAR;
	baddr = (char*)addr;
	for(i=0;i<size;++i) {
		if(*baddr != buf[i]) {
			printk("Flash Write verify fail %p = %02X\n", *baddr, buf[i]);

			FLASH_COMMAND_CLEAR;
			FLASH_COMMAND_READY;
			TOSHIBA_FLASH_CMDA(0xa0);
			*baddr = buf[i];

			if(flash_toshiba_state(baddr,buf[i])) {
				printk("Flash Write retry fail?! %p\n", *baddr);
				FLASH_COMMAND_CLEAR;
				return -1;
			} else {
				printk("Flash Write retry success %p\n", *baddr);
				FLASH_COMMAND_CLEAR;
			}
		}
		baddr++;
	}

	return 0;
}

#endif
