/* Taken from Ted's losetup.c - Mitch <m.dsouza@mrc-apu.cam.ac.uk> */
/* Added vfs mount options - aeb - 960223 */
/* Removed lomount - aeb - 960224 */

/* 1999-02-22 Arkadiusz Mikiewicz <misiek@pld.ORG.PL>
 * - added Native Language Support
 * Sun Mar 21 1999 - Arnaldo Carvalho de Melo <acme@conectiva.com.br>
 * - fixed strerr(errno) in gettext calls
 * 2000-09-24 Marc Mutz <Marc@Mutz.com>
 * - added long option names and the --pass-fd option to pass
 *   passphrases via fd's to losetup/mount. Used for encryption in
 *   non-interactive environments. The idea behind xgetpass() is stolen
 *   from GnuPG, v.1.0.3 (http://www.gnupg.org/).
 */

#define PROC_DEVICES	"/proc/devices"

/*
 * losetup.c - setup and control loop devices
 */

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <fcntl.h>
#include <errno.h>
#include <stdlib.h>
#include <unistd.h>
#include <limits.h>
#include <sys/ioctl.h>
#include <sys/stat.h>
#include <sys/mman.h>

#include "loop.h"
#include "lomount.h"
#include "rmd160.h"
#include "xstrncpy.h"
#include "nls.h"

#ifndef LO_CRYPT_CRYPTOAPI
#define LO_CRYPT_CRYPTOAPI 18
#endif
#ifndef LO_CRYPT_NONE
#define LO_CRYPT_NONE 0
#endif
#ifndef LO_CRYPT_XOR
#define LO_CRYPT_XOR 1
#endif
#ifndef LO_CRYPT_DES
#define LO_CRYPT_DES 2
#endif
#ifndef LO_CRYPT_FISH2
#define LO_CRYPT_FISH2 3
#endif
#ifndef LO_CRYPT_BLOW
#define LO_CRYPT_BLOW 4
#endif
#ifndef LO_CRYPT_CAST128
#define LO_CRYPT_CAST128 5
#endif
#ifndef LO_CRYPT_IDEA
#define LO_CRYPT_IDEA 6
#endif
#ifndef LO_CRYPT_SERPENT
#define LO_CRYPT_SERPENT 7
#endif
#ifndef LO_CRYPT_MARS
#define LO_CRYPT_MARS 8
#endif
#ifndef LO_CRYPT_RC6
#define LO_CRYPT_RC6 11
#endif
#ifndef LO_CRYPT_3DES
#define LO_CRYPT_3DES 12
#endif
#ifndef LO_CRYPT_DFC
#define LO_CRYPT_DFC 15
#endif
#ifndef LO_CRYPT_RIJNDAEL
#define LO_CRYPT_RIJNDAEL 16
#endif


extern int verbose;
extern char *xstrdup (const char *s);	/* not: #include "sundries.h" */
extern void error (const char *fmt, ...);	/* idem */


struct cipher_info
{
	const char *name;
	int blocksize;
	int keysize_mask;
	int ivsize;
	int key_schedule_size;
};

static int set_loop_passwd(struct loop_info *_loopinfo, int pfd, int keysz,
			   const char *encryption, int fd, int ffd);
static int get_cipher_info(const char *name, struct cipher_info *res);
static int name_to_id(const char *name);
#ifdef MAIN
static char *id_to_name(int id);
#endif
 

#ifdef LOOP_SET_FD
struct crypt_type_struct {
	int id;
	char *name;
	int keylength;
} crypt_type_tbl[] = {
        { LO_CRYPT_NONE, "none", 0 },
        { LO_CRYPT_XOR, "xor", 0 },
        { LO_CRYPT_DES, "des", 8 },
        { LO_CRYPT_FISH2, "twofish", 20 },
        { LO_CRYPT_BLOW, "blowfish", 20 },
        { LO_CRYPT_CAST128, "cast", 16 },
        { LO_CRYPT_SERPENT, "serpent", 16 },
        { LO_CRYPT_MARS, "mars", 16 },
        { LO_CRYPT_RC6, "rc6", 16 },
        { LO_CRYPT_3DES, "des-ede3", 24 },
        { LO_CRYPT_DFC, "dfc", 16 },
        { LO_CRYPT_IDEA, "idea", 16 },
        { LO_CRYPT_RIJNDAEL, "rijndael", 16 },
	{ -1, NULL,0   }
};

#ifdef MAIN
static int
show_loop (char *device) {
	struct loop_info loopinfo;
	int fd;

	if ((fd = open (device, O_RDONLY)) < 0) {
		int errsv = errno;
		fprintf(stderr, _("loop: can't open device %s: %s\n"),
			device, strerror (errsv));
		return 2;
	}
	if (ioctl (fd, LOOP_GET_STATUS, &loopinfo) < 0) {
		int errsv = errno;
		fprintf(stderr, _("loop: can't get info on device %s: %s\n"),
			device, strerror (errsv));
		close (fd);
		return 1;
	}
	printf (_("%s: [%04x]:%ld (%s) offset %d, %s encryption\n"),
		device, loopinfo.lo_device, loopinfo.lo_inode,
		loopinfo.lo_name, loopinfo.lo_offset,
		id_to_name(loopinfo.lo_encrypt_type));
	close (fd);

	return 0;
}
#endif

int
is_loop_device (const char *device) {
	struct stat statbuf;
	int loopmajor;
#if 1
	loopmajor = 7;
#else
	FILE *procdev;
	char line[100], *cp;

	loopmajor = 0;
	if ((procdev = fopen(PROC_DEVICES, "r")) != NULL) {
		while (fgets (line, sizeof(line), procdev)) {
			if ((cp = strstr (line, " loop\n")) != NULL) {
				*cp='\0';
				loopmajor=atoi(line);
				break;
			}
		}
		fclose(procdev);
	}
#endif
	return (loopmajor && stat(device, &statbuf) == 0 &&
		S_ISBLK(statbuf.st_mode) &&
		(statbuf.st_rdev>>8) == loopmajor);
}

#define SIZE(a) (sizeof(a)/sizeof(a[0]))

char *
find_unused_loop_device (void) {
	/* Just creating a device, say in /tmp, is probably a bad idea -
	   people might have problems with backup or so.
	   So, we just try /dev/loop[0-7]. */
	char dev[20];
	char *loop_formats[] = { "/dev/loop%d", "/dev/loop/%d" };
	int i, j, fd, somedev = 0, someloop = 0, loop_known = 0;
	struct stat statbuf;
	struct loop_info loopinfo;
	FILE *procdev;

	for (j = 0; j < SIZE(loop_formats); j++) {
	    for(i = 0; i < 256; i++) {
		sprintf(dev, loop_formats[j], i);
		if (stat (dev, &statbuf) == 0 && S_ISBLK(statbuf.st_mode)) {
			somedev++;
			fd = open (dev, O_RDONLY);
			if (fd >= 0) {
				if(ioctl (fd, LOOP_GET_STATUS, &loopinfo) == 0)
					someloop++;		/* in use */
				else if (errno == ENXIO) {
					close (fd);
					return xstrdup(dev);/* probably free */
				}
				close (fd);
			}
			continue;/* continue trying as long as devices exist */
		}
		break;
	    }
	}

	/* Nothing found. Why not? */
	if ((procdev = fopen(PROC_DEVICES, "r")) != NULL) {
		char line[100];
		while (fgets (line, sizeof(line), procdev))
			if (strstr (line, " loop\n")) {
				loop_known = 1;
				break;
			}
		fclose(procdev);
		if (!loop_known)
			loop_known = -1;
	}

	if (!somedev)
		error(_("mount: could not find any device /dev/loop#"));
	else if(!someloop) {
	    if (loop_known == 1)
		error(_(
		    "mount: Could not find any loop device.\n"
		    "       Maybe /dev/loop# has a wrong major number?"));
	    else if (loop_known == -1)
		error(_(
		    "mount: Could not find any loop device, and, according to %s,\n"
		    "       this kernel does not know about the loop device.\n"
		    "       (If so, then recompile or `modprobe loop'.)"),
		      PROC_DEVICES);
	    else
		error(_(
		    "mount: Could not find any loop device. Maybe this kernel does not know\n"
		    "       about the loop device (then recompile or `modprobe loop'), or\n"
		    "       maybe /dev/loop# has the wrong major number?"));
	} else
		error(_("mount: could not find any free loop device"));
	return 0;
}

#define HASHLENGTH 20
#define PASSWDBUFFLEN 130 /* getpass returns only max. 128 bytes, see man getpass */

/* A function to read the passphrase either from the terminal or from
 * an open file descriptor */
static char *
xgetpass (int pfd, const char *prompt)
{
        if (pfd < 0) /* terminal */
	        return (getpass(prompt));
	else {       /* file descriptor */
	        char *pass = NULL;
		int buflen, i;

		buflen=0;
		for (i=0; ; i++) {
		        if (i >= buflen-1) {
		                /* we're running out of space in the buffer. 
				 * Make it bigger: */
		                char *tmppass = pass;
				buflen += 128;
				pass = realloc(tmppass,buflen);
				if (pass == NULL) {
					/* realloc failed. Stop reading _now_. */
			                error("not enough memory while reading passphrase");
					pass = tmppass; /* the old buffer hasn't changed */
					break;
				}
			};
			if ( read(pfd,pass+i, 1) != 1 || pass[i] == '\n' )
			        break;
		}
		if (pass == NULL)
		        return "";
		else {
		        pass[i] = 0;
		        return pass;
		}
	}
}

int
set_loop (const char *device, const char *file, int offset,
	  const char *encryption, int pfd, int keysz, int *loopro) {
	struct loop_info loopinfo;
	int fd, ffd, mode, tried_old;

	mode = (*loopro ? O_RDONLY : O_RDWR);
	if ((ffd = open (file, mode)) < 0) {
		if (!*loopro && errno == EROFS)
			ffd = open (file, mode = O_RDONLY);
		if (ffd < 0) {
			perror (file);
			return 1;
		}
	}
	if ((fd = open (device, mode)) < 0) {
		perror (device);
		return 1;
	}
	*loopro = (mode == O_RDONLY);

	memset (&loopinfo, 0, sizeof (loopinfo));
	snprintf(loopinfo.lo_name, sizeof(loopinfo.lo_name),
		 "%s-cbc", encryption);
	loopinfo.lo_name[LO_NAME_SIZE - 1] = 0;
	loopinfo.lo_encrypt_type = LO_CRYPT_CRYPTOAPI;
	loopinfo.lo_offset = offset;

#ifdef MCL_FUTURE  
	/*
	 * Oh-oh, sensitive data coming up. Better lock into memory to prevent
	 * passwd etc being swapped out and left somewhere on disk.
	 */
                                                
	if(mlockall(MCL_CURRENT | MCL_FUTURE)) {
		perror("memlock");
		fprintf(stderr, _("Couldn't lock into memory, exiting.\n"));
		exit(1);
	}
#endif

	if (ioctl (fd, LOOP_SET_FD, ffd) < 0) {
		perror ("ioctl: LOOP_SET_FD");
		return 1;
	}

	tried_old = 0;
again:
	set_loop_passwd(&loopinfo, pfd, keysz, encryption, fd, ffd);

	if (ioctl (fd, LOOP_SET_STATUS, &loopinfo) < 0) {
		/* Try again with old-style LO_CRYPT_XX if
                   new-style LO_CRYPT_CRYPTOAPI ioctl didn't work */
		if (tried_old) {
			error("The cipher does not exist, or a cipher module "
			      "needs to be loaded into the kernel");
			perror ("ioctl: LOOP_SET_STATUS");
			goto out_ioctl;
		}
		strncpy (loopinfo.lo_name, file, LO_NAME_SIZE);
		loopinfo.lo_name[LO_NAME_SIZE - 1] = 0;
		loopinfo.lo_encrypt_type = name_to_id (encryption);
		tried_old = 1;
		goto again;
	}
	close (fd);
	close (ffd);
	if (verbose > 1)
		printf(_("set_loop(%s,%s,%d): success\n"),
		       device, file, offset);
	return 0;
out_ioctl:
	(void) ioctl (fd, LOOP_CLR_FD, 0);
	return 1;
}

int
set_loop_passwd(struct loop_info *loopinfo, int pfd, int keysz, 
		const char *encryption, int fd, int ffd)
{
	int i;
	int keylength;
	char *pass;
	char keybits[2*HASHLENGTH]; 
	char passwdbuff[PASSWDBUFFLEN];
	struct cipher_info info;

	switch (loopinfo->lo_encrypt_type) {
	case LO_CRYPT_NONE:
		loopinfo->lo_encrypt_key_size = 0;
		break;
	case LO_CRYPT_XOR:
          /* WARNING: xgetpass() can return massive amounts of data,
           * not only 128 bytes like the original getpass(3) */
		pass = xgetpass (pfd,_("Password: "));
		xstrncpy (loopinfo->lo_encrypt_key, pass, LO_KEY_SIZE);
		loopinfo->lo_encrypt_key_size = strlen(loopinfo->lo_encrypt_key);
		break;
	case LO_CRYPT_DES:
		printf(_("WARNING: Use of DES is depreciated.\n"));
		pass = xgetpass (pfd,_("Password: "));
		strncpy (loopinfo->lo_encrypt_key, pass, 8);
		loopinfo->lo_encrypt_key[8] = 0;
		loopinfo->lo_encrypt_key_size = 8;
		pass = getpass (_("Init (up to 16 hex digits): "));
		for (i = 0; i < 16 && pass[i]; i++)
			if (isxdigit (pass[i])) {
				loopinfo->lo_init[i >> 3] |= (pass[i] > '9' ?
				  (islower (pass[i]) ? toupper (pass[i]) :
				   pass[i])-'A'+10 : pass[i]-'0') << (i&7) * 4;
			} else {
				fprintf (stderr, _("Non-hex digit '%c'.\n"),
					 pass[i]);
				return 1;
			}
		break;
	case LO_CRYPT_FISH2:
	case LO_CRYPT_BLOW:
	case LO_CRYPT_IDEA:
	case LO_CRYPT_CAST128:
        case LO_CRYPT_SERPENT:
        case LO_CRYPT_MARS:
        case LO_CRYPT_RC6:
        case LO_CRYPT_3DES:
        case LO_CRYPT_DFC:
        case LO_CRYPT_RIJNDAEL:
		pass = xgetpass(pfd, _("Password :"));
		strncpy(passwdbuff+1,pass,PASSWDBUFFLEN-1);
		passwdbuff[PASSWDBUFFLEN-1] = '\0';
		passwdbuff[0] = 'A';
		rmd160_hash_buffer(keybits,pass,strlen(pass));
		rmd160_hash_buffer(keybits+HASHLENGTH,passwdbuff,strlen(pass)+1);
		memcpy((char*)loopinfo->lo_encrypt_key,keybits,2*HASHLENGTH);
		keylength=0;
		for(i=0; crypt_type_tbl[i].id != -1; i++){
		         if(loopinfo->lo_encrypt_type == crypt_type_tbl[i].id){
			         keylength = crypt_type_tbl[i].keylength;
				 break;
			 }
		}
		loopinfo->lo_encrypt_key_size=keylength;
		break;
	case LO_CRYPT_CRYPTOAPI:
		/* Give the kernel an opportunity to load the cipher */
		(void) ioctl (fd, LOOP_SET_STATUS, loopinfo);
		if (get_cipher_info(loopinfo->lo_name, &info) < 0) {
			return 1;
		}
		if (keysz > 0 &&
		    !((1 << ((keysz / 8) - 1)) & info.keysize_mask)) {
			error("The specified keysize is not supported by "
			      "the selected cipher");
			keysz = 0;
		}

		while (keysz <= 0 ||
		       !((1 << ((keysz / 8) - 1)) & info.keysize_mask)) {
			int i = 0;
			int available = 0;
			char keysize[200];
			printf("Available keysizes (bits): ");
			for (; i < 32; i++) {
				if (info.keysize_mask & (1 << i)) {
					printf("%d ", 8*(i+1));
					available = 1;
				}
			}
			if (!available) {
				printf("none");
			}
			printf("\nKeysize: ");
			fgets(keysize, sizeof(keysize), stdin);
			keysz = atoi(keysize);
		}

		pass = xgetpass(pfd, _("Password :"));
		strncpy(passwdbuff+1,pass,PASSWDBUFFLEN-1);
		passwdbuff[PASSWDBUFFLEN-1] = '\0';
		passwdbuff[0] = 'A';
		rmd160_hash_buffer(keybits,pass,strlen(pass));
		rmd160_hash_buffer(keybits+HASHLENGTH,passwdbuff,strlen(pass)+1);
		memcpy((char*)loopinfo->lo_encrypt_key,keybits,2*HASHLENGTH);

		loopinfo->lo_encrypt_key_size=keysz/8;

		break;
	default:
		fprintf (stderr,
			 _("Don't know how to get key for encryption system %d\n"),
			 loopinfo->lo_encrypt_type);
		return 1;
	}
        return 0;
}




int 
del_loop (const char *device) {
	int fd;

	if ((fd = open (device, O_RDONLY)) < 0) {
		int errsv = errno;
		fprintf(stderr, _("loop: can't delete device %s: %s\n"),
			device, strerror (errsv));
		return 1;
	}
	if (ioctl (fd, LOOP_CLR_FD, 0) < 0) {
		perror ("ioctl: LOOP_CLR_FD");
		return 1;
	}
	close (fd);
	if (verbose > 1)
		printf(_("del_loop(%s): success\n"), device);
	return 0;
}

#else /* no LOOP_SET_FD defined */
static void
mutter(void) {
	fprintf(stderr,
		_("This mount was compiled without loop support. "
		  "Please recompile.\n"));
}  

int
set_loop (const char *device, const char *file, int offset,
	  const char *encryption, int pfd, int *loopro) {
	mutter();
	return 1;
}

int
del_loop (const char *device) {
	mutter();
	return 1;
}

char *
find_unused_loop_device (void) {
	mutter();
	return 0;
}

#endif

#ifdef MAIN

#ifdef LOOP_SET_FD

#include <getopt.h>
#include <stdarg.h>

int verbose = 0;
static char *progname;

static struct option longopts[] = {
	{ "delete", 0, 0, 'd' },
	{ "detach", 0, 0, 'd' },
	{ "encryption", 1, 0, 'e' },
	{ "help", 0, 0, 'h' },
	{ "offset", 1, 0, 'o' },
	{ "pass-fd", 1, 0, 'p' },
	{ "verbose", 0, 0, 'v' },
	{ "keybits", 1, 0, 'k' },
	{ NULL, 0, 0, 0 }
};


static void
usage(void) {
	fprintf(stderr, _("usage:\n\
  %s loop_device                                      # give info\n\
  %s -d loop_device                                   # delete\n\
  %s [ options ] loop_device file                     # setup\n\
    where options include\n\
    --offset <num>, -o <num>\n\
        start at offset <num> into file.\n\
    --pass-fd <num>, -p <num>\n\
        read passphrase from file descriptor <num>\n\
        instead of the terminal.\n\
    --encryption <cipher>, -e <cipher>\n\
        encrypt with <cipher>.\n\
        Check /proc/cipher for available ciphers.\n\
    --keybits <num>, -k <num>\n\
        specify number of bits in the hashed key given\n\
        to the cipher.  Some ciphers support several key\n\
        sizes and might be more efficient with a smaller\n\
        key size.  Key sizes < 128 are generally not\n\
        recommended\n"), progname, progname, progname);
	exit(1);
}

char *
xstrdup (const char *s) {
	char *t;

	if (s == NULL)
		return NULL;

	t = strdup (s);

	if (t == NULL) {
		fprintf(stderr, _("not enough memory"));
		exit(1);
	}

	return t;
}

void
error (const char *fmt, ...) {
	va_list args;

	va_start (args, fmt);
	vfprintf (stderr, fmt, args);
	va_end (args);
	fprintf (stderr, "\n");
}

int
main(int argc, char **argv) {
	char *offset, *encryption, *passfd, *keysize;
	int delete,off,c;
	int pfd = -1; 
	int res = 0;
	int ro = 0;
	int keysz = 0;

	setlocale(LC_ALL, "");
	bindtextdomain(PACKAGE, LOCALEDIR);
	textdomain(PACKAGE);

	delete = off = 0;
	offset = encryption = passfd = keysize = NULL;
	progname = argv[0];
	while ((c = getopt_long(argc,argv,"de:hk:o:p:v",
				longopts, NULL)) != EOF) {
		switch (c) {
		case 'd':
			delete = 1;
			break;
		case 'e':
			encryption = optarg;
			break;
		case 'k':
		        keysize = optarg;
			break;
		case 'o':
			offset = optarg;
			break;
		case 'p':
		        passfd = optarg;
			break;
		case 'v':
			verbose = 1;
			break;
		default:
			usage();
		}
	}
	if (argc == 1) usage();
	if ((delete && (argc != optind+1 || encryption || offset || passfd)) ||
	    (!delete && (argc < optind+1 || argc > optind+2)))
		usage();
	if (argc == optind+1) {
		if (delete)
			res = del_loop(argv[optind]);
		else
			res = show_loop(argv[optind]);
	} else {
		if (offset && sscanf(offset,"%d",&off) != 1)
			usage();
		if (passfd && sscanf(passfd,"%d",&pfd) != 1)
		        usage();
		if (keysize && sscanf(keysize,"%d",&keysz) != 1)
		        usage();
		res = set_loop(argv[optind], argv[optind+1], off,
			       encryption, pfd, keysz, &ro);
	}
	return res;
}

#else /* LOOP_SET_FD not defined */

int
main(int argc, char **argv) {
	fprintf(stderr,
		_("No loop support was available at compile time. "
		  "Please recompile.\n"));
	return -1;
}
#endif
#endif

static int get_cipher_info(const char *name, struct cipher_info *res)
{
	char path[PATH_MAX];
	char buf[2000];
	FILE *f;
	struct {
		int *out;
		const char *prefix;
	} fields[] = {{&res->blocksize, "blocksize:"},
		      {&res->keysize_mask, "keysize_mask:"},
		      {&res->ivsize, "ivsize:"},
		      {&res->key_schedule_size, "key_schedule_size:"}};
	snprintf(path, sizeof(path), "/proc/crypto/cipher/%s", name);
	f = fopen(path, "r");
	while(f && fgets(buf, sizeof(buf), f)) {
		int i;
		for (i = 0; i < sizeof(fields)/sizeof(fields[0]); i++) {
			int len = strlen(fields[i].prefix);
			if (strncmp(buf, fields[i].prefix, len) == 0) {
				*fields[i].out = strtoul(&buf[len+1], NULL, 0);
				break;
			}
		}
		
	}
	if (!f) 
		return -1;
	return 0;
}


static int 
name_to_id(const char *name) 
{
	int i;

	if (name) {
		for (i = 0; crypt_type_tbl[i].id != -1; i++)
			if (!strcasecmp (name, crypt_type_tbl[i].name))
				return crypt_type_tbl[i].id;
	} else
		return LO_CRYPT_NONE;
	return LO_CRYPT_CRYPTOAPI;
}

#ifdef MAIN
static char *
id_to_name(int id) {
	int i;

	for (i = 0; crypt_type_tbl[i].id != -1; i++)
		if (id == crypt_type_tbl[i].id)
			return crypt_type_tbl[i].name;
	return "undefined";
}
#endif

