/* ------------------------------
  Set CheckSum for Kernel Image
  Composed by Inada.
  
------------------------------ */
#include <stdio.h>
#include <sys/stat.h>
#include <unistd.h>
#include <time.h>

#include "../../../../../include/buffalo/firminfo.h"
// ----------------------------------------------------
struct lineinfo {
	char index[16];
	int  id;
	};

#define INFO_ID          0x01
#define INFO_NAME        0x02
#define INFO_VERSION     0x04
#define INFO_SUBVERSION  0x08
#define INFO_ALL     (INFO_ID | INFO_NAME | INFO_VERSION | INFO_SUBVERSION)

static struct lineinfo info[]={
	{"ID",          INFO_ID},
	{"NAME",        INFO_NAME},
	{"VER" ,        INFO_VERSION},
	{"SUBVERSION" , INFO_SUBVERSION},
	{"",0x00},
	};
// ----------------------------------------------------
#define STRBUFF_MAX 256*4

/* usage 
   setsum config_file target_file
*/
struct firminfo *information;
unsigned long    information_size;
// ----------------------------------------------------
// Little -> Big 
#define l2bl(a) \
	((a & 0xff000000) >> 24) + \
	((a & 0x00ff0000) >> 8 ) + \
	((a & 0x0000ff00) << 8 ) + \
	((a & 0x000000ff) << 24)
#define l2bs(a) \
	((a & 0xff00) >> 8 ) + \
	((a & 0x00ff) << 8)
// ----------------------------------------------------

void print_usage() 
	{
	int i=0;
	printf("Check sum calicurate mode -----\n");
	printf("    Usage:setsum targetfile\n");
	printf("Add information header mode ---\n");
	printf("    Usage:setsum configfile targetfile [-r ramdiskfile] [-o outputfile]\n");
	printf("Support information tag:\n    ");
	while(info[i].id!=0) {printf("[%s] ",info[i].index);i++;}
	printf("\n");
	return;
	}
void print_info() 
	{
	struct firminfo* info = information;
	int              i = 0;
	printf("Firmware ID  :%08X\n",info->firmid);
	printf("         Name:");
	while(i<32 && info->firmname[i]!=0) {printf("%c",info->firmname[i]);i++;}
	printf(" ver %d.%02d\n"  ,info->ver_major,info->ver_minor);
	printf("         Date:%d/%d/%d %d:%d:%d\n",info->year+1900,info->mon,info->day,
												info->hour,info->min,info->sec);
	printf("kernel_offset:%08X (%dbytes)\n"     ,info->kernel_offset,info->kernel_size);
	printf("initrd_offset:%08X (%dbytes)\n"     ,info->initrd_offset,info->initrd_size);
	printf("total size: %dbytes\n"              ,info->size);
	}

void error(char* mes)
	{
	printf("Error:");printf(mes);printf("\n");
	}
void strtrim(char* s)
	{
	int i = strlen(s)-1;
	if(i<0) return;
	while((s[i]==' '||s[i]=='\t') && i>=0) {s[i]='\0';i--;}
	}
void atoA(char* str) 
	{
	while(*str!=0){
		if(*str>='a' && *str<='z') *str = *str-'a'+'A';
		str++;
		}
	}
int atoL(char* str,unsigned long* req)
	{
	int ishex = 0;
	*req = 0;
	if(strlen(str)>2 && str[0]=='0' && str[1]=='X') {
		ishex = 1;
		str  += 2;
		}
	while((*str>='0' && *str<='9') || (ishex && *str>='A' && *str<='F')) {
		if(ishex) *req *= 0x10; else *req *= 10;
		if(*str>='0' && *str<='9') *req += *str-'0';
		if(*str>='A' && *str<='Z') *req += *str-'A'+10;
		str++;
		}
	if(*str!='\0') return(-1);
	return 0;
	}
// ----------------------------------------------------
int setparam(int id,char* cmd)
	{
	unsigned long val;
	int                i;
	char          *major,*minor;
	switch(id) {
		case INFO_ID:
			atoA(cmd);
			if(atoL(cmd,&val)) {
				error("Firm ID number incorrect!");
				return(-1);
				}
			information->firmid = val;
			break;
		case INFO_NAME:
			if(strlen(cmd)>FIRMNAME_MAX) {
				error("Firm Name incorrect length! Please under 31 chars!");
				return(-1);
				}
			strcpy(information->firmname,cmd);
			break;
		case INFO_SUBVERSION:
			if(strlen(cmd)>SUBVERSION_MAX) {
				error("Subversion incorrect length! Please under 31 chars!");
				return(-1);
				}
			strcpy(information->subver,cmd);
			break;
		case INFO_VERSION:
			major  = cmd;
			minor = NULL;
			while(*cmd!='\0') {
				if(*cmd=='.') {minor = cmd+1;*cmd='\0';break;}
				cmd++;
				}
			if(minor == NULL) {
				error("Firm Version incorrect format!");
				return(-1);
				}
			if(atoL(major,&val) || val>99) {
				error("Firm Major Version incorrect format!");
				return(-1);
				}
			information->ver_major = (unsigned short)val;
			if(atoL(minor,&val) || val>99) {
				error("Firm minor Version incorrect format!");
				return(-1);
				}
			information->ver_minor = (unsigned short)val;
			break;
		default :
			error("unknown command!");
			return(-1);
		}
	return(0);
	}
// ----------------------------------------------------
int readconfig(char* filename)
	{
	FILE          *rfp;
	int           i,j,id;
	int           comp = 0;
	char tmp[STRBUFF_MAX+1],c,sep;
	char *index,*command;
	if((rfp = fopen(filename,"r"))==NULL) {
		error("can't open config file!");
		return(-1);
		}
	// line read -----
	while(comp != INFO_ALL) {
		if(fgets(tmp,STRBUFF_MAX,rfp) == NULL) break;
		index   = NULL;
		command = NULL;
		i=0;j=0;
		while(tmp[i]!='\0' && j<4) {
			c = tmp[i];
			switch(j) {
				case 0:
					if((c>='A' && c<='Z') || (c>='a' && c<='z') || (c>='0' && c<='9') || (c=='_')) {
						j++;
						index = &(tmp[i]);
						}
					break;
				case 1:
					if(c=='=') {
						j++;
						tmp[i]='\0';
						}
					if(c>='a' && c<='z') tmp[i] = tmp[i]-'a'+'A';
					break;
				case 2:
					if(c=='"' || c=='\'') {
						sep = c;
						j++;
						tmp[i]='\0';
						}
					break;
				case 3:
					if(c == sep) {
						j++;
						tmp[i]='\0';
						}
					else if(command == NULL) command=&(tmp[i]);
					break;
				}
			++i;
			}
		if(j<4 || index == NULL || command == NULL) continue;
		strtrim(index);
		i = 0;
		id = 0;
		while(info[i].id != 0) {
			if(strcmp(info[i].index,index)==0) {id = info[i].id;break;}
			++i;
			}
		if(id == 0) printf("Warning!:unknown index:%s\n",index);
		else if(setparam(id,command)==0) comp = comp | id;
		}
	if(comp != INFO_ALL) {
		error("incorrect config file!");
		return(-1);
		}
	return;
	}
// ----------------------------------------------------
unsigned long chksum(unsigned long nowsum,unsigned char* addr,unsigned long size)
	{
	unsigned long* laddr = (unsigned long*)addr;
	unsigned long  sum    = nowsum,
	               remain = 0;
	int i;
	while(size>=4) {
		sum += l2bl(*laddr);
		laddr++;
		size -= 4;
		}
	addr = (unsigned char*)laddr;
	for(i=0;i<4;++i) {
		remain = remain << 8;
		if(size>i) remain += *addr;
		addr++;
		}
	sum += remain;
	return sum;
	}
/*
long chksum(long nowsum,unsigned char* data,unsigned long size)
	{
	int  i;
	long retval = nowsum;
	long *ldata  = (long*)data;
	long lastone = 0;
	long lsize   = size/4;
	long amari   = size%4;
	while(lsize>0) {lsize--;retval += l2bl(*ldata);ldata++;}
	data = (unsigned char*)ldata;
	if(amari>0) {
		for(i=0;i<4;++i) {
			lastone = lastone << 8;
			if(amari>i) {lastone += *data;}
			}
		retval += lastone;
		}
	return retval;
	}
*/
// ----------------------------------------------------
unsigned long get_filesize(char* name)
{
	struct stat   filestat;
	if(stat(name,&filestat)) {
		error("can't get target file size!");
		return(-1);
		}
	return (unsigned long)(filestat.st_size);
}


// ----------------------------------------------------
int setinfomation(char* infn,char* outfn) 
	{
	struct stat   filestat;
	FILE          *rfp,*wfp;
	char          tmp[STRBUFF_MAX+1];
	unsigned long sum      = 0,
	              totallen = 0;
	long          len;
	
	int           info_size;
	int           file_size;
	
	// get target size --
	info_size = information_size;
	file_size = get_filesize(infn);
	
	information->size  = information_size + file_size; // firmware total size
	
	print_info();
	
	// Endian Change ----
	information->info_ver   = l2bl(information->info_ver);
	information->firmid     = l2bl(information->firmid);
	information->ver_major  = l2bs(information->ver_major);
	information->ver_minor  = l2bs(information->ver_minor);
	information->build      = l2bs(information->build);
	information->size       = l2bl(information->size);
	information->chksum     = 0; // clear sum
	information->kernel_offset = l2bl(information->kernel_offset);
	information->kernel_size   = l2bl(information->kernel_size);
	information->initrd_offset = l2bl(information->initrd_offset);
	information->initrd_size   = l2bl(information->initrd_size);
	
	
	// count sum from info header --
	sum = chksum(sum,(char*)information,info_size);
	
	// open target file --
	if((rfp = fopen(infn,"r"))==NULL) {
		error("can't open target file!");
		return(-1);
		}
	// count sum from target file --
	while((len=fread(tmp,1,STRBUFF_MAX,rfp))!=0) {
		totallen += len;
		sum = chksum(sum,(char*)tmp,len);
		}
	if(totallen != file_size) {
		error("Panic! Read size != information file!");
		fclose(rfp);
		return(-1);
		}
	// sum܂ŌvZ0ɂȂlɂ炵
	sum = (0xffffffff-sum)+1;
	printf("check sum = %08X %d\n",sum,sizeof(int));
	sum = l2bl(sum);
	information->chksum = sum;
	if(fseek(rfp,0L,SEEK_SET)) {
		error("Target file seeking fail!");
		fclose(rfp);
		return(-1);
		}
	
	if((wfp = fopen(outfn,"w"))==NULL) {
		error("can't open output file!");
		fclose(rfp);
		return(-1);
		}
	if(fwrite((char*)information,info_size,1,wfp)<1) {
		error("can't write output file!");
		fclose(wfp);
		fclose(rfp);
		return(-1);
		}
	while((len=fread(tmp,1,STRBUFF_MAX,rfp))!=0) {
		if(fwrite((char*)tmp,1,len,wfp)<1) {
			error("can't copy output file from target!");
			fclose(wfp);
			fclose(rfp);
			return(-1);
			}
		}
	fclose(wfp);
	fclose(rfp);
	return(0);
	}
// ----------------------------------------------------
int viewsum(char* infn) 
	{
	struct stat   filestat;
	FILE          *rfp;
	char          tmp[STRBUFF_MAX+1];
	long          sum      = 0;
	long          len;
	if((rfp = fopen(infn,"r"))==NULL) {
		error("can't open target file!");
		return(-1);
		}
	while((len=fread(tmp,1,STRBUFF_MAX,rfp))!=0) {
		sum = chksum(sum,(char*)tmp,len);
		}
	sum = -sum;
	printf("check sum = %08X\n",sum);
	fclose(rfp);
	return(0);
	}
// ----------------------------------------------------
static char *dummyfilename = "./kernelimage.bin";
static char *tmpfilename   = "./kernelimage.tmp";
// ----------------------------------------------------
int main(int argc, char *argv[])
	{
	char * configfile = NULL;
	char * kernelfile = NULL;
	char * initrdfile = NULL;
	char * outputfile = NULL;
	int    i,
	       infosize;
	time_t     t;
	struct tm* date;
	char * str;
	int    opt = 0;
	
	if(argc<2)  {print_usage();return(0);}
	if(argc==2) {viewsum((char*)argv[1]);return(0);}
	for(i=1;i<argc;++i) {
		str = (char*)argv[i];
		if(str[0] == '-') {
			switch(str[1]) {
				case 'o': 
				case 'O': 
					opt = 1;break;
				case 'r':
				case 'R':
					opt = 2;break;
				default:
					error("Invalid option!");printf("> %s\n",str);return(-1);
				}
			continue;
			}
		if(opt>0) {
			switch(opt) {
				case 1:outputfile  = str;break;
				case 2:initrdfile  = str;break;
				}
			opt = 0;
			continue;
			}
		if(configfile == NULL) {configfile = str;continue;}
		if(kernelfile == NULL) {kernelfile = str;continue;}
		}
	
	if(configfile == NULL || kernelfile == NULL) {print_usage();return(0);}
	if(outputfile == NULL) outputfile = dummyfilename;
	
	i = sizeof(struct firminfo);
	infosize = i/4;
	if((i%4)>0) infosize++;
	infosize *= 4;
	if((information = (struct firminfo*)malloc((size_t)infosize))==NULL) {
		error("Can't memory allocate!");
		return(-1);
		}
	memset(information,0,infosize);
	information_size = infosize;
	
	information->info_ver = FIRMINFO_VER;
	
	printf("Firm information struct size = %d %d\n",infosize,sizeof(short));
	information->kernel_offset = infosize;
	printf("read config file:%s\n",configfile);
	if(readconfig(configfile)) return(-1);
	
	time(&t);
	date = localtime(&t);
	printf("set date:%d/%d/%d %d:%d:%d\n",
		date->tm_year + 1900,
		date->tm_mon,
		date->tm_mday,
		date->tm_hour,
		date->tm_min,
		date->tm_sec
		);
	information->year = (char)date->tm_year;
	information->mon  = (char)date->tm_mon + 1;// month0`11炵c(T-T
	information->day  = (char)date->tm_mday;
	information->hour = (char)date->tm_hour;
	information->min  = (char)date->tm_min;
	information->sec  = (char)date->tm_sec;
	
	information->kernel_offset = information_size;
	information->kernel_size   = get_filesize(kernelfile);
	printf("read Kernel file:%s(%d bytes)\n",kernelfile,information->kernel_size);
	
	if(initrdfile != NULL) {
		FILE *wfp,
			 *rfp;
		char  tmp[STRBUFF_MAX+1];
		unsigned long  len;
		
		information->initrd_offset = information->kernel_offset + information->kernel_size;
		information->initrd_size = get_filesize(initrdfile);
		printf("     initrd file:%s(%d bytes)\n",initrdfile,information->initrd_size);
		
		if((wfp = fopen(tmpfilename,"w"))==NULL) {
			error("can't open tmp file!");
			return(-1);
			}
		
		// copy temp.
		if((rfp = fopen(kernelfile,"r"))==NULL) {
			error("can't open tmp file!");
			return(-1);
			}
		while((len=fread(tmp,1,STRBUFF_MAX,rfp))!=0) {
			if(fwrite((char*)tmp,1,len,wfp)<1) {
				error("can't copy output file from target!");
				fclose(wfp);
				fclose(rfp);
				return(-1);
				}
			}
		fclose(rfp);
		if((rfp = fopen(initrdfile,"r"))==NULL) {
			error("can't open tmp file!");
			return(-1);
			}
		while((len=fread(tmp,1,STRBUFF_MAX,rfp))!=0) {
			if(fwrite((char*)tmp,1,len,wfp)<1) {
				error("can't copy output file from target!");
				fclose(wfp);
				fclose(rfp);
				return(-1);
				}
			}
		fclose(rfp);
		fclose(wfp);
		if(setinfomation(tmpfilename,outputfile)) return(-1);
		remove(tmpfilename);
		}
	else {
		if(setinfomation(kernelfile,outputfile)) return(-1);
		}
	printf("Done..\n");
	free(information);
	return 0;
	}
