/*
 * ocp.c
 *
 *	The is drived from pci.c
 *
 * 	Current Maintainer
 *      Armin Kuster akuster@pacbell.net
 *      Jan, 2002
 *
 *
 *
 * This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General Public License as published by the
 *  Free Software Foundation;  either version 2 of the  License, or (at your
 *  option) any later version.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR   IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE    LIABLE FOR ANY   DIRECT,  INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *	Version 1.0 (02/01/26) - A. Kuster
 *	Initial version	 -
 *
 *	Version 1.1 (04/19/02) - A. kuster
 *	added the ability to give # of emacs and return count
 *	added the addr & irq routines that were in *_enet.c
 *	improved ocp_register
 *	added ocp_alloc_dev
 *	created ocp_dev_count & get_ocp_dev
 *
 * 	Version 1.2 (4/22/02) - Armin
 * 	 added UART,I2C USB & IDE reg support
 * 	 vaddr is now ioremapped
 *
 * 	 Version 1.3 (04/25/02) - Armin/Todd
 * 	   cleanups
 * 	   bug fixes - ifndef's
 *	Vesrion 1.4 (05/03/02) - Armin
 *	    fixed ghost reg when ocp_reg fails , Idea from David Mueller
 *	    added ocp_get_max used by ocp_dev_count
 *	    ocp_reg  and get_ocp_dev call ocp_dev_count
 *	    removed need for ifndefs
 *	    Added use of core_ocp[]
 *	Version 1.5 - Armin
 *	added api descriptions
 *	name change to <class>_<action>_<obj>
 *	added ocp_type_info for name, short desciption and size
 *
 *	Version 1.5 - Armin/David Gibson
 *	added most of David ocp patch to cleanup some of the 
 *	APIs
 *	
 */

#include <linux/list.h>
#include <linux/module.h>
#include <linux/config.h>
#include <linux/miscdevice.h>
#include <linux/slab.h>
#include <linux/types.h>
#include <asm/io.h>
#include <asm/ocp.h>
#include <asm/errno.h>

LIST_HEAD(ocp_list);

extern struct ocp_def core_ocp[];

struct type_info ocp_type_info[] = {
	{"pci", "PCI"},
	{"gpt", "General purpose timers"},
	{"uart", "NS16550 compatable serial device"},
	{"opb", "OBP"},
	{"I2C", "I2C bus"},
	{"gpio", "Gernal purpose I/O"},
	{"emac", "10/100 BaseT ethernet"},
	{"zmii", "ZMII bridge"},
	{"ide", "Independent Disk"},
	{"usb", "USB"},
	{"sci", "Smart card"},
	{"audio", "Sound interface"},
	{"ssp", "Sync serail port"},
	{"scp", "Serial Controller port"},
	{"scc", "Serial controller"},
	{"video", "General Video"},
	{"dma", "Dynamic memory allocation"},
	{"uic", "Universal Interrupt controller"},
	{"rtc", "Real Time Clock"},
	{"led", "Light Emitting diode"}

};

/**
 * ocp_get_max - get max for ocp type 
 * @type: OCP type such as PCI, GPT, UART, OPB, IIC, GPIO, EMAC, ZMII,
 *  
 * The routine returns max number of type that a given core supports
 */
int
ocp_get_max(int type)
{
	int max, i;
	for (i = 0, max = 0; core_ocp[i].type != OCP_NULL_TYPE; i++) {
		if (type == core_ocp[i].type)
			max++;
	}
	return (max);
}

/**
 * get_ocp_irq - get irq for ocp type and instance of it 
 * @type: OCP type such as PCI, GPT, UART, OPB, IIC, GPIO, EMAC, ZMII,
 * @dev_num: ocp device number whos paddr you want
 *  
 * The routine returns the valid irq, OCP_IRQ_NA or OCP_IRQ_MUL
 * of type and instance requested 
 */
int
ocp_get_irq(int type, int dev_num)
{
	int a, i;
	a = 0;

	for (i = 0; core_ocp[i].type != OCP_NULL_TYPE; i++) {
		if (type == core_ocp[i].type) {
			if (a == dev_num)
				return (core_ocp[i].irq);
			a++;

		}
	}

	return (-ENXIO);

}

/**
 * ocp_get_paddr - get phyical address for ocp type and instance of it 
 * @type: OCP type such as PCI, GPT, UART, OPB, IIC, GPIO, EMAC, ZMII
 * @dev_num: ocp device number whos paddr you want
 *  
 * The routine returns paddr of type requested and it's instance or
 * 0x0 on a failure
 */
unsigned long
ocp_get_paddr(int type, int dev_num)
{
	int a, i;
	a = 0;

	for (i = 0; core_ocp[i].type != OCP_NULL_TYPE; i++) {
		if (type == core_ocp[i].type) {
			if (a == dev_num)
				return ((unsigned long) core_ocp[i].paddr);
			a++;

		}
	}

	return 0;
}
/**
 * ocp_get_pm - get pm for ocp type and instance of it 
 * @type: OCP type such as PCI, GPT, UART, OPB, IIC, GPIO, EMAC, ZMII,
 * @dev_num: ocp device number whos power managment 
 * value you want
 *  
 * The routine returns the power management value or OCP_CPM_NA,
 * of type and instance requested 
 */
int
ocp_get_pm(int type, int dev_num)
{
	int a, i;
	a = 0;

	for (i = 0; core_ocp[i].type != OCP_NULL_TYPE; i++) {
		if (type == core_ocp[i].type) {
			if (a == dev_num)
				return (core_ocp[i].cpm);
			a++;

		}
	}

	return OCP_CPM_NA;
}

/* this sets the irq,paddr and anme to the device */
static int
ocp_set_dev(struct ocp_dev *ocp)
{
	int a, i;
	a = 0;

	strcpy(ocp->name, ocp_type_info[ocp->type].name);

	for (i = 0; core_ocp[i].type != OCP_NULL_TYPE; i++) {
		if (ocp->type == core_ocp[i].type) {
			if (a == ocp->num) {
				ocp->paddr = core_ocp[i].paddr;
				ocp->irq = core_ocp[i].irq;
				return 1;

			}
			a++;
		}
	}

	ocp->paddr = 0;
	ocp->irq = OCP_IRQ_NA;
	return 0;
}

/**
 * ocp_get_numtypes - This determines how many OCP devices of a givrn
 * type are registered
 * @type: OCP type such as PCI, GPT, UART, OPB, IIC, GPIO, EMAC, ZMII,
 *  
 * The routine returns number of types are registered
 */
static int
ocp_get_numtypes(int type)
{
	int count = 0;
	struct ocp_dev *ocp;
	struct list_head *ocp_l;

	list_for_each(ocp_l, &ocp_list) {
		ocp = list_entry(ocp_l, struct ocp_dev, ocp_list);
		if (type == ocp->type)
			count++;
	}

	return count;
}

/**
 * ocp_alloc_dev - alloc space for ocp driver
 * @sizeof_ocpdev: size of the device structure
 *  
 * The routine returns a valid 32 bit aligned
 * points the sizeof ocp_struct and the 
 * requested device structure
 */
struct ocp_dev *
ocp_alloc_dev(int sizeof_ocpdev)
{
	struct ocp_dev *drv;
	int alloc_size;

	/* ensure 32-byte alignment of the driver area */
	alloc_size = sizeof (*drv) + sizeof_ocpdev + 31;

	drv = (struct ocp_dev *) kmalloc(alloc_size, GFP_KERNEL);
	if (drv == NULL) {
		printk(KERN_ERR
		       "ocp_alloc_dev: Unable to allocate device memory.\n");
		return NULL;
	}

	memset(drv, 0, alloc_size);

	if (sizeof_ocpdev)
		drv->ocpdev = (void *) (((long) (drv + 1) + 31) & ~31);

	return (drv);
}

/**
 * ocp_free_dev - frees alloc space for ocp driver
 * @drv: the driver structure to register
 * 
 * Adds the driver structure to the list of registered drivers
 * Returns the number of ocp type devices which were claimed by the driver
 * during registration.  The driver remains registered even if the
 * return value is zero.
 */
/*
 * ocp_free_dev - frees alloc space for ocp driver
 * @dev: ocp device as allocated in ocp_alloc_dev
 */
void
ocp_free_dev(void *dev)
{
	kfree(dev);
}

/**
 * ocp_register - register a new ocp driver
 * @drv: the driver structure to register
 * 
 * Adds the driver structure to the list of registered drivers
 * Returns the number of ocp type devices which were claimed by the driver
 * during registration.  The driver remains registered even if the
 * return value is zero.
 */
int
ocp_register(struct ocp_dev *drv)
{
	int max;

	list_add(&drv->ocp_list, &ocp_list);
	max = ocp_get_max(drv->type);
	drv->num = ocp_get_numtypes(drv->type) - 1;

	if (drv->num >= max) {
		list_del(&drv->ocp_list);
		return -ENXIO;
	}

	if (!ocp_set_dev(drv)) {
		list_del(&drv->ocp_list);
		return -ENXIO;
	}
#ifdef CONFIG_OCP_PROC
	ocp_proc_attach_device(drv);
#endif
#ifdef OCP_DEBUG
	printk("Dev: %s Num:%d @ paddr:0x%x irq:%d\n", drv->name, drv->num,
	       drv->paddr, drv->irq);
#endif
	return (drv->num);
}

/**
 * ocp_unregister - unregister a ocp driver
 * @drv: the driver structure to unregister
 * 
 * Deletes the driver structure from the list of registered OCP drivers,
 * gives it a chance to clean up by calling its remove() function for
 * each device it was responsible for, and marks those devices as
 * driverless.
 */

void
ocp_unregister(struct ocp_dev *drv)
{
	list_del(&drv->ocp_list);
#ifdef CONFIG_OCP_PROC
	ocp_proc_detach_device(drv);
#endif
}

/**
 * ocp_get_dev - get ocp driver pointer for ocp type and instance of it 
 * @type: OCP type such as PCI, GPT, UART, OPB, IIC, GPIO, EMAC, ZMII
 * @dev_num: ocp device number whos paddr you want
 *  
 * The routine returns ocp device pointer 
 * in list based on type and instance of that device
 *
 */
struct ocp_dev *
ocp_get_dev(int type, int dev_num)
{
	struct ocp_dev *ocp;
	struct list_head *ocp_l;
	int count = 0;
	list_for_each(ocp_l, &ocp_list) {
		ocp = list_entry(ocp_l, struct ocp_dev, ocp_list);
		if (type == ocp->type) {
			if (dev_num == count)
				return ocp;
			count++;

		}
	}
	return NULL;
}

EXPORT_SYMBOL(ocp_get_irq);
EXPORT_SYMBOL(ocp_get_paddr);
EXPORT_SYMBOL(ocp_get_pm);
EXPORT_SYMBOL(ocp_get_max);
EXPORT_SYMBOL(ocp_get_dev);
EXPORT_SYMBOL(ocp_alloc_dev);
EXPORT_SYMBOL(ocp_free_dev);
EXPORT_SYMBOL(ocp_register);
EXPORT_SYMBOL(ocp_unregister);
