/*
 *  RTC Control Driver for RS5C372
 *
 *  Copyright (C) 2001-2004  BUFFALO INC.
 *
 *  This software may be used and distributed according to the terms of
 *  the GNU General Public License (GPL), incorporated herein by reference.
 *  Drivers based on or derived from this code fall under the GPL and must
 *  retain the authorship, copyright and license notice.  This file is not
 *  a complete program and may only be used when the entire operating
 *  system is licensed under the GPL.
 *
 */

#include <linux/errno.h>
#include <linux/init.h>
#include <linux/kernel.h>
#include <linux/time.h>
#include <linux/timex.h>
#include <linux/config.h>
#include <asm/io.h>
#include <asm/time.h>

#include "i2c_export.h"
#include "buffalo/5c372.h"
#include <buffalo/kernevnt.h>	/* 2005.5.10 BUFFALO */

rwlock_t mrtc_lock;

static __inline__ unsigned long set_HID0(unsigned long hid0)
{
	asm volatile("mtspr 1008, %0" : :"r" (hid0));
	return hid0;
}

static __inline__ unsigned long get_HID0(void)
{
	unsigned long hid0;
	asm volatile("mfspr %0, 1008" : "=r" (hid0));
	return hid0;
}

/***********************************************************************
 *  Func    : melco_rtc_init                                           *
 *  comment : Initialize RTC                                           *
 ***********************************************************************/
long __init melco_rtc_init(void)
{
	unsigned char buf[17];
	
	I2C_Initialize(0x31, 0, NULL);  // no interrupt

	buf[0] = 0xf0;
	buf[1] = 0x20;/* set 24h time */
	I2C_do_transaction(0, I2C_MASTER_XMIT, 0x32, (unsigned char)buf[0], 1, &buf[1], I2C_STOP, 5, 0);
		
	//	accuracy correction
	buf[0] = 0x70;		//register 7h
	// XSL | 6 5 4 3 2 1 0
	//  0  | 
#if defined(CONFIG_HTGL) || defined(CONFIG_HTGLSATA)
	buf[1] = 0x77;	// -9
#else
	buf[1] = 0x15;
#endif
	I2C_do_transaction(0, I2C_MASTER_XMIT, 0x32, (unsigned char)buf[0], 1, &buf[1], I2C_STOP, 5, 0);

	return 0;
}

/***********************************************************************
 *  Func    : melco_get_rtc                                            *
 *  comment : output current time/date                                 *
 ***********************************************************************/
unsigned long melco_get_rtc(void)
{
	unsigned int year, mon, day, hour, min, sec;
	unsigned char buf[8];
	unsigned long total_sec;
	int limit = 0;
	I2C_Status status;

	buf[0] = 0x00;
	do {
		read_lock_irq(&mrtc_lock);
		status = I2C_do_transaction(0, I2C_MASTER_RCV, 0x32, (unsigned char)buf[0], 7, &buf[1], I2C_STOP, 5, 0);
		if(status == I2C_ERROR) I2C_Initialize(0x31, 0, NULL);
		read_unlock_irq(&mrtc_lock);
		limit++;
	} while (status != I2C_SUCCESS && limit < 10);

	if(status == I2C_ERROR){
		kernevnt_I2cErr();
		total_sec=0;
	}else{
		sec = (volatile unsigned int)BCD_TO_BIN(buf[1]);
		min = (volatile unsigned int)BCD_TO_BIN(buf[2]);
		hour = (volatile unsigned int)BCD_TO_BIN(buf[3]);
		day = (volatile unsigned int)BCD_TO_BIN(buf[5]);
		mon = (volatile unsigned int)BCD_TO_BIN(buf[6]);
		year = (volatile unsigned int)BCD_TO_BIN(buf[7]);

		year = year + 1900;
		if (year < 1970) {
			year += 100;
		}

		total_sec = mktime(year, mon, day, hour, min, sec);
	}
	return total_sec;
}

/***********************************************************************
 *  Func    : melco_get_rtc2                                           *
 *  comment : output current time/date                                 *
 ***********************************************************************/
struct rtc_time melco_get_rtc2(void)
{
	struct rtc_time tm;
	unsigned char buf[8];
	int limit = 0;
	I2C_Status status;


	buf[0] = 0x00;
	do {
		read_lock_irq(&mrtc_lock);
		status = I2C_do_transaction(0, I2C_MASTER_RCV, 0x32, (unsigned char)buf[0], 7, &buf[1], I2C_STOP, 5, 0);
		if(status == I2C_ERROR) I2C_Initialize(0x31, 0, NULL);
		read_unlock_irq(&mrtc_lock);
		limit++;
	} while (status != I2C_SUCCESS && limit < 10);

	if(status == I2C_ERROR){
		kernevnt_I2cErr();
		memset(&tm, 0, sizeof(tm));
	}else{
		tm.tm_sec = BCD_TO_BIN(buf[1]);
		tm.tm_min = BCD_TO_BIN(buf[2]);
		tm.tm_hour = BCD_TO_BIN(buf[3]);
		tm.tm_mday = BCD_TO_BIN(buf[5]);
		tm.tm_mon = BCD_TO_BIN(buf[6]);
		tm.tm_year = BCD_TO_BIN(buf[7]);

		tm.tm_year = tm.tm_year + 1900;
		if (tm.tm_year < 1970) {
			tm.tm_year += 100;
		}
	}
	return tm;
}

/***********************************************************************
 *  Func    : melco_get_sec                                            *
 *  comment : output current second                                    *
 ***********************************************************************/
unsigned char melco_get_sec(void)
{
	unsigned char val;
	unsigned char buf[2];
	int limit = 0;
	I2C_Status status;


	buf[0] = 0x00;
	
	do {
		read_lock_irq(&mrtc_lock);
		status = I2C_do_transaction(0, I2C_MASTER_RCV, 0x32, (unsigned char)buf[0], 1, &buf[1], I2C_STOP, 5, 0);
		if(status == I2C_ERROR) I2C_Initialize(0x31, 0, NULL);
		read_unlock_irq(&mrtc_lock);

		limit++;
	} while (status != I2C_SUCCESS && limit < 10);
	
	if(status == I2C_ERROR){
		kernevnt_I2cErr();
		val = 0;
	}else{
		val = buf[1] & 0x7f;
		BCD_TO_BIN(val);
	}
	return val;
}

/***********************************************************************
 *  Func    : melco_set_rtc                                            *
 *  comment : output current time/date                                 *
 ***********************************************************************/
int melco_set_rtc(unsigned long nowtime)
{
	struct rtc_time	tm;
	unsigned char buf[8];
	int limit = 0;
	I2C_Status status;

	to_tm(nowtime, &tm);

	tm.tm_year = (tm.tm_year - 1900) % 100;

	BIN_TO_BCD(tm.tm_sec);
	BIN_TO_BCD(tm.tm_min);
	BIN_TO_BCD(tm.tm_hour);
	BIN_TO_BCD(tm.tm_mon);
	BIN_TO_BCD(tm.tm_mday);
	BIN_TO_BCD(tm.tm_year);
	
	buf[0] = 0x0;
	buf[1] = tm.tm_sec;
	buf[2] = tm.tm_min;
	buf[3] = tm.tm_hour;
	buf[4] = 0x00;
	buf[5] = tm.tm_mday;
	buf[6] = tm.tm_mon;
	buf[7] = tm.tm_year;

	do {
		write_lock_irq(&mrtc_lock);
		status = I2C_do_transaction(0, I2C_MASTER_XMIT, 0x32, (unsigned char)buf[0], 7, &buf[1], I2C_STOP, 5, 0);
		if(status == I2C_ERROR) I2C_Initialize(0x31, 0, NULL);
		write_unlock_irq(&mrtc_lock);
		limit++;
	} while (status != I2C_SUCCESS && limit < 10);

	if(status == I2C_ERROR){
		kernevnt_I2cErr();
		return -1;
	}
	return MELCO_RTC_SUCCESS;
}

/***********************************************************************
 *  Func    : melco_set_rtc2                                           *
 *  comment : output current time/date                                 *
 ***********************************************************************/
int melco_set_rtc2(struct rtc_time	rtm)
{
	struct rtc_time	tm = rtm;
	unsigned char buf[8];
	int limit = 0;
	I2C_Status status;

	tm.tm_year = (tm.tm_year - 1900) % 100;

	BIN_TO_BCD(tm.tm_sec);
	BIN_TO_BCD(tm.tm_min);
	BIN_TO_BCD(tm.tm_hour);
	BIN_TO_BCD(tm.tm_mon);
	BIN_TO_BCD(tm.tm_mday);
	BIN_TO_BCD(tm.tm_year);
	
	buf[0] = 0x0;
	buf[1] = tm.tm_sec;
	buf[2] = tm.tm_min;
	buf[3] = tm.tm_hour;
	buf[4] = 0x00;
	buf[5] = tm.tm_mday;
	buf[6] = tm.tm_mon;
	buf[7] = tm.tm_year;

	do {
		write_lock_irq(&mrtc_lock);
		status = I2C_do_transaction(0, I2C_MASTER_XMIT, 0x32, (unsigned char)buf[0], 7, &buf[1], I2C_STOP, 5, 0);
		if(status == I2C_ERROR) I2C_Initialize(0x31, 0, NULL);
		write_unlock_irq(&mrtc_lock);
		limit++;
	} while (status != I2C_SUCCESS && limit < 100);

	if(status == I2C_ERROR){
		kernevnt_I2cErr();
		return -1;
	}
	return MELCO_RTC_SUCCESS;
}

/***********************************************************************
 *  Func    : melco_caribrate_decr                                     *
 *  comment : output current time/date                                 *
 ***********************************************************************/
// 2005.5.17 BUFFALO : from sandpoint_setup.c
extern int IsUseRevClock;

void melco_caribrate_decr(void)
{
	ulong	freq;
	ulong rev_freq;
	
#if defined(CONFIG_HGLAN) || defined(CONFIG_HTGL)
	rev_freq=32522240;	/* {32.768MHz*(100% - 0.75%)} * 4 / 4 */
	freq=32768000;
#elif defined(CONFIG_HTGLSATA)
	IsUseRevClock = 0;		//gUȂ
	freq=32768000;
#elif defined(CONFIG_HLAN)
	rev_freq=24391680;	/* {24.576MHz*(100% - 0.75%)} * 3 / 4  */
	freq=24576000;
#else
	#error "not defined"
#endif
	
	if (IsUseRevClock){
		freq = rev_freq;
	}
	
	tb_ticks_per_jiffy = freq / HZ;
	tb_to_us = mulhwu_scale_factor(freq, 1000000);
	printk("decrementer frequency = %ld.%6ld MHz \n", freq / 1000000, freq % 1000000);
	return;
}
