/*
 *  LinkStation/TeraStation Micon port Driver
 *
 *  Copyright (C)  BUFFALO INC.
 *
 *  This software may be used and distributed according to the terms of
 *  the GNU General Public License (GPL), incorporated herein by reference.
 *  Drivers based on or derived from this code fall under the GPL and must
 *  retain the authorship, copyright and license notice.  This file is not
 *  a complete program and may only be used when the entire operating
 *  system is licensed under the GPL.
 *
 */
#include <linux/config.h>

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/delay.h>
#include <linux/errno.h>
#include <linux/init.h>
#include <linux/fs.h>
#include <linux/proc_fs.h>
#include <linux/miscdevice.h>

#include <asm/string.h>
#include <asm/io.h>
#include <asm/system.h>
#include <asm/uaccess.h>
#include <asm/serial.h>
#include <asm/irq.h>
#include <buffalo/miconcntl.h>	/* 2005.5.10 BUFFALO */
#include <buffalo/kernevnt.h>	/* 2005.5.10 BUFFALO */

#define bzero(p,sz) memset(p,0,sz)

//#define DEBUG
//#define MICONMSG


#ifdef DEBUG
 #define TRACE(x) x
#else
 #define TRACE(x)
#endif

	#define TX_EMPTY 0x20
	#define DATA_RDY 0x01
	#define RCV_FIFO_RST 0x02
	#define FIFO_ENABLE  0x01

#ifdef MICONMSG
//--------------------------------------------------------------
static void dumpdata(const char *title,const unsigned char *data, int len)
{
	int i;
	
	printk("--- %s %d ---\n",title,len);
	for (i=0; i<len; i++){
		printk("%02x ",data[i]);
	}
	printk("\n");
}

#endif
//--------------------------------------------------------------
//ppc only
static int MiconPortWrite(const unsigned char *buf, int count)
{
	int i;
	unsigned char ier;
	const unsigned port_ier=SANDPOINT_SERIAL_0+1;
	const unsigned port_lsr=SANDPOINT_SERIAL_0+5;
	const unsigned port_thr=SANDPOINT_SERIAL_0;
	
#ifdef MICONMSG
	printk(">%s:count=%d\n",__FUNCTION__,count);
	dumpdata(__FUNCTION__,buf,count);
#endif
	
	ier = readb(port_ier);
	//printk("ier=%x\n",ier);
	writeb(0, port_ier);
	for (i=0; i<count; i++){
		unsigned int status, tmout = 1000000;
		do{
			status = readb(port_lsr);
			if (--tmout == 0){
				printk("TX_EMPTY : Time out.\n");
				break;
			}
		} while ((status & TX_EMPTY) != TX_EMPTY);
		TRACE(printk("status=%x tmout=%x\n",status,tmout));
		if (tmout == 0){
			break;
		}
		writeb(buf[i],port_thr);
	}
	writeb(ier,port_ier);
	
	return 0;
}

//--------------------------------------------------------------
//ppc only
static int MiconPortRead(unsigned char *buf, int count)
{
	int i;
	unsigned char ier;
	const unsigned port_ier=SANDPOINT_SERIAL_0+1;
	const unsigned port_lsr=SANDPOINT_SERIAL_0+5;
	const unsigned port_thr=SANDPOINT_SERIAL_0;
	
#ifdef MICONMSG
	printk(">%s\n",__FUNCTION__);
#endif
	ier = readb(port_ier);
	//printk("ier=%x\n",ier);
	writeb(0, port_ier);
	for (i=0; i<count; i++){
		unsigned int status, tmout = 1000000;
		do{
			status = readb(port_lsr);
			if (--tmout == 0){
#ifdef MICONMSG
				printk("DATA_RDY : Time out.\n");
#endif
				break;
			}
		} while ((status & DATA_RDY) != DATA_RDY);
		TRACE(printk("status=%x tmout=%x\n",status,tmout));
		if (tmout == 0){
			break;
		}
		buf[i] = readb(port_thr);
	}
	writeb(ier,port_ier);
	
#ifdef MICONMSG
	printk(">%s:count=%d\n",__FUNCTION__,i);
	dumpdata(__FUNCTION__,buf,i);
#endif
	return i;		// return read bytes
}

//--------------------------------------------------------------
//ppc only
static void MiconPortFifoClr(void)
{
	u8* port_fcr = (u8 *)SANDPOINT_SERIAL_0+2;
	//Recive FIFO Clear
	out_8(port_fcr, RCV_FIFO_RST|FIFO_ENABLE);
}

//--------------------------------------------------------------
static void miconCntl_SendPreamble(void)
{
	unsigned char buff[40];
#ifdef MICONMSG
	printk(">%s\n",__FUNCTION__);
#endif
	memset(&buff,0xff,sizeof(buff));
	MiconPortWrite(buff,sizeof(buff));
	mdelay(100);
	// make dummy read.
	MiconPortRead(buff,sizeof(buff));
	mdelay(100);
	MiconPortRead(buff,sizeof(buff));
	mdelay(100);
	MiconPortRead(buff,sizeof(buff));
}

//--------------------------------------------------------------
static int miconCntl_SendCmd(const unsigned char *data, int count)
{
	int i;
	unsigned char parity;
	unsigned char recv_buf[35];
	int retry=2;
	
	TRACE(printk(">%s\n",__FUNCTION__));
	
	//Generate checksum
	parity = 0;
	for(i=0;i<count;i++){
		parity +=  data[i];
	}
	parity = 0 - parity ;
	
	mdelay(10);		//̃R}hƂ̊ԊuƂ
	MiconPortFifoClr();
	
	do {
#ifdef MICONMSG
		printk(">%s retry=%d\n",__FUNCTION__,retry);
#endif
		// send data
		MiconPortWrite(data,count);
		// send parity
		MiconPortWrite(&parity,1);
		if (MiconPortRead(recv_buf,sizeof(recv_buf))<=3){
			printk(">%s: recv fail.\n",__FUNCTION__);
			// send preamble
			miconCntl_SendPreamble();
		}else{
			//Generate Recive data
			unsigned char correct_ACK[4];
			correct_ACK[0] = 0x01;
			correct_ACK[1] = data[1];
			correct_ACK[2] = 0x00;
			correct_ACK[3] = 0 - (0x01 + data[1] + 0x00);
			
			//Parity Check
			if(0 != (0xFF & (recv_buf[0] + recv_buf[1] + recv_buf[2] + recv_buf[3]))){
				printk("Parity Error : Recive data[%02x, %02x, %02x, %02x]\n", 
						recv_buf[0], recv_buf[1], recv_buf[2], recv_buf[3]);
			}else{
				//Check Recive Data
				if( (correct_ACK[0] == recv_buf[0]) && (correct_ACK[1] == recv_buf[1])
						&& (correct_ACK[2] == recv_buf[2]) && (correct_ACK[3] == recv_buf[3])){
					//Recive ACK
					return 0;
				}
			}
			//Recive NAK or illegal Data
			printk("Error : NAK or Illegal Data Recived\n");
		}
	} while(retry--);
	return -1;
}


//--------------------------------------------------------------
static void miconCntl_ShutdownWait(void)
{
	const unsigned char WDkill_msg[] = {0x01,0x35,0x00};
	const unsigned char SdWait[] = {0x00,0x0c};
	const unsigned char BootEnd[] = {0x00,0x03};
	
	printk(">%s\n",__FUNCTION__);
	
	while(miconCntl_SendCmd(WDkill_msg,sizeof(WDkill_msg))) ;
	while(miconCntl_SendCmd(BootEnd,sizeof(BootEnd))) ;
	while(miconCntl_SendCmd(SdWait,sizeof(SdWait))) ;
}

//--------------------------------------------------------------
void miconCntl_Reboot(void)
{
	const unsigned char reboot_msg[] = {0x00,0x0E};
	printk(">%s\n",__FUNCTION__);
	
	miconCntl_ShutdownWait();
	
#ifdef MICONMSG
	printk("Send Reboot\n");
#endif
	//Send Reboot CMD
	// halt
	while (1){
		miconCntl_SendCmd(reboot_msg,sizeof(reboot_msg));
		mdelay(1000);
	}
}

//--------------------------------------------------------------
void miconCntl_PowerOff(void)
{
	const unsigned char poff_msg[] = {0x00,0x06};
	printk(">%s\n",__FUNCTION__);

	miconCntl_ShutdownWait();

#ifdef MICONMSG
	printk("Send Poweroff\n");
#endif
	//Send Poweroff CMD
	// halt
	while (1){
		miconCntl_SendCmd(poff_msg,sizeof(poff_msg));
		mdelay(1000);
	}
}

//--------------------------------------------------------------
static void micon_interrupts(int irq, void *dev_id, struct pt_regs *regs)
{
	disable_irq(irq);
	kernevnt_MiconInt();
	enable_irq(irq);
}

//--------------------------------------------------------------
/*
 * Initialize driver.
 */
int __init BuffaloMiconV2_init (void)
{
	request_irq(MICON_IRQ, micon_interrupts, 0, "MiCon", NULL);
	TRACE(printk(">%s\n",__FUNCTION__));
	printk("MICON V2 (C) BUFFALO INC. V.1.00 installed.\n"); 
	
	return 0;
}

//--------------------------------------------------------------
void BuffaloMiconV2_exit(void)
{
	free_irq(MICON_IRQ, NULL);
	TRACE(printk(">%s\n",__FUNCTION__));
	printk("MICON V2 removed.");
}

module_init(BuffaloMiconV2_init);
module_exit(BuffaloMiconV2_exit);
MODULE_LICENSE("GPL");

