/*
 * arch/ppc/platforms/prpmc800_pci.c
 * 
 * PCI support for Motorola PrPMC800
 *
 * Author: Dale Farnsworth <dale.farnsworth@mvista.com>
 *
 * Copyright 2001 MontaVista Software Inc.
 *
 * This program is free software; you can redistribute  it and/or modify it
 * under  the terms of  the GNU General Public License as published by the
 * Free Software Foundation;  either version 2 of the  License, or (at your
 * option) any later version.
 */

#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/pci.h>
#include <linux/slab.h>
#include <linux/harrier_defs.h>

#include <asm/byteorder.h>
#include <asm/io.h>
#include <asm/irq.h>
#include <asm/uaccess.h>
#include <asm/machdep.h>
#include <asm/pci-bridge.h>
#include <platforms/prpmc800.h>
#include <asm/harrier.h>

#define HARRIER_VENI_REG	(PRPMC800_HARRIER_XCSR_BASE + HARRIER_VENI_OFF)
#define HARRIER_MISC_CSR	(PRPMC800_HARRIER_XCSR_BASE + \
				 HARRIER_MISC_CSR_OFF)

static int monarch = 0;
static int found_self = 0;
static int self = 0;

#define MONARCH	(monarch != 0)
#define NON_MONARCH (monarch == 0)

/*
 * Motorola PrPMC750/PrPMC800 in PrPMCBASE or PrPMC-Carrier
 * Combined irq tables.  Only Base has IDSEL 14, only Carrier has 21 and 22.
 */
static inline int
prpmc_map_irq(struct pci_dev *dev, unsigned char idsel, unsigned char pin)
{
	static char pci_irq_table[][4] =
	/*
	 *      PCI IDSEL/INTPIN->INTLINE
	 *      A       B       C       D
	 */ 
	{
		{12,	0,	0,	0},  /* IDSEL 14 - Ethernet, base */
		{0,	0,	0,	0},  /* IDSEL 15 - unused */
		{10,	11,	12,	9},  /* IDSEL 16 - PMC A1, PMC1 */
		{10,	11,	12,	9},  /* IDSEL 17 - PrPMC-A-B, PMC2-B */
		{11,	12,	9,	10}, /* IDSEL 18 - PMC A1-B, PMC1-B */
		{0,	0,	0,	0},  /* IDSEL 19 - unused */
		{9,	10,	11,	12}, /* IDSEL 20 - P2P Bridge */
		{11,	12,	9,	10}, /* IDSEL 21 - PMC A2, carrier */
		{12,	9,	10,	11}, /* IDSEL 22 - PMC A2-B, carrier */
	};
	const long min_idsel = 14, max_idsel = 22, irqs_per_slot = 4;
	return PCI_IRQ_TABLE_LOOKUP;
};

static int
prpmc_read_config_dword(struct pci_controller *hose, u8 bus, u8 devfn, 
			int offset, u32 *val) {

	/* paranoia */
	if ((hose == NULL) ||
	    (hose->cfg_addr == NULL) ||
	    (hose->cfg_data == NULL))
		return PCIBIOS_DEVICE_NOT_FOUND;

	out_be32(hose->cfg_addr,
		((offset & 0xfc) << 24) | (devfn << 16)
		| ((bus - hose->bus_offset) << 8) | 0x80);
	*val = in_le32((u32*)(hose->cfg_data + (offset & 3)));

	return PCIBIOS_SUCCESSFUL;
}

#define HARRIER_PCI_VEND_DEV_ID	(PCI_VENDOR_ID_MOTOROLA | \
				 (PCI_DEVICE_ID_MOTOROLA_HARRIER << 16))
static int
prpmc_self(u8 bus, u8 devfn)
{
	/*
	 * Harriers always view themselves as being on bus 0. If we're not
	 * looking at bus 0, we're not going to find ourselves.
	 */
	if (bus != 0)
		return PCIBIOS_DEVICE_NOT_FOUND;
	else {
		int result;
		int val;
		struct pci_controller	*hose;

		hose = pci_bus_to_hose(bus);

		/* See if target device is a Harrier */
		result = prpmc_read_config_dword(hose, bus, devfn,
						 PCI_VENDOR_ID, &val);
		if ((result != PCIBIOS_SUCCESSFUL) ||
		    (val != HARRIER_PCI_VEND_DEV_ID))
			return PCIBIOS_DEVICE_NOT_FOUND;

		/*
		 * LBA bit is set if target Harrier == initiating Harrier
		 * (i.e. if we are reading our own PCI header).
		 */
		result = prpmc_read_config_dword(hose, bus, devfn,
						 HARRIER_LBA_OFF, &val);
		if ((result != PCIBIOS_SUCCESSFUL) ||
		    ((val & HARRIER_LBA_MSK) != HARRIER_LBA_MSK))
			return PCIBIOS_DEVICE_NOT_FOUND;

		/* It's us, save our location for later */
		self = devfn;
		found_self = 1;
		return PCIBIOS_SUCCESSFUL;
	}
}

static int
prpmc_exclude_device(u8 bus, u8 devfn)
{
	/*
	 * Monarch is allowed to access all PCI devices. Non-monarch is
	 * only allowed to access its own Harrier.
	 */

	if (MONARCH)
		return PCIBIOS_SUCCESSFUL;
	if (found_self)
		if ((bus == 0) && (devfn == self))
			return PCIBIOS_SUCCESSFUL;
		else
			return PCIBIOS_DEVICE_NOT_FOUND;
	else
		return prpmc_self(bus, devfn);
}
	

void __init
prpmc800_find_bridges(void)
{
	struct pci_controller* hose;
	int host_bridge;

	hose = pcibios_alloc_controller();
	if (!hose)
		return;

	hose->first_busno = 0;
	hose->last_busno = 0xff;

	ppc_md.pci_exclude_device = prpmc_exclude_device;
	ppc_md.pcibios_fixup = NULL;
	ppc_md.pcibios_fixup_bus = NULL;
	ppc_md.pci_swizzle = common_swizzle;
	ppc_md.pci_map_irq = prpmc_map_irq;

	setup_indirect_pci(hose,
			PRPMC800_PCI_CONFIG_ADDR,
			PRPMC800_PCI_CONFIG_DATA);

	/* Get host bridge vendor/dev id */

	host_bridge = in_be32((uint *)(HARRIER_VENI_REG)); 

	if (host_bridge != HARRIER_VEND_DEV_ID) {
		printk("Host bridge 0x%x not supported\n", host_bridge);
		return;
	}

	monarch = in_be32((uint *)HARRIER_MISC_CSR) & HARRIER_SYSCON;

	printk("Running as %s.\n", MONARCH ? "Monarch" : "Non-Monarch");

	hose->io_space.start = PRPMC800_PCI_IO_START;
	hose->io_space.end = PRPMC800_PCI_IO_END;
	hose->io_base_virt = (void *)PRPMC800_ISA_IO_BASE;
	hose->pci_mem_offset = PRPMC800_PCI_PHY_MEM_OFFSET;

	pci_init_resource(&hose->io_resource,
			PRPMC800_PCI_IO_START,
			PRPMC800_PCI_IO_END,
			IORESOURCE_IO,
			"PCI host bridge");

	if (MONARCH) {

		hose->mem_space.start = PRPMC800_PCI_MEM_START;
		hose->mem_space.end = PRPMC800_PCI_MEM_END;

		pci_init_resource(&hose->mem_resources[0],
				PRPMC800_PCI_MEM_START,
				PRPMC800_PCI_MEM_END,
				IORESOURCE_MEM,
				"PCI host bridge");

		if (harrier_init(hose,
				 PRPMC800_HARRIER_XCSR_BASE,
				 PRPMC800_PROC_PCI_MEM_START,
				 PRPMC800_PROC_PCI_MEM_END,
				 PRPMC800_PROC_PCI_IO_START,
				 PRPMC800_PROC_PCI_IO_END,
				 PRPMC800_HARRIER_MPIC_BASE) != 0)
			printk("Could not initialize HARRIER bridge\n");

		harrier_release_eready(PRPMC800_HARRIER_XCSR_BASE);
		harrier_wait_eready(PRPMC800_HARRIER_XCSR_BASE);
		hose->last_busno = pciauto_bus_scan(hose, hose->first_busno);

	} else {

		pci_init_resource(&hose->mem_resources[0],
				PRPMC800_NM_PCI_MEM_START,
				PRPMC800_NM_PCI_MEM_END,
				IORESOURCE_MEM,
				"PCI host bridge");

		hose->mem_space.start = PRPMC800_NM_PCI_MEM_START;
		hose->mem_space.end = PRPMC800_NM_PCI_MEM_END;

		if (harrier_init(hose,
				 PRPMC800_HARRIER_XCSR_BASE,
				 PRPMC800_NM_PROC_PCI_MEM_START,
				 PRPMC800_NM_PROC_PCI_MEM_END,
				 PRPMC800_PROC_PCI_IO_START,
				 PRPMC800_PROC_PCI_IO_END,
				 PRPMC800_HARRIER_MPIC_BASE) != 0)
			printk("Could not initialize HARRIER bridge\n");

		harrier_setup_nonmonarch(PRPMC800_HARRIER_XCSR_BASE,
					 HARRIER_ITSZ_1MB);
		harrier_release_eready(PRPMC800_HARRIER_XCSR_BASE);
	}
}
