/*
 *  dbmx1_wdt.c -- Dragonball MX1 Watch Dog Timer driver
 *
 */
/*
 *  Copyright 2002 Sony Corporation.
 *  Copyright (C) 2003 MontaVista Software Inc. <source@mvista.com>
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  version 2 of the  License.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE    LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
#include <linux/config.h>
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/types.h>
#include <linux/init.h>
#include <linux/ioport.h>
#include <linux/smp_lock.h>
#include <linux/interrupt.h>
#include <linux/miscdevice.h>
#include <linux/watchdog.h>

#include <asm/io.h>
#include <asm/uaccess.h>
#include <asm/system.h>

#include <linux/pm.h>

#include "mx1-wdt.h"

#undef WDOG_DEBUG
// #define WDOG_DEBUG 4
#ifdef WDOG_DEBUG
static int debug = WDOG_DEBUG;
MODULE_PARM(debug, "i");
#define wdt_printk( level, args... ) if( (level)<=debug ) printk( args )
#else
#define wdt_printk( level, args... )
#endif

#define ENTER() wdt_printk( 3, ">>> %s entered\n", __FUNCTION__ )
#define LEAVE() wdt_printk( 3, "<<< %s left out\n", __FUNCTION__ )
#define RETURN( something ) do { LEAVE(); return (something); } while( 0 )

static int wdt_is_open = 0;

/* -----------------------
 *  FUNCTION ROUTINES
 * -----------------------
 */
static unsigned long
read_data(unsigned long addr)
{
	unsigned long rd_data;

	ENTER();
	wdt_printk(4, "reading register %04X...", WDOG_BASE + addr);
	rd_data = *(__u32 *) (IO_ADDRESS(WDOG_BASE + addr));
	wdt_printk(4, "value =  %08X\n", rd_data);
	RETURN(rd_data);
}

static void
write_data(unsigned int wt_data, unsigned long addr)
{
	ENTER();
	wdt_printk(4, "writing %04X to register %08X\n",
		   wt_data, WDOG_BASE + addr);
	*(__u32 *) (IO_ADDRESS(WDOG_BASE + addr)) = wt_data;
	LEAVE();
}

static void
wdt_ping(void)
{
	ENTER();
	write_data(WDS_DATA1, WDOG_WSR);	/* reload */
	write_data(WDS_DATA2, WDOG_WSR);
	LEAVE();
}

static unsigned int
wdt_getsts(void)
{
	unsigned int sts;

	ENTER();
	sts = read_data(WDOG_WSTR);	/* Watchdog Status Reg Read */
	sts &= (WSTR_TINT | WSTR_TOUT);	/* WSTR TINT,TOUT bit get */
	RETURN(sts);
}

static void
wdt_setwdt(void)
{
	unsigned int out_data = 0;
	write_data(WDE_OFF, WDOG_WCR);	/* Watchdog Disable */

	/* Watchdog Parameter Set & Enable      */
	/*   WatchdogParameter                  */
	/*     WatchTime    =60sec              */
	/*     Interrupt    =WDT_RST            */
	/*     TestMode     =2Hz                */
	/*     SoftwareRest =Disable            */
	/*     EnableControl=more than one time */
	/*     Enable       =Enable             */
	out_data = WDT_TIMEOUT | WIE_OFF | TMD_OFF | SWR_OFF | WDEC_ON | WDE_ON;

	write_data(out_data, WDOG_WCR);
}

static int
wdt_open(struct inode *inode, struct file *file)
{
	ENTER();
	if (wdt_is_open) {
		RETURN(-EBUSY);
	}

	wdt_is_open = 1;
	wdt_setwdt();
	RETURN(0);
}

static int
wdt_close(struct inode *inode, struct file *file)
{
	ENTER();

	lock_kernel();
#ifndef CONFIG_WATCHDOG_NOWAYOUT
	wdt_printk(1,
		   "%s: disabling the watchdog. You may keep the device open\n",
		   __FUNCTION__);
	write_data(WDE_OFF, WDOG_WCR);	/* Watchdog Disable */
#endif
	wdt_is_open = 0;
	unlock_kernel();
	RETURN(0);
}

static ssize_t
wdt_read(struct file *file, char *buf, size_t count, loff_t * ptr)
{
	unsigned int status;

	ENTER();
	/*  Can't seek (pread) on this device  */
	if (ptr != &file->f_pos) {
		RETURN(-ESPIPE);
	}

	status = wdt_getsts();
	copy_to_user(buf, &status, sizeof (unsigned int));
	RETURN(sizeof (unsigned int));
}

static ssize_t
wdt_write(struct file *file, const char *buf, size_t count, loff_t * ppos)
{
	ENTER();
	/*  Can't seek (pwrite) on this device  */
	if (ppos != &file->f_pos) {
		RETURN(-ESPIPE);
	}

	if (count > 0) {
		wdt_ping();
		RETURN(1);
	}

	RETURN(0);
}

static int
wdt_ioctl(struct inode *inode, struct file *file,
	  unsigned int cmd, unsigned long arg)
{
	return -ENOTTY;
}

static struct file_operations wdt_fops = {
	owner:THIS_MODULE,
	open:wdt_open,		/* open  */
	release:wdt_close,	/* close */
	read:wdt_read,		/* read  */
	write:wdt_write,	/* write */
	ioctl:wdt_ioctl		/* ioctl */
};

static struct miscdevice wdt_miscdev = {
	WATCHDOG_MINOR,
	WDT_DEV_NAME,
	&wdt_fops
};

static int
handle_pm_event(struct pm_dev *dev, pm_request_t rq, void *data)
{
	switch (rq) {
	case PM_SUSPEND:
		lock_kernel();
#ifndef CONFIG_WATCHDOG_NOWAYOUT
		wdt_printk(1,
			   "%s: disabling the watchdog. You may keep the device open\n",
			   __FUNCTION__);
		write_data(WDE_OFF, WDOG_WCR);	/* Watchdog Disable */
#endif
		unlock_kernel();
		break;
	case PM_RESUME:
		wdt_printk(2, ("%s: resume\n", __FUNCTION__));
		if (wdt_is_open)
			wdt_setwdt();
		break;
	default:
		break;
	}
	return 0;
}

static int __init
wdt_init(void)
{
	unsigned int in_data = 0;
	int ret;

	ENTER();
	in_data = read_data(WDOG_WSTR);	/* Watchdog interrupt reset */

	ret = misc_register(&wdt_miscdev);

	if (ret < 0) {
		return ret;
	}
	pm_register(PM_SYS_DEV, PM_SYS_UNKNOWN, handle_pm_event);

	RETURN(0);
}

static void __exit
wdt_exit(void)
{
	unsigned int in_data;

	ENTER();
	in_data = read_data(WDOG_WSTR);
	pm_unregister_all(handle_pm_event);
	misc_deregister(&wdt_miscdev);
	LEAVE();
}

module_init(wdt_init);
module_exit(wdt_exit);

MODULE_LICENSE("GPL");
MODULE_DESCRIPTION("MX1 Watchdog timer");
