/*******************************************************************************************************

	mx1ads-keyp.c
	MX1ADS on-board keypad driver

	Uses PCF8575 installed on MX1ADS.

	Author: MontaVista Software, Inc. <source@mvista.com>
	2003 (c) MontaVista Software, Inc. This file is licensed under
	the terms of the GNU General Public License version 2. This program
	is licensed "as is" without any warranty of any kind, whether express
	or implied.
	
	Modifications:
	Nov 2003 - ver 1.1, MontaVista Software, Inc: removed an unbalanced enable-irq
	added i2c search for dev-fs

*******************************************************************************************************/

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/config.h>
#include <linux/init.h>
#include <linux/sched.h>
#include <linux/delay.h>
#include <linux/i2c.h>
#include <linux/input.h>

#include <asm/uaccess.h>
#include <asm/arch/platform.h>
#include <asm/irq.h>
#include <asm/arch/mx1ads-gpio.h>
#include <asm/io.h>

#ifdef CONFIG_PM
#include <linux/pm.h>
#endif

#define MX1KEY_INPUT_DEV_SUPPORT 1	/*enable dev/input support */

#define MX1_KEYPAD_I2CADDR (0x48>>1)
#define MX1_CTLPCF_I2CADDR (0x44>>1)

static struct file *mx1key_i2c;
static char mesgbuf[4];
static struct i2c_client *mx1key_client;

/*initialization progress indicators*/
static int mx1key_initstate_gpio;
static int mx1key_initstate_i2c;
static int mx1key_initstate_irq;
static int mx1key_initstate_thread;

#ifdef CONFIG_PM
static struct pm_dev *mx1key_pmdev;
#endif

DECLARE_WAIT_QUEUE_HEAD(wait_for_key);

static DECLARE_COMPLETION(keypth_exit);

static int keypad_thread_terminating = 0;

/*this function is used to wake up and terminate a thread*/
#define __mx1key_wait_event(wq, condition) 					\
do {									\
	wait_queue_t __wait;						\
	init_waitqueue_entry(&__wait, current);				\
	add_wait_queue(&wq, &__wait);					\
		set_current_state(TASK_UNINTERRUPTIBLE);		\
		if (!(condition))						\
		schedule();						\
	current->state = TASK_RUNNING;					\
	remove_wait_queue(&wq, &__wait);				\
} while (0)

#define mx1key_wait_event(wq, condition) 					\
do {									\
	if (condition)	 						\
		break;							\
	__mx1key_wait_event(wq, condition);					\
} while (0)

static int keypad_thread(void *data);
static void mx1_keyp_isr(int irq, void *dev_id, struct pt_regs *reg);
int __init mx1ads_keyp_init(void);
void __init mx1ads_keyp_exit(void);
#ifdef MX1KEY_INPUT_DEV_SUPPORT
static int mx1ads_keyp_open(struct input_dev *dev);
static void mx1ads_keyp_close(struct input_dev *dev);
#endif
#ifdef CONFIG_PM
static int mx1key_pm_callback(struct pm_dev *pmdev,
			      pm_request_t rqst, void *data);
#endif

static struct input_dev mx1key_dev;

/*initialize i2c*/
static void
mx1ads_keyp_i2c_init(void)
{
	int tmp;
	char filename[20];
	/*find the I2C driver we need */
	for (tmp = 0; tmp < I2C_ADAP_MAX; tmp++) {
#ifdef CONFIG_DEVFS_FS
		sprintf(filename, "/dev/i2c/%d", tmp);
#else
		sprintf(filename, "/dev/i2c-%d", tmp);
#endif

		if (!IS_ERR(mx1key_i2c = filp_open(filename, O_RDWR, 0))) {
			/*found some driver */
			mx1key_client = (struct i2c_client *)
			    mx1key_i2c->private_data;
			if (strlen(mx1key_client->adapter->name) >= 9) {
				if (!memcmp
				    (mx1key_client->
				     adapter->name, "DBMX1 I2C", 9)) {
					mx1key_initstate_i2c = 1;
					break;	/*we found our driver! */
				}
			}
			filp_close(mx1key_i2c, NULL);
		}
	}
}

/*initialize i2c, if not initialized, and read key states*/
static void
mx1ads_keyp_read_state(void)
{

	if (!mx1key_initstate_i2c)
		mx1ads_keyp_i2c_init();
	if (mx1key_initstate_i2c) {
		mx1key_client->addr = MX1_KEYPAD_I2CADDR;
		i2c_master_recv(mx1key_client, &mesgbuf[0], 2);
		mx1key_client->addr = MX1_CTLPCF_I2CADDR;
		i2c_master_recv(mx1key_client, &mesgbuf[2], 2);
#ifdef MX1KEY_INPUT_DEV_SUPPORT
		input_report_key(&mx1key_dev, KEY_1, ~mesgbuf[0] & 0x01);
		input_report_key(&mx1key_dev, KEY_2, ~mesgbuf[0] & 0x02);
		input_report_key(&mx1key_dev, KEY_3, ~mesgbuf[0] & 0x04);
		input_report_key(&mx1key_dev, KEY_4, ~mesgbuf[0] & 0x08);
		input_report_key(&mx1key_dev, KEY_5, ~mesgbuf[0] & 0x10);
		input_report_key(&mx1key_dev, KEY_6, ~mesgbuf[0] & 0x20);
		input_report_key(&mx1key_dev, KEY_7, ~mesgbuf[0] & 0x40);
		input_report_key(&mx1key_dev, KEY_8, ~mesgbuf[0] & 0x80);
		input_report_key(&mx1key_dev, KEY_9, ~mesgbuf[1] & 0x01);
		input_report_key(&mx1key_dev, KEY_0, ~mesgbuf[1] & 0x02);
		input_report_key(&mx1key_dev, KEY_A, ~mesgbuf[1] & 0x04);
		input_report_key(&mx1key_dev, KEY_B, ~mesgbuf[1] & 0x08);
		input_report_key(&mx1key_dev, KEY_C, ~mesgbuf[1] & 0x10);
		input_report_key(&mx1key_dev, KEY_D, ~mesgbuf[1] & 0x20);
		input_report_key(&mx1key_dev, KEY_E, ~mesgbuf[1] & 0x40);
		input_report_key(&mx1key_dev, KEY_F, ~mesgbuf[1] & 0x80);
		input_report_key(&mx1key_dev, KEY_G, ~mesgbuf[3] & 0x01);
		input_report_key(&mx1key_dev, KEY_H, ~mesgbuf[3] & 0x02);
#else
		printk(KERN_INFO
		       "mx1ads key state = %x, %x, ctl pcf state = %x, %x  \n",
		       mesgbuf[0], mesgbuf[1], mesgbuf[2], mesgbuf[3]);
#endif
	}
}

static int
keypad_thread(void *data)
{
	daemonize();
	reparent_to_init();
	strcpy(current->comm, "mx1adskey");

	for (;;) {
		mx1key_wait_event(wait_for_key, keypad_thread_terminating);
		if (keypad_thread_terminating == 1)
			break;
		mdelay(1);
		mx1ads_keyp_read_state();

	}

	complete_and_exit(&keypth_exit, 0);
	return 0;
}

/*keypad interrupt handler*/
static void
mx1_keyp_isr(int irq, void *dev_id, struct pt_regs *reg)
{
	if (mx1_gpio_intr_status_bit(PORT_B, 15)) {
		/*write 1 to clear interrupt status */
		mx1_gpio_clear_intr(PORT_B, 15);
		wake_up(&wait_for_key);
	}
}

#ifdef CONFIG_PM
/*power management event handling*/
static int
mx1key_pm_callback(struct pm_dev *pmdev, pm_request_t rqst, void *data)
{
	switch (rqst) {
	case PM_SUSPEND:
		mx1_gpio_mask_intr(PORT_B, 15);	/*IMR */
		break;

	case PM_RESUME:
		mx1_gpio_clear_intr(PORT_B, 15);
		mx1_gpio_unmask_intr(PORT_B, 15);	/*IMR */
		break;
	}
	return 0;
}
#endif

#ifdef MX1KEY_INPUT_DEV_SUPPORT
static int
mx1ads_keyp_open(struct input_dev *dev)
{
	/*printk("*********MX1ADS Keypad Driver opened!\n"); */
	/* The interrupts from both of the PCFs are tied - see schematic. If one of PCFs has some lines input high at powerup,
	   it'll drive the !int line low, preventing the other PCF from generating
	   interrupts, until we read from the PCFs, and now is a good time */
	MOD_INC_USE_COUNT;
	mx1ads_keyp_read_state();
	if (!mx1key_initstate_i2c)
		printk(KERN_WARNING
		       "MX1ADS KEYPAD WARNING: driver will not function until DBMX1 I2C driver is loaded\n");
	/*also, no caller of this function seem to care about the return code */
	return 0;
}

static void
mx1ads_keyp_close(struct input_dev *dev)
{
	MOD_DEC_USE_COUNT;
}
#endif

/*initialize keypad*/
int __init
mx1ads_keyp_init(void)
{
	int tmp;

	mx1key_initstate_gpio = 0;
	mx1key_initstate_i2c = 0;
	mx1key_initstate_irq = 0;
	mx1key_initstate_thread = 0;

	printk(KERN_INFO "MX1ADS Keypad Driver Ver. 1.1\n");
	/*printk(KERN_INFO "%s %s\n", __TIME__,__DATE__); */

	/*configure PB15 as input */
	/*enable pull-up for the PCF8575 open-drain interrupt source */
	tmp = mx1_register_gpio(PORT_B, 15, GPIO | INPUT | PULLUP);	/*GIUS, DDIR, PUEN */
	if (tmp < 0) {
		printk(KERN_ERR
		       "MX1ADS KEYPAD ERR: could not register PORT_B 15\n");
		mx1ads_keyp_exit();
		return tmp;
	}
	mx1key_initstate_gpio = 1;

#ifdef MX1KEY_INPUT_DEV_SUPPORT
	mx1key_dev.evbit[LONG(EV_KEY)] |= BIT(EV_KEY);
	mx1key_dev.keybit[LONG(KEY_1)] |= BIT(KEY_1);
	mx1key_dev.keybit[LONG(KEY_2)] |= BIT(KEY_2);
	mx1key_dev.keybit[LONG(KEY_3)] |= BIT(KEY_3);
	mx1key_dev.keybit[LONG(KEY_4)] |= BIT(KEY_4);
	mx1key_dev.keybit[LONG(KEY_5)] |= BIT(KEY_5);
	mx1key_dev.keybit[LONG(KEY_6)] |= BIT(KEY_6);
	mx1key_dev.keybit[LONG(KEY_7)] |= BIT(KEY_7);
	mx1key_dev.keybit[LONG(KEY_8)] |= BIT(KEY_8);
	mx1key_dev.keybit[LONG(KEY_9)] |= BIT(KEY_9);
	mx1key_dev.keybit[LONG(KEY_0)] |= BIT(KEY_0);
	mx1key_dev.keybit[LONG(KEY_A)] |= BIT(KEY_A);
	mx1key_dev.keybit[LONG(KEY_B)] |= BIT(KEY_B);
	mx1key_dev.keybit[LONG(KEY_C)] |= BIT(KEY_C);
	mx1key_dev.keybit[LONG(KEY_D)] |= BIT(KEY_D);
	mx1key_dev.keybit[LONG(KEY_E)] |= BIT(KEY_E);
	mx1key_dev.keybit[LONG(KEY_F)] |= BIT(KEY_F);
	mx1key_dev.keybit[LONG(KEY_G)] |= BIT(KEY_G);
	mx1key_dev.keybit[LONG(KEY_H)] |= BIT(KEY_H);
	mx1key_dev.open = &mx1ads_keyp_open;
	mx1key_dev.close = &mx1ads_keyp_close;

	input_register_device(&mx1key_dev);
#endif

	tmp = request_irq(GPIO_INT_PORTB,
			  (void *) mx1_keyp_isr,
			  SA_INTERRUPT, "MX1_KEYP_IRQ", "mx1keypirq");

	if (tmp) {
		printk(KERN_ERR "MX1ADS KEYPAD ERR: could not get IRQ\n");
		mx1ads_keyp_exit();
		return tmp;
	}
	mx1key_initstate_irq = 1;

	/*negative edge sensitive interrupt */
	mx1_gpio_config_intr(PORT_B, 15, NEGATIVE_EDGE);	/*ICR */
	mx1_gpio_clear_intr(PORT_B, 15);
	mx1_gpio_unmask_intr(PORT_B, 15);	/*IMR */

	/*init thread */
	keypad_thread_terminating = 0;
	tmp =
	    kernel_thread(&keypad_thread, NULL,
			  CLONE_FS | CLONE_FILES | CLONE_SIGHAND);
	if (tmp < 0) {
		printk(KERN_ERR "MX1ADS KEYPAD ERR: could not start thread\n");
		mx1ads_keyp_exit();
		return tmp;
	}
	mx1key_initstate_thread = 1;
#ifdef CONFIG_PM
	mx1key_pmdev =
	    pm_register(PM_UNKNOWN_DEV, PM_SYS_UNKNOWN, mx1key_pm_callback);

	if (!mx1key_pmdev)
		printk(KERN_WARNING
		       "%s: failed to init power management\n", __FILE__);
#endif

	return 0;
}

/*deinitialize keypad*/
void __init
mx1ads_keyp_exit(void)
{
#ifdef CONFIG_PM
	pm_unregister(mx1key_pmdev);
#endif
	if (mx1key_initstate_thread) {
		keypad_thread_terminating = 1;
		wake_up(&wait_for_key);
		wait_for_completion(&keypth_exit);
	}

	/* release the mx1 keypad isr handler */
	if (mx1key_initstate_irq) {
		mx1_gpio_mask_intr(PORT_B, 15);	/*IMR */
		free_irq(GPIO_INT_PORTB, "mx1keypirq");
	}

	if (mx1key_initstate_i2c) {
		filp_close(mx1key_i2c, NULL);
	}

	/*disable the interrupt input */
	if (mx1key_initstate_gpio) {
#ifdef MX1KEY_INPUT_DEV_SUPPORT
		input_unregister_device(&mx1key_dev);
#endif
		mx1_unregister_gpio(PORT_B, 15);
	}
}

MODULE_AUTHOR("MontaVista Software, Inc. <source@mvista.com>");
MODULE_DESCRIPTION("MX1 ADS Keypad Driver");
MODULE_LICENSE("GPL");

module_init(mx1ads_keyp_init);
module_exit(mx1ads_keyp_exit);
