/*
 *  MXL touchscreen driver ( SPI mode)
 *  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
 *  Author: MontaVista Software, Inc. <source@mvista.com>
 *  2003 (c) MontaVista Software, Inc. This file is licensed under
 *  the terms of the GNU General Public License version 2. This program
 *  is licensed "as is" without any warranty of any kind, whether express
 *  or implied.
 *		
 */
#ifndef __KERNEL__
#  define __KERNEL__
#endif

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/version.h>
#include <linux/init.h>
#include <linux/fs.h>
#include <linux/delay.h>
#include <linux/poll.h>
#include <asm/uaccess.h>	/* get_user,copy_to_user */

#include <linux/string.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/types.h>
#include <linux/fcntl.h>
#include <linux/interrupt.h>
#include <linux/ptrace.h>
#include <linux/ioport.h>
#include <linux/in.h>
#include <linux/slab.h>
#include <linux/vmalloc.h>
#include <linux/string.h>
#include <linux/init.h>
#include <asm/bitops.h>
#include <asm/io.h>
#include <linux/errno.h>
#include <linux/tqueue.h>
#include <linux/wait.h>

#include <asm/irq.h>
#include <asm/arch/hardware.h>
#include <asm/arch/irqs.h>
#include <asm/arch/mx1ads-gpio.h>

#include "mx1ads-spi.h"

#define SPI_DEBUG 0
//#undef SPI_DEBUG

#ifdef SPI_DEBUG
static int debug = SPI_DEBUG;
MODULE_PARM(debug, "i");
#define spi_printk( level, args... ) if( (level) <= debug ) printk( args )
#else
#define spi_printk( level, args... )
#endif

#define ENTER() spi_printk( 4, "%s: entered\n", __FUNCTION__ )
#define LEAVE() spi_printk( 4, "%s: leaving\n", __FUNCTION__ )

int spi_open(struct inode *inode, struct file *filp);
int spi_release(struct inode *inode, struct file *filp);
static int spi_fasync(int fd, struct file *filp, int mode);
int spi_ioctl(struct inode *inode, struct file *filp, unsigned int cmd,
	      unsigned long arg);
ssize_t spi_read(struct file *filp, char *buf, size_t count, loff_t * l);
unsigned int spi_poll(struct file *filp, struct poll_table_struct *wait);
void spi_sam_callback(unsigned long context);

struct file_operations spi_fops = {
	open:spi_open,
	release:spi_release,
	read:spi_read,
	poll:spi_poll,
	ioctl:spi_ioctl,
	fasync:spi_fasync,
};

static int spi_checkdevice(struct inode *pInode);
static int getblock(ts_event_t * *block, int size);
static ts_event_t *getdata(void);
static void spi_tx_data(__u16 data);

static void add_x_y(unsigned x, unsigned y, unsigned flag);
static int initbuf(void);
void spi_flush_fifo(void);
void spi_init_dev(void);

struct timer_list spi_sampler;
int spi_timer_in_use = 0;
wait_queue_head_t spi_wait;
struct fasync_struct *ts_fasync;
int spi_major;
unsigned short rptr, wptr;

#define NODATA()	( rptr==wptr )

void
spi_configure_gpio(void)
{
	ENTER();

	mx1_register_gpios(PORT_A, (1 << 0) | (1 << 17), GPIO | INPUT);
	mx1_register_gpios(PORT_D,
			   (1 << 7) | (1 << 8) | (1 << 10), GPIO | INPUT);
	mx1_register_gpio(PORT_A, 1, GPIO | OUTPUT);

	mx1_register_gpio(PORT_C, 15, PRIMARY | OUTPUT /* | OCR_DATA */ );
	mx1_register_gpios(PORT_C,
			   (1 << 14) | (1 << 13) | (1 << 17), PRIMARY | OUTPUT);
	mx1_register_gpio(PORT_C, 16, PRIMARY | INPUT);

	LEAVE();
}

void
spi_unconfigure_gpio(void)
{
	ENTER();

	mx1_unregister_gpios(PORT_C,
			     (1 << 13) | (1 << 14) | (1 << 15) | (1 << 16));
	mx1_unregister_gpios(PORT_A, (1 << 0) | (1 << 17) | (1 << 1));
	mx1_unregister_gpios(PORT_D, (1 << 7) | (1 << 8) | (1 << 10));

	LEAVE();
}

#define TOUCHPAN_BIT 31
#define SPI_BIT 15

#define touchpan_enable()  mx1_gpio_set_bit( PORT_C, SPI_BIT, 0 )
#define touchpan_disable() mx1_gpio_set_bit( PORT_C, SPI_BIT, 1 )

#define touchpan_enable_int( )   mx1_gpio_unmask_intr( PORT_D, TOUCHPAN_BIT )
#define touchpan_disable_int( )  mx1_gpio_mask_intr( PORT_D, TOUCHPAN_BIT)
#define touchpan_clear_int(  )   mx1_gpio_clear_intr( PORT_D, TOUCHPAN_BIT )

#define touchpan_pos_polarity( )  do {    \
	spi_printk( 3, "POS_POLARITY\n" );\
	mx1_gpio_config_intr( PORT_D, TOUCHPAN_BIT, POSITIVE_EDGE ); } while(0)
#define touchpan_neg_polarity( )  do {    \
	spi_printk( 3, "NEG_POLARITY\n" );\
	mx1_gpio_config_intr( PORT_D, TOUCHPAN_BIT, NEGATIVE_EDGE ); } while(0)

#define touchpan_get_polarity( ) mx1_gpio_get_intr_config( PORT_D, TOUCHPAN_BIT )
#define touchpan_pen_is_down() (touchpan_get_polarity() == POSITIVE_EDGE)

void
touchpan_init(void)
{
	ENTER();
	mx1_gpio_set_bit(PORT_D, TOUCHPAN_BIT, 1);
	LEAVE();
}

void
touchpan_config(void)
{
	ENTER();
	spi_init_dev();
/*    mx1_register_gpio( PORT_D, TOUCHPAN_BIT, GPIO | INPUT | PULLUP ); */
	mx1_gpio_mask_intr(PORT_D, TOUCHPAN_BIT);
	touchpan_disable_int();
	touchpan_init();
	touchpan_enable();

	LEAVE();
}

void
touchpan_unconfig(void)
{
	ENTER();
	mx1_gpio_mask_intr(PORT_D, TOUCHPAN_BIT);
	mx1_unregister_gpio(PORT_D, TOUCHPAN_BIT);
	LEAVE();
}

void
touchpan_readdev(__u32 * x, __u32 * y)
{
	__u32 x_upper, x_lower, y_upper, y_lower;

	ENTER();

	cli();
	touchpan_disable_int();
	touchpan_enable();

	spi_tx_data(0xD0);
	x_upper = SPI_REG(SPI_RXD);	/* the same as below */
	spi_tx_data(0x00);
	x_upper = SPI_REG(SPI_RXD);
	spi_tx_data(0x90);
	x_lower = SPI_REG(SPI_RXD);
	spi_tx_data(0x00);
	y_upper = SPI_REG(SPI_RXD);
	spi_tx_data(0x00);
	y_lower = SPI_REG(SPI_RXD);

	spi_printk(3, "%s: x_upper = %d, x_lower = %d\n", __FUNCTION__, x_upper,
		   x_lower);
	spi_printk(3, "%s: y_upper = %d, y_lower = %d\n", __FUNCTION__, y_upper,
		   y_lower);

	*x = (((x_upper << 5) & 0xFE0) | ((x_lower >> 3) & 0x1F));
	*y = (((y_upper << 5) & 0xFE0) | ((y_lower >> 3) & 0x1F));

	touchpan_disable();
	SPI_REG(SPI_CONT) &= (~SPI_EN);

	touchpan_enable_int();

	sti();
	spi_printk(1, "%s: coordinates are (%08X,%08X)\n", __FUNCTION__, *x,
		   *y);
	LEAVE();
}

void
touchpan_readdata(__u32 * x, __u32 * y)
{
	int i;
#define NUM_OF_SAMPLE  (2)
#define MAX_POS_DIFF   (2)
	ENTER();

	for (*x = *y = i = 0; i < NUM_OF_SAMPLE; i++) {
		__u32 xp, yp;

		touchpan_readdev(&xp, &yp);
		if ((xp < 100) || (xp > 5000) || (yp < 100) || (yp > 5000)) {
			*x = TPNL_PEN_UPVALUE;
			*y = TPNL_PEN_UPVALUE;
			return;
		}
		*x += xp;
		*y += yp;
	}

	*x /= NUM_OF_SAMPLE;
	*y /= NUM_OF_SAMPLE;

	LEAVE();

	return;
}

static void
touchpan_isr(int irq, void *dev_id, struct pt_regs *regs)
{
	__u32 touchflags;
	__u32 newX = TPNL_PEN_UPVALUE;
	__u32 newY;
	__u32 maxDelay = 0;

	ENTER();
	save_flags(touchflags);
	cli();

	/* check the interrupt polarity to see it is pen up or down */
	if (touchpan_pen_is_down()) {
		if (mx1_gpio_get_bit(PORT_D, TOUCHPAN_BIT) == 0) {
			touchpan_clear_int();
			return;
		}

		/* check if the pen sampling timer is in use */
		if (spi_timer_in_use) {
			/* then just change the interrupt polarity to be pen up */
			touchpan_neg_polarity();
			touchpan_clear_int();
			restore_flags(touchflags);
			LEAVE();
			return;
		} else {	/* no sampling timer */

			/* disable pen down interrupt */
			touchpan_clear_int();
			touchpan_readdata(&newX, &newY);

			while (newX == TPNL_PEN_UPVALUE && ++maxDelay <= 100) {
				touchpan_readdata(&newX, &newY);
			}
			if (newX == TPNL_PEN_UPVALUE) {
				touchpan_clear_int();
				restore_flags(touchflags);
				spi_printk(1, "%s: newX is incorrect\n",
					   __FUNCTION__);
				return;
			}

			add_x_y(newX, newY, PENDOWN);
			wake_up_interruptible(&spi_wait);

			if (ts_fasync) {
				kill_fasync(&ts_fasync, SIGIO, POLL_IN);
			}

			touchpan_neg_polarity();
			touchpan_clear_int();

			spi_timer_in_use = 1;
			spi_sampler.expires = jiffies + 2;

			spi_printk(2,
				   "%s: scheduling timer to %ld(%ld), function %p ( %p )\n",
				   __FUNCTION__, jiffies + 2,
				   jiffies + 2 - jiffies, spi_sampler.function,
				   spi_sam_callback);

			add_timer(&spi_sampler);

			restore_flags(touchflags);
			LEAVE();
			return;
		}
	}

	else {			/* pen up interrupt  */

		if (mx1_gpio_get_bit(PORT_D, TOUCHPAN_BIT) == 0) {
			touchpan_clear_int();
			restore_flags(touchflags);
			LEAVE();
			return;
		}
		//check if the pen sampling timer is in use
		if (spi_timer_in_use) {
			//get Pen up
			add_x_y(0, 0, PENUP);

			// if there is a pen up, shall tell apps to handle the event, and it shall
			// be data there
			if (!NODATA()) {
				wake_up_interruptible(&spi_wait);
				if (ts_fasync) {
					kill_fasync(&ts_fasync, SIGIO, POLL_IN);
				}
			}

			spi_timer_in_use = 0;
		}

		touchpan_pos_polarity();
		touchpan_clear_int();
		restore_flags(touchflags);
	}
	LEAVE();
}

void
spi_sam_callback(unsigned long context)
{
	__u32 samflags;
	__u32 newX = TPNL_PEN_UPVALUE, newY = TPNL_PEN_UPVALUE;
	__u32 maxDelay = 0;

	ENTER();
	save_flags(samflags);
	cli();
	if (spi_timer_in_use) {
		spi_timer_in_use = 0;
		/*
		   disable pen down interrupt
		   get Pen up event
		 */
		add_x_y(0, 0, PENUP);
		/*
		   if there is a pen up, shall tell apps to handle the event,
		   and it shall be data there
		 */
		if (!NODATA()) {
			wake_up_interruptible(&spi_wait);
			if (ts_fasync) {
				kill_fasync(&ts_fasync, SIGIO, POLL_IN);
			}
		}

		touchpan_pos_polarity();
		touchpan_clear_int();

		restore_flags(samflags);
		return;
	}

	if (touchpan_pen_is_down()) {
		/*
		 * The case below means that pen is DOWN
		 */

		while ((newX == TPNL_PEN_UPVALUE) && (++maxDelay <= 100)) {
			touchpan_readdata(&newX, &newY);
		}

		if (newX == TPNL_PEN_UPVALUE) {
			restore_flags(samflags);
			return;
		}

		/* report input event */
		add_x_y(newX, newY, PENDOWN);
		wake_up_interruptible(&spi_wait);
		if (ts_fasync) {
			kill_fasync(&ts_fasync, SIGIO, POLL_IN);
		}
		restore_flags(samflags);
		return;
	} else {
		spi_printk(0,
			   "%s: you should not see this at all. Incorrect call\n",
			   __FUNCTION__);
		restore_flags(samflags);
		return;
	}
	LEAVE();
}

/************************************************
Misc functions for SPI
*************************************************/
void
spi_init_dev(void)
{
	ENTER();
#ifdef USE_SPI1
	spi_configure_gpio();
	SPI_REG(SPI_RESET) = 1;
	udelay(10);

	spi_printk(2, "%s: writing %08X to SPI_CONT ( expected 0xE607 ) \n",
		   __FUNCTION__,
		   SPI_RATE_512 | SPI_RDY_IGNORE | SPI_MASTER |
		   SPI_EN | SPI_SS_ACTIVE_HIGH | SPI_SS_INSERT_PULSES |
		   SPI_PHA0 | SPI_CLK_ACTIVE_HIGH | 7);

	SPI_REG(SPI_CONT) = (SPI_RATE_512 | SPI_RDY_IGNORE | SPI_MASTER |
			     SPI_EN | SPI_SS_ACTIVE_HIGH | SPI_SS_INSERT_PULSES
			     | SPI_PHA0 | SPI_CLK_ACTIVE_HIGH | 7);
	LEAVE();
#endif
}

void
spi_free_dev(void)
{
	spi_unconfigure_gpio();
}

#define UINT16 __u16

/* To polling when Spi transfer is not finished */
static void
spi_poll_done(void)
{
	/* wait for SPI_TE_INT bit */
	while (!(SPI_REG(SPI_INT) & SPI_TE_INT)) {
		continue;
	}
	/*    Reset SPI IRQ bit */
	SPI_REG(SPI_INT) &= SPI_TE_INT;
	/* wait for SPI_XCH bit */
	while (!(SPI_REG(SPI_CONT) & SPI_XCH)) {
		continue;
	}
}

static void
spi_tx_data(UINT16 data)
{
	/* ensure data will not be transmitted while writing to SPI */
	SPI_REG(SPI_CONT) &= ~SPI_XCH;
	SPI_REG(SPI_CONT) |= SPI_EN;
	SPI_REG(SPI_TXD) /* REG_SPI_TXDATA */  = data;	/* transfer data */
	/* exchange data */
	SPI_REG(SPI_CONT) |= SPI_XCH;
	spi_poll_done();
	SPI_REG(SPI_CONT) &= ~SPI_XCH;
}

static int
spi_checkdevice(struct inode *pInode)
{
	int minor;
	kdev_t dev = pInode->i_rdev;

	if (MAJOR(dev) != spi_major) {
		spi_printk(1, "%s: bad major = %d\n", __FUNCTION__, MAJOR(dev));
		return -1;
	}

	minor = MINOR(dev);
	if (minor >= MAX_ID) {
		spi_printk(1, "%s: bad minor = %d\n", __FUNCTION__, minor);
		return -1;
	}
	return minor;
}

/**************************************************
Misc functions for buffer
***************************************************/
/* local buffer for store data
 * speed  200 f/s
 *
 * a new feature is read block, assume the buffer is a continuous buffer
 * so, if there is enough data, read data from current position to
 * buffer end, then next time, from buffer head to data end. -- the buffer
 * is a loop buffer.
 */
ts_event_t *buffer;
#define BUFLEN	250
#define BUFSIZE (BUFLEN*sizeof(ts_event_t))
#define NEXTI(i)	{i=(i==BUFLEN-1)?0:i+1;}
#define GETNEXTI(i)	((i==BUFLEN-1)?0:i+1)
#define BLKEND(i)	((i)==0)

#if 1
#define DSIZE()		((wptr<rptr)?(BUFLEN-rptr):(wptr-rptr))
#else
#define DSIZE()		((wptr<rptr)?(BUFLEN-rptr+wptr):(wptr-rptr))
#endif

static int
initbuf(void)
{
	if (NULL == (buffer = (ts_event_t *) vmalloc(BUFSIZE))) {
		spi_printk(0,
			   KERN_ERR
			   "%s: not enough kernel memory for spi data buffer\n",
			   __FUNCTION__);
		return -1;
	}

	rptr = wptr = 0;
	return 1;
}

/******************************************************************************
 * Function Name: add_x_y
 *
 * Input: 		i	:
 * Value Returned:	void	:
 *
 * Description: add pen data to buffer
 *
 *****************************************************************************/
static void
add_x_y(unsigned x, unsigned y, unsigned flag)
{
	ENTER();
	if (GETNEXTI(wptr) == rptr) {
		return;
	}
	spi_printk(2, "%s: Added %d, %d, flag = %X\n", __FUNCTION__, x, y,
		   flag);
	buffer[wptr].x = x;
	buffer[wptr].y = y;
	buffer[wptr].pressure = flag;
	NEXTI(wptr);
	LEAVE();
}

/******************************************************************************
 * Function Name: getblock
 *
 * Description: read a block of 'touch' data from buffer, read count shall less
 *	than size,but shall be as more as possible. the data shall not really copy
 *  to upper layer,untill copy_to_user is invoked.
 *****************************************************************************/
static int
getblock(ts_event_t * *block, int size)
{
	int cnt, rd;
	unsigned long flags;
	ts_event_t *p;

	if (NODATA()) {
		return 0;
	}

	save_flags(flags);
	cli();
	rd = (DSIZE() * sizeof (ts_event_t) >=
	      size) ? size / sizeof (ts_event_t) : DSIZE();

	for (*block = p = getdata(), cnt = 1; (NULL != p) && (cnt < rd); cnt++) {
		if (rptr == 0)
			break;
		p = getdata();
	}

	restore_flags(flags);
	return (cnt) * sizeof (ts_event_t);

}

/******************************************************************************
 * Function Name: getdata
 *
 * Input: 		void	:
 * Value Returned:	ts_event_t	: a 'touch' event data format
 *
 * Description: read a 'touch' event from data buffer
 *
 *****************************************************************************/
static ts_event_t *
getdata(void)
{
	ts_event_t *data;

	if (NODATA()) {
		return NULL;
	}

	data = &(buffer[rptr]);
	NEXTI(rptr);
	return data;
}

/******************************************************************************
 * Function Name: spi_release
 *
 * Input: 		inode	:
 * 			filp	:
 * Value Returned:	int	: Return status.If no error, return 0.
 *
 * Description: release resource when close the inode
 *
 *****************************************************************************/
int
spi_release(struct inode *inode, struct file *filp)
{
	ENTER();
	spi_fasync(-1, filp, 0);

	touchpan_disable_int();
	MOD_DEC_USE_COUNT;

	LEAVE();
	return 0;
}

void
spi_flush_fifo(void)
{
	__u32 j;
	int i;

	for (i = 0; i < 8; i++) {
		if (SPI_REG(SPI_INT) & SPI_RR_INT) {
			j = SPI_REG(SPI_RXD);
		}
	}
}

/******************************************************************************
 * Function Name: spi_open
 *
 * Input: 		inode	:
 * 			filp	:
 * Value Returned:	int	: Return status.If no error, return 0.
 *
 * Description: allocate resource when open the inode
 *
 *****************************************************************************/
int
spi_open(struct inode *inode, struct file *filp)
{
	MOD_INC_USE_COUNT;

	spi_flush_fifo();
	touchpan_disable_int();
	touchpan_pos_polarity();
	touchpan_clear_int();
	touchpan_enable_int();

	spi_timer_in_use = 0;
	return 0;
}

/******************************************************************************
 * Function Name: spi_fasync
 *
 * Input: 		fd	:
 * 			filp	:
 * 			mode	:
 * Value Returned:	int	: Return status.If no error, return 0.
 *
 * Description: provide fasync functionality for select system call
 *
 *****************************************************************************/
static int
spi_fasync(int fd, struct file *filp, int mode)
{
	/* TODO TODO put this data into file private data */
	if (spi_checkdevice(filp->f_dentry->d_inode) < 0) {
		spi_printk(1, "%s: incorrect device nr\n", __FUNCTION__);
		return -ENODEV;
	}
	return (fasync_helper(fd, filp, mode, &ts_fasync));
}

/******************************************************************************
 * Function Name: spi_ioctl
 *
 * Input: 		inode	:
 * 			filp	:
 * 			cmd	: command for ioctl
 * 			arg	: parameter for command
 * Value Returned:	int	: Return status.If no error, return 0.
 *
 * Description: ioctl for this device driver
 *
 *****************************************************************************/
int
spi_ioctl(struct inode *inode,
	  struct file *filp, unsigned int cmd, unsigned long arg)
{
	int ret = -EIO;

	if (spi_checkdevice(inode) < 0) {
		spi_printk(1, "%s: incorrect device nr\n", __FUNCTION__);
		return -ENODEV;
	}
	return ret;
}

/******************************************************************************
 * Function Name: spi_poll
 *
 * Input: 		filp	:
 * 			wait	:
 * Value Returned:	int	: Return status.If no error, return 0.
 *
 * Description: support poll and select
 *
 *****************************************************************************/
unsigned int
spi_poll(struct file *filp, struct poll_table_struct *wait)
{
	if (spi_checkdevice(filp->f_dentry->d_inode) < 0) {
		spi_printk(1, "%s: incorrect device nr\n", __FUNCTION__);
		return -ENODEV;
	}

	poll_wait(filp, &spi_wait, wait);

	return (NODATA())? 0 : (POLLIN | POLLRDNORM);
}

ssize_t
spi_read(struct file * filp, char *buf, size_t count, loff_t * l)
{
	int nonBlocking = filp->f_flags & O_NONBLOCK;
	ts_event_t *ev;
	int cnt = 0;

	if (spi_checkdevice(filp->f_dentry->d_inode) < 0) {
		spi_printk(1, "%s: incorrect device nr\n", __FUNCTION__);
		return -ENODEV;
	}

	ENTER();

	if (nonBlocking) {
		if (NODATA())
			return -EINVAL;

		cnt = getblock(&ev, count);
		if (cnt > count) {
			spi_printk(1, KERN_ERR "Error read spi buffer\n");
			return -EINVAL;
		}

		copy_to_user(buf, (char *) ev, cnt);
		LEAVE();
		return cnt;

	} else {

		/* check , when woken, there is a complete event to read */
		for (;;) {
			if (!NODATA()) {
				cnt = getblock(&ev, count);
				if (cnt > count) {
					spi_printk(1,
						   KERN_ERR
						   "Error read spi buffer\n");
					return -EINVAL;
				}
				LEAVE();
				/* returns length to be copied otherwise errno -Exxx */
				copy_to_user(buf, (char *) ev, cnt);
				return cnt;
			} else {
				spi_printk(3, "%s: waiting for data...\n",
					   __FUNCTION__);
				interruptible_sleep_on(&spi_wait);
				if (signal_pending(current)) {
					LEAVE();
					return -ERESTARTSYS;
				}
			}
		}
	}
}

/*******************************************************
*SPI init function
*Parameters:None
*Return	
*	0	indicates SUCCESS
* 	-1	indicates FAILURE
*Description: in this function should initialize the SPI device
as well as the external touch pannel hardware
1, touch pannel to port D[31], set portD interrupt setting??
2, request SPI interrupt
********************************************************/
int __init
spi_init(void)
{
	int result;
	int i;
	__u16 sequence[] = { 0xD000, 0x0000, 0x9000, 0x0000,
		0xD000, 0x9000, 0x0000, 0x0000
	};

	printk("SPI driver( %s) compiled %s %s\n", __FILE__, __TIME__,
	       __DATE__);
	ENTER();

	if ((result = register_chrdev(0, MOD_NAME, &spi_fops)) < 0) {
		spi_printk(1, "%s: failed to register_chrdev, code %d\n",
			   __FUNCTION__, result);
		return result;
	}
	if (spi_major == 0) {
		spi_major = result;
	}

	spi_printk(1, "%s: request_irq for %d\n", __FUNCTION__, TPNL_IRQ);
	/*Request for Touch Pannel ISR handler */
	if ((result = request_irq(TPNL_IRQ,
				  (void *) touchpan_isr,
				  TPNL_INTR_MODE, "spi", "spi")) < 0) {
		spi_printk(1, "%s: failed to request_irq(%d), code %d\n",
			   __FUNCTION__, TPNL_IRQ, result);
		unregister_chrdev(spi_major, MOD_NAME);
		return -1;
	}

	if (-1 == initbuf()) {
		spi_printk(1, "%s: failed to initbuf by some reason\n",
			   __FUNCTION__);
		free_irq(TPNL_IRQ, "spi");

		unregister_chrdev(spi_major, MOD_NAME);
		disable_irq(TPNL_IRQ);
		return -1;
	}

	init_waitqueue_head(&spi_wait);

	init_timer(&spi_sampler);
	spi_sampler.function = spi_sam_callback;

	touchpan_config();

	for (i = 0; i < sizeof (sequence) / sizeof (sequence[0]); i++) {
		spi_tx_data(sequence[i]);
	}

	touchpan_disable();
	udelay(10);
	LEAVE();
	return 0;
}

void __exit
spi_cleanup(void)
{
	spi_free_dev();

	free_irq(TPNL_IRQ, "spi");

	touchpan_unconfig();

	unregister_chrdev(spi_major, MOD_NAME);
	disable_irq(SPI_IRQ);
	disable_irq(TPNL_IRQ);
}

module_init(spi_init);
module_exit(spi_cleanup);
MODULE_LICENSE("GPL");
