/*******************************************************************************************************

	sim-dbmx1.c
	SmartCard Interface Module Linux Driver for DragonBall MX1

	Author: MontaVista Software, Inc. <source@mvista.com>
	Copyright (c) 2003 MontaVista Software, Inc.

	ATR handling routine in mx1sim_get_atr() is based on atrhandler.c from pcsc-lite-1.1.1
	Copyright (c) 1999-2002 David Corcoran <corcoran@linuxnet.com>
	http://www.linuxnet.com (MUSCLE)

	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

	Modifications:
	Nov 2003 - ver 1.1, MontaVista Software, Inc: changed to use misc device

*******************************************************************************************************/
/*******************************************************************************************************
ATR handling routine in mx1sim_get_atr() is based on atrhandler.c from pcsc-lite-1.1.1

This is a copy of pcsc-lite-1.1.1 license:

Copyright (c) 1999-2002 David Corcoran <corcoran@linuxnet.com>
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

1. Redistributions of source code must retain the above copyright
   notice, this list of conditions and the following disclaimer.
2. Redistributions in binary form must reproduce the above copyright
   notice, this list of conditions and the following disclaimer in the
   documentation and/or other materials provided with the distribution.
3. All advertising materials mentioning features or use of this software
   must display the following acknowledgement:
     This product includes software developed by:
      David Corcoran <corcoran@linuxnet.com>
      http://www.linuxnet.com (MUSCLE)
4. The name of the author may not be used to endorse or promote products
   derived from this software without specific prior written permission.

Changes to this license can be made only by the copyright author with
explicit written consent.

THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*******************************************************************************************************/

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/config.h>
#include <linux/init.h>
#include <linux/sched.h>
#include <linux/delay.h>
#include <linux/fs.h>
#include <linux/ioport.h>
#include <linux/slab.h>
#include <linux/miscdevice.h>

#ifdef CONFIG_PM
#include <linux/pm.h>
#endif

#include <asm/uaccess.h>
#include <asm/arch/platform.h>
#include <asm/irq.h>
#include <asm/arch/mx1ads-gpio.h>
#include <asm/io.h>

#include "sim-dbmx1.h"

/*This defines the base frequency for the SIM and some other modules.
If you change it here, please also adjust
an output delay within MX1SIM_HW_BUG_WORKAROUND in mx1sim_write_buf() and
the timeout periods in mx1sim_wait_for_tx() mx1sim_wait_for_rx()
and mx1sim_wait_for_tx_complete().
All delays are based on 1MHz clock rate*/
#define MX1SIM_PLLCLK2 0x5	/*16 MHz base frequency */

/*register access macros*/
#define mx1_sim_reg_out(r,x) outl(x,IO_ADDRESS((SIM_BASE + r)))
#define mx1_sim_reg_in(r) inl(IO_ADDRESS((SIM_BASE + r)))

/* This SIM controller has problems transmitting and receiving bytes.
For transmission, a delay is inserted after the fifth transmitted byte.
For receiving, garbage data in FIFO is removed.
The garbage is usually 1 byte identical to the second byte of the command sent.
It appears before the actual response data and only when a non-error response is received*/
#define MX1SIM_HW_BUG_WORKAROUND	/*hw bug workaround */

/*#define DEBUG_MX1SIM */
#ifdef DEBUG_MX1SIM
#define debugprintk(fmt, args...) printk("%s: " fmt, __FUNCTION__ , ## args)
#else
#define debugprintk(fmt, args...)
#endif

#define MX1SIM_PORTB_OUTP_MASK (0x35<<14)	/*configure PB 14,16,18,19 as primary output */
#define MX1SIM_PORTB_INP_MASK (1<<17)	/*configure PB 17 as primary input */
#define MX1SIM_PORTB_PD_INP_MASK (1<<15)	/*configure PB 15 as primary input */

#define MX1SIM_PROTO_T0 0
#define MX1SIM_PROTO_T1 1
#define MX1SIM_PROTO_AVAIL_T0 1
#define MX1SIM_PROTO_AVAIL_T1 2

static int mx1sim_open(struct inode *inode, struct file *file);
static int mx1sim_release(struct inode *inode, struct file *file);
static ssize_t mx1sim_read(struct file *file, char *buf, size_t count,
			   loff_t * offset);
static ssize_t mx1sim_write(struct file *file, const char *buf,
			    size_t count, loff_t * offset);
static int mx1sim_ioctl(struct inode *inode, struct file *file,
			unsigned int cmd, unsigned long arg);

static int mx1sim_get_atr(void);
static int mx1sim_turn_power_off(void);
static int mx1sim_turn_power_on(void);
static int mx1sim_wait_for_tx(void);
static int mx1sim_wait_for_rx(void);
#if defined (MX1SIM_MANAGE_TX_RX) || defined (MX1SIM_HW_BUG_WORKAROUND)
static int mx1sim_wait_for_tx_complete(void);
#endif
static int mx1sim_do_reset(void);
static int mx1sim_set_power(int power);
static int mx1sim_set_presence(int presence);
static int mx1sim_get_presence(void);
static int mx1sim_write_buf(char *buf, size_t count);
static int mx1sim_read_buf(char *buf, size_t count);
static void mx1sim_set_settings(struct mx1sim_settings_struct *inp_struct);
static void mx1sim_reg_init(void);
static int mx1sim_get_data(void);

static void mx1sim_isr(int irq, void *dev_id, struct pt_regs *reg);
static void mx1sim_data_isr(int irq, void *dev_id, struct pt_regs *reg);

#ifdef CONFIG_PM
static int mx1sim_pm_callback(struct pm_dev *pmdev,
			      pm_request_t rqst, void *data);
#endif

int __init dbmx1_sim_init(void);
void __init dbmx1_sim_exit(void);

static int mx1sim_power;
static int mx1sim_busy;
static char mx1sim_atr_buf[MX1SIM_MAX_ATR];
static int mx1sim_atr_len;
static struct mx1sim_settings_struct mx1sim_settings;
static int mx1sim_presence;
static int mx1sim_presence_stat;
#ifdef MX1SIM_HW_BUG_WORKAROUND
static int mx1sim_debug_data_save = -1;
#endif
#ifdef CONFIG_PM
static struct pm_dev *mx1sim_pmdev;
#endif

static struct file_operations mx1sim_fops = {
	.owner = THIS_MODULE,
	.llseek = no_llseek,
	.read = mx1sim_read,
	.write = mx1sim_write,
	.ioctl = mx1sim_ioctl,
	.open = mx1sim_open,
	.release = mx1sim_release,
};

/*initialization progress indicators*/
static int mx1sim_initstate_sim;
static int mx1sim_initstate_outp_gpio;
static int mx1sim_initstate_inp_gpio;
static int mx1sim_initstate_irq;
static int mx1sim_initstate_data_irq;
static int mx1sim_initstate_misc_dev;

DECLARE_WAIT_QUEUE_HEAD(mx1sim_rx_wait);
DECLARE_WAIT_QUEUE_HEAD(mx1sim_tx_wait);
DECLARE_WAIT_QUEUE_HEAD(mx1sim_tx_complete_wait);

/*interrupt handler*/
static void
mx1sim_isr(int irq, void *dev_id, struct pt_regs *reg)
{
	unsigned long tmp_rx_stat, tmp_tx_stat, tmp_intmask, tmp_pd_cntl;

	tmp_rx_stat = mx1_sim_reg_in(SIM_RCV_STATUS);
	tmp_tx_stat = mx1_sim_reg_in(SIM_XMT_STATUS);
	tmp_intmask = mx1_sim_reg_in(SIM_INT_MASK);
	tmp_pd_cntl = mx1_sim_reg_in(SIM_PORT_DETECT);

	if ((tmp_pd_cntl & (1 << 1)) && (!(tmp_pd_cntl & 1))) {	/*removal detected */
		debugprintk("int remove rx %x, tx %x, mask %x, pd %x \n",
			    tmp_rx_stat, tmp_tx_stat, tmp_intmask, tmp_pd_cntl);
		mx1_sim_reg_out(SIM_PORT_DETECT, tmp_pd_cntl);
		if (mx1sim_power & MX1SIM_POWER_ON) {
			mx1sim_presence_stat = MX1SIM_CARD_REMOVED;
			wake_up(&mx1sim_rx_wait);
			wake_up(&mx1sim_tx_wait);
			wake_up(&mx1sim_tx_complete_wait);
			mx1sim_turn_power_off();
		}
	}

	if ((tmp_rx_stat & (1 << 5)) && (!(tmp_intmask & (1 << 9)))) {	/*Char Wait Time Cnt */
		debugprintk("int cwt exp rx %x, tx %x, mask %x, pd %x \n",
			    tmp_rx_stat, tmp_tx_stat, tmp_intmask, tmp_pd_cntl);
		tmp_intmask |= (1 << 9);	/*disable rx fifo thresh, CWT int */
		mx1_sim_reg_out(SIM_INT_MASK, tmp_intmask);
		wake_up(&mx1sim_rx_wait);
	}

}

static void
mx1sim_data_isr(int irq, void *dev_id, struct pt_regs *reg)
{
	unsigned long tmp_rx_stat, tmp_tx_stat, tmp_intmask, tmp_pd_cntl;

	tmp_rx_stat = mx1_sim_reg_in(SIM_RCV_STATUS);
	tmp_tx_stat = mx1_sim_reg_in(SIM_XMT_STATUS);
	tmp_intmask = mx1_sim_reg_in(SIM_INT_MASK);
	tmp_pd_cntl = mx1_sim_reg_in(SIM_PORT_DETECT);

	if ((tmp_rx_stat & (1 << 1)) && (!(tmp_intmask & 1))) {	/*rxdata */
		debugprintk("int rxdata rx %x, tx %x, mask %x, pd %x \n",
			    tmp_rx_stat, tmp_tx_stat, tmp_intmask, tmp_pd_cntl);
		tmp_intmask |= 1;	/*disable rx fifo thresh */
		mx1_sim_reg_out(SIM_INT_MASK, tmp_intmask);
		wake_up(&mx1sim_rx_wait);
	}
	if ((tmp_tx_stat & (1 << 5)) && (!(tmp_intmask & (1 << 7)))) {	/*txdata */
		debugprintk("int txdata rx %x, tx %x, mask %x, pd %x \n",
			    tmp_rx_stat, tmp_tx_stat, tmp_intmask, tmp_pd_cntl);
		tmp_intmask |= (1 << 7);	/*disable tx fifo thresh int */
		mx1_sim_reg_out(SIM_INT_MASK, tmp_intmask);
		wake_up(&mx1sim_tx_wait);
	}
	if ((tmp_tx_stat & (1 << 3)) && (!(tmp_intmask & (1 << 1)))) {	/*txcomplete */
		debugprintk("int txcompl rx %x, tx %x, mask %x, pd %x \n",
			    tmp_rx_stat, tmp_tx_stat, tmp_intmask, tmp_pd_cntl);
		tmp_intmask |= (1 << 1);	/*disable tx complete int */
		mx1_sim_reg_out(SIM_INT_MASK, tmp_intmask);
		wake_up(&mx1sim_tx_complete_wait);

	}

}

/*returns success or timeout*/
static int
mx1sim_wait_for_tx(void)
{
	unsigned long tx_status;
	wait_queue_t __wait;

	if (mx1_sim_reg_in(SIM_XMT_STATUS) & (1 << 5))
		return 0;

	init_waitqueue_entry(&__wait, current);
	add_wait_queue(&mx1sim_tx_wait, &__wait);
	set_current_state(TASK_INTERRUPTIBLE);

	mx1_sim_reg_out(SIM_INT_MASK, mx1_sim_reg_in(SIM_INT_MASK) & ~(1 << 7));	/*enable tx fifo thresh int */

	schedule_timeout(HZ / 2);	/*adjust this delay if the MX1SIM_PLLCLK2 is changed. */
	current->state = TASK_RUNNING;
	remove_wait_queue(&mx1sim_tx_wait, &__wait);

	mx1_sim_reg_out(SIM_INT_MASK, mx1_sim_reg_in(SIM_INT_MASK) | (1 << 7));	/*disable tx fifo thresh int */
	tx_status = mx1_sim_reg_in(SIM_XMT_STATUS);
	tx_status &= (1 << 5);
	mx1_sim_reg_out(SIM_XMT_STATUS, tx_status);
	if (tx_status & (1 << 5))
		return 0;
	else
		return -1;	/*timeout happened */
}

/*returns success or timeout (schedule or CWT)*/
static int
mx1sim_wait_for_rx(void)
{
	wait_queue_t __wait;
	unsigned long tmp_status;

	if (mx1_sim_reg_in(SIM_RCV_STATUS) & (1 << 1))
		return 0;

	init_waitqueue_entry(&__wait, current);
	add_wait_queue(&mx1sim_rx_wait, &__wait);
	set_current_state(TASK_INTERRUPTIBLE);

	mx1_sim_reg_out(SIM_CNTL, mx1_sim_reg_in(SIM_CNTL) & ~(1 << 12));	/*clear CWT */
	mx1_sim_reg_out(SIM_CNTL, mx1_sim_reg_in(SIM_CNTL) | (1 << 12));	/*enable CWT */

	mx1_sim_reg_out(SIM_INT_MASK, mx1_sim_reg_in(SIM_INT_MASK) & ~(1 | (1 << 9)));	/*enable rx fifo thresh and CWT int */

	schedule_timeout(HZ * 4);	/*adjust this delay if the MX1SIM_PLLCLK2 is changed. */
	current->state = TASK_RUNNING;
	remove_wait_queue(&mx1sim_rx_wait, &__wait);

	mx1_sim_reg_out(SIM_CNTL, mx1_sim_reg_in(SIM_CNTL) & ~(1 << 12));	/*clear CWT */
	mx1_sim_reg_out(SIM_INT_MASK, mx1_sim_reg_in(SIM_INT_MASK) | (1 | (1 << 9)));	/*disable rx fifo thresh and CWT int */

	tmp_status = mx1_sim_reg_in(SIM_RCV_STATUS);
	tmp_status &= ((1 << 1) | (1 << 5));
	mx1_sim_reg_out(SIM_RCV_STATUS, tmp_status);

	if (tmp_status & (1 << 1))
		return 0;
	else
		return -1;	/*timeout happened */
}

#if defined (MX1SIM_MANAGE_TX_RX) || defined (MX1SIM_HW_BUG_WORKAROUND)
/*wait for tx full complete*/
static int
mx1sim_wait_for_tx_complete(void)
{

	wait_queue_t __wait;
	unsigned long tx_status;

	if (mx1_sim_reg_in(SIM_XMT_STATUS) & (1 << 3))
		return 0;
	debugprintk("\n");
	init_waitqueue_entry(&__wait, current);
	add_wait_queue(&mx1sim_tx_complete_wait, &__wait);
	set_current_state(TASK_INTERRUPTIBLE);

	mx1_sim_reg_out(SIM_INT_MASK, mx1_sim_reg_in(SIM_INT_MASK) & ~(1 << 1));	/*enable tx complete int */

	schedule_timeout(HZ);	/*adjust this delay if the MX1SIM_PLLCLK2 is changed. */
	current->state = TASK_RUNNING;
	remove_wait_queue(&mx1sim_tx_complete_wait, &__wait);

	mx1_sim_reg_out(SIM_INT_MASK, mx1_sim_reg_in(SIM_INT_MASK) | (1 << 1));	/*disable tx complete int */
	tx_status = mx1_sim_reg_in(SIM_XMT_STATUS);
	tx_status &= (1 << 3);
	mx1_sim_reg_out(SIM_XMT_STATUS, tx_status);
	if (tx_status & (1 << 3))
		return 0;
	else
		return -1;	/*timeout happened */
}
#endif

static void
mx1sim_reg_init(void)
{
	mx1_sim_reg_out(SIM_INT_MASK, 0x3FF);
	mx1_sim_reg_out(SIM_GPCNT, 0);
	mx1_sim_reg_out(SIM_XMT_THRESHOLD, 0xF);
	mx1_sim_reg_out(SIM_RCV_THRESHOLD, 1);
	mx1_sim_reg_out(SIM_PORT_DETECT, 1);
	mx1sim_set_presence(mx1sim_presence);
	mx1sim_set_settings(&mx1sim_settings);
}

static int
mx1sim_open(struct inode *inode, struct file *file)
{
	debugprintk("\n");
	if (mx1sim_busy)
		return -EBUSY;
	mx1sim_busy = 1;
	mx1_sim_reg_out(SIM_ENABLE, 1);	/*enable SIM, but disable SIM Tx  and Rx */

	return 0;
}

static int
mx1sim_release(struct inode *inode, struct file *file)
{
	debugprintk("\n");
	mx1sim_busy = 0;
	mx1sim_turn_power_off();
	mx1sim_set_presence(0);	/*disable presence detect */
	mx1_sim_reg_out(SIM_ENABLE, 0);	/*disable SIM */
	return 0;
}

static int
mx1sim_get_data(void)
{
	int tmpdata, tmp_dat, tmp_err, i, attempts;
	if ((mx1sim_settings.anack_en == 1)
	    || (mx1sim_settings.onack_en == 1))
		attempts = 15;
	else
		attempts = 1;
	for (i = 0; i < attempts; i++) {
		if (mx1sim_wait_for_rx()) {
			/*timeout error */
			debugprintk(" timeout\n");
			return -ETIMEDOUT;
		}

		tmpdata = mx1_sim_reg_in(SIM_PORT_RCV_BUF);

		debugprintk("%x ", tmpdata);
		tmp_dat = tmpdata & 0xFF;
		tmp_err = (tmpdata >> 8) & 0x3;

		if (tmp_err) {
			debugprintk(" err\n");
			continue;
		}

		return tmp_dat;
	}
	return -EIO;
}

static int
mx1sim_read_buf(char *buf, size_t count)
{
	int tmp;
	size_t i;
	debugprintk("\n");

	for (i = 0; i < count; i++) {
		tmp = mx1sim_get_data();
		if (tmp < 0) {
			if (i == 0) {
				/*timeout error */
				debugprintk("init timeout\n");
				return -ETIMEDOUT;	/*report error if no bytes recvd */
			} else {
				/*timeout error */
				debugprintk("timeout\n");
				break;
			}
		}
#ifdef MX1SIM_HW_BUG_WORKAROUND
		if ((mx1sim_debug_data_save >= 0)
		    && (mx1sim_debug_data_save == tmp) && (i == 0))
			count++;
#endif
		*(buf + i) = tmp;
	}
	debugprintk(" ok %d\n", i);
	if ((mx1_sim_reg_in(SIM_RCV_STATUS) & 1)) {
		printk(KERN_WARNING "%s: rx buffer overflow happened\n",
		       __FILE__);
		mx1_sim_reg_out(SIM_RCV_STATUS, 1);
	}
	return i;
}

/*returns requested number of bytes received or less in case of IO error
there is also a special ATR mode (returns ATR), enabled by resetting the device*/
static ssize_t
mx1sim_read(struct file *file, char *buf, size_t count, loff_t * offset)
{
	char *tmp_buf;
	int ret;
	debugprintk("\n");

	if (mx1sim_power & MX1SIM_POWER_RESET) {	/*reset mode. return ATR in this case */
		debugprintk("send atr len %d\n", mx1sim_atr_len);
		ret =
		    copy_to_user(buf, &mx1sim_atr_buf[0],
				 mx1sim_atr_len) ? -EFAULT : mx1sim_atr_len;
		if (ret >= 0)
			mx1sim_power &= ~MX1SIM_POWER_RESET;
		return mx1sim_atr_len;
	}
	if (!(mx1sim_power & MX1SIM_POWER_ON))
		return -EPERM;	/*turn power on first */
	debugprintk("power ok\n");

	/*regular mode */

#ifdef MX1SIM_HW_BUG_WORKAROUND
	if (count < 2)
		return -EPERM;	/*response is always at least 2 bytes long */
	if (count > 8191)
		count = 8191;
	tmp_buf = kmalloc(count + 1, GFP_KERNEL);
#else
	if (count < 1)
		return -EPERM;
	if (count > 8192)
		count = 8192;
	tmp_buf = kmalloc(count, GFP_KERNEL);
#endif
	if (tmp_buf == NULL)
		return -ENOMEM;	/*out of memory */

	if ((ret = mx1sim_read_buf(tmp_buf, count)) >= 0) {
#ifdef MX1SIM_HW_BUG_WORKAROUND
		if (ret > 2) {
			ret--;
			ret =
			    copy_to_user(buf, (tmp_buf + 1),
					 ret) ? -EFAULT : ret;
		} else if (ret == 2)
			ret = copy_to_user(buf, tmp_buf, ret) ? -EFAULT : ret;
		else
			ret = -ETIMEDOUT;
		mx1sim_debug_data_save = -1;
#else
		ret = copy_to_user(buf, tmp_buf, ret) ? -EFAULT : ret;
#endif
	}

	kfree(tmp_buf);
	debugprintk("ret %d\n", ret);
	return ret;
}

static int
mx1sim_write_buf(char *buf, size_t count)
{
	size_t i;

	debugprintk("\n");

	mx1_sim_reg_out(SIM_RESET_CNTL, 0x3);	/*reset SIM rx and tx */
	mx1_sim_reg_out(SIM_RESET_CNTL, 0);	/*clear reset for SIM rx and tx */

#ifdef MX1SIM_MANAGE_TX_RX
	mx1_sim_reg_out(SIM_ENABLE, 7);	/*enable SIM Rx and SIM Tx */
#endif

	for (i = 0; i < count; i++) {
		if (mx1sim_wait_for_tx()) {
			/*timeout error */
			debugprintk("timeout\n");
			return -ETIMEDOUT;
		}
#ifdef MX1SIM_HW_BUG_WORKAROUND
		if (i == 5) {
			mx1sim_wait_for_tx_complete();
			mdelay(30);	/*adjust this delay if the MX1SIM_PLLCLK2 is changed. */
		}
		if (i == 1)
			mx1sim_debug_data_save = *(buf + i);
#endif
#if defined (MX1SIM_MANAGE_TX_RX) || defined (MX1SIM_HW_BUG_WORKAROUND)
		mx1_sim_reg_out(SIM_XMT_STATUS, (1 << 3));	/*clear tx complete flag */
#endif
		mx1_sim_reg_out(SIM_PORT_XMT_BUF, *(buf + i));
		debugprintk("%x ", *(buf + i));

	}

#ifdef MX1SIM_MANAGE_TX_RX
	mx1sim_wait_for_tx_complete();
	mx1_sim_reg_out(SIM_ENABLE, 3);	/*enable SIM and SIM Rx */
#endif

	return i;
}

/*returns requested number of bytes trasmitted or less in case of IO error*/
static ssize_t
mx1sim_write(struct file *file, const char *buf, size_t count, loff_t * offset)
{
	int ret;
	char *tmp_buf;
	debugprintk("\n");
	if (mx1sim_power & MX1SIM_POWER_RESET)
		return -EPERM;	/*read ATR first */
	debugprintk("atr ok\n");
	if (!(mx1sim_power & MX1SIM_POWER_ON))
		return -EPERM;	/*turn power on first */
	debugprintk("power ok\n");

#ifdef MX1SIM_HW_BUG_WORKAROUND
	if (count < 4)
		return -EPERM;	/*command is always at least 4 bytes long */
#else
	if (count < 1)
		return -EPERM;
#endif

	if (count > 8192)
		count = 8192;

	/* copy user space data to kernel space. */
	tmp_buf = kmalloc(count, GFP_KERNEL);
	if (tmp_buf == NULL) {
		debugprintk("no mem\n");
		return -ENOMEM;	/*out of memory */
	}
	if (copy_from_user(tmp_buf, buf, count)) {
		kfree(tmp_buf);
		debugprintk("copy from usr err\n");
		return -EFAULT;
	}

	ret = mx1sim_write_buf(tmp_buf, count);

	kfree(tmp_buf);
	debugprintk("ret %d\n", ret);
	return ret;

}

static int
mx1sim_ioctl(struct inode *inode, struct file *file,
	     unsigned int cmd, unsigned long arg)
{
	int tmp_arg;

	switch (cmd) {
/*setting*/

	case MX1SIM_IOW_POWER:	/*Set On, Off, Reset */
		if (copy_from_user(&tmp_arg, (int *) arg, sizeof (int)))
			return -EFAULT;
		return mx1sim_set_power(tmp_arg);

	case MX1SIM_IOW_PRESENCE:	/*enable/disable presence detect functionality */
		if (copy_from_user(&tmp_arg, (int *) arg, sizeof (int)))
			return -EFAULT;
		return mx1sim_set_presence(tmp_arg);

	case MX1SIM_IOW_SETTINGS:	/*set settings */
		{
			struct mx1sim_settings_struct tmp_settings;
			if (copy_from_user
			    (&tmp_settings,
			     (struct mx1sim_settings_struct *) arg,
			     sizeof (struct mx1sim_settings_struct)))
				return -EFAULT;
			mx1sim_set_settings(&tmp_settings);
			mx1sim_settings = tmp_settings;
			return 0;
		}

/*reading*/
	case MX1SIM_IOR_POWER:	/*return power state (On, Off, Reset) */
		tmp_arg = mx1sim_power;
		return (copy_to_user((int *) arg, &tmp_arg, sizeof (int))) ?
		    -EFAULT : 0;

	case MX1SIM_IOR_PRESENCE:	/*return SCard presence (yes/no) */
		tmp_arg = mx1sim_get_presence();
		return (copy_to_user((int *) arg, &tmp_arg, sizeof (int))) ?
		    -EFAULT : 0;

	case MX1SIM_IOR_SETTINGS:	/*get settings */
		return (copy_to_user
			((struct mx1sim_settings_struct *) arg,
			 &mx1sim_settings,
			 sizeof (struct mx1sim_settings_struct))) ? -EFAULT : 0;

	default:
		return -ENOIOCTLCMD;
	}
}

static int
mx1sim_get_atr(void)
{
	int i, j, tmp;
	int tmp_T, parse_K, parse_TCK, parse_Y1i, parse_TAi, parse_TBi,
	    parse_TCi, parse_TDi;
	int tmp_proto;
	int tmp_proto_avail;
	int tmp_inverse_data;
	int tmp_FI_DI;
	int tmp_guard_time;

	debugprintk("\n");

	mx1_sim_reg_out(SIM_CNTL, mx1_sim_reg_in(SIM_CNTL) | (0x1 << 1));	/*enable initial char mode */

#ifdef MX1SIM_MANAGE_TX_RX
	mx1_sim_reg_out(SIM_ENABLE, 3);	/*enable SIM and SIM Rx */
#else
	mx1_sim_reg_out(SIM_ENABLE, 7);	/*enable SIM , SIM Rx and SIM Tx */
#endif
	mx1_sim_reg_out(SIM_PORT_CNTL, (mx1_sim_reg_in(SIM_PORT_CNTL) | (0x1 << 3)));	/*release SC reset (SRST) */

	i = 0;
	tmp_proto = -1;
	tmp_proto_avail = 0;
	tmp_FI_DI = -1;
	tmp_guard_time = -1;

	if ((tmp = mx1sim_get_data()) < 0) {
		debugprintk(" bad init char\n");
		return tmp;
	}
	if (mx1_sim_reg_in(SIM_CNTL) & (0x1 << 1)) {
		debugprintk(" init char not rcvd\n");
		return -EPROTO;
	}

	if (tmp == 0x3F)
		tmp_inverse_data = 1;	/* inverse convention */
	else if (tmp == 0x3B)
		tmp_inverse_data = 0;	/* direct convention */
	else
		return -EPROTO;	/*error */
	mx1sim_atr_buf[i++] = tmp & 0xFF;

	if ((tmp = mx1sim_get_data()) < 0)
		return tmp;
	else
		mx1sim_atr_buf[i++] = tmp & 0xFF;
	parse_Y1i = tmp >> 4;
	parse_K = tmp & 0x0F;

	do {

		if (parse_Y1i & 0x01) {
			if (i >= MX1SIM_MAX_ATR)
				return -EPROTO;
			if ((tmp = mx1sim_get_data()) < 0)
				return tmp;
			else
				mx1sim_atr_buf[i++] = tmp & 0xFF;
			parse_TAi = tmp;
		} else
			parse_TAi = -1;
		if (parse_Y1i & 0x02) {
			if (i >= MX1SIM_MAX_ATR)
				return -EPROTO;
			if ((tmp = mx1sim_get_data()) < 0)
				return tmp;
			else
				mx1sim_atr_buf[i++] = tmp & 0xFF;
			parse_TBi = tmp;
		} else
			parse_TBi = -1;
		if (parse_Y1i & 0x04) {
			if (i >= MX1SIM_MAX_ATR)
				return -EPROTO;
			if ((tmp = mx1sim_get_data()) < 0)
				return tmp;
			else
				mx1sim_atr_buf[i++] = tmp & 0xFF;
			parse_TCi = tmp;
		} else
			parse_TCi = -1;
		if (parse_Y1i & 0x08) {
			if (i >= MX1SIM_MAX_ATR)
				return -EPROTO;
			if ((tmp = mx1sim_get_data()) < 0)
				return tmp;
			else
				mx1sim_atr_buf[i++] = tmp & 0xFF;
			parse_TDi = tmp;
		} else
			parse_TDi = -1;

		if ((parse_TAi >= 0) && (tmp_FI_DI == -1)) {
			tmp_FI_DI = parse_TAi;
		}

		if ((parse_TCi >= 0) && (tmp_guard_time == -1))
			tmp_guard_time = parse_TCi;

		/*process protocol bytes */
		if (parse_TDi >= 0) {
			parse_Y1i = parse_TDi >> 4;
			tmp_T = parse_TDi & 0x0F;

			if (tmp_proto == -1) {
				if (tmp_T == 0)
					tmp_proto = MX1SIM_PROTO_T0;
				else if (tmp_T == 1)
					tmp_proto = MX1SIM_PROTO_T1;
				else
					return -EPROTO;
			}

			if (tmp_T == 0)
				tmp_proto_avail |= MX1SIM_PROTO_AVAIL_T0;
			else if (tmp_T == 1)
				tmp_proto_avail |= MX1SIM_PROTO_AVAIL_T1;
			else
				return -EPROTO;

		} else
			parse_Y1i = 0;

	}
	while (parse_Y1i != 0);

	/*T0 is the default protocol */
	if (tmp_proto == -1) {
		tmp_proto = MX1SIM_PROTO_T0;
		tmp_proto_avail = MX1SIM_PROTO_AVAIL_T0;
	}
	/*process historical chars */
	for (j = 0; j < parse_K; j++) {
		if (i >= MX1SIM_MAX_ATR)
			return -EPROTO;
		if ((tmp = mx1sim_get_data()) < 0)
			return tmp;
		else
			mx1sim_atr_buf[i++] = tmp & 0xFF;
	}

	/*verify Check Character */
	if (tmp_proto_avail & MX1SIM_PROTO_AVAIL_T1) {
		if (i >= MX1SIM_MAX_ATR)
			return -EPROTO;
		if ((tmp = mx1sim_get_data()) < 0)
			return tmp;
		else
			mx1sim_atr_buf[i++] = tmp & 0xFF;
		parse_TCK = tmp;
	}

	mx1sim_atr_len = i;	/* atr length */

	debugprintk
	    ("\nok, proto T%d, avail:%x, inv %d, len %d, guar %x, Fi %x, Di %x\n",
	     tmp_proto, tmp_proto_avail, tmp_inverse_data, i,
	     tmp_guard_time, (tmp_FI_DI >> 4) & 0xF, tmp_FI_DI & 0xF);

	if (tmp_guard_time == -1) {
		tmp_guard_time = 0;
	}
	if (tmp_FI_DI == -1) {
		tmp_FI_DI = 0x11;
	}

	/*save data format. This is the only setting that is determined automatically by HW */
	mx1sim_settings.inv_en = mx1_sim_reg_in(SIM_CNTL) & 1;
	return i;
}

static int
mx1sim_turn_power_on(void)
{
	int tmp;

	debugprintk("\n");
	if (mx1sim_power & MX1SIM_POWER_ON)
		return -EPERM;
	if (!(mx1sim_get_presence() & MX1SIM_CARD_PRESENT))
		return -EPERM;
	mx1_sim_reg_out(SIM_ENABLE, 0);	/*disable SIM */

	mx1_sim_reg_out(SIM_RESET_CNTL, (1 << 2));	/*reset SIM */
	mdelay(10);		/*wait for at least 4 ref clk cycles */

	mx1_sim_reg_out(SIM_ENABLE, 1);	/*enable SIM */
	mx1sim_reg_init();
	mx1_sim_reg_out(SIM_PORT_CNTL, (mx1_sim_reg_in(SIM_PORT_CNTL) | (0x1 << 1)));	/*enable power to SC (SVEN) */
	udelay(100);
	mx1_sim_reg_out(SIM_PORT_CNTL, (mx1_sim_reg_in(SIM_PORT_CNTL) | (0x1 << 2)));	/*enable tx data outp (STEN) */
	udelay(100);
	mx1_sim_reg_out(SIM_PORT_CNTL, (mx1_sim_reg_in(SIM_PORT_CNTL) | (0x1 << 4)));	/*enable SC clock (SCEN) */
	udelay(100);
	mx1sim_power = MX1SIM_POWER_ON;
	if ((tmp = mx1sim_get_atr()) < 0) {
		mx1sim_turn_power_off();
		return tmp;
	}
	mx1sim_power = MX1SIM_POWER_ON | MX1SIM_POWER_RESET;

	return tmp;
}

static int
mx1sim_turn_power_off(void)
{
	debugprintk("\n");
	if (!(mx1sim_power & MX1SIM_POWER_ON))
		return -EPERM;
	mx1_sim_reg_out(SIM_ENABLE, 1);	/*disable SIM Tx  and Rx */
	mx1_sim_reg_out(SIM_PORT_CNTL, (mx1_sim_reg_in(SIM_PORT_CNTL) & ~(0x1 << 3)));	/*put up SC reset (SRST) */
	udelay(30);
	mx1_sim_reg_out(SIM_PORT_CNTL, (mx1_sim_reg_in(SIM_PORT_CNTL) & ~(0x1 << 4)));	/*disable SC clock (SCEN) */
	udelay(30);
	mx1_sim_reg_out(SIM_PORT_CNTL, (mx1_sim_reg_in(SIM_PORT_CNTL) & ~(0x1 << 2)));	/*disable tx data outp (STEN) */
	udelay(30);
	mx1_sim_reg_out(SIM_PORT_CNTL, (mx1_sim_reg_in(SIM_PORT_CNTL) & ~(0x1 << 1)));	/*disable power to SC (SVEN) */

	mx1sim_power = 0;

	return 0;
}

static int
mx1sim_do_reset(void)
{
	int tmp;
	debugprintk("\n");
	if (!(mx1sim_power & MX1SIM_POWER_ON))
		return mx1sim_turn_power_on();	/*turn power on first */

	mx1_sim_reg_out(SIM_PORT_CNTL, (mx1_sim_reg_in(SIM_PORT_CNTL) & ~(0x1 << 3)));	/*put up SC reset (SRST) */

	mx1_sim_reg_out(SIM_ENABLE, 1);	/*enable SIM */
	mx1_sim_reg_out(SIM_RESET_CNTL, 0x3);	/*reset SIM rx and tx */
	mx1_sim_reg_out(SIM_RESET_CNTL, 0);	/*clear reset for SIM rx and tx */
	mdelay(10);

	if ((tmp = mx1sim_get_atr()) < 0) {
		mx1sim_turn_power_off();
		return -EIO;	/*failed to communicate */
	}
	mx1sim_power = MX1SIM_POWER_ON | MX1SIM_POWER_RESET;
	return tmp;
}

/*turn power on, off, or reset*/
static int
mx1sim_set_power(int power)
{
	debugprintk("\n");
	if (power & MX1SIM_POWER_RESET) {
		return mx1sim_do_reset();
	}
	if (power & MX1SIM_POWER_ON) {
		return mx1sim_turn_power_on();
	} else {
		return mx1sim_turn_power_off();
	}
}

/*enable/disable presence/absence detect*/
static int
mx1sim_set_presence(int presence)
{
	unsigned long tmp_data;
	int tmp;
	debugprintk("\n");
	if (mx1sim_power & MX1SIM_POWER_ON)
		return -EPERM;	/*not allowed if power is on */

	if (mx1sim_presence & MX1SIM_PRESENCE_EN) {	/*if already enabled */
		if (presence & MX1SIM_PRESENCE_EN) {	/*possible presence definition change */
			debugprintk("possible presence definition change\n");
			tmp_data = 1;	/*mask out PD interrupt */
			mx1_sim_reg_out(SIM_PORT_DETECT, tmp_data);
			tmp_data = (1 << 1);
			if (presence & MX1SIM_REMOVAL_ON_RISING_EDGE)
				tmp_data |= (1 << 3);
			mx1_sim_reg_out(SIM_PORT_DETECT, tmp_data);
			mx1sim_presence = presence;
			return 0;
		} else {	/*disable presence detect */
			debugprintk("disable presence detect\n");
			mx1_unregister_gpios(PORT_B, MX1SIM_PORTB_PD_INP_MASK);
			tmp_data = 1;	/*mask out PD interrupt */
			mx1_sim_reg_out(SIM_PORT_DETECT, tmp_data);
			mx1sim_presence = 0;
			return 0;
		}

	} else {		/*if not enabled */

		if (presence & MX1SIM_PRESENCE_EN) {	/*enable presence detect */
			debugprintk("enable presence detect\n");
			mx1sim_presence_stat = 0;
			if ((tmp =
			     mx1_register_gpios(PORT_B,
						MX1SIM_PORTB_PD_INP_MASK,
						PRIMARY | INPUT |
						TRISTATE)) < 0)
				return tmp;
			mdelay(1);
			tmp_data = (1 << 1);
			if (presence & MX1SIM_REMOVAL_ON_RISING_EDGE)
				tmp_data |= (1 << 3);
			mx1_sim_reg_out(SIM_PORT_DETECT, tmp_data);
			mx1sim_presence = presence;
			return 0;
		}
	}
	return -EPERM;
}

/*return current presence/absence status*/
static int
mx1sim_get_presence(void)
{
	unsigned long tmp_data;
	int tmp = 0;
	debugprintk("\n");
	if (mx1sim_presence & MX1SIM_PRESENCE_EN) {
		tmp_data = mx1_sim_reg_in(SIM_PORT_DETECT);
		tmp = mx1sim_presence_stat;
		mx1sim_presence_stat = 0;
		if ((!(mx1sim_presence & MX1SIM_REMOVAL_ON_RISING_EDGE))
		    && (tmp_data & (1 << 2)))
			tmp |= MX1SIM_CARD_PRESENT;
		if ((mx1sim_presence & MX1SIM_REMOVAL_ON_RISING_EDGE)
		    && (!(tmp_data & (1 << 2))))
			tmp |= MX1SIM_CARD_PRESENT;
		debugprintk("reg: %x result: %x\n", tmp_data, tmp);
		return tmp;
	} else
		return (MX1SIM_CARD_PRESENT | MX1SIM_PRESENCE_DETECT_DISABLED);
}

static void
mx1sim_set_settings(struct mx1sim_settings_struct *inp_struct)
{
	unsigned long tmp_state, tmp_data;

	debugprintk("\n");

	tmp_state = mx1_sim_reg_in(SIM_ENABLE);	/*remeber what is enabled */
	mx1_sim_reg_out(SIM_ENABLE, 1);	/*enable SIM, but disable SIM Tx and Rx */

	/*setup open-drain or push-pull mode */
	tmp_data = inp_struct->open_drain_en & 1;
	mx1_sim_reg_out(SIM_OD_CONFIG, tmp_data);

	/*setup bidir mode and clock stop polarity */
	tmp_data = mx1_sim_reg_in(SIM_PORT_CNTL);
	if (inp_struct->bidir_en & 1)
		tmp_data |= (1 << 6);
	else
		tmp_data &= ~(1 << 6);
	if (inp_struct->clk_stop_pol & 1)
		tmp_data |= (1 << 5);
	else
		tmp_data &= ~(1 << 5);
	mx1_sim_reg_out(SIM_PORT_CNTL, tmp_data);

	/*set up char wait */
	tmp_data = inp_struct->char_wait & 0xFFFF;
	mx1_sim_reg_out(SIM_CHAR_WAIT, tmp_data);

	/*setup anack, onack, clk, baud, format */
	tmp_data = mx1_sim_reg_in(SIM_CNTL);
	if (inp_struct->anack_en & 1)
		tmp_data |= (1 << 2);
	else
		tmp_data &= ~(1 << 2);
	if (inp_struct->onack_en & 1)
		tmp_data |= (1 << 3);
	else
		tmp_data &= ~(1 << 5);
	if (inp_struct->inv_en & 1)
		tmp_data |= (1);
	else
		tmp_data &= ~(1);
	tmp_data &= ~(0x7 << 4);
	tmp_data |= (((unsigned long) inp_struct->clk) << 4) & (0x7 << 4);
	tmp_data &= ~(0x7 << 7);
	tmp_data |= (((unsigned long) inp_struct->baud) << 7) & (0x7 << 7);
	mx1_sim_reg_out(SIM_CNTL, tmp_data);

	/*setup divisor */
	tmp_data = inp_struct->div & 0x7F;
	mx1_sim_reg_out(SIM_DIVISOR, tmp_data);

	/*setup tx nack thresh */
	tmp_data = mx1_sim_reg_in(SIM_XMT_THRESHOLD);
	tmp_data &= ~(0xf << 4);
	tmp_data |=
	    (((unsigned long) inp_struct->tx_nack_thresh) << 4) & (0xf << 4);
	mx1_sim_reg_out(SIM_XMT_THRESHOLD, tmp_data);

	/*setup tx guard and # of rx stop bits */
	tmp_data = mx1_sim_reg_in(SIM_GUARD_CNTL);
	if (inp_struct->rcv_1_stop_en & 1)
		tmp_data |= (1 << 8);
	else
		tmp_data &= ~(1 << 8);
	tmp_data &= ~0xff;
	tmp_data |= ((unsigned long) inp_struct->tx_guard) & 0xff;
	mx1_sim_reg_out(SIM_GUARD_CNTL, tmp_data);
	debugprintk("SIM_PORT_CNTL: %x\n", mx1_sim_reg_in(SIM_PORT_CNTL));
	debugprintk("SIM_CNTL: %x\n", mx1_sim_reg_in(SIM_CNTL));
	debugprintk("SIM_OD_CONFIG: %x\n", mx1_sim_reg_in(SIM_OD_CONFIG));
	debugprintk("SIM_CHAR_WAIT: %x\n", mx1_sim_reg_in(SIM_CHAR_WAIT));
	debugprintk("SIM_DIVISOR: %x\n", mx1_sim_reg_in(SIM_DIVISOR));
	debugprintk("SIM_XMT_THRESHOLD: %x\n",
		    mx1_sim_reg_in(SIM_XMT_THRESHOLD));
	debugprintk("SIM_GUARD_CNTL: %x\n", mx1_sim_reg_in(SIM_GUARD_CNTL));
	debugprintk("SIM_PORT_DETECT: %x\n", mx1_sim_reg_in(SIM_PORT_DETECT));
	mx1_sim_reg_out(SIM_ENABLE, tmp_state);
}

#ifdef CONFIG_PM
/*power management event handling*/
static int
mx1sim_pm_callback(struct pm_dev *pmdev, pm_request_t rqst, void *data)
{
	switch (rqst) {
	case PM_SUSPEND:
		mx1sim_turn_power_off();
		mx1_sim_reg_out(SIM_ENABLE, 0);	/*disable SIM, enter power-saving mode */
		break;
	case PM_RESUME:
		mx1_sim_reg_out(SIM_ENABLE, 1);	/*enable SIM */
		mx1sim_set_presence(mx1sim_presence);	/*restore presence detect mode */
		/*the user will have to start over because the card will reset once the power is up.
		   Therefore, there is no need to restore power to the card now */
		break;
	}
	return 0;
}
#endif

static struct miscdevice mx1sim_misc_dev = {
	minor:MISC_DYNAMIC_MINOR,
	name:"sim-dbmx1",
	fops:&mx1sim_fops,
};

/*initialize*/
int __init
dbmx1_sim_init(void)
{
	int tmp = 0;
	unsigned long tmp_reg;
	mx1sim_initstate_outp_gpio = 0;
	mx1sim_initstate_inp_gpio = 0;
	mx1sim_initstate_irq = 0;
	mx1sim_initstate_data_irq = 0;
	mx1sim_initstate_sim = 0;
	mx1sim_initstate_misc_dev = 0;

	printk(KERN_INFO "DBMX1 SIM Driver Ver. 1.1\n");
	debugprintk("%s %s\n", __TIME__, __DATE__);

	tmp = (int) request_region(IO_ADDRESS(SIM_BASE), 0x44, "dbmx1_sim");
	if (!tmp) {
		printk(KERN_ERR "%s: SIM is already in use\n", __FILE__);
		dbmx1_sim_exit();
		return -1;
	}
	mx1sim_initstate_sim = 1;

	tmp =
	    mx1_register_gpios(PORT_B, MX1SIM_PORTB_OUTP_MASK,
			       PRIMARY | OUTPUT | TRISTATE);
	if (tmp < 0) {
		printk(KERN_ERR
		       "%s: could not register PORT_B mask 0x%x\n",
		       __FILE__, MX1SIM_PORTB_OUTP_MASK);
		dbmx1_sim_exit();
		return tmp;
	}
	mx1sim_initstate_outp_gpio = 1;

	tmp =
	    mx1_register_gpios(PORT_B, MX1SIM_PORTB_INP_MASK,
			       PRIMARY | INPUT | TRISTATE);
	if (tmp < 0) {
		printk(KERN_ERR
		       "%s: could not register PORT_B mask 0x%x\n",
		       __FILE__, MX1SIM_PORTB_INP_MASK);
		dbmx1_sim_exit();
		return tmp;
	}
	mx1sim_initstate_inp_gpio = 1;

	tmp =
	    request_irq(SIM_INT, (void *) mx1sim_isr, SA_INTERRUPT,
			"MX1_SIM_IRQ", "mx1simirq");

	if (tmp) {
		printk(KERN_ERR "%s: could not get SIM IRQ #%d\n",
		       __FILE__, SIM_INT);
		dbmx1_sim_exit();
		return tmp;
	}
	mx1sim_initstate_irq = 1;

	tmp =
	    request_irq(SIM_DATA_INT, (void *) mx1sim_data_isr,
			SA_INTERRUPT, "MX1_SIM_DATA_IRQ", "mx1simdatairq");

	if (tmp) {

		printk(KERN_ERR "%s: could not get SIM DATA IRQ #%d\n",
		       __FILE__, SIM_DATA_INT);
		dbmx1_sim_exit();
		return tmp;
	}
	mx1sim_initstate_data_irq = 1;

	tmp = misc_register(&mx1sim_misc_dev);
	if (tmp < 0) {
		printk(KERN_ERR "%s: unable to register char dev\n", __FILE__);
		dbmx1_sim_exit();
		return tmp;
	}
	mx1sim_initstate_misc_dev = 1;

#ifdef CONFIG_PM
	mx1sim_pmdev =
	    pm_register(PM_UNKNOWN_DEV, PM_SYS_UNKNOWN, mx1sim_pm_callback);

	if (!mx1sim_pmdev)
		printk(KERN_WARNING
		       "%s: failed to init power management\n", __FILE__);
#endif

	if (((tmp_reg =
	      inl(IO_ADDRESS((PLL_PCDR + PLL_BASE)))) & PCDR_PCLKDIV2_MASK)
	    != ((MX1SIM_PLLCLK2 << PCDR_PCLKDIV2_BIT) & PCDR_PCLKDIV2_MASK)) {
		printk(KERN_WARNING
		       "%s: changing PERCLK2 from 0x%x to 0x%x\n",
		       __FILE__,
		       (unsigned int) ((tmp_reg & PCDR_PCLKDIV2_MASK) >>
				       PCDR_PCLKDIV2_BIT), MX1SIM_PLLCLK2);
		tmp_reg &= ~PCDR_PCLKDIV2_MASK;
		tmp_reg |=
		    (MX1SIM_PLLCLK2 << PCDR_PCLKDIV2_BIT) & PCDR_PCLKDIV2_MASK;
		outl(tmp_reg, IO_ADDRESS((PLL_PCDR + PLL_BASE)));
	}

	mx1sim_settings.clk_stop_pol = 0;	/*clock stop polarity */
	mx1sim_settings.anack_en = 0;	/*anack en */
	mx1sim_settings.onack_en = 0;	/*onack en */
	mx1sim_settings.tx_nack_thresh = 0xf;	/*tx nack threshold */
	mx1sim_settings.baud = 0;	/*baud sel */
	mx1sim_settings.div = 0;	/*divisor sel */
	mx1sim_settings.clk = 1;	/*clk sel */
	mx1sim_settings.open_drain_en = 1;	/*open-drain */
	mx1sim_settings.bidir_en = 0;	/*bidirectional */
	mx1sim_settings.tx_guard = 0;	/*tx guard */
	mx1sim_settings.rcv_1_stop_en = 1;	/*rcv 1 (1) or 2 (0) stop bits */
	mx1sim_settings.char_wait = 9600 - 12;	/*char wait time */
	mx1sim_settings.inv_en = 0;	/*format direct (0)/inverse(1) */
	mx1sim_presence = 0;	/*presence detect disabled */

	return 0;
}

/*deinitialize*/
void __init
dbmx1_sim_exit(void)
{
	if (mx1sim_initstate_misc_dev) {
		disable_irq(SIM_INT);
		disable_irq(SIM_DATA_INT);
#ifdef CONFIG_PM
		pm_unregister(mx1sim_pmdev);
#endif
		misc_deregister(&mx1sim_misc_dev);
	}

	if (mx1sim_initstate_irq)
		free_irq(SIM_INT, "mx1simirq");
	if (mx1sim_initstate_data_irq)
		free_irq(SIM_DATA_INT, "mx1simdatairq");
	if (mx1sim_initstate_outp_gpio)
		mx1_unregister_gpios(PORT_B, MX1SIM_PORTB_OUTP_MASK);
	if (mx1sim_initstate_inp_gpio)
		mx1_unregister_gpios(PORT_B, MX1SIM_PORTB_INP_MASK);
	if (mx1sim_initstate_sim)
		release_region(IO_ADDRESS(SIM_BASE), 0x44);
}

MODULE_AUTHOR("MontaVista Software, Inc. <source@mvista.com>");
MODULE_DESCRIPTION("DBMX1 SIM Driver");
MODULE_LICENSE("GPL");

module_init(dbmx1_sim_init);
module_exit(dbmx1_sim_exit);
