/*
* drivers/misc/ti1500_tpanel.c
*
* Touch screen driver for the XXS_1500 board
* Based on Touch screen driver for the TI Innovator (OMAP1510).
*
* Author: Sergey Podstavin <source@mvista.com>
*
* 1999-2003 (c) MontaVista Software, Inc. This file is licensed under
* the terms of the GNU General Public License version 2. This program
* is licensed "as is" without any warranty of any kind, whether express
* or implied.
*/

#include <linux/autoconf.h>
#include <linux/config.h>
#include <linux/module.h>
#include <linux/version.h>
#include <linux/init.h>
#include <linux/fs.h>
#include <linux/miscdevice.h>
#include <linux/delay.h>
#include <linux/poll.h>
#include <linux/string.h>
#include <linux/ioport.h>	/* request_region */
#include <linux/interrupt.h>
#include <linux/pci.h>
#include <linux/pagemap.h>	/* pci mapping */
#include <asm/uaccess.h>	/* get_user,copy_to_user */
#include <asm/irq.h>
#include <asm/io.h>

#define TS_NAME "xxs1500_tpanel"
#define TS_MINOR 14
#define PFX TS_NAME

#define XXS1500_TPANEL_DEBUG 1

#ifdef XXS1500_TPANEL_DEBUG
#define dbg(format, arg...)	printk(KERN_DEBUG PFX ": " format "\n" , ## arg)
#else
#define dbg(format, arg...)
#endif
#define err(format, arg...) 	printk(KERN_ERR PFX ": " format "\n" , ## arg)
#define info(format, arg...) 	printk(KERN_INFO PFX ": " format "\n" , ## arg)
#define warn(format, arg...) 	printk(KERN_WARNING PFX ": " format "\n" , ## arg)

// Touch Panel registers

#define TP_CONV 0x0
#define TP_CTRL 0x4
#define TP_WAIT 0x8
#define TP_MASK 0xC
#define TP_STAT 0x8

// Touch Panel registers bits

#define TP_MASK_PEN_EN    0x1	// PEN Interrupt enable
#define TP_MASK_CC_EN     0x2	// Conversion Interrupt enable
#define TP_STAT_BUSY      0x4	// busy during conversion
#define TP_STAT_CC        0x2	// conversion completed
#define TP_STAT_PEN       0x1	// by an active/PENIRQ (low)

// Ti1500 touch panel PCI Interrupt Control Registers

#define FUNC0_INT_LATCH   0x20
#define FUNC0_INT_MASK    0x24
#define FUNC0_INT_MRES    0x28
#define FUNC0_INT_MSET    0x2c

/*Register Settings*/
#define TP_OFFSET         0x00
#define TP_LENGTH         20

// AD7873 Control Byte bit defines
#define AD7873_ADDR_BIT   4
#define AD7873_ADDR_MASK  (0x7<<AD7873_ADDR_BIT)
#define AD7873_MEASURE_X  (0x5<<AD7873_ADDR_BIT)
#define AD7873_MEASURE_Y  (0x1<<AD7873_ADDR_BIT)
#define AD7873_MEASURE_Z1 (0x3<<AD7873_ADDR_BIT)
#define AD7873_MEASURE_Z2 (0x4<<AD7873_ADDR_BIT)
#define AD7873_8BITS      (1<<3)
#define AD7873_12BITS     0
#define AD7873_SER        (1<<2)
#define AD7873_DFR        0
#define AD7873_PWR_BIT    0
#define AD7873_PD         0
#define AD7873_ADC_ON     (0x1<<AD7873_PWR_BIT)
#define AD7873_REF_ON     (0x2<<AD7873_PWR_BIT)
#define AD7873_REF_ADC_ON (0x3<<AD7873_PWR_BIT)
#define AD7873_START_BIT  (1<<7)
#define MEASURE_12BIT_X \
    (AD7873_START_BIT | AD7873_MEASURE_X | AD7873_12BITS | \
     AD7873_DFR | AD7873_REF_ADC_ON)
#define MEASURE_12BIT_Y \
    (AD7873_START_BIT | AD7873_MEASURE_Y | AD7873_12BITS | \
     AD7873_DFR | AD7873_REF_ADC_ON)
#define MEASURE_12BIT_Z1 \
    (AD7873_START_BIT | AD7873_MEASURE_Z1 | AD7873_12BITS | \
     AD7873_DFR | AD7873_REF_ADC_ON)
#define MEASURE_12BIT_Z2 \
    (AD7873_START_BIT | AD7873_MEASURE_Z2 | AD7873_12BITS | \
     AD7873_DFR | AD7873_REF_ADC_ON)
#ifdef CONFIG_TI1500_TPANEL_INT
/* Getting irq has the problems, so we set irq manually */
static int irq = 2;		/* select whether a manual irq is used */
MODULE_PARM(irq, "i");
#endif				//CONFIG_TI1500_TPANEL_INT
/*
 * The definition of the following structure is copied from ucb1x00-ts.c
 * so the touchscreen will "just work" with tslib.
 */
struct ts_event {
	u16 pressure;
	u16 x;
	u16 y;
	u16 pad;
	struct timeval stamp;
};

#define EVENT_BUFSIZE 32	// must be power of two

/*
 * Which pressure equation to use from AD7346 datasheet.
 * The first equation requires knowing only the X plate
 * resistance, but needs 4 measurements (X, Y, Z1, Z2).
 * The second equation requires knowing both X and Y plate
 * resistance, but only needs 3 measurements (X, Y, Z1).
 * The second equation is preferred because of the shorter
 * acquisition time required.
 */
enum {
	PRESSURE_EQN_1 = 0,
	PRESSURE_EQN_2
};

/*
 * The touch screen's X and Y plate resistances, used by
 * pressure equations.
 */
#define DEFAULT_X_PLATE_OHMS 330
#define DEFAULT_Y_PLATE_OHMS 485

/* Minimum pen down pressure */
#define DEFAULT_PEN_DOWN_P 15	/* Xenarc TP */

struct xxs1500_tpanel_t {
	unsigned long intctl_base;	// base address for fpga function unit #0
	unsigned long intctl_io;	// mapped address for fpga function unit #0
	unsigned long intctl_iolen;	// size of fpga function unit #0 memory
	unsigned long tp_base;	// mapped touch panel base
	// The X and Y plate resistance, needed to calculate pressure
	int x_plate_ohms, y_plate_ohms;
	int pressure_eqn;	// eqn to use for pressure calc
	u8 pendown_irq;		// IRQ of pendown interrupt
	int pen_is_down;	// 1 = pen is down, 0 = pen is up
	int pen_was_down;	// Was pen down last time?
	int irq_enabled;
	struct ts_event event_buf[EVENT_BUFSIZE];	// The event queue
	int nextIn, nextOut;
	int event_count;
	struct fasync_struct *fasync;	// asynch notification
	struct timer_list acq_timer;	// Timer for triggering acquisitions
	wait_queue_head_t wait;	// read wait queue
	spinlock_t lock;
};

/*identifies FPGA version*/
int fpga_ver = 0;

static struct xxs1500_tpanel_t xxs1500_tpanel;

static inline u32
fpga_ts_read(void)
{
	return readl(xxs1500_tpanel.tp_base + TP_STAT);
}

static inline void
fpga_ts_write(u32 val)
{
	writel(val, xxs1500_tpanel.tp_base + TP_CTRL);
}

static inline int
BUSY(void)
{
	return ((fpga_ts_read() & TP_STAT_BUSY) ? 1 : 0);
}

static inline u8
DOUT(void)
{
	return ((fpga_ts_read() & TP_STAT_CC) ? 1 : 0);
}

static inline int
PenIsDown(void)
{
	return ((fpga_ts_read() & TP_STAT_PEN) ? 1 : 0);
}

static u32
ad7873_do(u32 cmd)
{
	u32 val = 0;
	fpga_ts_write(cmd);
	while (BUSY()) ;
	// now read returned data
	val = readl(xxs1500_tpanel.tp_base + TP_CONV);
	return val;
}

// hold the spinlock before calling.
static void
event_add(struct xxs1500_tpanel_t *ts, struct ts_event *event)
{
	// add this event to the event queue
	ts->event_buf[ts->nextIn] = *event;
	ts->nextIn = (ts->nextIn + 1) & (EVENT_BUFSIZE - 1);
	if (ts->event_count < EVENT_BUFSIZE) {
		ts->event_count++;
	} else {
		// throw out the oldest event
		ts->nextOut = (ts->nextOut + 1) & (EVENT_BUFSIZE - 1);
	}
	// async notify
	if (ts->fasync)
		kill_fasync(&ts->fasync, SIGIO, POLL_IN);
	// wake up any read call
	if (waitqueue_active(&ts->wait))
		wake_up_interruptible(&ts->wait);
}

static int
event_pull(struct xxs1500_tpanel_t *ts, struct ts_event *event)
{
	unsigned long flags;
	int ret;
	spin_lock_irqsave(&ts->lock, flags);
	ret = ts->event_count;
	if (ts->event_count) {
		*event = ts->event_buf[ts->nextOut];
		ts->nextOut = (ts->nextOut + 1) & (EVENT_BUFSIZE - 1);
		ts->event_count--;
	}
	spin_unlock_irqrestore(&ts->lock, flags);
	return ret;
}

#ifdef CONFIG_TI1500_TPANEL_INT
static void
xxs1500_tpanel_interrupt(int irq, void *dev_id, struct pt_regs *regs)
{
	struct xxs1500_tpanel_t *ts = (struct xxs1500_tpanel_t *) dev_id;
	u32 int_stat;
	spin_lock(&ts->lock);
	int_stat = readl(ts->tp_base + TP_STAT);
	if ((!(int_stat & TP_STAT_CC)) && (!(int_stat & TP_STAT_BUSY))) {
		if (int_stat & TP_STAT_PEN) {
			writel(0x8, ts->intctl_io + FUNC0_INT_LATCH);	//acknowledge pen interrupt
			if (ts->irq_enabled) {
				ts->irq_enabled = 0;
				disable_irq(irq);
			}
			ts->pen_is_down = 1;
			// restart acquire
			ts->acq_timer.expires = jiffies + HZ / 100;
			add_timer(&ts->acq_timer);
			spin_unlock(&ts->lock);
			return;
		} else {
			writel(0x8, ts->intctl_io + FUNC0_INT_LATCH);	//acknowledge conversion interrupt
			spin_unlock(&ts->lock);
			return;
		}
	}
	if (int_stat & TP_STAT_BUSY) {
		spin_unlock(&ts->lock);
		return;
	}
	printk(KERN_ERR "%s: interrupt error - we shouldn't have been here\n",
	       __FUNCTION__);
	writel(0x8, ts->intctl_io + FUNC0_INT_LATCH);	//acknowledge conversion interrupt
	spin_unlock(&ts->lock);
}
#endif				//CONFIG_TI1500_TPANEL_INT
/*
 * Acquire Raw pen coodinate data and compute touch screen
 * pressure resistance. Hold spinlock when calling.
 */
static void
AcquireEvent(struct xxs1500_tpanel_t *ts, struct ts_event *event)
{
	unsigned long x_raw = 0, y_raw = 0, z1_raw = 0, z2_raw = 0;
	unsigned long Rt = 0;
	y_raw += ad7873_do(MEASURE_12BIT_X);
	x_raw += ad7873_do(MEASURE_12BIT_Y);
	z1_raw += ad7873_do(MEASURE_12BIT_Z1);
	if (ts->pressure_eqn == PRESSURE_EQN_1) {
		z2_raw += ad7873_do(MEASURE_12BIT_Z2);
	}
	// Calculate touch pressure resistance
	if (z1_raw) {
		if (ts->pressure_eqn == PRESSURE_EQN_1) {
			Rt = (ts->x_plate_ohms * (u32) x_raw *
			      ((u32) z2_raw - (u32) z1_raw)) / (u32) z1_raw;
		} else {
			Rt = (ts->x_plate_ohms * (u32) x_raw *
			      (4096 - (u32) z1_raw)) / (u32) z1_raw;
			Rt -= (ts->y_plate_ohms * (u32) y_raw);
		}
		Rt = (Rt + 2048) >> 12;	// round up to nearest ohm
	}
	event->x = x_raw;
	event->y = y_raw;
	event->pressure = (u16) Rt;
	// timestamp this new event.
	do_gettimeofday(&event->stamp);
}

/*
 * Raw X,Y,pressure acquisition timer function. It gets scheduled
 * only while pen is down. Its duration between calls is the polling
 * rate.
 */
static void
xxs1500_tpanel_acq_timer(unsigned long data)
{
	struct xxs1500_tpanel_t *ts = (struct xxs1500_tpanel_t *) data;
	unsigned long flags;
	struct ts_event event;
#ifndef CONFIG_TI1500_TPANEL_INT
	struct ts_event eventnul;
#else
	int pen_was_down = ts->pen_is_down;
#endif				// no CONFIG_TI1500_TPANEL_INT
	spin_lock_irqsave(&ts->lock, flags);
#ifdef CONFIG_TI1500_TPANEL_INT
	if (PenIsDown()) {
#else
	if (ts->pen_is_down) {
#endif				//CONFIG_TI1500_TPANEL_INT
		ts->pen_is_down = 1;
		AcquireEvent(ts, &event);
		udelay(1000);
		// schedule next acquire
		ts->acq_timer.expires = jiffies + HZ / 100;
		add_timer(&ts->acq_timer);
#ifndef CONFIG_TI1500_TPANEL_INT
		if ((event.pressure < DEFAULT_PEN_DOWN_P)
		    || (event.pressure > 0xf00) || (event.y == 0xfff)) {
			ts->pen_is_down = 0;
			ts->pen_was_down = 1;
			spin_unlock_irqrestore(&ts->lock, flags);
			return;
		}
#endif				// no CONFIG_TI1500_TPANEL_INT
		event_add(ts, &event);
	} else {
		if (!ts->irq_enabled) {
			ts->irq_enabled = 1;
#ifdef CONFIG_TI1500_TPANEL_INT
			enable_irq(ts->pendown_irq);
#endif				//CONFIG_TI1500_TPANEL_INT
		}
		ts->pen_is_down = 0;
#ifdef CONFIG_TI1500_TPANEL_INT
		if (pen_was_down) {
#else
		AcquireEvent(ts, &eventnul);
		if ((eventnul.pressure > DEFAULT_PEN_DOWN_P)
		    && (eventnul.y != 0xfff)) {
			ts->pen_is_down = 1;
		}
		if (ts->pen_was_down) {
			ts->pen_was_down = 0;
#endif				//CONFIG_TI1500_TPANEL_INT
			event.x = event.y = event.pressure = 0;
			do_gettimeofday(&event.stamp);
			event_add(ts, &event);
		}
#ifndef CONFIG_TI1500_TPANEL_INT
		udelay(1000);
		// schedule next acquire
		ts->acq_timer.expires = jiffies + HZ / 100;
		add_timer(&ts->acq_timer);
#endif				// no CONFIG_TI1500_TPANEL_INT
	}
	spin_unlock_irqrestore(&ts->lock, flags);
}

/* +++++++++++++ File operations ++++++++++++++*/
static int
xxs1500_tpanel_fasync(int fd, struct file *filp, int mode)
{
	struct xxs1500_tpanel_t *ts =
	    (struct xxs1500_tpanel_t *) filp->private_data;
	return fasync_helper(fd, filp, mode, &ts->fasync);
}

static unsigned int
xxs1500_tpanel_poll(struct file *filp, poll_table * wait)
{
	struct xxs1500_tpanel_t *ts =
	    (struct xxs1500_tpanel_t *) filp->private_data;
	poll_wait(filp, &ts->wait, wait);
	if (ts->event_count)
		return POLLIN | POLLRDNORM;
	return 0;
}

static ssize_t
xxs1500_tpanel_read(struct file *filp, char *buffer, size_t count,
		    loff_t * ppos)
{
	DECLARE_WAITQUEUE(wait, current);
	struct xxs1500_tpanel_t *ts =
	    (struct xxs1500_tpanel_t *) filp->private_data;
	char *ptr = buffer;
	struct ts_event event;
	int err = 0;
	add_wait_queue(&ts->wait, &wait);
	while (count >= sizeof (struct ts_event)) {
		err = -ERESTARTSYS;
		if (signal_pending(current))
			break;
		if (event_pull(ts, &event)) {
			err = copy_to_user(ptr, &event,
					   sizeof (struct ts_event));
			if (err)
				break;
			ptr += sizeof (struct ts_event);
			count -= sizeof (struct ts_event);
		} else {
			set_current_state(TASK_INTERRUPTIBLE);
			err = -EAGAIN;
			if (filp->f_flags & O_NONBLOCK)
				break;
			schedule();
		}
	}
	current->state = TASK_RUNNING;
	remove_wait_queue(&ts->wait, &wait);
	return ptr == buffer ? err : ptr - buffer;
}

static int
xxs1500_tpanel_open(struct inode *inode, struct file *filp)
{
	struct xxs1500_tpanel_t *ts;
	filp->private_data = ts = &xxs1500_tpanel;
	MOD_INC_USE_COUNT;
	return 0;
}

static int
xxs1500_tpanel_release(struct inode *inode, struct file *filp)
{
	xxs1500_tpanel_fasync(-1, filp, 0);
	MOD_DEC_USE_COUNT;
	return 0;
}

static struct file_operations ts_fops = {
	.owner = THIS_MODULE,
	.read = xxs1500_tpanel_read,
	.poll = xxs1500_tpanel_poll,
	.fasync = xxs1500_tpanel_fasync,
	.open = xxs1500_tpanel_open,
	.release = xxs1500_tpanel_release,
};

/* +++++++++++++ End File operations ++++++++++++++*/

static struct miscdevice xxs1500_ts_dev = {
	.minor = TS_MINOR,
	.name = TS_NAME,
	.fops = &ts_fops,
};

int __init
xxs1500_tpanel_init_module(void)
{
	struct xxs1500_tpanel_t *ts = &xxs1500_tpanel;
	int ret;
	struct pci_dev *dev = NULL;
#ifdef CONFIG_TI1500_TPANEL_INT
	struct pci_dev *func0dev = NULL;
#endif				//CONFIG_TI1500_TPANEL_INT
	unsigned long flags;
	unsigned int tmp = 0;
	if (!pci_present())
		return -ENODEV;
	/*Search for the device */
	dev =
	    pci_find_device(PCI_VENDOR_ID_AMM_052, PCI_DEVISE_ID_AMM_052_FUNC3,
			    dev);
	fpga_ver = 052;
	if (!dev) {
		dev =
		    pci_find_device(PCI_VENDOR_ID_AMM_050,
				    PCI_DEVICE_ID_AMM_050, dev);
		if (!dev) {
			printk(KERN_CRIT "%s: Device NOT found\n",
			       __FUNCTION__);
			return -1;
		}
		dev =
		    pci_find_device(PCI_VENDOR_ID_AMM_050,
				    PCI_DEVICE_ID_AMM_050, dev);
		dev =
		    pci_find_device(PCI_VENDOR_ID_AMM_050,
				    PCI_DEVICE_ID_AMM_050, dev);
		dev =
		    pci_find_device(PCI_VENDOR_ID_AMM_050,
				    PCI_DEVICE_ID_AMM_050, dev);
		fpga_ver = 050;
	}
	/*Enable PCI Device */
	if (pci_enable_device(dev)) {
		printk(KERN_CRIT "%s: PCI tpanel dev enable failed\n",
		       __FUNCTION__);
		return -1;
	}
	/*Read the mapped memory space address */
	tmp = pci_resource_start(dev, 0);
	/* register our character device */
	if ((ret = misc_register(&xxs1500_ts_dev)) < 0) {
		printk(KERN_CRIT "%s: can't get major number\n", __FUNCTION__);
		return ret;
	}
	memset(ts, 0, sizeof (struct xxs1500_tpanel_t));
	init_waitqueue_head(&ts->wait);
	spin_lock_init(&ts->lock);
	/* Map the different memory space regions for this function 
	 * into kernel mem space*/
	tmp = (unsigned long) ioremap(tmp, TP_LENGTH);
	if (!tmp) {
		printk(KERN_CRIT "%s: cannot map memory\n", __FUNCTION__);
		return -1;
	}
	if (check_mem_region(tmp, TP_LENGTH)) {
		printk(KERN_CRIT "%s: memory already in use\n", __FUNCTION__);
		return -EBUSY;
	}
	if (!request_mem_region(tmp, TP_LENGTH, TS_NAME)) {
		iounmap((char *) tmp);
		printk(KERN_CRIT "%s: cannot request memory\n", __FUNCTION__);
		return -1;
	}
	ts->tp_base = (unsigned long) tmp;
#ifdef CONFIG_TI1500_TPANEL_INT
	if (pci_read_config_byte(dev, PCI_INTERRUPT_LINE, &ts->pendown_irq)) {	/* deal with error */
		printk(KERN_INFO "%s: can't get assigned irq %i\n",
		       __FUNCTION__, ts->pendown_irq);
		ts->pendown_irq = -1;
	}
	ts->pendown_irq = irq;	// got at loading, default irq=2
	ts->irq_enabled = 1;
	if (request_irq
	    (ts->pendown_irq, xxs1500_tpanel_interrupt, SA_SHIRQ, TS_NAME,
	     ts)) {
		printk(KERN_INFO "%s: can't get assigned irq %i\n",
		       __FUNCTION__, ts->pendown_irq);
		ts->pendown_irq = -1;
	} else {		/* actually enable it */
		tmp = readl(ts->tp_base + TP_MASK);
		tmp |= TP_MASK_PEN_EN | TP_MASK_CC_EN;
		writel(tmp, ts->tp_base + TP_MASK);
	}
	func0dev =
	    pci_find_device(PCI_VENDOR_ID_AMM_052, PCI_DEVISE_ID_AMM_052_FUNC0,
			    func0dev);
	if (!func0dev) {
		printk(KERN_CRIT "%s: func0 device NOT found\n", __FUNCTION__);
		return -1;
	}
	pci_enable_device(func0dev);
	ts->intctl_base = pci_resource_start(func0dev, 0);
	ts->intctl_iolen = pci_resource_len(func0dev, 0);
	tmp = 0;
	tmp = (unsigned long) ioremap(ts->intctl_base, ts->intctl_iolen);
	if (!tmp) {
		printk(KERN_CRIT "%s: func0 device - cannot map memory\n",
		       __FUNCTION__);
		return -1;
	}
	if (check_mem_region(tmp, ts->intctl_iolen)) {
		printk(KERN_CRIT "%s: func0 device - memory already in use\n",
		       __FUNCTION__);
		return -EBUSY;
	}
	if (!request_mem_region(tmp, ts->intctl_iolen, "ti1500_tpanel_func0")) {
		iounmap((char *) tmp);
		printk(KERN_CRIT "%s: func0 device - cannot request memory\n",
		       __FUNCTION__);
		return -1;
	}
	ts->intctl_io = tmp;
	// enable PCI interrupts in function unit #0
	writel(0x8, ts->intctl_io + FUNC0_INT_MSET);
#endif				//CONFIG_TI1500_TPANEL_INT
	ts->pressure_eqn = PRESSURE_EQN_1;
	// init X and Y plate resistances
	ts->x_plate_ohms = DEFAULT_X_PLATE_OHMS;
	ts->y_plate_ohms = DEFAULT_Y_PLATE_OHMS;
	spin_lock_irqsave(&ts->lock, flags);
	ts->pen_is_down = 0;	// start with pen up
	// flush event queue
	ts->nextIn = ts->nextOut = ts->event_count = 0;
	// Init acquisition timer function
	init_timer(&ts->acq_timer);
	ts->acq_timer.function = xxs1500_tpanel_acq_timer;
	ts->acq_timer.data = (unsigned long) ts;
	spin_unlock_irqrestore(&ts->lock, flags);
#ifndef CONFIG_TI1500_TPANEL_INT
	ts->pen_is_down = 1;	// start with pen down
	ts->pen_was_down = 0;
	// start acquire
	ts->acq_timer.expires = jiffies + HZ / 100;
	add_timer(&ts->acq_timer);
#endif				//no CONFIG_TI1500_TPANEL_INT
	return 0;
}

void
xxs1500_tpanel_cleanup_module(void)
{
	struct xxs1500_tpanel_t *ts = &xxs1500_tpanel;
	/* Free interrupt */
	del_timer_sync(&ts->acq_timer);
#ifdef CONFIG_TI1500_TPANEL_INT
	writel(0x0, ts->tp_base + TP_MASK);
	writel(0x8, ts->intctl_io + FUNC0_INT_MRES);
	free_irq(ts->pendown_irq, ts);
	/*Free the mapped memory regions */
	release_mem_region(ts->intctl_io, ts->intctl_iolen);
	iounmap((unsigned long *) ts->intctl_io);
#endif				//CONFIG_TI1500_TPANEL_INT
	release_mem_region((unsigned long) ts->tp_base, TP_LENGTH);
	iounmap((unsigned long *) ts->tp_base);
	misc_deregister(&xxs1500_ts_dev);
}

EXPORT_NO_SYMBOLS;
/* Module information */
MODULE_AUTHOR("MontaVista Software, Inc. <source@mvista.com>");
MODULE_DESCRIPTION("XXS1500/AD7873 Touch Panel Driver");
MODULE_LICENSE("GPL");

module_init(xxs1500_tpanel_init_module);
module_exit(xxs1500_tpanel_cleanup_module);
