/*
 * drivers/mtd/maps/fujitsu_camelot.c
 *
 * Flash memory access on Fujitsu Camelot boards
 *
 * I managed to make this more complex than it needs to be by originally
 * doing complex mappings and then switching later to the simple mapping.
 * Brad Parker <brad@heeltoe.com>
 *
 * Author: <source@mvista.com>
 *
 * 2003 (c) MontaVista Software, Inc. This file is licensed under
 * the terms of the GNU General Public License version 2. This program
 * is licensed "as is" without any warranty of any kind, whether express
 * or implied.
 */


#include <linux/config.h>
#include <linux/module.h>
#include <linux/types.h>
#include <linux/kernel.h>
#include <linux/init.h>

#include <linux/mtd/mtd.h>
#include <linux/mtd/map.h>
#include <linux/mtd/partitions.h>

#include <asm/io.h>

#ifdef 	DEBUG_RW
#define	DBG(x...)	printk(x)
#else
#define	DBG(x...)	
#endif

#ifdef CONFIG_MACH_FUJITSU_CAMELOT
#define WINDOW_ADDR 0x00000000
#define WINDOW_SIZE 0x200000
#define BUSWIDTH    2
#define USE_COMPLEX_MAPPING
#endif

#ifdef USE_COMPLEX_MAPPING

#ifndef CONFIG_MTD_COMPLEX_MAPPINGS
#error need to define CONFIG_MTD_COMPLEX_MAPPINGS!
#endif

__u8 physmap_read8(struct map_info *map, unsigned long ofs)
{
	__u8 ret;
	ret = __raw_readb(map->virt + ofs);
	DBG("read8 from %x, %x\n", (unsigned)(map->virt + ofs), ret);
	return ret;
}

__u16 physmap_read16(struct map_info *map, unsigned long ofs)
{
	__u16 ret;
	ret = __raw_readw(map->virt + ofs);
	DBG("read16 from %x, %x\n", (unsigned)(map->virt + ofs), ret);
	return ret;
}

__u32 physmap_read32(struct map_info *map, unsigned long ofs)
{
	__u32 ret;
	ret = __raw_readl(map->virt + ofs);
	DBG("read32 from %x, %x\n", (unsigned)(map->virt + ofs), ret);
	return ret;
}

void physmap_copy_from(struct map_info *map, void *to, unsigned long from, ssize_t len)
{
	DBG("physmap_copy from %x to %x\n", (unsigned)from, (unsigned)to);
	memcpy_fromio(to, map->virt + from, len);
}

void physmap_write8(struct map_info *map, __u8 d, unsigned long adr)
{
	DBG("write8 at %x, %x\n", (unsigned)(map->virt + adr), d);
	__raw_writeb(d, map->virt + adr);
	mb();
}

void physmap_write16(struct map_info *map, __u16 d, unsigned long adr)
{
	DBG("write16 at %x, %x\n", (unsigned)(map->virt + adr), d);
	__raw_writew(d, map->virt + adr);
	mb();
}

void physmap_write32(struct map_info *map, __u32 d, unsigned long adr)
{
	DBG("write32 at %x, %x\n", (unsigned)(map->virt + adr), d);
	__raw_writel(d, map->virt + adr);
	mb();
}

void physmap_copy_to(struct map_info *map, unsigned long to, const void *from, ssize_t len)
{
	DBG("physmap_copy_to %x from %x\n", (unsigned)to, (unsigned)from);
	memcpy_toio(map->virt + to, from, len);
}

#endif

static struct map_info fujitsu_camelot_map = {
	.name = "Fujitsu Camelot flash",
	.size = WINDOW_SIZE,
	.buswidth = BUSWIDTH,
	.phys = WINDOW_ADDR,

#ifdef USE_COMPLEX_MAPPING
	read8: physmap_read8,
	read16: physmap_read16,
	read32: physmap_read32,
	copy_from: physmap_copy_from,
	write8: physmap_write8,
	write16: physmap_write16,
	write32: physmap_write32,
	copy_to: physmap_copy_to,
#endif
};

#ifdef CONFIG_MACH_FUJITSU_CAMELOT

//static unsigned long flash_size = 0x00200000;
//static unsigned char flash_buswidth = 2;
static struct mtd_partition fujitsu_camelot_partitions[] = {
        {
                name: "boot",
                size: 0x00100000,
                offset: 0,
//                mask_flags: MTD_WRITEABLE
        },{
                name: "kernel",
                size: 0x00100000,
                offset: 0x00100000
        }
};

#else
#error Unsupported board
#endif

#define NB_OF(x)  (sizeof(x)/sizeof(x[0]))

static struct mtd_info *mymtd;

int __init fujitsu_camelot_mtd_init(void)
{
	struct mtd_partition *parts;
	int nb_parts = 0;
	
	/*
	 * Static partition definition selection
	 */
	parts = fujitsu_camelot_partitions;
	nb_parts = NB_OF(fujitsu_camelot_partitions);

	/*
	 * Now let's probe for the actual flash.  Do it here since
	 * specific machine settings might have been set above.
	 */
	printk(KERN_NOTICE "Fujitsu_Camelot flash: probing %d-bit flash\n", 
	       fujitsu_camelot_map.buswidth*8);

	fujitsu_camelot_map.virt = 
		(unsigned long)ioremap(WINDOW_ADDR, WINDOW_SIZE);

#ifndef USE_COMPLEX_MAPPING
	simple_map_init(&fujitsu_camelot_map);
#endif

	mymtd = do_map_probe("cfi_probe", &fujitsu_camelot_map);
	if (!mymtd) {
		iounmap((void *)fujitsu_camelot_map.virt);
		return -ENXIO;
	}

	mymtd->owner = THIS_MODULE;

#ifdef CONFIG_MTD_PARTITIONS
	add_mtd_partitions(mymtd, parts, nb_parts);
#endif

	return 0;
}

static void __exit fujitsu_camelot_mtd_cleanup(void)
{
	if (mymtd) {
		del_mtd_partitions(mymtd);
		map_destroy(mymtd);
	}
	if (fujitsu_camelot_map.virt) {
		iounmap((void *)fujitsu_camelot_map.virt);
		fujitsu_camelot_map.virt = 0;
	}

}

module_init(fujitsu_camelot_mtd_init);
module_exit(fujitsu_camelot_mtd_cleanup);


MODULE_AUTHOR("Brad Parker");
MODULE_DESCRIPTION("Fujitsu_Camelot CFI map driver");
MODULE_LICENSE("GPL");
