/*
 * $Id: jmr3927-flash.c,v 1.1.2.1 2003/10/16 21:07:30 ahennessy Exp $
 *
 * Mapping for the Toshiba jmr3927 board
 *
 * Basically physmap.c with the addition of partitions and 
 * an array of mapping info to accomodate more than one flash type per board.
 *
 * Author: source@mvista.com
 *
 * 2003 (c) MontaVista Software, Inc. This file is licensed under the
 * terms of the GNU General Public License version 2. This program is
 * licensed "as is" without any warranty of any kind, whether express
 * or implied.
 */

#include <linux/module.h>
#include <linux/types.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <asm/io.h>
#include <linux/mtd/mtd.h>
#include <linux/mtd/map.h>
#include <linux/mtd/partitions.h>
#include <linux/config.h>
#include <linux/delay.h>

u8
jmr3927_flash_read8(struct map_info *map, unsigned long ofs)
{
	return *(u8 *) (map->virt + ofs);
}

u16
jmr3927_flash_read16(struct map_info * map, unsigned long ofs)
{
	return *(u16 *) (map->virt + ofs);
}

u32
jmr3927_flash_read32(struct map_info * map, unsigned long ofs)
{
	return *(u32 *) (map->virt + ofs);
}

void
jmr3927_flash_copy_from(struct map_info *map, void *to, unsigned long from,
			ssize_t len)
{
	memcpy_fromio(to, map->virt + from, len);
}

void
jmr3927_flash_write8(struct map_info *map, unsigned char d, unsigned long adr)
{
	*(u8 *) (map->virt + adr) = d;
}

void
jmr3927_flash_write16(struct map_info *map, unsigned short d, unsigned long adr)
{
	*(u16 *) (map->virt + adr) = d;
}

void
jmr3927_flash_write32(struct map_info *map, unsigned long d, unsigned long adr)
{
	*(u32 *) (map->virt + adr) = d;
}

void
jmr3927_flash_copy_to(struct map_info *map, unsigned long to, const void *from,
		      ssize_t len)
{
	memcpy_toio(map->virt + to, from, len);
}

const struct map_info basic_jmr3927_flash_map = {
	name:NULL,
	size:0,
	buswidth:0,
	read8:jmr3927_flash_read8,
	read16:jmr3927_flash_read16,
	read32:jmr3927_flash_read32,
	copy_from:jmr3927_flash_copy_from,
	write8:jmr3927_flash_write8,
	write16:jmr3927_flash_write16,
	write32:jmr3927_flash_write32,
	copy_to:jmr3927_flash_copy_to,
	virt:0,
	map_priv_2:0
};

/* board and partition description */

#define MAX_PHYSMAP_PARTITIONS    8
struct jmr3927_flash_info {
	char *name;
	unsigned long window_addr;
	unsigned long window_size;
	int buswidth;
	int num_partitions;
};

#define PHYSMAP_NUMBER  1	// number of board desc structs needed, one per contiguous flash type
const struct jmr3927_flash_info jmr3927_flash_board_desc[PHYSMAP_NUMBER] = {
	{
	 "MTD flash",		// name
	 0x1e000000,		// window_addr
	 0x00400000,		// window_size
	 4,			// buswidth
	 1,			// num_partitions
	 },

};
static struct mtd_partition
    jmr3927_flash_partitions[PHYSMAP_NUMBER][MAX_PHYSMAP_PARTITIONS] = {
	{
	 {
	  .name = "User FS",
	  .size = 0x00400000,
	  .offset = 0,
	  },
	 },
};

struct map_info jmr3927_flash_map[PHYSMAP_NUMBER];

int __init
init_jmr3927_flash(void)
{
	int i;
	struct mtd_info *mymtd;
	struct mtd_partition *parts;

	/* Initialize mapping */
	for (i = 0; i < PHYSMAP_NUMBER; i++) {
		printk(KERN_NOTICE
		       "jmr3927_flash flash device: 0x%lx at 0x%lx\n",
		       jmr3927_flash_board_desc[i].window_size,
		       jmr3927_flash_board_desc[i].window_addr);

		memcpy((char *) &jmr3927_flash_map[i],
		       (char *) &basic_jmr3927_flash_map,
		       sizeof (struct map_info));

		jmr3927_flash_map[i].phys =
		    jmr3927_flash_board_desc[i].window_addr;
		jmr3927_flash_map[i].virt =
		    (unsigned long) ioremap(jmr3927_flash_board_desc[i].
					    window_addr,
					    jmr3927_flash_board_desc[i].
					    window_size);
		if (!jmr3927_flash_map[i].virt) {
			printk(KERN_WARNING "Failed to ioremap\n");
			return -EIO;
		}
		jmr3927_flash_map[i].name = jmr3927_flash_board_desc[i].name;
		jmr3927_flash_map[i].size =
		    jmr3927_flash_board_desc[i].window_size;
		jmr3927_flash_map[i].buswidth =
		    jmr3927_flash_board_desc[i].buswidth;
		//printk(KERN_NOTICE "jmr3927_flash: ioremap is %x\n",(unsigned int)(jmr3927_flash_map[i].virt));
	}

	for (i = 0; i < PHYSMAP_NUMBER; i++) {
		parts = &jmr3927_flash_partitions[i][0];
		mymtd =
		    (struct mtd_info *) do_map_probe("cfi_probe",
						     &jmr3927_flash_map[i]);
		//printk(KERN_NOTICE "phymap %d cfi_probe: mymtd is %x\n",i,(unsigned int)mymtd);
		if (mymtd) {
			mymtd->owner = THIS_MODULE;

			jmr3927_flash_map[i].map_priv_2 = (unsigned long) mymtd;
			add_mtd_partitions(mymtd, parts,
					   jmr3927_flash_board_desc[i].
					   num_partitions);
		} else
			return -ENXIO;
	}
	return 0;
}

static void __exit
cleanup_jmr3927_flash(void)
{
	int i;
	struct mtd_info *mymtd;

	for (i = 0; i < PHYSMAP_NUMBER; i++) {
		mymtd = (struct mtd_info *) jmr3927_flash_map[i].map_priv_2;
		if (mymtd) {
			del_mtd_partitions(mymtd);
			map_destroy(mymtd);
		}
		if (jmr3927_flash_map[i].virt) {
			iounmap((void *) jmr3927_flash_map[i].virt);
			jmr3927_flash_map[i].virt = 0;
		}
	}
}

module_init(init_jmr3927_flash);
module_exit(cleanup_jmr3927_flash);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("Alice Hennessy <ahennessy@mvista.com>");
MODULE_DESCRIPTION("MTD map driver for Toshiba JMR3927 board");
