/*
 * Mapping for K2 user flash
 *
 * Author: Matt Porter <mporter@mvista.com>
 *
 * 2002 (c) MontaVista Software, Inc. This file is licensed under
 * the terms of the GNU General Public License version 2. This program
 * is licensed "as is" without any warranty of any kind, whether express
 * or implied.
 */

#include <linux/module.h>
#include <linux/init.h>
#include <linux/types.h>
#include <linux/kernel.h>
#include <linux/mtd/mtd.h>
#include <linux/mtd/map.h>
#include <linux/mtd/partitions.h>
#include <linux/config.h>
#include <asm/io.h>

static struct mtd_info *flash;

static struct map_info k2_map = {
	name: "K2 flash",
	size: 0x200000,
	buswidth: 1,
};

static struct mtd_partition k2_partitions[] = {
	{
		name: "Partition 0",
		offset: 0,
		size: (1024*1024)
	},
	{
		name: "Partition 1",
		offset: MTDPART_OFS_NXTBLK,
		size: (1024*1024)
	}
};

#define NB_OF(x)  (sizeof(x)/sizeof(x[0]))

int __init init_k2(void)
{
	k2_map.map_priv_1 =
		(unsigned long)ioremap(0xffe00000, k2_map.size);

	if (!k2_map.map_priv_1) {
		printk("Failed to ioremap flash\n");
		return -EIO;
	}

	simple_map_init(&k2_map);

	flash = do_map_probe("cfi_probe", &k2_map);
	if (flash) {
		flash->owner = THIS_MODULE;
		add_mtd_partitions(flash, k2_partitions,
					NB_OF(k2_partitions));
	} else {
		printk("map probe failed for flash\n");
		return -ENXIO;
	}

	return 0;
}

static void __exit cleanup_k2(void)
{
	if (flash) {
		del_mtd_partitions(flash);
		map_destroy(flash);
	}

	if (k2_map.map_priv_1) {
		iounmap((void *)k2_map.map_priv_1);
		k2_map.map_priv_1 = 0;
	}
}

module_init(init_k2);
module_exit(cleanup_k2);


MODULE_LICENSE("GPL");
MODULE_DESCRIPTION("K2 flash map");
