/*
 * Flash memory access on MyCable XXS1500 board
 * 
 * (C) 2003 Pete Popov <ppopov@mvista.com>
 * 
 * 2003 (c) MontaVista, Software, Inc. This file is licensed under
 * the terms of the GNU General Public License version 2.1. This program
 * is licensed "as is" without any warranty of any kind, whether express
 * or implied.
 */

#include <linux/config.h>
#include <linux/module.h>
#include <linux/types.h>
#include <linux/init.h>
#include <linux/kernel.h>

#include <linux/mtd/mtd.h>
#include <linux/mtd/map.h>
#include <linux/mtd/partitions.h>

#include <asm/io.h>
#include <asm/au1000.h>

#ifdef 	DEBUG_RW
#define	DBG(x...)	printk(x)
#else
#define	DBG(x...)	
#endif

#ifdef CONFIG_MIPS_XXS1500
#define WINDOW_ADDR 0x1F000000
#define WINDOW_SIZE 0x1000000
#endif

#define NAME "XXS1500 flash"

static struct map_info xxs1500_map = {
	.name		= NAME,
};


static unsigned long flash_size = 0x00800000;
static unsigned char flash_buswidth = 4;
static struct mtd_partition xxs1500_partitions[] = {
        {
                .name	= "kernel image",
                .size	= 0x00200000,
                .offset	= 0,
        },{
                .name	= "user fs 0",
                .size	= (0x00C00000-0x200000),
                .offset	= MTDPART_OFS_APPEND,
        },{
                .name	= "yamon",
                .size	= 0x00100000,
                .offset	= MTDPART_OFS_APPEND,
                .mask_flags	= MTD_WRITEABLE,
        },{
                .name	= "user fs 1",
                .size	= 0x2c0000,
                .offset	= MTDPART_OFS_APPEND,
        },{
                .name	= "yamon env vars",
                .size	= 0x040000,
                .offset	= MTDPART_OFS_APPEND,
                .mask_flags	= MTD_WRITEABLE,
        }
};


#define NB_OF(x)  (sizeof(x)/sizeof(x[0]))

static struct mtd_partition *parsed_parts;
static struct mtd_info *mymtd;

int __init xxs1500_mtd_init(void)
{
	struct mtd_partition *parts;
	int nb_parts = 0;
	char *part_type;
	
	/* Default flash buswidth */
	xxs1500_map.buswidth = flash_buswidth;

	/*
	 * Static partition definition selection
	 */
	part_type = "static";
	parts = xxs1500_partitions;
	nb_parts = NB_OF(xxs1500_partitions);
	xxs1500_map.size = flash_size;

	/*
	 * Now let's probe for the actual flash.  Do it here since
	 * specific machine settings might have been set above.
	 */
	printk(KERN_NOTICE "XXS1500 flash: probing %d-bit flash bus\n", 
			xxs1500_map.buswidth*8);
	xxs1500_map.virt = 
		(unsigned long)ioremap(WINDOW_ADDR, WINDOW_SIZE);
	mymtd = do_map_probe("cfi_probe", &xxs1500_map);
	if (!mymtd) return -ENXIO;
	mymtd->owner = THIS_MODULE;

	add_mtd_partitions(mymtd, parts, nb_parts);
	return 0;
}

static void __exit xxs1500_mtd_cleanup(void)
{
	if (mymtd) {
		del_mtd_partitions(mymtd);
		map_destroy(mymtd);
		if (parsed_parts)
			kfree(parsed_parts);
	}
}

module_init(xxs1500_mtd_init);
module_exit(xxs1500_mtd_cleanup);

MODULE_AUTHOR("Pete Popov");
MODULE_DESCRIPTION("XXS1500 CFI map driver");
MODULE_LICENSE("GPL");
