/*
 *  linux/drivers/net/irda/mx1_ir.c
 *
 *  Infra-red SIR and driver for the MX1 embedded microprocessors
 *  Based on linux/drivers/net/irda/pxa_ir.c
 *
 *  Author: MontaVista Software, Inc. <source@mvista.com>
 *  2003 (c) MontaVista Software, Inc. This file is licensed under
 *  the terms of the GNU General Public License version 2. This program
 *  is licensed "as is" without any warranty of any kind, whether express
 *  or implied.
 *
 */

#undef CONFIG_PM

#include <linux/config.h>
#include <linux/module.h>
#include <linux/types.h>
#include <linux/init.h>
#include <linux/errno.h>
#include <linux/netdevice.h>
#include <linux/slab.h>
#include <linux/rtnetlink.h>
#include <linux/interrupt.h>
#include <linux/delay.h>
#include <linux/ioport.h>
#include <linux/delay.h>

#include <linux/pm.h>

#include <net/irda/irda.h>
#include <net/irda/irmod.h>
#include <net/irda/wrapper.h>
#include <net/irda/irda_device.h>

#include <asm/irq.h>
#include <asm/dma.h>
#include <asm/hardware.h>
#include <asm/mach-types.h>
#include <asm/arch/platform.h>
#include <asm/arch/mx1ads-gpio.h>

#define TX_LVL   2
#define RX_LVL   1

#define _reg_PCDR		(*((volatile unsigned long *)(IO_ADDRESS(0x21B020))))
#define _reg_UART2_URXD		(*((volatile unsigned long *)(UART2_BASE + UART_RXDATA)))
#define _reg_UART2_UTXD		(*((volatile unsigned long *)(UART2_BASE + UART_TXDATA)))
#define _reg_UART2_UCR1		(*((volatile unsigned long *)(UART2_BASE + UART_CR1)))
#define _reg_UART2_UCR2		(*((volatile unsigned long *)(UART2_BASE + UART_CR2)))
#define _reg_UART2_UCR3		(*((volatile unsigned long *)(UART2_BASE + UART_CR3)))
#define _reg_UART2_UCR4		(*((volatile unsigned long *)(UART2_BASE + UART_CR4)))
#define _reg_UART2_UFCR		(*((volatile unsigned long *)(UART2_BASE + UART_UFCR)))
#define _reg_UART2_USR1		(*((volatile unsigned long *)(UART2_BASE + UART_SR1)))
#define _reg_UART2_USR2		(*((volatile unsigned long *)(UART2_BASE + UART_SR2)))
#define _reg_UART2_UBIR		(*((volatile unsigned long *)(UART2_BASE + UART_BIR)))
#define _reg_UART2_UBMR		(*((volatile unsigned long *)(UART2_BASE + UART_BMR)))
#define  PLL_FREQUENCY               96000000

static int rx_count = 0;
static int tx_count = 0;

/*
 * Our netdevice.  There is only ever one of these.
 */

static struct net_device *netdev;

struct mx1_irda {

	unsigned char open;

	int speed;
	int newspeed;

	struct sk_buff *txskb;
	struct sk_buff *rxskb;

	struct net_device_stats stats;
	struct irlap_cb *irlap;
	struct pm_dev *pmdev;
	struct qos_info qos;

	/* => SIR */
	iobuff_t tx_buff;
	iobuff_t rx_buff;
	/* <= SIR */
};

#define HPSIR_MAX_RXLEN		2050
#define HPSIR_MAX_TXLEN		2050
#define TXBUFF_MAX_SIZE		HPSIR_MAX_TXLEN
#define SET_SIR_MODE            STISR = STISR_RCVEIR | STISR_XMITIR | STISR_XMODE

/*
 * If you want to disable debug information
 * please uncomment line bellow
 */

//#define MX1_SIR_DEBUG_ENABLE
#undef MX1_SIR_DEBUG_ENABLE

//#define MX1_SIR_IRQ_DEBUG_ENABLE
#undef MX1_SIR_IRQ_DEBUG_ENABLE

#ifdef MX1_SIR_DEBUG_ENABLE
#define __ECHO_IN printk(KERN_ERR "%s: enter\n",__FUNCTION__);
#define __ECHO_OUT printk(KERN_ERR "%s: exit\n",__FUNCTION__);
#define DBG(args...) printk(KERN_ERR __FUNCTION__"():"args);
#else
#define __ECHO_IN
#define __ECHO_OUT
#define DBG(args...)
#endif

#ifdef MX1_SIR_IRQ_DEBUG_ENABLE
#define DBG_IRQ(args...) printk(KERN_ERR __FUNCTION__"():"args);
#else
#define DBG_IRQ(args...)
#endif

static int mx1_irda_set_speed(struct net_device *dev, int speed);

/*
 * Allocate the receive buffer, unless it is already allocated.
 */

static int
mx1_irda_rx_alloc(struct mx1_irda *si)
{
	__ECHO_IN;

	if (si->rxskb)
		return 0;

	si->rxskb = alloc_skb(HPSIR_MAX_RXLEN + 1, GFP_ATOMIC);

	if (!si->rxskb) {
		printk(KERN_ERR "mx1_ir: out of memory for RX SKB\n");
		return -ENOMEM;
	}

	/*
	 * Align any IP headers that may be contained
	 * within the frame.
	 */
	skb_reserve(si->rxskb, 1);

	__ECHO_OUT;

	return 0;
}

static void
mx1_irda_hpsir_irq(struct net_device *dev)
{
	struct mx1_irda *si = dev->priv;
	int rx;

	/*
	 * Deal with any receive errors first.  The bytes in error may be
	 * the only bytes in the receive FIFO, so we do this first.
	 */
	__ECHO_IN;

	while ((rx = _reg_UART2_URXD) & UART_CHARRDY) {
		DBG("receive_char=%x", rx);
		if (rx & (UART_FRMERR | UART_PRERR | UART_OVRRUN)) {
			si->stats.rx_errors++;
			if (rx & UART_FRMERR) {
				si->stats.rx_frame_errors++;
				printk(KERN_WARNING "frame error \n");
			}
			if (rx & UART_OVRRUN) {
				si->stats.rx_fifo_errors++;
				printk(KERN_WARNING "overrun error \n");
			}

		} else {
			rx_count++;
			async_unwrap_char(dev, &si->stats, &si->rx_buff,
					  (rx & UART_RXDATA_MASK));
		}

	}

	__ECHO_OUT;
}

static void
mx1_sir_irda_shutdown(struct mx1_irda *si)
{

}

/************************************************************************************/

/*Low level init/uninstall function PM control and IrDA protocol stack registration */

/*
 * Set the IrDA communications speed.
 * Interrupt have to be disabled here.
 */

static int
mx1_irda_startup(struct net_device *dev)
{
	int retval;
	int rx;

	__ECHO_IN;

	/*
	 * Ensure that the ports for this device are setup correctly.
	 */

	retval = mx1_register_gpios(PORT_B, (1 << 31) |	/* UART2: RXD */
				    (1 << 30) |	/* UART2: TXD */
				    (1 << 29) |	/* UART2: RTS */
				    (1 << 28),	/* UART2: CTS */
				    PRIMARY | NOINTERRUPT /* ,"uart2" */ );
	if (retval < 0)
		goto errorB;

	retval = mx1_register_gpios(PORT_C, (1 << 9) |	/* UART2: RXD */
				    (1 << 10) |	/* UART2: TXD */
				    (1 << 11) |	/* UART2: RTS */
				    (1 << 12),	/* UART2: CTS */
				    PRIMARY | NOINTERRUPT /* ,"uart2" */ );
	if (retval < 0)
		goto errorC;

	retval = mx1_register_gpios(PORT_D, (1 << 10) |	/* UART2: DSR */
				    (1 << 9) |	/* UART2: RI */
				    (1 << 8) |	/* UART2: DCD */
				    (1 << 7),	/* UART2: DTR */
				    SECONDARY /* "irda" */ );
	if (retval < 0)
		goto errorD;

	_reg_UART2_UCR1 |= (UART_UARTEN | UART_IREN);
	_reg_UART2_UCR1 |= UART_UARTCLKEN;
	_reg_UART2_UCR2 |= UART_IRTS;
	_reg_UART2_UCR2 |= UART_8BIT;
	_reg_UART2_UCR2 &= ~UART_CTSC;
	_reg_UART2_UCR2 &= ~(UART_TXEN);
	_reg_UART2_UCR3 &= ~UART_INVT;
	_reg_UART2_UCR4 &= ~UART_INVR;

	_reg_UART2_UCR4 = 0x8000 | UART_REF16;
	_reg_UART2_UFCR =
	    (TX_LVL << UART_TXTL_BIT) | (RX_LVL << UART_RXTL_BIT) |
	    UART_RFDIV_1;

	_reg_UART2_UCR2 |= UART_SRST;
	_reg_UART2_UCR4 |= UART_IRSC;

	_reg_UART2_UCR2 |= (UART_RXEN | UART_TXEN);
	_reg_UART2_UCR1 &= ~(UART_TRDYEN | UART_TXEMPTYEN);
	udelay(1000);
	while ((rx = _reg_UART2_URXD) & UART_CHARRDY) ;	/* clear RX FIFO */
	_reg_UART2_UCR4 |= UART_DREN;

	return 0;

	__ECHO_OUT;

      errorD:
	mx1_unregister_gpios(PORT_D,
			     (1 << 10) | (1 << 9) | (1 << 8) | (1 << 7));
      errorB:
	mx1_unregister_gpios(PORT_B,
			     (1 << 28) | (1 << 29) | (1 << 30) | (1 << 31));
      errorC:
	mx1_unregister_gpios(PORT_C,
			     (1 << 12) | (1 << 11) | (1 << 10) | (1 << 9));

	return retval;

}

#ifdef CONFIG_PM
/*
 * Suspend the IrDA interface.
 */

static int
mx1_irda_shutdown(struct mx1_irda *si)
{

	mx1_sir_irda_shutdown(si);
	return 0;

}

static int
mx1_irda_suspend(struct net_device *dev, int state)
{
	struct mx1_irda *si = dev->priv;

	if (si && si->open) {
		/*
		 * Stop the transmit queue
		 */

		netif_stop_queue(dev);
		disable_irq(dev->irq);
		mx1_sir_irda_shutdown(si);
	}

	return 0;
}

/*
 * Resume the IrDA interface.
 */

static int
mx1_irda_resume(struct net_device *dev)
{
	struct mx1_irda *si = dev->priv;

	__ECHO_IN;

	if (si && si->open) {
		/*
		 * If we missed a speed change, initialise at the new speed
		 * directly.  It is debatable whether this is actually
		 * required, but in the interests of continuing from where
		 * we left off it is desireable.  The converse argument is
		 * that we should re-negotiate at 9600 baud again.
		 */
		if (si->newspeed) {
			si->speed = si->newspeed;
			si->newspeed = 0;
		}

		mx1_irda_startup(dev);
		enable_irq(dev->irq);
		enable_irq(UART2_MINT_TX);

		/*
		 * This automatically wakes up the queue
		 */
		netif_wake_queue(dev);
		mx1_irda_set_speed(dev, si->speed = 9600);

	}

	__ECHO_OUT;
	return 0;
}

static int
mx1_irda_pmproc(struct pm_dev *dev, pm_request_t rqst, void *data)
{
	int ret;

	if (!dev->data)
		return -EINVAL;

	switch (rqst) {
	case PM_SUSPEND:
		ret = mx1_irda_suspend((struct net_device *) dev->data,
				       (int) data);
		break;

	case PM_RESUME:
		ret = mx1_irda_resume((struct net_device *) dev->data);
		break;

	default:

		ret = -EINVAL;
		break;
	}

	return ret;
}
#endif

static void
transmit_chars(int irq, void *dev_id, struct pt_regs *regs)
{
	struct net_device *dev = dev_id;
	struct mx1_irda *si = dev->priv;
	int rx;
	__ECHO_IN;
	if (si->tx_buff.len) {

		DBG(" %x\n", (unsigned int) *si->tx_buff.data);

		_reg_UART2_UTXD = (unsigned int) *si->tx_buff.data++;
		si->tx_buff.len -= 1;
		tx_count++;
	}

	if (si->tx_buff.len == 0) {

		DBG(" transmited\n");
		si->stats.tx_packets++;
		si->stats.tx_bytes += si->tx_buff.data - si->tx_buff.head;
		do {
			udelay(10);

		} while (!(_reg_UART2_USR2 & UART_TXDC));

		if (si->newspeed) {
			mx1_irda_set_speed(dev, si->newspeed);
			si->newspeed = 0;
		}
		netif_wake_queue(dev);
		DBG("Disable transmiter  and  enable receiver...");
		_reg_UART2_UCR1 &= ~(UART_TRDYEN | UART_TXEMPTYEN);
		while ((rx = _reg_UART2_URXD) & UART_CHARRDY) ;	/* clear RX FIFO */
		_reg_UART2_UCR4 |= UART_DREN;

	}

}

static void
mx1_irda_irq(int irq, void *dev_id, struct pt_regs *regs)
{
	struct net_device *dev = dev_id;

	mx1_irda_hpsir_irq(dev);

}

static int
mx1_irda_hard_xmit(struct sk_buff *skb, struct net_device *dev)
{
	struct mx1_irda *si = dev->priv;
	int speed;
	int mtt;

	__ECHO_IN;
	speed = irda_get_next_speed(skb);

	/*
	 * Does this packet contain a request to change the interface
	 * speed?  If so, remember it until we complete the transmission
	 * of this frame.
	 */
	if (speed != si->speed && speed != -1)
		si->newspeed = speed;

	/*
	 * If this is an empty frame, we can bypass a lot.
	 */
	if (skb->len == 0) {
		if (si->newspeed) {
			si->newspeed = 0;
			mx1_irda_set_speed(dev, speed);
		}
		dev_kfree_skb(skb);
		return 0;
	}

	DBG("stop queue\n");
	netif_stop_queue(dev);

	si->tx_buff.data = si->tx_buff.head;
	si->tx_buff.len = async_wrap_skb(skb, si->tx_buff.data,
					 si->tx_buff.truesize);

	DBG("Disable receiver and  enable transmiter\n");
	_reg_UART2_UCR4 &= ~UART_DREN;
	if ((mtt = irda_get_mtt(skb)) > 0) {
		udelay(mtt);
	}
	_reg_UART2_UCR1 |= (UART_TRDYEN | UART_TXEMPTYEN);

	dev_kfree_skb(skb);

	dev->trans_start = jiffies;

	return 0;

}

static int
mx1_irda_ioctl(struct net_device *dev, struct ifreq *ifreq, int cmd)
{
	struct if_irda_req *rq = (struct if_irda_req *) ifreq;
	struct mx1_irda *si = dev->priv;
	int ret = -EOPNOTSUPP;

	__ECHO_IN;

	switch (cmd) {
	case SIOCSBANDWIDTH:
		if (capable(CAP_NET_ADMIN)) {
			/*
			 * We are unable to set the speed if the
			 * device is not running.
			 */
			if (si->open) {
				ret = mx1_irda_set_speed(dev, rq->ifr_baudrate);
			} else {
				printk
				    ("mx1_irda_ioctl: SIOCSBANDWIDTH: !netif_running\n");
				ret = 0;
			}
		}
		break;

	case SIOCSMEDIABUSY:
		ret = -EPERM;
		if (capable(CAP_NET_ADMIN)) {
			irda_device_set_media_busy(dev, TRUE);
			ret = 0;
		}
		break;

	case SIOCGRECEIVING:
		rq->ifr_receiving = 0 ? 0 : si->rx_buff.state != OUTSIDE_FRAME;
		break;

	default:
		break;
	}

	__ECHO_OUT;

	return ret;
}

static struct net_device_stats *
mx1_irda_stats(struct net_device *dev)
{
	struct mx1_irda *si = dev->priv;
	return &si->stats;
}

static int
mx1_irda_start(struct net_device *dev)
{
	struct mx1_irda *si = dev->priv;
	int err;
	unsigned int flags;

	MOD_INC_USE_COUNT;

	__ECHO_IN;
	si->speed = 9600;

	local_irq_save(flags);

	err = request_irq(dev->irq, mx1_irda_irq, 0, dev->name, dev);
	if (err) {
		printk(KERN_ERR "Request IRQ  %d failed\n", mx1_irda_irq);
		goto err_irq;
	}

	err = request_irq(UART2_MINT_TX, transmit_chars, 0, dev->name, dev);
	if (err) {
		printk(KERN_ERR "Request IRQ  %d failed\n", UART2_MINT_TX);
		goto err_tx_irq;
	}

	/*
	 * The interrupt must remain disabled for now.
	 */

	disable_irq(dev->irq);

	local_irq_restore(flags);

	/* Alocate skb for receiver */
	err = mx1_irda_rx_alloc(si);
	if (err)
		goto err_rx_alloc;

	/*
	 * Setup the serial port for the specified config.
	 */
	err = mx1_irda_startup(dev);
	if (err)
		goto err_startup;

	mx1_irda_set_speed(dev, si->speed = 9600);

	/*
	 * Open a new IrLAP layer instance.
	 */
	si->irlap = irlap_open(dev, &si->qos, "mx1");
	err = -ENOMEM;
	if (!si->irlap)
		goto err_irlap;

	/*
	 * Now enable the interrupt and start the queue
	 */
	si->open = 1;
	enable_irq(dev->irq);
	enable_irq(UART2_MINT_TX);
	netif_start_queue(dev);
	return 0;

      err_irlap:
	si->open = 0;
	mx1_sir_irda_shutdown(si);
      err_startup:
	dev_kfree_skb(si->rxskb);
      err_rx_alloc:
      err_tx_irq:
	free_irq(dev->irq, dev);
      err_irq:
	MOD_DEC_USE_COUNT;
	return err;
}

static int
mx1_irda_stop(struct net_device *dev)
{
	struct mx1_irda *si = dev->priv;

	printk(KERN_ERR "Irda stop... RX = %d TX = %d\n", rx_count, tx_count);

	disable_irq(dev->irq);

	/*
	 * If we have been doing DMA receive, make sure we
	 * tidy that up cleanly.
	 */
	if (si->rxskb) {
		dev_kfree_skb(si->rxskb);
		si->rxskb = NULL;
	}

	/* Stop IrLAP */
	if (si->irlap) {
		irlap_close(si->irlap);
		si->irlap = NULL;
	}

	netif_stop_queue(dev);
	si->open = 0;

	/*
	 * Free resources
	 */
	free_irq(dev->irq, dev);
	free_irq(UART2_MINT_TX, dev);

	MOD_DEC_USE_COUNT;

	return 0;
}

static int
mx1_irda_init_iobuf(iobuff_t * io, int size)
{
	io->head = kmalloc(size, GFP_KERNEL | GFP_DMA);
	if (io->head != NULL) {
		io->truesize = size;
		io->in_frame = FALSE;
		io->state = OUTSIDE_FRAME;
		io->data = io->head;
	}
	return io->head ? 0 : -ENOMEM;
}

static int
mx1_irda_set_speed(struct net_device *dev, int speed)
{
	struct mx1_irda *si = dev->priv;
	int brd, ret = -EINVAL;
	unsigned long frequency;

	__ECHO_IN;

	frequency = (PLL_FREQUENCY / ((_reg_PCDR & 0xF) + 1)) / 10000;

	switch (speed) {

	case 9600:
	case 19200:
	case 38400:
	case 57600:
	case 115200:
		_reg_UART2_UBIR = ((speed * 16) / frequency) - 1;
		_reg_UART2_UBMR = 9999;
		si->speed = speed;
		ret = 0;
		break;

	case 4000000:
		break;

	default:
		break;
	}

	__ECHO_OUT;

	return ret;
}

static int
mx1_irda_net_init(struct net_device *dev)
{
	struct mx1_irda *si = dev->priv;
	unsigned int baudrate_mask;
	int err = -ENOMEM;

	si = kmalloc(sizeof (struct mx1_irda), GFP_KERNEL);
	if (!si)
		goto out;

	memset(si, 0, sizeof (*si));

	/*
	 * Initialise the HP-SIR buffers
	 */

	err = mx1_irda_init_iobuf(&si->rx_buff, 14384);
	if (err)
		goto out;
	err = mx1_irda_init_iobuf(&si->tx_buff, 4000);
	if (err)
		goto out_free_rx;

	dev->priv = si;
	dev->hard_start_xmit = mx1_irda_hard_xmit;
	dev->open = mx1_irda_start;
	dev->stop = mx1_irda_stop;
	dev->do_ioctl = mx1_irda_ioctl;
	dev->get_stats = mx1_irda_stats;

	irda_device_setup(dev);
	irda_init_max_qos_capabilies(&si->qos);

	/*
	 * We support original IRDA up to 115k2. (we don't currently
	 * support 4Mbps).  Min Turn Time set to 1ms or greater.
	 */
	baudrate_mask = IR_9600 | IR_19200 | IR_38400 | IR_57600 | IR_115200;
	si->qos.baud_rate.bits &= baudrate_mask;
	si->qos.min_turn_time.bits = 7;

	irda_qos_bits_to_value(&si->qos);

#ifdef CONFIG_PM
	/*
	 * Power-Management is optional.
	 */
	si->pmdev = pm_register(PM_SYS_DEV, PM_SYS_IRDA, mx1_irda_pmproc);
	if (si->pmdev)
		si->pmdev->data = dev;
#endif

	return 0;

	kfree(si->tx_buff.head);
      out_free_rx:
	kfree(si->rx_buff.head);
      out:
	kfree(si);

	return err;
}

/*
 * Remove all traces of this driver module from the kernel, so we can't be
 * called.  Note that the device has already been stopped, so we don't have
 * to worry about interrupts or dma.
 */
static void
mx1_irda_net_uninit(struct net_device *dev)
{
	struct mx1_irda *si = dev->priv;

	dev->hard_start_xmit = NULL;
	dev->open = NULL;
	dev->stop = NULL;
	dev->do_ioctl = NULL;
	dev->get_stats = NULL;
	dev->priv = NULL;

	pm_unregister(si->pmdev);

	kfree(si->tx_buff.head);
	kfree(si->rx_buff.head);
	kfree(si);
}

static int __init
mx1_irda_init(void)
{
	struct net_device *dev;
	int err;

	printk("IrDA Init.\n");

	rtnl_lock();
	dev = dev_alloc("irda%d", &err);
	if (dev) {
		dev->irq = UART2_MINT_RX;
		dev->init = mx1_irda_net_init;
		dev->uninit = mx1_irda_net_uninit;

		err = register_netdevice(dev);

		if (err)
			kfree(dev);
		else
			netdev = dev;
	}
	rtnl_unlock();
	return (0);
}

static void __exit
mx1_irda_exit(void)
{
	struct net_device *dev = netdev;

	_reg_UART2_UCR1 &= ~UART_IREN;
	_reg_UART2_UCR1 &= ~UART_UARTEN;

	mx1_unregister_gpios(PORT_C,
			     (1 << 12) | (1 << 11) | (1 << 10) | (1 << 9));
	mx1_unregister_gpios(PORT_B,
			     (1 << 31) | (1 << 30) | (1 << 29) | (1 << 28));
	mx1_unregister_gpios(PORT_D,
			     (1 << 10) | (1 << 9) | (1 << 8) | (1 << 7));

	netdev = NULL;
	if (dev) {
		rtnl_lock();
		unregister_netdevice(dev);
		rtnl_unlock();
	}

	/*
	 * We now know that the netdevice is no longer in use, and all
	 * references to our driver have been removed.  The only structure
	 * which may still be present is the netdevice, which will get
	 * cleaned up by net/core/dev.c
	 */
}

module_init(mx1_irda_init);
module_exit(mx1_irda_exit);

MODULE_DESCRIPTION("MX1 SIR");
MODULE_LICENSE("GPL");
EXPORT_NO_SYMBOLS;
