/*
 * drivers/pcmcia/vr4181a_ecu.c
 *
 * CompactFlash/PC Card Control Unit driver for NEC VR4181A ECU
 *
 * Author: Yoichi Yuasa <yyuasa@mvista.com, or source@mvista.com>
 *
 * 2003 (c) MontaVista, Software, Inc. This file is licensed under
 * the terms of the GNU General Public License version 2. This program
 * is licensed "as is" without any warranty of any kind, whether express
 * or implied.
 */
#include <linux/init.h>
#include <linux/module.h>
#include <linux/proc_fs.h>

#include <pcmcia/cs_types.h>
#include <pcmcia/cs.h>
#include <pcmcia/cistpl.h>
#include <pcmcia/bulkmem.h>
#include <pcmcia/ss.h>

#include <asm/vr4181a/vr4181a_ecu.h>

#include "cs_internal.h"

extern struct socket_info_t *pcmcia_register_socket(int slot,
						    struct pccard_operations
						    *vtable, int use_bus_pm);
extern void pcmcia_unregister_socket(struct socket_info_t *s);

#define ECU_NR_SLOTS			2
#define ECU_IO_NR_MAPS			2
#define ECU_MEM_NR_MAPS			5

typedef struct ecu_socket {
	int slot;
	int irq;
	void (*handler) (void *, unsigned int);
	void *info;
	spinlock_t event_lock;
	unsigned int events;
	socket_info_t *pcmcia_socket;
	struct tq_struct tq_task;
	char *name;
} ecu_socket_t;

static ecu_socket_t ecu_sockets[ECU_NR_SLOTS] = {
	{
	 .irq = -1,
	 .name = "NEC VR4181A ECU Slot0",
	 },
	{
	 .irq = -1,
	 .name = "NEC VR4181A ECU Slot1",
	 },
};

static int
ecu_init(unsigned int sock)
{
	ecu_socket_t *socket;

	if (sock >= ECU_NR_SLOTS)
		return -EINVAL;

	socket = &ecu_sockets[sock];
	socket->events = 0;
	spin_lock_init(socket->event_lock);

	return 0;
}

static int
ecu_suspend(unsigned int sock)
{
	return -EINVAL;
}

static int
ecu_register_callback(unsigned int sock,
		      void (*handler) (void *, unsigned int), void *info)
{
	ecu_socket_t *socket;

	if (sock >= ECU_NR_SLOTS)
		return -EINVAL;

	socket = &ecu_sockets[sock];

	socket->handler = handler;
	socket->info = info;

	if (handler == NULL)
		MOD_DEC_USE_COUNT;
	else
		MOD_INC_USE_COUNT;

	return 0;
}

static int
ecu_inquire_socket(unsigned int sock, socket_cap_t * cap)
{
	if (sock >= ECU_NR_SLOTS)
		return -EINVAL;

	cap->features = SS_CAP_PCCARD | SS_CAP_STATIC_MAP | SS_CAP_PAGE_REGS;
	cap->irq_mask = 0;
	cap->map_size = 0x1000;
	cap->io_offset = vr4181a_ecu_get_io_offset(sock);
	if (cap->io_offset != 0xffffffff)
		cap->io_offset -= PHYSADDR(mips_io_port_base);
	if (sock == 0)
		cap->pci_irq = GPIO_IRQ(CONFIG_IREQ0_GPIO_NO);
	else
		cap->pci_irq = GPIO_IRQ(CONFIG_IREQ1_GPIO_NO);

	return 0;
}

static int
ecu_get_status(unsigned int sock, u_int * value)
{
	uint8_t val;
	u_int status = 0;

	if (sock >= ECU_NR_SLOTS || value == NULL)
		return -EINVAL;

	val = exca_readb(sock, IF_STAT_REG);
	if ((val & POWER_STATUS) == POWER_ON)
		status |= SS_POWERON;
	if ((val & CARD_STATUS) == CARD_READY)
		status |= SS_READY;
	if ((val & WRITE_PROTECT) == WRITE_PROTECT_ON)
		status |= SS_WRPROT;
	if ((val & CARD_DETECT) == CARD_ACTIVE)
		status |= SS_DETECT;

	if ((exca_readb(sock, ITGENCTREG) & CARD_TYPE) == IO_CARD) {
		if ((val & STATUS_CHANGE) == STATUS_CHANGE_ACTIVE)
			val |= SS_STSCHG;
	} else {
		if ((val & BATTRY_VOLTAGE_DETECT) == BATTRY_VOLTAGE_DEAD)
			status |= SS_BATDEAD;
	}

	val = exca_readb(sock, VOLTSENREG);
	if ((val & VOLTAGE_SENSE) == VOLTAGE_SENSE_33V)
		status |= SS_3VCARD;

	*value = status;

	return 0;
}

static inline u_char
get_Vcc_value(uint8_t val)
{
	switch (val & VCC_SELECT) {
	case VCC_33V:
		return 33;
	default:
		break;
	}

	return 0;
}

static int
ecu_get_socket(unsigned int sock, socket_state_t * state)
{
	uint8_t val;

	if (sock >= ECU_NR_SLOTS || state == NULL)
		return -EINVAL;

	val = exca_readb(sock, VOLTSELREG);
	state->Vcc = state->Vpp = get_Vcc_value(val);
	state->flags = 0;

	val = exca_readb(sock, PWRRSETDRV);
	if ((val & POWER_OUTPUT) == POWER_OUTPUT_ENABLE)
		state->flags |= SS_OUTPUT_ENA;

	val = exca_readb(sock, ITGENCTREG);
	if ((val & CARD_RESET) == CARD_RESET_ACTIVE)
		state->flags |= SS_RESET;
	if ((val & CARD_TYPE) == IO_CARD)
		state->flags |= SS_IOCARD;

	state->csc_mask = 0;
	val = exca_readb(sock, CRDSTATREG);
	if (val & CARD_DETECT_ENABLE)
		state->csc_mask |= SS_DETECT;
	if (val & CARD_STATUS_ENABLE)
		state->csc_mask |= SS_READY;
	if (val & STATUS_CHANGE_ENABLE)
		state->csc_mask |= SS_STSCHG;

	return 0;
}

static int
ecu_set_socket(unsigned int sock, socket_state_t * state)
{
	ecu_socket_t *socket;
	uint8_t val;

	if (sock >= ECU_NR_SLOTS || state == NULL ||
	    (state->Vpp != 33 && state->Vpp != 0) ||
	    (state->Vcc != 33 && state->Vcc != 0))
		return -EINVAL;

	socket = &ecu_sockets[sock];

	spin_lock_irq(&socket->event_lock);
	if (state->Vcc == 33) {
		val = POWER_ENABLE;
		exca_writeb(sock, PWRRSETDRV, val);
		if (state->flags & SS_OUTPUT_ENA) {
			val |= POWER_OUTPUT_ENABLE;
			exca_writeb(sock, PWRRSETDRV, val);
		}
	} else
		exca_writeb(sock, PWRRSETDRV, 0);

	val = exca_readb(sock, ITGENCTREG);
	val &= ~(CARD_RESET | CARD_TYPE);
	if (state->flags & SS_RESET)
		val |= CARD_RESET_ACTIVE;
	else
		val |= CARD_RESET_INACTIVE;
	if (state->flags & SS_IOCARD)
		val |= IO_CARD;
	else
		val |= MEMORY_CARD;
	exca_writeb(sock, ITGENCTREG, val);

	val = 0;
	if (state->csc_mask & SS_DETECT)
		val |= CARD_DETECT_ENABLE;
	if (state->csc_mask & SS_READY)
		val |= CARD_STATUS_ENABLE;
	if (state->csc_mask & SS_STSCHG)
		val |= STATUS_CHANGE_ENABLE;
	exca_writeb(sock, CRDSTATREG, val);
	spin_unlock_irq(&socket->event_lock);

	return 0;
}

static int
ecu_get_io_map(unsigned int sock, struct pccard_io_map *io)
{
	uint8_t ioctrl, adwinen;
	u_char map;

	if (sock >= ECU_NR_SLOTS || io == NULL || io->map > 1)
		return -EINVAL;

	map = io->map;

	io->start = exca_readw(sock, IOADSLBREG(map));
	io->stop = exca_readw(sock, IOSLBREG(map));

	io->flags = 0;
	adwinen = exca_readb(sock, ADWINENREG);
	if (adwinen & IO_WINDOW_ENABLE(map))
		io->flags |= MAP_ACTIVE;

	ioctrl = exca_readb(sock, IOCTRL_REG);
	if ((ioctrl & IO_WINDOW_DATA_WIDTH(map)) ==
	    IO_WINDOW_DATA_WIDTH_WITH_SIGNAL(map))
		io->flags |= MAP_AUTOSZ;
	if ((ioctrl & IO_WINDOW_DATA_WIDTH(map)) ==
	    IO_WINDOW_DATA_WIDTH_16BIT(map))
		io->flags |= MAP_16BIT;
	if ((ioctrl & IO_WINDOW_ACCESS(map)) == IO_WINDOW_ACCESS_NO_WAIT(map))
		io->flags |= MAP_0WS;
	if ((ioctrl & IO_WINDOW_ACCESS(map)) == IO_WINDOW_ACCESS_ADD_WAIT(map))
		io->flags |= MAP_USE_WAIT;

	return 0;
}

static int
ecu_set_io_map(unsigned int sock, struct pccard_io_map *io)
{
	uint8_t ioctrl, adwinen;
	u_char map;

	if (sock >= ECU_NR_SLOTS || io == NULL || io->map >= ECU_IO_NR_MAPS)
		return -EINVAL;

	map = io->map;

	adwinen = exca_readb(sock, ADWINENREG);
	adwinen &= ~IO_WINDOW_ENABLE(map);
	exca_writeb(sock, ADWINENREG, adwinen);

	exca_writew(sock, IOADSLBREG(map), io->start);
	exca_writew(sock, IOSLBREG(map), io->stop);

	ioctrl = exca_readb(sock, IOCTRL_REG) & ~IO_WINDOW(map);
	if (io->flags & MAP_AUTOSZ)
		ioctrl |= IO_WINDOW_DATA_WIDTH_WITH_SIGNAL(map);
	if (io->flags & MAP_16BIT)
		ioctrl |= IO_WINDOW_DATA_WIDTH_16BIT(map);
	if (io->flags & MAP_USE_WAIT)
		ioctrl |= IO_WINDOW_ACCESS_ADD_WAIT(map);
	exca_writeb(sock, IOCTRL_REG, ioctrl);

	if (io->flags & MAP_ACTIVE)
		exca_writeb(sock, ADWINENREG, adwinen | IO_WINDOW_ENABLE(map));

	return 0;
}

static inline u_int
get_speed_value(unsigned int sock, u_char map)
{
	uint8_t val;
	u_int wait;

	val = exca_readb(sock, MEMSEL_REG(map)) & MEMORY_CARD_ACCESS_WAIT_VALUE;
	switch (val) {
	case MEMORY_CARD_ACCESS_ADD_4WAIT:
		wait = 4;
		break;
	case MEMORY_CARD_ACCESS_ADD_3WAIT:
		wait = 3;
		break;
	case MEMORY_CARD_ACCESS_ADD_2WAIT:
		wait = 2;
		break;
	case MEMORY_CARD_ACCESS_ADD_1WAIT:
	default:
		wait = 1;
		break;
	}

	return (wait * 1000000000) / vr4181a_get_ecu_sysclock();
}

static int
ecu_get_mem_map(unsigned int sock, struct pccard_mem_map *mem)
{
	u_long sys_start, sys_stop;
	uint8_t adwinen, memoffh;
	u_char map;

	if (sock >= ECU_NR_SLOTS || mem == NULL || mem->map >= ECU_MEM_NR_MAPS
	    || mem->sys_start > mem->sys_stop)
		return -EINVAL;

	map = mem->map;

	mem->flags = 0;
	adwinen = exca_readb(sock, ADWINENREG);
	if (adwinen & MEMORY_WINDOW_ENABLE(map))
		mem->flags |= MAP_ACTIVE;

	sys_start = exca_readb(sock, MEMWID_REG(map));
	if ((sys_start & MEMORY_CARD_DATA_WIDTH) ==
	    MEMORY_CARD_DATA_WIDTH_16BIT)
		mem->flags |= MAP_16BIT;
	if ((sys_start & MEMORY_CARD_ACCESS) == MEMORY_CARD_ACCESS_NO_WAIT) {
		mem->flags |= MAP_0WS;
	} else {
		mem->flags |= MAP_USE_WAIT;
		mem->speed = get_speed_value(sock, map);
	}

	memoffh = exca_readb(sock, MEMOFFHREG(map));
	if ((memoffh & MEMORY_CARD_WRITE) == MEMORY_CARD_WRITE_PROHIBIT)
		mem->flags |= MAP_WRPROT;
	if ((memoffh & ACCESS_MEMORY) == ATTRIBUTE_MEMORY)
		mem->flags |= MAP_ATTRIB;

	sys_start <<= 8;
	sys_start |= exca_readb(sock, SYSMEMSLREG(map));
	sys_start &= MEMORY_WINDOW_START_MASK;
	mem->sys_start = sys_start << 12;

	sys_stop = exca_readw(sock, SYSMEMELREG(map));
	sys_stop &= MEMORY_WINDOW_STOP_MASK;
	mem->sys_stop = sys_stop << 12;

	mem->card_start = 0;

	return 0;
}

static inline uint16_t
get_wait_value(u_int speed)
{
	int wait;

	/*
	 * The unit of speed is nano seconds.
	 */
	wait = (speed / 1000000000) * vr4181a_get_ecu_sysclock();

	switch (wait) {
	case 1:
		return MEMORY_CARD_ACCESS_ADD_1WAIT;
	case 2:
		return MEMORY_CARD_ACCESS_ADD_2WAIT;
	case 3:
		return MEMORY_CARD_ACCESS_ADD_3WAIT;
	case 4:
		return MEMORY_CARD_ACCESS_ADD_4WAIT;
	default:
		break;
	}

	return 0;
}

static int
ecu_set_mem_map(unsigned int sock, struct pccard_mem_map *mem)
{
	uint8_t adwinen;
	u_long sys_start, sys_stop;
	uint8_t memoffh;
	u_char map;

	if (sock >= ECU_NR_SLOTS || mem == NULL || mem->map >= ECU_MEM_NR_MAPS
	    || mem->sys_start > mem->sys_stop)
		return -EINVAL;

	map = mem->map;

	adwinen = exca_readb(sock, ADWINENREG);
	adwinen &= ~MEMORY_WINDOW_ENABLE(map);
	exca_writeb(sock, ADWINENREG, adwinen);

	mem->sys_start = sys_start = vr4181a_ecu_get_sys_start(sock);
	mem->sys_stop = sys_stop = sys_start + 0x1000;
	mem->card_start = 0;

	sys_start = (sys_start >> 12) & MEMORY_WINDOW_START_MASK;
	if (mem->flags & MAP_16BIT)
		sys_start |= MEMORY_CARD_DATA_WIDTH_16BIT;
	if (mem->flags & MAP_0WS)
		sys_start |= MEMORY_CARD_ACCESS_NO_WAIT;
	exca_writew(sock, SYSMEMSLREG(map), sys_start);

	sys_stop = (sys_stop >> 12) & MEMORY_WINDOW_STOP_MASK;
	if (mem->flags & MAP_USE_WAIT) {
		sys_stop |= get_wait_value(mem->speed);
	}
	exca_writew(sock, SYSMEMELREG(map), sys_stop);

	memoffh = 0;
	if (mem->flags & MAP_WRPROT)
		memoffh |= MEMORY_CARD_WRITE_PROHIBIT;
	if (mem->flags & MAP_ATTRIB)
		memoffh |= ATTRIBUTE_MEMORY;
	exca_writeb(sock, MEMOFFHREG(map), memoffh);

	if (mem->flags & MAP_ACTIVE)
		exca_writeb(sock, ADWINENREG,
			    adwinen | MEMORY_WINDOW_ENABLE(map));

	return 0;
}

static void
ecu_proc_setup(unsigned int sock, struct proc_dir_entry *entry)
{
}

static struct pccard_operations ecu_operations = {
	.init = ecu_init,
	.suspend = ecu_suspend,
	.register_callback = ecu_register_callback,
	.inquire_socket = ecu_inquire_socket,
	.get_status = ecu_get_status,
	.get_socket = ecu_get_socket,
	.set_socket = ecu_set_socket,
	.get_io_map = ecu_get_io_map,
	.set_io_map = ecu_set_io_map,
	.get_mem_map = ecu_get_mem_map,
	.set_mem_map = ecu_set_mem_map,
	.proc_setup = ecu_proc_setup,
};

/* ========================================================================== */

static void
ecu_bh(void *data)
{
	ecu_socket_t *socket = (ecu_socket_t *) data;
	unsigned int events;

	spin_lock_irq(&socket->event_lock);
	events = socket->events;
	socket->events = 0;
	spin_unlock_irq(&socket->event_lock);

	if (socket->handler)
		socket->handler(socket->info, events);
}

static inline unsigned int
get_events(int slot)
{
	unsigned int events = 0;
	uint8_t cdstchg;

	cdstchg = exca_readb(slot, CDSTCHGREG);
	if (cdstchg & CARD_DETECT_CHANGE)
		events |= SS_DETECT;

	if (cdstchg & CARD_STATUS_CHANGE)
		events |= SS_READY;

	if (cdstchg & STATUS_CHANGE_CHANGE) {
		if ((exca_readb(slot, ITGENCTREG) & CARD_TYPE) == IO_CARD)
			events |= SS_STSCHG;
		else
			events |= SS_BATDEAD;
	}

	return events;
}

static void
ecu_interrupt(int irq, void *dev_id, struct pt_regs *regs)
{
	ecu_socket_t *socket = (ecu_socket_t *) dev_id;
	unsigned int events;

	socket->tq_task.routine = ecu_bh;
	socket->tq_task.data = socket;

	events = get_events(socket->slot);
	if (events) {
		spin_lock(&socket->event_lock);
		socket->events |= events;
		spin_unlock(&socket->event_lock);
		schedule_task(&socket->tq_task);
	}
}

static inline void
ecu_remove_socket(int slot)
{
	ecu_socket_t *socket = &ecu_sockets[slot];

	if (socket->pcmcia_socket != NULL)
		pcmcia_unregister_socket(socket->pcmcia_socket);

	socket->pcmcia_socket = NULL;

	exca_writeb(slot, CRDSTATREG, 0);
	if (socket->irq >= 0)
		free_irq(socket->irq, socket);

	socket->irq = -1;
}

static int __devinit
vr4181a_pcmcia_init(void)
{
	ecu_socket_t *socket;
	int slot, ret;

	/*
	 * ECU Slot0
	 */
	slot = 0;
	if (vr4181a_ecu_get_mode() == ECU_MODE_CF) {
		socket = &ecu_sockets[slot];
		socket->slot = slot;
		socket->pcmcia_socket =
		    pcmcia_register_socket(slot, &ecu_operations, 0);
		if (socket->pcmcia_socket == NULL)
			return -ENOMEM;

		socket->irq = ECU_SLOT0_STATUS_IRQ;
		ret =
		    request_irq(socket->irq, ecu_interrupt, 0, socket->name,
				socket);
		if (ret < 0) {
			ecu_remove_socket(slot);
			return ret;
		}

		exca_writeb(slot, DTGENCLREG, 0);
		exca_writeb(slot, GLOCTRLREG, 0);

		printk(KERN_INFO "%s, IRQ %d\n", socket->name, socket->irq);
	}

	/*
	 * ECU Slot 1
	 */
	slot = 1;
	if (vr4181a_get_pinmode0() & PINMODE0_ECU1) {
		socket = &ecu_sockets[slot];
		socket->slot = slot;
		socket->pcmcia_socket =
		    pcmcia_register_socket(slot, &ecu_operations, 0);
		if (socket->pcmcia_socket == NULL) {
			ecu_remove_socket(0);
			return -ENOMEM;
		}

		socket->irq = ECU_SLOT1_STATUS_IRQ;
		ret =
		    request_irq(socket->irq, ecu_interrupt, 0, socket->name,
				socket);
		if (ret < 0) {
			ecu_remove_socket(0);
			ecu_remove_socket(slot);
			return ret;
		}

		exca_writeb(slot, DTGENCLREG, 0);
		exca_writeb(slot, GLOCTRLREG, 0);

		printk(KERN_INFO "%s, IRQ %d\n", socket->name, socket->irq);
	}

	return 0;
}

static void __devexit
vr4181a_pcmcia_exit(void)
{
	int slot;

	for (slot = 0; slot < ECU_NR_SLOTS; slot++)
		ecu_remove_socket(slot);
}

module_init(vr4181a_pcmcia_init);
module_exit(vr4181a_pcmcia_exit);
