/******************************************************************************
	vio-dbmx1.c
	driver for Motorola DragonBall MX1 ASP - Voice I/O support
	Features:
	- 8, 16 or 32KHz sampling rate
	- signed 16 bit data format (only 9 lower bits are significant)
	NOTE: Touchscreen support is in ts-dbmx1.c

	Author: MontaVista Software, Inc. <source@mvista.com>
	2003 (c) MontaVista Software, Inc. This file is licensed under
	the terms of the GNU General Public License version 2. This program
	is licensed "as is" without any warranty of any kind, whether express
	or implied.

********************************************************************************/

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/config.h>
#include <linux/init.h>

#include <linux/errno.h>
#include <linux/i2c.h>

#include <asm/uaccess.h>
#include <asm/arch/platform.h>
#include <asm/irq.h>
#include <asm/arch/mx1ads-gpio.h>
#include <asm/io.h>
#include <asm/dma.h>
#include <asm/arch/dma.h>
#include <asm/mach/dma.h>

#ifdef CONFIG_PM
#include <linux/pm.h>
#endif

#include "sound_config.h"

#define MX1VIO_PLLCLK2 0x5

//#define MX1VIO_FIR_COEF      /*enable loading FIR coefficients as described in the HW man */
//#define MX1VIO_ENABLE_DEBUG_CTL /*enable extended control commands in mixer device to debug gains */
//#define DEBUG_MX1VIO
#ifdef DEBUG_MX1VIO
#define debugprintk(fmt, args...) printk("%s: " fmt, __FUNCTION__ , ## args)
#else
#define debugprintk(fmt, args...)
#endif

/*register access macros*/
#define mx1_asp_reg_out(r,x) outl(x,IO_ADDRESS((ASP_BASE + r)))
#define mx1_asp_reg_in(r) inl(IO_ADDRESS((ASP_BASE + r)))
#define mx1_dma_chan_reg_out(r,c,x) outl(x,IO_ADDRESS((DMA_CH0_BASE + r + c * DMA_REG_SET_OFS * sizeof(VU32))))

/*ioctl data*/
static unsigned int mx1vio_bits = AFMT_S16_LE;
static signed short mx1vio_channels = 1;
static int      mx1vio_speed = 8000;

/*mixer data*/
static int      mx1vio_vol = 50;

extern void     mx1ads_request_dma_intr(dmach_t channel,
					callback_t dma_callback,
					err_callback_t dma_err_callback);
extern void     mx1ads_free_dma_intr(dmach_t channel);

static dmach_t  mx1vio_outp_dma = -1;
static dmach_t  mx1vio_inp_dma = -1;

#ifdef MX1VIO_ENABLE_DEBUG_CTL
static dmach_t  mx1vio_loop_dma = -1;
static int      mx1vio_loop = 0;
#endif

#define MX1VIO_IDLE_MODE 0
#define MX1VIO_WRITE_MODE 1
#define MX1VIO_READ_MODE  2
#define MX1VIO_LOOP_MODE 4
static int      mx1vio_busy = MX1VIO_IDLE_MODE;

static int      mx1vio_dev = -1;

static int      mx1vio_mixerdev = -1;
static int      mx1vio_initstate_region_vio;
static int      mx1vio_initstate_region_asp;

#ifdef CONFIG_PM
static struct pm_dev *mx1vio_pmdev;
#endif

extern void     DMAbuf_outputintr(int dev, int notify_only);
extern int      DMAbuf_start_dma(int dev, unsigned long physaddr,
				 int count, int dma_mode);
extern void     DMAbuf_inputintr(int dev);

static void     mx1vio_start_input(int dev, unsigned long buf, int count,
				   int intrflag);
static int      mx1vio_prepare_for_input(int dev, int bufsize, int nbufs);
static int      mx1vio_prepare_for_output(int dev, int bufsize, int nbufs);
static void     mx1vio_trigger(int dev, int state);
static void     mx1vio_output_block(int dev, unsigned long buf, int count,
				    int intrflag);
static void     mx1vio_close(int dev);
static int      mx1vio_open(int dev, int mode);
static void     mx1vio_halt_io(int dev);
static int      mx1vio_ioctl(int dev, unsigned int cmd, caddr_t arg);
static int      mx1vio_set_speed(int dev, int speed);
static int      mx1vio_set_vol(int vol);
static signed short mx1vio_set_channels(int dev, signed short channels);
static unsigned int mx1vio_set_bits(int dev, unsigned int bits);

static void     mx1vio_outp_dma_handler(void);
static void     mx1vio_outp_dma_error(int errno);
static void     mx1vio_inp_dma_handler(void);
static void     mx1vio_inp_dma_error(int errno);

#ifdef MX1VIO_ENABLE_DEBUG_CTL
#define MX1VIO_MIXER_VDA_CTL 111
#define MX1VIO_MIXER_VAD_CTL 112
#define MX1VIO_MIXER_LOOP_CTL 113
#endif
static int      mx1vio_mixer_ioctl(int dev, unsigned int cmd, caddr_t arg);

static void     mx1vio_reg_init(void);
static void     mx1vio_reg_clear(void);
static void     mx1vio_asp_reg_init(void);
static void     mx1vio_asp_reg_clear(void);

int __init      dbmx1_vio_init(void);
void __init     dbmx1_vio_exit(void);

#ifdef CONFIG_PM
static int      mx1vio_pm_callback(struct pm_dev *pmdev,
				   pm_request_t rqst, void *data);
#endif

static struct audio_driver mx1vio_driver = {
	.owner = THIS_MODULE,
	.open = mx1vio_open,
	.close = mx1vio_close,
	.output_block = mx1vio_output_block,
	.start_input = mx1vio_start_input,
	.ioctl = mx1vio_ioctl,
	.prepare_for_input = mx1vio_prepare_for_input,
	.prepare_for_output = mx1vio_prepare_for_output,
	.halt_io = mx1vio_halt_io,
	.local_qlen = 0,	   /*optional */
	.copy_user = 0,		   /*optional */
	.halt_input = 0,	   /*optional */
	.halt_output = 0,	   /*optional */
	.trigger = mx1vio_trigger,
	.set_speed = mx1vio_set_speed,
	.set_bits = mx1vio_set_bits,
	.set_channels = mx1vio_set_channels,
	.postprocess_write = 0,	   /*optional */
	.preprocess_read = 0,	   /*optional */
	.mmap = 0,		   /*optional */
};

static struct mixer_operations mx1vio_mixer_operations = {
	.owner = THIS_MODULE,
	.id = "DBMX1VIO",
	.name = "DBMX1 VIO Volume Ctl",
	.ioctl = mx1vio_mixer_ioctl,
};

static int
mx1vio_mixer_ioctl(int dev, unsigned int cmd, caddr_t arg)
{
	int             val, left, right;

	if (((cmd >> 8) & 0xff) != 'M')
		return -EINVAL;

	if (!(_SIOC_DIR(cmd) & (_SIOC_WRITE | _SIOC_READ)))
		return -EINVAL;	/*no void commands supported */

	if (_SIOC_DIR(cmd) & _SIOC_WRITE)	/*writing commands */
	  {

		  if (get_user(val, (int *) arg))
			  return -EFAULT;

		  switch (cmd & 0xff)
		    {

		    case SOUND_MIXER_VOLUME:
			    left = val & 0x00ff;
			    right = (val >> 8) & 0x00ff;

			    if (left < 0)
				    left = 0;
			    if (left > 100)
				    left = 100;
			    if (right < 0)
				    right = 0;
			    if (right > 100)
				    right = 100;

			    if (mx1vio_vol == 100)
			      {
				      if (left < right)
					      val = left;
				      else
					      val = right;
			    } else if (left > right)
				    val = left;
			    else
				    val = right;

			    int             tmp = mx1vio_vol;

			    if (val > tmp)
				    mx1vio_vol = 100;
			    if (val < tmp)
				    mx1vio_vol = 50;
			    if (mx1vio_busy != MX1VIO_IDLE_MODE)
			      {
				      mx1vio_set_vol(mx1vio_vol);

			      }
			    break;
#ifdef MX1VIO_ENABLE_DEBUG_CTL
		    case MX1VIO_MIXER_VDA_CTL:	/*Set VDA Control Reg */
			    mx1_asp_reg_out(ASP_VDAGAIN, val);
			    break;
		    case MX1VIO_MIXER_VAD_CTL:	/*Set VAD Control Reg */
			    mx1_asp_reg_out(ASP_VADGAIN, val);
			    break;
		    case MX1VIO_MIXER_LOOP_CTL:	/*toggle Mic to Voice-Out loopback test. */
			    /* NOTE: this test may disrupt normal ASP function. ASP may require HW reset after */
			    if (mx1vio_loop != (val & 1))
			      {
				      if (val & 1)
					{
						if (mx1vio_busy ==
						    MX1VIO_IDLE_MODE)
						  {
							  mx1vio_loop = 1;
							  mx1vio_busy =
							      MX1VIO_LOOP_MODE;
							  mx1_asp_reg_out(ASP_ACNTLCR, (mx1_asp_reg_in(ASP_ACNTLCR) | 0xC));	/*enable VAD/VDA controllers */
							  mx1_asp_reg_out(ASP_ICNTLR, (mx1_asp_reg_in(ASP_ICNTLR) | 0x180));	/*enable VDA/VAD DMAs */
							  enable_dma
							      (mx1vio_loop_dma);
						  }
				      } else
					{
						mx1vio_loop = 0;
						mx1vio_busy = MX1VIO_IDLE_MODE;
						disable_dma(mx1vio_loop_dma);
						mx1_asp_reg_out(ASP_ICNTLR, (mx1_asp_reg_in(ASP_ICNTLR) & ~0x180));	/*disable VAD/VDA DMA's */
						mx1_asp_reg_out(ASP_ACNTLCR, (mx1_asp_reg_in(ASP_ACNTLCR) & ~0xC));	/*disable VAD/VDA controllers */
					}
			      }
			    break;
#endif
		    default:
			    return -EINVAL;
		    }
	  }
	if (_SIOC_DIR(cmd) & _SIOC_READ)	/*reading commands */
	  {
		  switch (cmd & 0xff)
		    {

		    case SOUND_MIXER_VOLUME:
			    val = mx1vio_vol + ((mx1vio_vol << 8) & 0xFF00);
			    break;
		    case SOUND_MIXER_DEVMASK:
			    val = SOUND_MASK_VOLUME;
			    break;
#ifdef MX1VIO_ENABLE_DEBUG_CTL
		    case MX1VIO_MIXER_VDA_CTL:	/*Get VDA Control Reg */
			    val = (int) mx1_asp_reg_in(ASP_VDAGAIN);	/*bits 16-31 are not used anyway */
			    break;
		    case MX1VIO_MIXER_VAD_CTL:	/*Get VAD Control Reg */
			    val = (int) mx1_asp_reg_in(ASP_VADGAIN);	/*bits 16-31 are not used anyway */
			    break;
		    case MX1VIO_MIXER_LOOP_CTL:
			    val = mx1vio_loop;
			    break;
#endif
		    default:
			    return -EINVAL;
		    }
		  return put_user(val, (int *) arg);
	  }
	return 0;
}

#ifdef MX1VIO_FIR_COEF
#define MX1VIO_FIR_RAM_SIZE 128
static unsigned long int mx1vio_fir_ram_vals[MX1VIO_FIR_RAM_SIZE] = {
	0x0000FFFD, 0x0000FFFE, 0x0000FFFF, 0x00000005, 0x00000010,
	0x0000001D, 0x00000027, 0x00000027,
	0x0000001B, 0x00000002, 0x0000FFE6, 0x0000FFD0, 0x0000FFCE,
	0x0000FFE4, 0x0000000C, 0x00000037,
	0x0000004E, 0x00000043, 0x00000013, 0x0000FFD2, 0x0000FF9A,
	0x0000FF8C, 0x0000FFB6, 0x0000000C,
	0x0000006B, 0x000000A5, 0x00000094, 0x00000037, 0x0000FFAF,
	0x0000FF3A, 0x0000FF16, 0x0000FF62,
	0x00000008, 0x000000C4, 0x0000013A, 0x00000124, 0x00000078,
	0x0000FF77, 0x0000FE94, 0x0000FE46,
	0x0000FECA, 0x0000FFFD, 0x0000015F, 0x00000249, 0x00000231,
	0x000000FA, 0x0000FF15, 0x0000FD54,
	0x0000FCA0, 0x0000FD81, 0x0000FFD0, 0x000002A9, 0x000004BE,
	0x000004D9, 0x0000027C, 0x0000FE3A,
	0x0000F9A8, 0x0000F6E4, 0x0000F7CB, 0x0000FD2D, 0x0000065C,
	0x0000113F, 0x00001AE9, 0x00002095,
	0x00002095, 0x00001AE9, 0x0000113F, 0x0000065C, 0x0000FD2D,
	0x0000F7CB, 0x0000F6E4, 0x0000F9A8,
	0x0000FE3A, 0x0000027C, 0x000004D9, 0x000004BE, 0x000002A9,
	0x0000FFD0, 0x0000FD81, 0x0000FCA0,
	0x0000FD54, 0x0000FF15, 0x000000FA, 0x00000231, 0x00000249,
	0x0000015F, 0x0000FFFD, 0x0000FECA,
	0x0000FE46, 0x0000FE94, 0x0000FF77, 0x00000078, 0x00000124,
	0x0000013A, 0x000000C4, 0x00000008,
	0x0000FF62, 0x0000FF16, 0x0000FF3A, 0x0000FFAF, 0x00000037,
	0x00000094, 0x000000A5, 0x0000006B,
	0x0000000C, 0x0000FFB6, 0x0000FF8C, 0x0000FF9A, 0x0000FFD2,
	0x00000013, 0x00000043, 0x0000004E,
	0x00000037, 0x0000000C, 0x0000FFE4, 0x0000FFCE, 0x0000FFD0,
	0x0000FFE6, 0x00000002, 0x0000001B,
	0x00000027, 0x00000027, 0x0000001D, 0x00000010, 0x00000005,
	0x0000FFFF, 0x0000FFFE, 0x0000FFFD,
};
#endif

/*Common voice/pen registers:
ASP_ACNTLCR - control
ASP_ICNTLR - dma/int control
ASP_ISTATR - int status
ASP_CLKDIV - clock divider

Pen-related registers:
ASP_PADFIFO - fifo
ASP_PSMPLRG - sample rate control
ASP_CMPCNTL - compare & compare control

Voice-related registers:
ASP_VADFIFO - input fifo
ASP_VDAFIFO - output fifo
ASP_VADGAIN - input FIR/Comb gains/decimations
ASP_VDAGAIN - output FIR/Comb gains/decimations
ASP_VADCOEF - access to input FIR RAM with coefficients
ASP_VDACOEF - access to output FIR RAM with coefficients
*/

/*initialize common registers*/
static void
mx1vio_asp_reg_init(void)
{
	mx1_asp_reg_out(ASP_ACNTLCR, (1UL << 25));	/*enable clock */

	mx1_asp_reg_out(ASP_ACNTLCR, (mx1_asp_reg_in(ASP_ACNTLCR) | (1UL << 23)));	/*software reset */
	for (;;)
		if (!(mx1_asp_reg_in(ASP_ACNTLCR) & (1UL << 23)))
			break;	/*wait for sw reset ready */

	mx1_asp_reg_out(ASP_ACNTLCR, (mx1_asp_reg_in(ASP_ACNTLCR) | 0x1));	/*enable  BandGap */
	for (;;)
		if (mx1_asp_reg_in(ASP_ISTATR) & (1UL << 9))
			break;	/*wait for bandgap ready */
}

/* Initialize voice registers: */
static void
mx1vio_reg_init(void)
{
#ifdef MX1VIO_FIR_COEF
	int             i;

#ifdef DEBUG_MX1VIO
	int             j;
#endif
#endif

	mx1vio_reg_clear();

#ifdef MX1VIO_FIR_COEF
	for (i = 0; i < MX1VIO_FIR_RAM_SIZE; i++)	/*fill up FIR RAM */
	  {
		  mx1_asp_reg_out(ASP_VDACOEF, mx1vio_fir_ram_vals[i]);
		  mx1_asp_reg_out(ASP_VADCOEF, mx1vio_fir_ram_vals[i]);
	  }

#ifdef DEBUG_MX1VIO
	debugprintk("ASP DA FIR Coefficients\n");
	for (j = 0; j < 16; j++)
	  {
		  for (i = 0; i < 7; i++)
			  printk("%x ", mx1_asp_reg_in(ASP_VDACOEF));
		  printk("%x\n", mx1_asp_reg_in(ASP_VDACOEF));
	  }
	debugprintk("ASP AD FIR Coefficients\n");
	for (j = 0; j < 16; j++)
	  {
		  for (i = 0; i < 7; i++)
			  printk("%x ", mx1_asp_reg_in(ASP_VADCOEF));
		  printk("%x\n", mx1_asp_reg_in(ASP_VADCOEF));
	  }
#endif
#endif

#ifdef MX1VIO_FIR_COEF
	mx1_asp_reg_out(ASP_VDAGAIN, 0x5a39);	/*set output FIR/Comb gains and decimations */
	mx1_asp_reg_out(ASP_VADGAIN, 0x1a39);	/*set input FIR/Comb gains and decimations */
#else
	mx1_asp_reg_out(ASP_VDAGAIN, 0x54B9);	/*set output FIR/Comb gains and decimations */
	mx1_asp_reg_out(ASP_VADGAIN, 0x14B9);	/*set input FIR/Comb gains and decimations */
#endif

	mx1vio_set_speed(0, mx1vio_speed);
	mx1vio_set_vol(mx1vio_vol);

	/*ASP_ACNTLCR bit 3 - VDA ctrl en, bit 2- VAD ctrl en */
	/*ASP_ICNTLR bit 8 - VDAC (outp) DMA En, bit 7 - VADC (inp) DMA En */
	if (mx1vio_busy & MX1VIO_READ_MODE)
	  {
		  mx1_asp_reg_out(ASP_ACNTLCR, (mx1_asp_reg_in(ASP_ACNTLCR) | 0x4));	/*enable VAD controller */
		  mx1_asp_reg_out(ASP_ICNTLR, (mx1_asp_reg_in(ASP_ICNTLR) | 0x80));	/*enable VAD DMA */
	  }
	if (mx1vio_busy & MX1VIO_WRITE_MODE)
	  {
		  mx1_asp_reg_out(ASP_ACNTLCR, (mx1_asp_reg_in(ASP_ACNTLCR) | 0x8));	/*enable VDA controller */
		  mx1_asp_reg_out(ASP_ICNTLR, (mx1_asp_reg_in(ASP_ICNTLR) | 0x100));	/*enable VDA DMA */
	  }
}

/*Disable ASP, clear ASP registers*/
static void
mx1vio_asp_reg_clear(void)
{
	mx1_asp_reg_out(ASP_ICNTLR, 0);
	mx1_asp_reg_out(ASP_CLKDIV, 0);
	mx1_asp_reg_out(ASP_ACNTLCR, 0);	/*disable ASP */
}

/*Disable VIO, clear VIO registers*/
static void
mx1vio_reg_clear(void)
{
	mx1_asp_reg_out(ASP_ICNTLR, (mx1_asp_reg_in(ASP_ICNTLR) & ~0x180));	/*disable VAD/VDA DMA's */
	mx1_asp_reg_out(ASP_ACNTLCR, (mx1_asp_reg_in(ASP_ACNTLCR) & ~0xC));	/*disable VAD/VDA controllers */
	mx1_asp_reg_out(ASP_VDAFIFO, 0);
}

/* set number of bits, only 16 supported*/
static unsigned int
mx1vio_set_bits(int dev, unsigned int bits)
{
	switch (bits)
	  {
	  case 0:
		  return mx1vio_bits;
	  case AFMT_S16_LE:
		  mx1vio_bits = AFMT_S16_LE;
		  break;
	  default:
		  return -EINVAL;
	  }
	debugprintk("bits %d\n", bits);
	return mx1vio_bits;
}

/* set number of channels, only MONO supported*/
static signed short
mx1vio_set_channels(int dev, signed short channels)
{
	switch (channels)
	  {
	  case 0:
		  return mx1vio_channels;
	  case 1:
		  mx1vio_channels = 1;
		  break;
	  default:
		  return -EINVAL;
	  }
	debugprintk("channels %d\n", channels);
	return mx1vio_channels;
}

/* set ADC/DAC sampling rate, only 8k supported*/
static int
mx1vio_set_speed(int dev, int speed)
{
	switch (speed)
	  {
	  case 0:
		  return mx1vio_speed;
	  case 8000:
		  mx1vio_speed = 8000;
		  mx1_asp_reg_out(ASP_CLKDIV, ((mx1_asp_reg_in(ASP_CLKDIV) & 0x1F) | (0x420 * 7)));	/*set 8KHz freq */
		  break;
	  case 16000:
		  mx1vio_speed = 16000;
		  mx1_asp_reg_out(ASP_CLKDIV, ((mx1_asp_reg_in(ASP_CLKDIV) & 0x1F) | (0x420 * 3)));	/*set 16KHz freq */
		  break;
	  case 32000:
		  mx1vio_speed = 32000;
		  mx1_asp_reg_out(ASP_CLKDIV, ((mx1_asp_reg_in(ASP_CLKDIV) & 0x1F) | 0x420));	/*set 32KHz freq */

		  break;
	  default:
		  return -EINVAL;
	  }
	debugprintk("speed %d\n", speed);
	return mx1vio_speed;
}

/*set volume*/
static int
mx1vio_set_vol(int vol)
{
	if (vol == 50)
		mx1_asp_reg_out(ASP_ACNTLCR, (mx1_asp_reg_in(ASP_ACNTLCR) | (1UL << 16)));	/*enable low DAC gain */
	else
		mx1_asp_reg_out(ASP_ACNTLCR, (mx1_asp_reg_in(ASP_ACNTLCR) & ~(1UL << 16)));	/*enable high DAC gain */

	return 0;
}

static int
mx1vio_ioctl(int dev, unsigned int cmd, caddr_t arg)
{
	int             val, ret = 0;

	switch (cmd)
	  {
	  case SOUND_PCM_WRITE_RATE:
		  if (get_user(val, (int *) arg))
			  return -EFAULT;
		  ret = mx1vio_set_speed(dev, val);
		  break;
	  case SOUND_PCM_READ_RATE:
		  ret = mx1vio_speed;
		  break;
	  case SNDCTL_DSP_STEREO:
		  if (get_user(val, (int *) arg))
			  return -EFAULT;
		  ret = mx1vio_set_channels(dev, (val + 1)) - 1;
		  break;
	  case SOUND_PCM_WRITE_CHANNELS:
		  if (get_user(val, (int *) arg))
			  return -EFAULT;
		  ret = mx1vio_set_channels(dev, val);
		  break;
	  case SOUND_PCM_READ_CHANNELS:
		  ret = mx1vio_channels;
		  break;
	  case SNDCTL_DSP_SETFMT:
		  if (get_user(val, (int *) arg))
			  return -EFAULT;
		  ret = mx1vio_set_bits(dev, val);
		  break;
	  case SOUND_PCM_READ_BITS:
		  ret = mx1vio_bits;
		  break;
	  default:
		  return -EINVAL;
	  }
	return put_user(ret, (int *) arg);
}

static void
mx1vio_outp_dma_handler(void)
{
	debugprintk("\n");
	DMAbuf_outputintr(mx1vio_dev, 1);
}

static void
mx1vio_outp_dma_error(int errno)
{
	printk(KERN_ERR "DBMX1 VIO ERROR: output DMA error %d \n", errno);
}

static void
mx1vio_inp_dma_handler(void)
{
	debugprintk("\n");
	DMAbuf_inputintr(mx1vio_dev);
}

static void
mx1vio_inp_dma_error(int errno)
{
	printk(KERN_ERR "DBMX1 VIO ERROR: input DMA error %d \n", errno);
}

/*start the device*/
static int
mx1vio_open(int dev, int mode)
{
	debugprintk("\n");
	if (mx1vio_busy != MX1VIO_IDLE_MODE)
		return -EBUSY;

	/*read, write, or readwrite possible */

	if (mode & OPEN_WRITE)
	  {
		  mx1vio_busy |= MX1VIO_WRITE_MODE;
		  debugprintk("write\n");
	  }
	if (mode & OPEN_READ)
	  {
		  mx1vio_busy |= MX1VIO_READ_MODE;
		  debugprintk("read\n");
	  }

	mx1vio_reg_init();
	return (0);
}

/*restart the device*/
static void
mx1vio_halt_io(int dev)
{
	debugprintk("\n");
	mx1vio_reg_init();
}

/*stop the device*/
static void
mx1vio_close(int dev)
{
	mx1vio_reg_clear();
	mx1vio_busy = MX1VIO_IDLE_MODE;
}

/*this function is a part of output dma buffer processing*/
static int
mx1vio_prepare_for_output(int dev, int bufsize, int nbufs)
{
	debugprintk("dma chan %d\n", audio_devs[dev]->dmap_out->dma);
	audio_devs[dev]->dmap_out->flags |= DMA_NODMA;
	return 0;
}

/*this function is a part of input dma buffer processing*/
static int
mx1vio_prepare_for_input(int dev, int bufsize, int nbufs)
{
	debugprintk("dma chan %d\n", audio_devs[dev]->dmap_in->dma);
	audio_devs[dev]->dmap_in->flags |= DMA_NODMA;
	return 0;
}

/*this function is a part of output dma buffer processing*/
static void
mx1vio_output_block(int dev, unsigned long buf, int count, int intrflag)
{
	debugprintk("count %d\n", count);
	DMAbuf_start_dma(dev, buf, count, DMA_MODE_WRITE);
}

/*this function is a part of input dma buffer processing*/
static void
mx1vio_start_input(int dev, unsigned long buf, int count, int intrflag)
{
	debugprintk("count %d\n", count);
	DMAbuf_start_dma(dev, buf, count, DMA_MODE_READ);
}

/*this function is a part of input and output dma buffer processing*/
static void
mx1vio_trigger(int dev, int state)
{
	debugprintk("\n");
}

#ifdef CONFIG_PM
/*power management event handling*/
static int
mx1vio_pm_callback(struct pm_dev *pmdev, pm_request_t rqst, void *data)
{
	switch (rqst)
	  {
	  case PM_SUSPEND:
		  if (mx1vio_busy != MX1VIO_IDLE_MODE)
			  mx1vio_reg_clear();
		  if (!(check_region(IO_ADDRESS(ASP_PADFIFO), 0x4)))
			  mx1vio_asp_reg_clear();	/*touchscreen not loaded */
		  if (mx1vio_initstate_region_vio)
			  release_region(IO_ADDRESS(ASP_VADFIFO), 0x4);
		  if (mx1vio_initstate_region_asp)
			  release_region(IO_ADDRESS(ASP_ACNTLCR), 0x4);
		  mx1vio_initstate_region_vio = 0;
		  mx1vio_initstate_region_asp = 0;
		  break;
	  case PM_RESUME:
		  /*NOTE: ASP registers support sound and also touchscreen in a separate driver */
		  if (request_region(IO_ADDRESS(ASP_ACNTLCR), 0x4,
				     "dbmx1_asp(vio)"))
		    {
			    debugprintk("taking common controls of ASP\n");
			    mx1vio_initstate_region_asp = 1;
		    }
		  /*NOTE: ASP registers support sound and also touchscreen in a separate driver */

		  if (!(request_region(IO_ADDRESS(ASP_VADFIFO), 0x4,
				       "dbmx1_vio")))
		    {
			    printk(KERN_ERR
				   "DBMX1 VIO ERROR: ASP-Voice is already in use\n");
		  } else
			  mx1vio_initstate_region_vio = 1;
		  if (mx1vio_initstate_region_asp)
			  mx1vio_asp_reg_init();
		  if (mx1vio_busy != MX1VIO_IDLE_MODE)
			  mx1vio_reg_init();
		  break;
	  }
	return 0;
}
#endif

/*initialize the device*/
int             __init
dbmx1_vio_init(void)
{
	int             tmp;
	unsigned long   tmp_reg;

	printk(KERN_INFO "DBMX1 VIO Driver Ver. 1.0\n");
	debugprintk("%s %s\n", __TIME__, __DATE__);
	mx1vio_outp_dma = -1;
	mx1vio_inp_dma = -1;
#ifdef MX1VIO_ENABLE_DEBUG_CTL
	mx1vio_loop_dma = -1;
#endif
	mx1vio_dev = -1;
	mx1vio_initstate_region_vio = 0;
	mx1vio_initstate_region_asp = 0;

	/*NOTE: ASP registers support sound and also touchscreen in a separate driver */
	tmp =
	    (int) request_region(IO_ADDRESS(ASP_ACNTLCR), 0x4,
				 "dbmx1_asp(vio)");
	if (tmp)
	  {
		  debugprintk("taking common controls of ASP\n");
		  mx1vio_initstate_region_asp = 1;
	  }

	/*NOTE: ASP registers support sound and also touchscreen in a separate driver */
	tmp = (int) request_region(IO_ADDRESS(ASP_VADFIFO), 0x4, "dbmx1_vio");
	if (!tmp)
	  {
		  printk(KERN_ERR
			 "DBMX1 VIO ERROR: ASP-Voice is already in use\n");
		  dbmx1_vio_exit();
		  return -EPERM;
	  }
	mx1vio_initstate_region_vio = 1;

/*ASP Signal Descriptions from DBMX1 Reference Manual
UIN Positive U analog input (for low voltage, temperature measurement)
UIP Negative U analog input (for low voltage, temperature measurement)
PX1 Positive pen-X analog input
PY1 Positive pen-Y analog input
PX2 Negative pen-X analog input
PY2 Negative pen-Y analog input
R1A Positive resistance input (a)
R1B Positive resistance input (b)
R2A Negative resistance input (a)
R2B Negative resistance input (b)
MIP Positive voice analog input
MIM Negative voice analog input
RVP Positive reference for pen ADC
RVM Negative reference for pen ADC
RVP1 Positive reference for voice ADC
RVM1 Negative reference for voice ADC
RP Positive bandgap reference
RM Negative bandgap reference
DAC_OP Voice DAC positive output
DAC_OM Voice DAC negative output
AVDD Analog power supply
AGND Analog ground
all pins are reserved just for the ASP*/

	/*configure output DMA support */
	for (mx1vio_outp_dma = 0; mx1vio_outp_dma < MAX_DMA_CHANNELS;
	     mx1vio_outp_dma++)
	  {
		  if (!sound_alloc_dma(mx1vio_outp_dma, "DBMX1_VIO_OUT"))
			  break;
	  }
	if (mx1vio_outp_dma == MAX_DMA_CHANNELS)
	  {
		  mx1vio_outp_dma = -1;
		  printk(KERN_ERR
			 "DBMX1 VIO ERROR: can't get a DMA channel for output\n");
		  dbmx1_vio_exit();
		  return -1;
	  }

	/*configure input DMA support */
	for (mx1vio_inp_dma = 0; mx1vio_inp_dma < MAX_DMA_CHANNELS;
	     mx1vio_inp_dma++)
	  {
		  if (!sound_alloc_dma(mx1vio_inp_dma, "DBMX1_VIO_IN"))
			  break;
	  }
	if (mx1vio_inp_dma == MAX_DMA_CHANNELS)
	  {
		  mx1vio_inp_dma = -1;
		  printk(KERN_ERR
			 "DBMX1 VIO ERROR: can't get a DMA channel for input\n");
		  dbmx1_vio_exit();
		  return -1;
	  }
#ifdef MX1VIO_ENABLE_DEBUG_CTL
/* request a DMA channel for MIC-Voice Loop data */
	for (mx1vio_loop_dma = 0; mx1vio_loop_dma < MAX_DMA_CHANNELS;
	     mx1vio_loop_dma++)
	  {
		  if (!request_dma(mx1vio_loop_dma, "DBMX1_VIO"))
			  break;
	  }

	if (mx1vio_loop_dma == MAX_DMA_CHANNELS)
	  {
		  printk(KERN_ERR
			 "DBMX1 VIO ERROR: could not allocate DMA for MIC-Voice Loop\n");
		  mx1vio_loop_dma = -1;
		  dbmx1_vio_exit();
		  return -1;
	  }
#endif

	if ((mx1vio_dev = sound_install_audiodrv(AUDIO_DRIVER_VERSION, "dbmx1_vio_dev", &mx1vio_driver, sizeof (struct audio_driver), DMA_DUPLEX, AFMT_S16_LE, NULL, mx1vio_outp_dma,	/*output dma */
						 mx1vio_inp_dma
						 /*input dma */ )) < 0)
	  {
		  printk(KERN_ERR
			 "DBMX1 VIO ERROR: too many audio devices in the system!\n");
		  dbmx1_vio_exit();
		  return mx1vio_dev;
	  }

	if ((mx1vio_mixerdev = sound_alloc_mixerdev()) >= 0)
	  {
		  mixer_devs[mx1vio_mixerdev] = &mx1vio_mixer_operations;
	} else
		printk(KERN_WARNING
		       "DBMX1 VIO WARNING: failed to load mixer device\n");

	if (((tmp_reg =
	      inl(IO_ADDRESS((PLL_PCDR + PLL_BASE)))) & PCDR_PCLKDIV2_MASK)
	    != ((MX1VIO_PLLCLK2 << PCDR_PCLKDIV2_BIT) & PCDR_PCLKDIV2_MASK))
	  {
		  printk(KERN_WARNING
			 "DBMX1 VIO WARNING: changing PERCLK2 from 0x%x to 0x%x\n",
			 (unsigned int) ((tmp_reg & PCDR_PCLKDIV2_MASK) >>
					 PCDR_PCLKDIV2_BIT), MX1VIO_PLLCLK2);
		  tmp_reg &= ~PCDR_PCLKDIV2_MASK;
		  tmp_reg |=
		      (MX1VIO_PLLCLK2 << PCDR_PCLKDIV2_BIT) &
		      PCDR_PCLKDIV2_MASK;
		  outl(tmp_reg, IO_ADDRESS((PLL_PCDR + PLL_BASE)));
	  }

	/*configure output DMA */
	mx1_dma_chan_reg_out(DMA_DAR, mx1vio_outp_dma, (ASP_BASE + ASP_VDAFIFO));	/*destination address */
	/*destination: fifo 16 bit; source: linear memory 32 bit; addr inc; req en; 0010 0000 1000 1000 */
	mx1_dma_chan_reg_out(DMA_CCR, mx1vio_outp_dma, 0x2088);
	mx1_dma_chan_reg_out(DMA_RSSR, mx1vio_outp_dma, 18);	/*dma request */
	mx1_dma_chan_reg_out(DMA_BLR, mx1vio_outp_dma, 16);	/*16 bytes per burst - to fill 8x16 FIFO */
	mx1_dma_chan_reg_out(DMA_BUCR, mx1vio_outp_dma, 0);	/*delay btw bursts */

	mx1ads_request_dma_intr(mx1vio_outp_dma,
				(callback_t) mx1vio_outp_dma_handler,
				(err_callback_t) mx1vio_outp_dma_error);

	/*configure input DMA */
	mx1_dma_chan_reg_out(DMA_SAR, mx1vio_inp_dma, (ASP_BASE + ASP_VADFIFO));	/*source address */
	/*destination: linear memory 32 bit; source: fifo 16 bit; addr inc; req en; 0000 1000 0010 1000 */
	mx1_dma_chan_reg_out(DMA_CCR, mx1vio_inp_dma, 0x0828);
	mx1_dma_chan_reg_out(DMA_RSSR, mx1vio_inp_dma, 19);	/*dma request */
	mx1_dma_chan_reg_out(DMA_BLR, mx1vio_inp_dma, 16);	/*16 bytes per burst - to fill 8x16 FIFO */
	mx1_dma_chan_reg_out(DMA_BUCR, mx1vio_inp_dma, 0);	/*delay btw bursts */

	mx1ads_request_dma_intr(mx1vio_inp_dma,
				(callback_t) mx1vio_inp_dma_handler,
				(err_callback_t) mx1vio_inp_dma_error);
#ifdef MX1VIO_ENABLE_DEBUG_CTL
	mx1_dma_chan_reg_out(DMA_SAR, mx1vio_loop_dma, (ASP_BASE + ASP_VADFIFO));	/*source address */
	mx1_dma_chan_reg_out(DMA_DAR, mx1vio_loop_dma, (ASP_BASE + ASP_VDAFIFO));	/*destination address */
	/*destination: fifo 16 bit; source: fifo 16 bit;  req dis, rpt en; 0010 1000 1010 0100 */
	mx1_dma_chan_reg_out(DMA_CCR, mx1vio_loop_dma, 0x28A4);	/*28A4 */
	mx1_dma_chan_reg_out(DMA_CNTR, mx1vio_loop_dma, 0);
	mx1_dma_chan_reg_out(DMA_BLR, mx1vio_loop_dma, 16);	/*16 bytes per burst - to fill 8x16 FIFO */

	mx1_dma_chan_reg_out(DMA_RSSR, mx1vio_loop_dma, 0);	/* no DMA request */
	mx1_dma_chan_reg_out(DMA_RTOR, mx1vio_loop_dma, 0);	/* burst timeout, not used */
	mx1_dma_chan_reg_out(DMA_BUCR, mx1vio_loop_dma, 100);	/* bus utilization */
#endif
	if (mx1vio_initstate_region_asp)
		mx1vio_asp_reg_init();

#ifdef CONFIG_PM
	mx1vio_pmdev =
	    pm_register(PM_UNKNOWN_DEV, PM_SYS_UNKNOWN, mx1vio_pm_callback);

	if (!mx1vio_pmdev)
		printk(KERN_WARNING
		       "DBMX1 VIO WARNING: failed to init power management\n");
#endif
	return (0);
}

/*deinitialize the device*/
void            __init
dbmx1_vio_exit(void)
{
	if (mx1vio_dev >= 0)
	  {
#ifdef CONFIG_PM
		  pm_unregister(mx1vio_pmdev);
#endif
		  if (!(check_region(IO_ADDRESS(ASP_PADFIFO), 0x4)))
			  mx1vio_asp_reg_clear();	/*touchscreen not loaded */

		  if (mx1vio_mixerdev >= 0)
			  sound_unload_mixerdev(mx1vio_mixerdev);

		  sound_unload_audiodev(mx1vio_dev);
	  }
#ifdef MX1VIO_ENABLE_DEBUG_CTL
	if (mx1vio_loop_dma != -1)
	  {

		  disable_dma(mx1vio_loop_dma);
		  free_dma(mx1vio_loop_dma);
	  }
#endif

	if (mx1vio_outp_dma != -1)
	  {
		  mx1ads_free_dma_intr(mx1vio_outp_dma);
		  sound_free_dma(mx1vio_outp_dma);
	  }

	if (mx1vio_inp_dma != -1)
	  {
		  mx1ads_free_dma_intr(mx1vio_inp_dma);
		  sound_free_dma(mx1vio_inp_dma);
	  }

	if (mx1vio_initstate_region_vio)
		release_region(IO_ADDRESS(ASP_VADFIFO), 0x4);

	if (mx1vio_initstate_region_asp)
		release_region(IO_ADDRESS(ASP_ACNTLCR), 0x4);
}

MODULE_AUTHOR("MontaVista Software, Inc. <source@mvista.com>");
MODULE_DESCRIPTION("DBMX1 VIO Driver");
MODULE_LICENSE("GPL");

module_init(dbmx1_vio_init);
module_exit(dbmx1_vio_exit);
