/*
 * linux/drivers/usbd/bi/mx1_usbd.c -- USB Device Controller driver.
 *
 * Copyright (c) 2000, 2001, 2002 Lineo
 * Copyright (c) 2003 MontaVista Software Inc. <source@mvista.com>
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/*****************************************************************************/

#include <linux/config.h>
#include <linux/module.h>

#include "../usbd-export.h"
#include "../usbd-build.h"
#include "../usbd-module.h"
#include "../usbd-debug.h"
#include "../usbd-arch.h"

#include "mx1.h"

static int storage_flag = 0;

/*
 * ep_register - endpoint register addresses
 *
 */

volatile EP_REGS_CH *ep_regs_ch[UDC_MAX_ENDPOINTS];
static int cable_connect = 1;
static int first_connection;
U32 gEPconfigData[55] = {
	// EP0
	0x00000000,
	0x00000000,
	0x00000008,
	0x00000000,
	0x00000000,
	// EP1
	0x00000014,		// EP#:1, CF#: 1, IF#:0
	0x00000014,		// BULK, IN
	0x00000020,		// MAX PS: 32
	0x000000C0,		// 0xC0, except for EP0
	0x00000001,		// FIFO#: 1
	// EP2
	0x00000024,		// EP#:2, CF#: 1, IF#:0
	0x00000010,		// BULK, OUT
	0x00000020,		// MAX PS: 32
	0x000000C0,		// 0xC0, except for EP0
	0x00000002,		// FIFO#: 2
	// EP3
	0x00000034,		// EP#:3, CF#: 1, IF#:0
	0x0000001C,		// INTR, IN
	0x00000010,		// MAX PS: 16
	0x000000C0,		// 0xC0, except for EP0
	0x00000003,		// FIFO#: 3
	// EP4
	0x00000044,		// EP#:4, CF#: 1, IF#:0
	0x00000018,		// INTR, OIUT
	0x00000010,		// MAX PS: 16
	0x000000C0,		// 0xC0, except for EP0
	0x00000004,		// FIFO#: 4
	// EP5
	0x00000054,		// EP#:5, CF#: 1, IF#:0
	0x0000001C,		// INTR, IN
	0x00000010,		// MAX PS: 16
	0x000000C0,		// 0xC0, except for EP0
	0x00000005,		// FIFO#: 5
	// EP6
	0x00000018,		// EP#:1, CF#: 2, IF#:0
	0x00000014,		// BULK, IN
	0x00000020,		// MAX PS: 32
	0x000000C0,		// 0xC0, except for EP0
	0x00000001,		// FIFO#: 1
	// EP7
	0x00000028,		// EP#:2, CF#: 2, IF#:0
	0x00000010,		// BULK, OUT
	0x00000020,		// MAX PS: 32
	0x000000C0,		// 0xC0, except for EP0
	0x00000002,		// FIFO#: 2
	// EP8
	0x00000038,		// EP#:3, CF#: 2, IF#:0
	0x0000001C,		// INTR, IN
	0x00000010,		// MAX PS: 16
	0x000000C0,		// 0xC0, except for EP0
	0x00000003,		// FIFO#: 3
	// EP9
	0x00000048,		// EP#:4, CF#: 2, IF#:0
	0x00000018,		// INTR, OIUT
	0x00000010,		// MAX PS: 16
	0x000000C0,		// 0xC0, except for EP0
	0x00000004,		// FIFO#: 4
	// EP10
	0x00000058,		// EP#:5, CF#: 2, IF#:0
	0x0000001C,		// INTR, IN
	0x00000010,		// MAX PS: 16
	0x000000C0,		// 0xC0, except for EP0
	0x00000005,		// FIFO#: 5
};

MODULE_DESCRIPTION
    ("USB MX1 Device Bus Interface for network and serial drivers");

#include <linux/kernel.h>
#include <linux/slab.h>
#include <linux/interrupt.h>
#include <linux/pci.h>
#include <linux/init.h>

#include <asm/atomic.h>
#include <asm/io.h>

#include <linux/netdevice.h>

#include <asm/irq.h>
#include <asm/dma.h>
#include <asm/system.h>

#include <asm/types.h>

#include <asm/uaccess.h>
#include <asm/io.h>
#include <asm/arch/mx1ads-gpio.h>
#include <linux/delay.h>

#include "../usbd.h"
#include "../usbd-func.h"
#include "../usbd-bus.h"
#include "../usbd-inline.h"
#include "usbd-bi.h"

#define MIN(a,b) ((a) < (b) ? (a) : (b))
#define MAX(a,b) ((a) > (b) ? (a) : (b))

static struct usb_device_instance *udc_device;	// required for the interrupt handler

/*
 * ep_endpoints - map physical endpoints to logical endpoints
 */
static struct usb_endpoint_instance *ep_endpoints[UDC_MAX_ENDPOINTS];

static struct urb *ep0_urb;
extern unsigned int udc_interrupts;

/**
 * send_data - send packet via endpoint
 * @ep: logical endpoint number
 * @bp: pointer to data
 * @size: bytes to write
 */
static void __inline__
send_data(unsigned char ep, unsigned char *bp, unsigned char size)
{
#ifdef WRITESIZE32
	U32 *_fifo;
#endif

	if (ep == 1) {
		int counter = 0xFFFF;
		while (((ep_regs_ch[ep]->STAT >> 16) & 0x7F) > 0x20) {
			udelay(1);
			counter--;
			if (counter == 0) {
				return;
			}
		}
	}
	// copy data from buffer to chip
	if (bp) {

#ifdef WRITESIZE32

		if (size == 0) {	//end of transfer
			ep_regs_ch[ep]->STAT |= ZLPS_MASK;	// next one is a zero length packet
			return;
		}
		while (size > 4) {

			_fifo = (int *) bp;
			ep_regs_ch[ep]->FDAT.FDAT_32 = __swab32(*_fifo);
			bp = bp + 4;
			size = size - 4;
		}
		if (size == 4) {
			ep_regs_ch[ep]->FCTRL |= 0x20000000;	// next write is last one of packet

			_fifo = (int *) bp;
			ep_regs_ch[ep]->FDAT.FDAT_32 = __swab32(*_fifo);
			bp = bp + 4;
			size = size - 4;

		} else if (size < 4) {
			while (--size) {
				ep_regs_ch[ep]->FDAT.FDAT_8[3] = *bp++;
			}
			ep_regs_ch[ep]->FCTRL |= 0x20000000;	// next write is last one of packet
			ep_regs_ch[ep]->FDAT.FDAT_8[3] = *bp;
		}
# else
		if (size == 0) {	//end of transfer
			ep_regs_ch[ep]->STAT |= ZLPS_MASK;	// next one is a zero length packet
			return;
		} else {
			while (--size) {
				ep_regs_ch[ep]->FDAT.FDAT_8[3] = *bp++;
			}
			ep_regs_ch[ep]->FCTRL |= 0x20000000;	// next write is last one of packet
			ep_regs_ch[ep]->FDAT.FDAT_8[3] = *bp;
		}

#endif

	}
}

/* ********************************************************************************************* */

/**
 * mx1_start - start transmit
 * @ep:
 */

static void __inline__
mx1_start(unsigned int ep, struct usb_endpoint_instance *endpoint, int restart)
{

	int sent;
	if (endpoint->tx_urb) {
		struct urb *urb = endpoint->tx_urb;

		while (TRUE) {

			if ((urb->actual_length - endpoint->sent) > 0) {
				sent = endpoint->last =
				    MIN(urb->actual_length - endpoint->sent,
					endpoint->tx_packetSize);
				send_data(ep, urb->buffer + endpoint->sent,
					  endpoint->last);
				endpoint->sent += sent;
				endpoint->last -= sent;
			}

			if ((urb->actual_length - endpoint->sent) == 0) {
				if (sent == endpoint->tx_packetSize) {

					// XXX ZLP
					endpoint->last = 0;
					if (ep == 0) {
						send_data(ep,
							  urb->buffer +
							  endpoint->sent, 0);
					}
				}
				if (ep == 0) {
					ep_regs_ch[ep]->INTR_MASK |=
					    FIFO_LOW_MASK;
				}
				usbd_tx_complete_irq(endpoint, 0);
				break;
			}
			if (ep == 0) {
				break;
			}
		}

	}

}

/* ********************************************************************************************* */
void
ep0_interrupt(int irq, void *dev_id, struct pt_regs *regs)
{

	int i;
	unsigned char *cp = (unsigned char *) &ep0_urb->device_request;
	struct usb_endpoint_instance *endpoint;
	U32 fifoWord;

	dbg_intr(1, "ep0_interrupt");

	endpoint = ep_endpoints[0];

	if ((ep_regs_ch[0]->INTR_STAT & MDEVREQ_MASK) > 0) {
		ep_regs_ch[0]->STAT = FLUSH_MASK;
		ep_regs_ch[0]->INTR_STAT = MDEVREQ_MASK;	// clear MDEVREQ

	} else if ((ep_regs_ch[0]->INTR_STAT & DEVREQ_MASK) > 0) {

		for (i = 0; i < 2; i++) {
			fifoWord = ep_regs_ch[0]->FDAT.FDAT_32;
			*cp++ = (U8) (fifoWord >> 24);
			*cp++ = (U8) (fifoWord >> 16);
			*cp++ = (U8) (fifoWord >> 8);
			*cp++ = (U8) fifoWord;
		}

		if (storage_flag) {
			if (ep0_urb->device_request.bmRequestType == 0x80 && ep0_urb->device_request.bRequest == 0x06 && __swab16(ep0_urb->device_request.wValue) == 0x01 && ep0_urb->device_request.wLength == 0x40) {	//Win 2000 bug
				ep0_urb->device_request.wLength = 0x8;
			}

			if (ep0_urb->device_request.wLength >= 0xff) {
				ep0_urb->device_request.wLength = 32;
			}

		}

		if (usbd_recv_setup(ep0_urb)) {
			printk(KERN_ERR "BAD SETUP");
			ep_regs_ch[0]->INTR_STAT = 0x1FF;	// clear all interrupts
			return;
		}

		if ((ep0_urb->device_request.
		     bmRequestType & USB_REQ_DIRECTION_MASK) ==
		    USB_REQ_HOST2DEVICE) {

			// Control Write - we are receiving more data from the host
			endpoint->rcv_urb = ep0_urb;
			endpoint->rcv_urb->actual_length = 0;

		} else {
			// Control Read - we are sending data to the host

			if (!(ep0_urb->device_request.wLength)) {
				udc_stall_ep(0);
				return;
			}
			// verify that we have non-zero length response
			if (!ep0_urb->actual_length) {
				udc_stall_ep(0);
				return;
			}
			// start sending
			endpoint->tx_urb = ep0_urb;
			endpoint->sent = 0;
			endpoint->last = 0;
			endpoint->state = 1;	//DATA_STATE_XMIT;

			ep_regs_ch[0]->INTR_MASK &= ~(FIFO_LOW_MASK);
			mx1_start(0, endpoint, 0);

		}
		ep_regs_ch[0]->INTR_STAT = 0x1FF;	// clear all interrupts

	} else if ((ep_regs_ch[0]->INTR_STAT & FIFO_LOW_MASK) > 0) {
		ep_regs_ch[0]->INTR_STAT = FIFO_LOW_MASK;
		mx1_start(0, endpoint, 0);

	}

}

void
ep1_interrupt(int irq, void *dev_id, struct pt_regs *regs)
{
	struct usb_endpoint_instance *endpoint;
	endpoint = ep_endpoints[1];

	dbg_intr(1, "ep1_interrupt");

	ep_regs_ch[1]->INTR_STAT = EOF_MASK;

	if (endpoint) {
		mx1_start(1, endpoint, 0);
	}

	ep_regs_ch[1]->INTR_STAT = ~EOF_MASK;

}

// handle OUT endpoint activities (command and data from host)

void
ep2_interrupt(int irq, void *dev_id, struct pt_regs *regs)
{
	U8 i;
	struct usb_endpoint_instance *endpoint;
	U32 len;
	struct urb *rcv_urb;

	endpoint = ep_endpoints[2];

	dbg_intr(1, "ep2_interrupt");

	if (endpoint) {
		if (!endpoint->rcv_urb) {
			endpoint->rcv_urb = first_urb_detached(&endpoint->rdy);
		}

		if (endpoint->rcv_urb) {

			rcv_urb = endpoint->rcv_urb;

			if ((ep_regs_ch[2]->INTR_STAT & EOT_MASK) > 0) {
				dbg_intr(3, "EOT interrupt");
				len = 0;
				unsigned char *cp =
				    endpoint->rcv_urb->buffer +
				    endpoint->rcv_urb->actual_length;

				if (cp) {
					// read available bytes into urb buffer
					while (TRUE) {
						if (((ep_regs_ch[2]->
						      STAT & BYTECOUNTMASK) >>
						     16) == 0) {
							break;
						}
						for (i = 0; i < 32; i++) {
							*cp =
							    ep_regs_ch[2]->FDAT.
							    FDAT_8[3];
							len++;
							cp++;
							if (((ep_regs_ch[2]->
							      STAT &
							      BYTECOUNTMASK) >>
							     16) == 0) {
								break;
							}
						}

						if (storage_flag || len > 0) {
							usbd_rcv_complete_irq
							    (endpoint, len, 0);
							if (endpoint->rcv_urb
							    && rcv_urb !=
							    endpoint->rcv_urb) {
								rcv_urb =
								    endpoint->
								    rcv_urb;
								cp = endpoint->
								    rcv_urb->
								    buffer +
								    endpoint->
								    rcv_urb->
								    actual_length;
							}
							len = 0;
						}
					}
				} else {
					printk(KERN_ERR
					       "read[%d] bad arguements\n",
					       udc_interrupts);
				}

				ep_regs_ch[2]->INTR_STAT = 0x1FF;	// clear all interrupts
			} else if ((ep_regs_ch[2]->INTR_STAT & EOF_MASK) > 0) {

				ep_regs_ch[2]->INTR_STAT = EOF_MASK;
				dbg_intr(3, "EOF interrupt");
				len = 0;
				unsigned char *cp =
				    endpoint->rcv_urb->buffer +
				    endpoint->rcv_urb->actual_length;

				if (cp) {
					while (((ep_regs_ch[2]->
						 STAT >> 16) & 0x7F) >= 0x20) {
						for (i = 0; i < 32; i++) {
							*cp =
							    ep_regs_ch[2]->FDAT.
							    FDAT_8[3];
							len++;
							cp++;
						}
						usbd_rcv_complete_irq(endpoint,
								      len, 0);
						if (endpoint->rcv_urb
						    && rcv_urb !=
						    endpoint->rcv_urb) {
							rcv_urb =
							    endpoint->rcv_urb;
							cp = endpoint->rcv_urb->
							    buffer +
							    endpoint->rcv_urb->
							    actual_length;
						}
						if (storage_flag
						    && endpoint->rcv_urb->
						    actual_length ==
						    STORAGE_BLOCK_SIZE) {
							usbd_rcv_complete_irq
							    (endpoint, 0, 0);
							if (endpoint->rcv_urb
							    && rcv_urb !=
							    endpoint->rcv_urb) {
								rcv_urb =
								    endpoint->
								    rcv_urb;
								cp = endpoint->
								    rcv_urb->
								    buffer +
								    endpoint->
								    rcv_urb->
								    actual_length;
							}
						}
						len = 0;
					}

				} else {
					printk(KERN_ERR
					       "read[%d] bad arguements\n",
					       udc_interrupts);
				}

			}

		} else {

			printk(KERN_ERR " no rcv_urb\n");
			char sp;
			for (i = 0; i < 32; i++) {
				if (((ep_regs_ch[2]->
				      STAT & BYTECOUNTMASK) >> 16) == 0) {
					break;
				}
				sp = ep_regs_ch[2]->FDAT.FDAT_8[3];
			}

		}

	}
}

void
ep3_interrupt(int irq, void *dev_id, struct pt_regs *regs)
{
	struct usb_endpoint_instance *endpoint;
	endpoint = ep_endpoints[3];

	dbg_intr(1, "ep3_interrupt");

	ep_regs_ch[3]->INTR_STAT = 0x1FF;	// clear all interrupts

}

void
usb_interrupt(int irq, void *dev_id, struct pt_regs *regs)
{

	if (_reg_USBD_INTR_STAT & SOF_MASK) {
		_reg_USBD_CTRL |= CMD_OVER_MASK;	// signal CMD_OVER
	}

	if (cable_connect == 0) {
		cable_connect = 1;
	}
	_reg_USBD_INTR_STAT = 0x800000FF;
}

/***********************************************************************************************/
void
_iUsbDelay(U32 count)
{
	U32 i;

	i = 0;
	while (i < count) {
		i++;
	}
}

void
_iUsbReset()
{
	U8 i;

	/* reset and enable USB device module */
	_reg_USBD_ENABLE &= ~0x40000000;	/* disable module */
	_iUsbDelay(10);
	_reg_USBD_ENABLE &= ~0x00000001;	/* select 32-bit mode FIFO */
	_reg_USBD_ENABLE &= ~0x10000000;	/* Set big endian */
	_reg_USBD_ENABLE |= 0x40000000;	/* enable module */

	/* remark: resetting the module will enable module automatically */
	while (!(_reg_USBD_ENABLE & 0x40000000)) ;	// wait until it is enabled

	// fill endpoint configuration buffer
	for (i = 0; i < 55; i++) {
		_reg_USBD_EPBUF = gEPconfigData[i];
		while (_reg_USBD_CFGBSY & 0x40000000) ;	// wait until busy bit is cleared
	}

	// mask interrupts
	_reg_USBD_INTR_STAT = 0x800000FF;	// clear general interrupts
	_reg_USBD_INTR_MASK = 0x800000FF;	// mask all general interrupts
	ep_regs_ch[0]->INTR_STAT = 0x000001FF;	// clear EP interrupts
	ep_regs_ch[0]->INTR_MASK = 0x000001FF;	// mask all EP interrupts
	ep_regs_ch[1]->INTR_STAT = 0x000001FF;	// clear EP interrupts
	ep_regs_ch[1]->INTR_MASK = 0x000001FF;	// mask all EP interrupts
	ep_regs_ch[2]->INTR_STAT = 0x000001FF;	// clear EP interrupts
	ep_regs_ch[2]->INTR_MASK = 0x000001FF;	// mask all EP interrupts
	ep_regs_ch[3]->INTR_STAT = 0x000001FF;	// clear EP interrupts
	ep_regs_ch[3]->INTR_MASK = 0x000001FF;	// mask all EP interrupts
	ep_regs_ch[4]->INTR_STAT = 0x000001FF;	// clear EP interrupts
	ep_regs_ch[4]->INTR_MASK = 0x000001FF;	// mask all EP interrupts
	ep_regs_ch[5]->INTR_STAT = 0x000001FF;	// clear EP interrupts
	ep_regs_ch[5]->INTR_MASK = 0x000001FF;	// mask all EP interrupts

	// configure USB_EPn_STAT registers and flush FIFOs
	ep_regs_ch[0]->STAT = 0x00000002;
	ep_regs_ch[1]->STAT = 0x000000D2;	// IN, MAX PS 32, BULK
	ep_regs_ch[2]->STAT = 0x00000052;	// OUT, MAX pS 32, BULK
	ep_regs_ch[3]->STAT = 0x000000BA;	// IN, MAX PS 16, INTR
	ep_regs_ch[4]->STAT = 0x0000003A;	// OUT, MAX PS 16, INTR
	ep_regs_ch[5]->STAT = 0x000000BA;	// IN, MAX PS 16, INTR

	// configure USB_EPn_FCTRL registers
	ep_regs_ch[0]->FCTRL = 0x0F000000;
	ep_regs_ch[1]->FCTRL = 0x0B000000;
	ep_regs_ch[2]->FCTRL = 0x0F000000;
	ep_regs_ch[3]->FCTRL = 0x0B000000;
	ep_regs_ch[4]->FCTRL = 0x0F000000;
	ep_regs_ch[5]->FCTRL = 0x0B000000;

	// configure USB_EPn_FALRM registers
	ep_regs_ch[0]->FALRM = 0x00000000;
	ep_regs_ch[1]->FALRM = 0x00000000;
	ep_regs_ch[2]->FALRM = 0x00000020;
	ep_regs_ch[3]->FALRM = 0x00000010;
	ep_regs_ch[4]->FALRM = 0x00000010;
	ep_regs_ch[5]->FALRM = 0x00000010;

	while (_reg_USBD_INTR_STAT & RST_START_MASK) ;	// wait until reset signaling finished
	_reg_USBD_CTRL = 0x0000001A;	// enable module
}

/* ********************************************************************************************* */
/**
 * mx1_usbd_probe - initialize
 *
 *
 **/
static int
mx1_usbd_probe(void)
{
	int retval;

	_reg_CCM_CSCR &= ~USB_DIV_MASK;
	_reg_CCM_CSCR |= 0x04000000;    // set USB DIV to divide-by-2

	// config port for USBD
	retval = mx1_register_gpios(PORT_B, (1 << 23) |	/* USB_SUS */
				    (1 << 27) |	/* USB_VMO */
				    (1 << 26) |	/* USB_VPO */
				    (1 << 21),	/* USB_OE */
				    PRIMARY | OUTPUT);

	if (retval < 0)
		goto errorA;

	retval = mx1_register_gpios(PORT_B, (1 << 25) |	/* USB_VM */
				    (1 << 24) |	/* USB_VP */
				    (1 << 22),	/* USB_RCV */
				    PRIMARY | INPUT);

	if (retval < 0)
		goto errorB;

	ep_regs_ch[0] = (EP_REGS_CH *) (USBD_BASE + EP0_OFFSET);
	ep_regs_ch[1] = (EP_REGS_CH *) (USBD_BASE + EP1_OFFSET);
	ep_regs_ch[2] = (EP_REGS_CH *) (USBD_BASE + EP2_OFFSET);
	ep_regs_ch[3] = (EP_REGS_CH *) (USBD_BASE + EP3_OFFSET);
	ep_regs_ch[4] = (EP_REGS_CH *) (USBD_BASE + EP4_OFFSET);
	ep_regs_ch[5] = (EP_REGS_CH *) (USBD_BASE + EP5_OFFSET);

	_iUsbReset();

	ep_regs_ch[0]->INTR_MASK &= ~(DEVREQ_MASK);	// enable DEVREQ for EP0
	ep_regs_ch[2]->INTR_MASK &= ~(EOT_MASK);	// enable EOT for EP2
	ep_regs_ch[2]->INTR_MASK &= ~(EOF_MASK);	// enable EOT for EP2
	_reg_USBD_INTR_MASK &= ~(SOF_MASK);
	return 0;
      errorB:
	mx1_unregister_gpios(PORT_B,
			     (1 << 23) | (1 << 27) | (1 << 26) | (1 << 21));
      errorA:
	mx1_unregister_gpios(PORT_B, (1 << 25) | (1 << 24) | (1 << 22));

	return retval;
}

/* ********************************************************************************************* */
/*
 * Start of public functions.
 */

/**
 * udc_start_in_irq - start transmit
 * @eendpoint: endpoint instance
 *
 * Called by bus interface driver to see if we need to start a data transmission.
 */
void
udc_start_in_irq(struct usb_endpoint_instance *endpoint)
{

	if (((endpoint->endpoint_address) & 0x7f) == 0) {
		ep_regs_ch[0]->INTR_MASK &= ~(FIFO_LOW_MASK);
	}
	mx1_start(endpoint->endpoint_address & 0x7f, endpoint, 0);
}

/**
 * udc_init - initialize
 *
 * Return non-zero if we cannot see device.
 **/
int
udc_init(void)
{
	// probe for mx1 usbd
	return mx1_usbd_probe();
}

/**
 * udc_start_in - start transmit
 * @eendpoint: endpoint instance
 *
 * Called by bus interface driver to see if we need to start a data transmission.
 */
void
udc_start_in(struct usb_endpoint_instance *endpoint)
{
	if (endpoint) {
		unsigned long flags;
		local_irq_save(flags);
		if (!endpoint->tx_urb) {
			mx1_start(endpoint->endpoint_address & 0x7f, endpoint,
				  0);
		}
		local_irq_restore(flags);
	}
}

/**
 * udc_stall_ep - stall endpoint
 * @ep: physical endpoint
 *
 * Stall the endpoint.
 */
void
udc_stall_ep(unsigned int ep)
{
	dbg_stall(3, " udc_stall_ep \n");
	if (ep < UDC_MAX_ENDPOINTS) {
		ep_regs_ch[ep]->STAT |= FORCE_STALL_MASK;
	}
}

/**
 * udc_reset_ep - reset endpoint
 * @ep: physical endpoint
 * reset the endpoint.
 *
 * returns : 0 if ok, -1 otherwise
 */
void
udc_reset_ep(unsigned int ep)
{
	dbg_tx(6, " udc_reset_ep ep=%d \n", ep);
	if (ep < UDC_MAX_ENDPOINTS) {
		// reset
	}
}

/**
 * udc_endpoint_halted - is endpoint halted
 * @ep:
 *
 * Return non-zero if endpoint is halted
 */
int
udc_endpoint_halted(unsigned int ep)
{
	dbg_tx(6, " udc_endpoint_halted \n");
	return 0;
}

/**
 * udc_set_address - set the USB address for this device
 * @address:
 *
 * Called from control endpoint function after it decodes a set address setup packet.
 */
void
udc_set_address(unsigned char address)
{
	dbg_tx(6, " udc_set_address  %d \n", address);
	// address cannot be setup until ack received
}

/**
 * udc_serial_init - set a serial number if available
 */
int __init
udc_serial_init(struct usb_bus_instance *bus)
{
	return -EINVAL;
}

/* ********************************************************************************************* */

/**
 * udc_max_endpoints - max physical endpoints
 *
 * Return number of physical endpoints.
 */
int
udc_max_endpoints(void)
{
	dbg_tx(6, " udc_max_endpoints = UDC_MAX_ENDPOINTS \n");
	return UDC_MAX_ENDPOINTS;
}

/**
 * udc_check_ep - check logical endpoint
 * @lep:
 *
 * Return physical endpoint number to use for this logical endpoint or zero if not valid.
 */
int
udc_check_ep(int logical_endpoint, int packetsize)
{

	return (((logical_endpoint & 0xf) >= UDC_MAX_ENDPOINTS)
		|| (packetsize > 64)) ? 0 : (logical_endpoint & 0xf);
}

/**
 * udc_set_ep - setup endpoint
 * @ep:
 * @endpoint:
 *
 * Associate a physical endpoint with endpoint_instance
 */
void
udc_setup_ep(struct usb_device_instance *device, unsigned int ep,
	     struct usb_endpoint_instance *endpoint)
{
	dbg_init(4, " udc_setup_ep: ep=%d \n", ep);
	if (ep < UDC_MAX_ENDPOINTS) {

		ep_endpoints[ep] = endpoint;

		// ep0
		if (ep == 0) {
		}
		// IN
		else if (endpoint->endpoint_address & 0x80) {
		}
		// OUT
		else if (endpoint->endpoint_address) {

			usbd_fill_rcv(device, endpoint, storage_flag ? 22 : 5);
			endpoint->rcv_urb = first_urb_detached(&endpoint->rdy);
		}
	}
}

/**
 * udc_disable_ep - disable endpoint
 * @ep:
 *
 * Disable specified endpoint
 */
void
udc_disable_ep(unsigned int ep)
{
	dbg_tx(6, " udc_disable_ep \n");
	if (ep < UDC_MAX_ENDPOINTS) {
		struct usb_endpoint_instance *endpoint;

		if ((endpoint = ep_endpoints[ep])) {
			ep_endpoints[ep] = NULL;
			usbd_flush_ep(endpoint);
		}
	}
}

/* ********************************************************************************************* */

/**
 * udc_connected - is the USB cable connected
 *
 * Return non-zeron if cable is connected.
 */
int
udc_connected()
{
	dbg_tx(6, " udc_connected \n");
	if (!first_connection)
		cable_connect = 0;
	if (first_connection) {
		first_connection = 0;
	}
	udelay(10000);
	return cable_connect;
}

/**
 * udc_connect - enable pullup resistor
 *
 * Turn on the USB connection by enabling the pullup resistor.
 */
void
udc_connect(void)
{
	dbg_tx(6, " udc_connect \n");
}

/**
 * udc_disconnect - disable pullup resistor
 *
 * Turn off the USB connection by disabling the pullup resistor.
 */
void
udc_disconnect(void)
{
	dbg_tx(6, " udc_disconnect \n");
}

/* ********************************************************************************************* */

/**
 * udc_enable_interrupts - enable interrupts
 *
 * Switch on UDC interrupts.
 *
 */
void
udc_all_interrupts(struct usb_device_instance *device)
{
	dbg_tx(6, "udc_all_interrupts enable \n");
	// set interrupt mask
	// mask interrupts

	ep_regs_ch[0]->INTR_MASK &= ~(DEVREQ_MASK);	// enable DEVREQ for EP0
	ep_regs_ch[2]->INTR_MASK &= ~(EOT_MASK);	// enable EOT for EP2
	ep_regs_ch[2]->INTR_MASK &= ~(EOF_MASK);	// enable EOT for EP2
	_reg_USBD_INTR_MASK &= ~(SOF_MASK);
}

/**
 * udc_suspended_interrupts - enable suspended interrupts
 *
 * Switch on only UDC resume interrupt.
 *
 */
void
udc_suspended_interrupts(struct usb_device_instance *device)
{
	dbg_tx(6, "udc_suspended_interrupts enable \n");

}

/**
 * udc_disable_interrupts - disable interrupts.
 *
 * switch off interrupts
 */
void
udc_disable_interrupts(struct usb_device_instance *device)
{

	_reg_USBD_INTR_STAT = 0x800000FF;	// clear general interrupts
	_reg_USBD_INTR_MASK = 0x800000FF;	// mask all general interrupts
	ep_regs_ch[0]->INTR_STAT = 0x000001FF;	// clear EP interrupts
	ep_regs_ch[0]->INTR_MASK = 0x000001FF;	// mask all EP interrupts
	ep_regs_ch[1]->INTR_STAT = 0x000001FF;	// clear EP interrupts
	ep_regs_ch[1]->INTR_MASK = 0x000001FF;	// mask all EP interrupts
	ep_regs_ch[2]->INTR_STAT = 0x000001FF;	// clear EP interrupts
	ep_regs_ch[2]->INTR_MASK = 0x000001FF;	// mask all EP interrupts
	ep_regs_ch[3]->INTR_STAT = 0x000001FF;	// clear EP interrupts
	ep_regs_ch[3]->INTR_MASK = 0x000001FF;	// mask all EP interrupts
	ep_regs_ch[4]->INTR_STAT = 0x000001FF;	// clear EP interrupts
	ep_regs_ch[4]->INTR_MASK = 0x000001FF;	// mask all EP interrupts
	ep_regs_ch[5]->INTR_STAT = 0x000001FF;	// clear EP interrupts
	ep_regs_ch[5]->INTR_MASK = 0x000001FF;	// mask all EP interrupts

}

/* ********************************************************************************************* */

/**
 * udc_ep0_packetsize - return ep0 packetsize
 */
int
udc_ep0_packetsize(void)
{
	//dbg_tx(6," udc_ep0_packetsize  = %d\n" ,  EP0_PACKETSIZE );
	return EP0_PACKETSIZE;
}

/**
 * udc_enable - enable the UDC
 *
 * Switch on the UDC
 */
void
udc_enable(struct usb_device_instance *device)
{

	// save the device structure pointer
	udc_device = device;

	// ep0 urb   alloc
	if (!ep0_urb) {
		if (!
		    (ep0_urb =
		     usbd_alloc_urb(device, device->function_instance_array, 0,
				    512))) {
			//printk (KERN_ERR "udc_enable: usbd_alloc_urb failed\n");
		}
	} else {
		//printk (KERN_ERR "udc_enable: ep0_urb already allocated\n");
	}

	// enable UDC
	_reg_USBD_ENABLE |= 0x40000000;	/* enable module */

}

/**
 * udc_disable - disable the UDC
 *
 * Switch off the UDC
 */
void
udc_disable(void)
{
	//dbg_tx(6," udc_disable \n" );
	//dbg_tx(6,"************************* udc_disable:\n");

	// disable UDC
	_reg_USBD_ENABLE &= ~0x40000000;	/* disable module */

	// reset device pointer
	udc_device = NULL;

	// ep0 urb
	if (ep0_urb) {
		usbd_dealloc_urb(ep0_urb);
		ep0_urb = NULL;
	}
	mx1_unregister_gpios(PORT_B,
			     (1 << 23) | (1 << 27) | (1 << 26) | (1 << 21));
	mx1_unregister_gpios(PORT_B, (1 << 25) | (1 << 24) | (1 << 22));
}

/**
 * udc_startup - allow udc code to do any additional startup
 */
void
udc_startup_events(struct usb_device_instance *device)
{
	struct usb_function_instance *function;
	struct usb_configuration_description *configuration;
	struct usb_interface_description *interface;
	int i, j;

	dbg_tx(6, " udc_startup_events:device->name=%s \n", device->name);

	function = device->function_instance_array;
	for (i = 0; i < function->function_driver->configurations; i++) {
		configuration =
		    function->function_driver->configuration_description + i;
		for (j = 0; j < configuration->interfaces; j++) {
			int logical_endpoints = 1;
			interface = configuration->interface_list + j;
			if (interface->bInterfaceClass ==
			    USB_CLASS_MASS_STORAGE) {
				storage_flag = 1;
			}
		}
	}

	usbd_device_event(device, DEVICE_INIT, 0);
	usbd_device_event(device, DEVICE_CREATE, 0);
	usbd_device_event(device, DEVICE_HUB_CONFIGURED, 0);
	usbd_device_event(device, DEVICE_RESET, 0);
	usbd_device_event(device, DEVICE_CONFIGURED, 0);
	if (!first_connection)
		first_connection = 1;
}

/* ********************************************************************************************* */

/**
 * udc_name - return name of USB Device Controller
 */
char *
udc_name(void)
{
	//dbg_tx(6," udc_name= %s\n ", UDC_NAME  );
	return UDC_NAME;
}

/**
 * udc_request_udc_irq - request UDC interrupt
 *
 * Return non-zero if not successful.
 */
int
udc_request_udc_irq()
{

	if (request_irq
	    (USBD_INT0, ep0_interrupt, SA_INTERRUPT, "USB EP0", 0) < 0)
		printk("Intr request for source %d failed !\n", USBD_INT0);
	if (request_irq
	    (USBD_INT1, ep1_interrupt, SA_INTERRUPT, "USB EP1", 0) < 0)
		printk("Intr request for source %d failed !\n", USBD_INT1);
	if (request_irq
	    (USBD_INT2, ep2_interrupt, SA_INTERRUPT, "USB EP2", 0) < 0)
		printk("Intr request for source %d failed !\n", USBD_INT2);
	if (request_irq
	    (USBD_INT3, ep3_interrupt, SA_INTERRUPT, "USB EP3", 0) < 0)
		printk("Intr request for source %d failed !\n", USBD_INT3);
	if (request_irq
	    (USBD_INT4, usb_interrupt, SA_INTERRUPT, "USB EP4", 0) < 0)
		printk("Intr request for source %d failed !\n", USBD_INT4);
	if (request_irq
	    (USBD_INT5, usb_interrupt, SA_INTERRUPT, "USB EP5", 0) < 0)
		printk("Intr request for source %d failed !\n", USBD_INT5);
	if (request_irq(USBD_INT6, usb_interrupt, SA_INTERRUPT, "USB", 0)
	    < 0)
		printk("Intr request for source %d failed !\n", USBD_INT6);

	return 0;
}

/**
 * udc_request_cable_irq - request Cable interrupt
 *
 * Return non-zero if not successful.
 */
int
udc_request_cable_irq()
{
	return 0;
}

/**
 * udc_request_udc_io - request UDC io region
 *
 * Return non-zero if not successful.
 */
int
udc_request_io()
{
	return 0;
}

/**
 * udc_release_udc_irq - release UDC irq
 */
void
udc_release_udc_irq()
{
	free_irq(USBD_INT0, 0);
	free_irq(USBD_INT1, 0);
	free_irq(USBD_INT2, 0);
	free_irq(USBD_INT3, 0);
	free_irq(USBD_INT4, 0);
	free_irq(USBD_INT5, 0);
	free_irq(USBD_INT6, 0);
}

/**
 * udc_release_cable_irq - release Cable irq
 */
void
udc_release_cable_irq()
{
}

/**
 * udc_release_release_io - release UDC io region
 */
void
udc_release_io()
{
}

/**
 * udc_regs - dump registers
 *
 * Dump registers with printk
 */
void
udc_regs(void)
{
}
