/*
 * Copyright (C) 2002 MontaVista Software Inc.
 * Author: Jun Sun, jsun@mvista.com or jsun@junsun.net
 *
 * This program is free software; you can redistribute  it and/or modify it
 * under  the terms of  the GNU General  Public License as published by the
 * Free Software Foundation;  either version 2 of the  License, or (at your
 * option) any later version.
 *
 */
#ifndef _ASM_FPU_H
#define _ASM_FPU_H

#include <linux/config.h>
#include <linux/sched.h>

#include <asm/mipsregs.h>
#include <asm/cpu.h>
#include <asm/bitops.h>
#include <asm/processor.h>
#include <asm/current.h>

struct sigcontext;

extern asmlinkage int (*save_fp_context)(struct sigcontext *sc);
extern asmlinkage int (*restore_fp_context)(struct sigcontext *sc);

extern void fpu_emulator_init_fpu(void);
extern void _init_fpu(void);
extern void _save_fp(struct task_struct *);
extern void _restore_fp(struct task_struct *);

#if defined(CONFIG_CPU_SB1)
#define __enable_fpu_hazard()						\
do {									\
	asm(".set push		\n\t"					\
	    ".set mips64	\n\t"					\
	    ".set noreorder	\n\t"					\
	    "ssnop		\n\t"					\
	    "bnezl $0, .+4	\n\t"					\
	    "ssnop		\n\t"					\
	    ".set pop");						\
} while (0)
#else
#define __enable_fpu_hazard()                                           \
do {                                                                    \
        asm("nop;nop;nop;nop");         /* max. hazard */               \
} while (0)
#endif

#define __enable_fpu()							\
do {									\
        set_c0_status(ST0_CU1);						\
        __enable_fpu_hazard();						\
} while (0)

#define __disable_fpu()							\
do {									\
	clear_c0_status(ST0_CU1);					\
	/* We don't care about the c0 hazard here  */			\
} while (0)

#define enable_fpu()							\
do {									\
	if (cpu_has_fpu)						\
		__enable_fpu();						\
} while (0)

#define disable_fpu()							\
do {									\
	if (cpu_has_fpu)						\
		__disable_fpu();					\
} while (0)


#define clear_fpu_owner() do { current->flags &= ~PF_USEDFPU; } while(0)

static inline int is_fpu_owner(void)
{
	return cpu_has_fpu && ((current->flags & PF_USEDFPU) != 0); 
}

static inline void own_fpu(void)
{
	if (cpu_has_fpu) {
		__enable_fpu();
		KSTK_STATUS(current) |= ST0_CU1;
		current->flags |= PF_USEDFPU;
	}
}

static inline void loose_fpu(void)
{
	if (cpu_has_fpu) {
		KSTK_STATUS(current) &= ~ST0_CU1;
		current->flags &= ~PF_USEDFPU;
		__disable_fpu();
	}
}

static inline void init_fpu(void)
{
	if (cpu_has_fpu) {
		_init_fpu();
	} else {
		fpu_emulator_init_fpu();
	}
}

static inline void save_fp(struct task_struct *tsk)
{
	if (cpu_has_fpu)
		_save_fp(tsk);
}

static inline void restore_fp(struct task_struct *tsk)
{
	if (cpu_has_fpu)
		_restore_fp(tsk);
}

static inline unsigned long long *get_fpu_regs(struct task_struct *tsk)
{
	preempt_disable();
	if (cpu_has_fpu) {
		if ((tsk == current) && is_fpu_owner()) 
			_save_fp(current);
		preempt_enable();
		return (unsigned long long *)&tsk->thread.fpu.hard.fp_regs[0];
	} else {
		preempt_enable();
		return (unsigned long long *)tsk->thread.fpu.soft.regs;
	}
}

#endif /* _ASM_FPU_H */
