/*
 * include/asm-mips/vr4181a/vr4181a.h
 *
 * Include for NEC VR4181A.
 *
 * Author: Yoichi Yuasa <yyuasa@mvista.com, or source@mvista.com>
 *
 * 2002,2003 (c) MontaVista, Software, Inc. This file is licensed under
 * the terms of the GNU General Public License version 2. This program
 * is licensed "as is" without any warranty of any kind, whether express
 * or implied.
 */
#ifndef __NEC_VR4181A
#define __NEC_VR4181A

#include <linux/interrupt.h>

#include <asm/addrspace.h>
#include <asm/io.h>

/*
 * Interrupt Numbers
 */
#define NMI_IRQ			8
#define POWER_SWITCH_IRQ	9
#define RTC_LONG1_IRQ		10
#define RTC_LONG2_IRQ		11
#define RTC_ELAPSEDTIME_IRQ	12
#define PIU_IRQ			13
#define AIU_IRQ			14
#define KIU_IRQ			15
#define GPIO_CASCADE0_IRQ	16
#define GPIO_CASCADE1_IRQ	17
#define GPIO_CASCADE2_IRQ	18
#define GPIO_CASCADE3_IRQ	19
#define SIU0_IRQ		20
#define SIU1_IRQ		21
#define SIU2_IRQ		22
#define SOFTWARE_IRQ		23
#define ECU_SLOT0_STATUS_IRQ	24
#define ECU_SLOT0_IRQ		25
#define ECU_SLOT1_STATUS_IRQ	26
#define ECU_SLOT1_IRQ		27
#define PWMU0_IRQ		28
/* RFU	*/
#define CSI_IRQ			31
#define LCU_IRQ			32
#define I2CU0_IRQ		33
#define I2CU1_IRQ		34
#define I2SU_IRQ		35
#define USBHU_IRQ		36
#define USBFU_IRQ		37
#define AC97U_IRQ		38
#define AC97U_DMA_IRQ		39
#define PCIIF_BUS_ERR_IRQ	40
/* RFU	*/
#define EXIBU_TIMEOUT_IRQ	48
/* RFU	*/
#define CPUIF_UNDECR_ERR_IRQ	56
/* RFU	*/
#define CPUIF_EXIBU_TIMEOUT_IRQ	58
#define CPUIF_PCI_ERR_IRQ	59
/* RFU */
#define PCIIF_SERR_IRQ		64
/* RFU */

#define GPIO_IRQ_BASE		72
#define GPIO_IRQ(x)		(GPIO_IRQ_BASE + (x))

/*
 * Internal Registers access
 */
extern unsigned long vr4181a_internal_registers_base;

#define set_vr4181a_internal_registers_base(base)  \
	do { vr4181a_internal_registers_base = (base); } while (0)

#define vr4181a_readb(offset)	\
	(*(volatile unsigned char *)(vr4181a_internal_registers_base + (offset)))
#define vr4181a_readw(offset)	\
	(*(volatile unsigned short *)(vr4181a_internal_registers_base + (offset)))
#define vr4181a_readl(offset)	\
	(*(volatile unsigned int *)(vr4181a_internal_registers_base + (offset)))

#define vr4181a_writeb(b,offset)	\
	(*(volatile unsigned char *)(vr4181a_internal_registers_base + (offset))) = (b)
#define vr4181a_writew(b,offset)	\
	(*(volatile unsigned short *)(vr4181a_internal_registers_base + (offset))) = (b)
#define vr4181a_writel(b,offset)	\
	(*(volatile unsigned int *)(vr4181a_internal_registers_base + (offset))) = (b)

/*
 * For the evasion about the problem of internal ISA register write access,
 * after write access to ECU, it is necessary to perform "vr4181a_write_fixed".
 */
#define vr4181a_write_fixed	readb(KSEG1ADDR(0x1fc00000))

/*
 * Basic Timer
 */
extern void vr4181a_timer_setup(struct irqaction *irq);

/*
 * Architecture-specific implementations of sys_reboot commands.
 */
extern void vr4181a_restart(char *command);
extern void vr4181a_halt(void);
extern void vr4181a_power_off(void);

/*
 * Clock Control Unit
 */
extern void vr4181a_ccu_init(unsigned long ecu_sysclock);

extern void vr4181a_clock_supply(unsigned short clock);
extern void vr4181a_clock_mask(unsigned short clock);

extern unsigned long vr4181a_get_tclock(void);
extern unsigned long vr4181a_get_ecu_sysclock(void);

enum {
	ADU18M_CLOCK,
	ADU_CLOCK,
	PIU_CLOCK,
	AIU_CLOCK,
	CSU_CLOCK,
	SIU0_CLOCK,
	SIU1_CLOCK,
	SIU2_CLOCK,
	PWM2_CLOCK,
	AC97_CLOCK,
	I2C_CLOCK,
	USB_HOST_CLOCK,
	USB_FUNCTION_CLOCK,
	I2S_CLOCK,
	PWM01_CLOCK,
	DMA_CLOCK,
	PCI_BRIDGE_CLOCK,
	LCD_CLOCK,
	USB_HOST_PCI_CLOCK,
	USB_FUNCTION_PCI_CLOCK,
	AC97_PCI_CLOCK,
};

/*
 * Host Bridge Unit
 */
struct vr4181a_address_window {
	unsigned long base_address;
	unsigned long size;
	unsigned char data_width;
	unsigned char pci_master_access;
};

enum {
	DATA_WIDTH_8BIT,
	DATA_WIDTH_16BIT,
	DATA_WIDTH_32BIT,
};

enum {
	PCI_MASTER_ACCESS_PROHIBIT,
	PCI_MASTER_ACCESS_ENABLE,
};

struct vr4181a_address_maps {
	struct vr4181a_address_window dram;
	struct vr4181a_address_window sync_flash_memory;
	struct vr4181a_address_window pcs0;
	struct vr4181a_address_window pcs1;
	struct vr4181a_address_window pcs2;
	struct vr4181a_address_window pcs3;
	struct vr4181a_address_window pcs4;
	struct vr4181a_address_window isa_bus;
	struct vr4181a_address_window pci_bus0;
	struct vr4181a_address_window pci_bus1;
	struct vr4181a_address_window internal_registers;
	struct vr4181a_address_window rom;
};

extern void vr4181a_hbu_init(struct vr4181a_address_maps *maps);

/*
 * External System Bus Control Unit
 */
struct vr4181a_access_config {
	unsigned char io_signal_setup_time;
	unsigned char page_access_length;
	unsigned char control_signal_active_level;
	unsigned char chip_select_active_level;
	unsigned char control_signal_deassert_timing;
	unsigned char iordy_sync_times;
	unsigned char ready_mode;
	unsigned char bus_idel_time;
	unsigned char control_signal_width_min;
	unsigned char control_signal_hold_time;
	unsigned char control_signal_assert_time_page_access;
	unsigned char control_signal_assert_time;
	unsigned short ready_signal_timeout_counter;
	unsigned char control_signal_setup_time;
	unsigned char config_active;
};

enum {
	ACTIVE_LOW,
	ACTIVE_HIGH,
};

enum {
	LATCH_1_TIME,
	LATCH_2_TIMES,
};

enum {
	READY_UNUSE,
	READY_USE,
};

struct vr4181a_access_setup {
	struct vr4181a_access_config pcs0;
	struct vr4181a_access_config pcs1;
	struct vr4181a_access_config pcs2;
	struct vr4181a_access_config pcs3;
	struct vr4181a_access_config pcs4;
	struct vr4181a_access_config isa_bus;
	struct vr4181a_access_config rom;
};

extern void vr4181a_exibu_init(struct vr4181a_access_setup *setup,
                               unsigned long lclock_max);

/*
 * Internal PCI Bridge Unit
 */
struct vr4181a_pci_base_address {
	unsigned long pci_base_address;
	unsigned char prefetch;
	unsigned char address_active;
};

enum {
	PREFETCH_PROHIBIT,
	PREFETCH_ENABLE,
};

struct vr4181a_pci_window_config {
	unsigned long pci_base_address;
	unsigned char access_width;
	unsigned char command_type;
};

enum {
	ACCESS_WIDTH_8BIT,
	ACCESS_WIDTH_16BIT,
	ACCESS_WIDTH_32BIT,
};

enum {
	COMMAND_TYPE_IO,
	COMMAND_TYPE_MEMORY,
	COMMAND_TYPE_CONFIGURATION,
};

struct vr4181a_pci_config {
	struct vr4181a_pci_base_address internal_registers;
	struct vr4181a_pci_base_address dram;
	struct vr4181a_pci_base_address sync_flash_memory;
	struct vr4181a_pci_base_address pcs0;
	struct vr4181a_pci_base_address pcs1;
	struct vr4181a_pci_base_address pcs2;
	struct vr4181a_pci_base_address pcs3;
	struct vr4181a_pci_base_address pcs4;
	struct vr4181a_pci_base_address isa_bus;
	struct vr4181a_pci_base_address rom;
	struct vr4181a_pci_window_config pci_window0;
	struct vr4181a_pci_window_config pci_window1;
	unsigned char master_latency;
	unsigned short delayed_transaction_abort;
	unsigned short retry_limit;
};

extern struct pci_ops vr4181a_pci_ops;

extern void vr4181a_iopciu_init(struct vr4181a_pci_config *config);

/*
 * Power Management Unit
 */
extern void vr4181a_reset_haltimer(void);
extern void vr4181a_soft_reset(void);
extern void vr4181a_wait(void);
extern void vr4181a_change_divide(int mode, int action);

enum {
	DIVIDE_DEFAULT,
	DIVIDE_MODE2,
	DIVIDE_MODE3,
};

enum {
	SOFT_RESET,
	HIBERNATE,
};

/*
 * CompactFlash/PC Card/IDE(ATA) Control Unit
 */
extern void vr4181a_ecu_set_window(struct vr4181a_address_window *win);
extern unsigned long vr4181a_ecu_get_map_size(int slot);
extern unsigned long vr4181a_ecu_get_io_offset(int slot);
extern unsigned long vr4181a_ecu_get_sys_start(int slot);

extern int vr4181a_ecu_get_mode(void);
extern void vr4181a_ecu_set_mode(int mode);

enum {
	ECU_MODE_UNKNOWN,
	ECU_MODE_CF,
	ECU_MODE_IDE,
};

extern void vr4181a_ecu_init(int mode);

/*
 * Serial Interface Unit
 */
extern void vr4181a_siu_init(unsigned char unit, int line);

#ifdef CONFIG_KGDB
extern void vr4181a_siu_debug_init(unsigned char unit, unsigned int divisor,
                                   unsigned char bits, unsigned char parity,
				   unsigned char stop_bits);

#define DATA_5BIT		0x00
#define DATA_6BIT		0x01
#define DATA_7BIT		0x02
#define DATA_8BIT		0x03

#define PARITY_NONE		0x00
#define PARITY_ODD		0x08
#define PARITY_EVEN		0x18
#define PARITY_MARK		0x28
#define PARITY_SPACE		0x38

#define STOP_1BIT		0x00
#define STOP_2BIT		0x04

#define BAUD_RATE_50		23040
#define BAUD_RATE_75		15360
#define BAUD_RATE_110		10473
#define BAUD_RATE_134_5		8565
#define BAUD_RATE_150		7680
#define BAUD_RATE_300		3840
#define BAUD_RATE_600		1920
#define BAUD_RATE_1200		960
#define BAUD_RATE_1800		640
#define BAUD_RATE_2000		576
#define BAUD_RATE_2400		640
#define BAUD_RATE_3600		320
#define BAUD_RATE_4800		240
#define BAUD_RATE_7200		160
#define BAUD_RATE_9600		120
#define BAUD_RATE_19200		60
#define BAUD_RATE_38400		30
#define BAUD_RATE_57600		20
#define BAUD_RATE_115200	10
#define BAUD_RATE_128000	9
#define BAUD_RATE_144000	8
#define BAUD_RATE_192000	6
#define BAUD_RATE_288000	4
#define BAUD_RATE_384000	3
#define BAUD_RATE_576000	2
#define BAUD_RATE_1152000	1
#endif

/*
 * General-purpose I/O Unit
 */
extern void vr4181a_set_irq_trigger(unsigned short pin, int trigger);

enum {
	TRIGGER_EDGE,
	TRIGGER_LEVEL,
};

extern void vr4181a_set_irq_level(unsigned short pin, int level);

enum {
	LEVEL_LOW,
	LEVEL_HIGH,
};

extern int vr4181a_set_gpio_mode(unsigned short pin, int mode, int input_mode);

enum {
	GPIO_INPUT,
	GPIO_OUTPUT,
	GPIO_OTHER,
};

enum {
	GPIO_INPUT_PROHIBIT,
	GPIO_INPUT_ENABLE,
};

extern int vr4181a_get_gpio_data(unsigned short pin);
extern int vr4181a_set_gpio_data(unsigned short pin , int data);

extern u16 vr4181a_read_nvreg(int regno);
extern void vr4181a_write_nvreg(int regno, u16 value);

extern int vr4181a_get_pinmode(void);
extern void vr4181a_set_pinmode(int mode);

#define	PINMODE_PWM0			0x01
#define PINMODE_PWM1			0x02
#define PINMODE_KSCAN6			0x04
#define PINMODE_KSCAN7			0x08
#define PINMODE_CF1_RESET_PROHIBIT	0x10
#define PINMODE_CF1_RESET_FIXED1	0x20
#define PINMODE_CF1_RESET_FIXED0	0x40
#define PINMODE_CF1_RESET_ENABLE	0x80

extern int vr4181a_get_pinmode0(void);
extern void vr4181a_set_pinmode0(int mode);

#define PINMODE0_SIU0		0x00001
#define PINMODE0_SIU1		0x00002
#define PINMODE0_CSI		0x00004
#define PINMODE0_I2S		0x00008
#define PINMODE0_KSCAN4		0x00010
#define PINMODE0_KSCAN8		0x00020
#define PINMODE0_KSCAN9		0x00040
#define PINMODE0_KSCAN10	0x00080
#define PINMODE0_KSCAN11	0x00100
#define PINMODE0_KPORT4		0x00200
#define PINMODE0_KPORT5		0x00400
#define PINMODE0_ECU1		0x00800
#define PINMODE0_FPD10		0x01000
#define PINMODE0_FPD11		0x02000
#define PINMODE0_FPD12		0x04000
#define PINMODE0_FPD13		0x08000
#define PINMODE0_FPD14		0x10000
#define PINMODE0_FPD15		0x20000

extern int vr4181a_get_pinmode1(void);
extern void vr4181a_set_pinmode1(int mode);

#define	PINMODE1_PWM2		0x01
#define	PINMODE1_I2C0		0x02
#define	PINMODE1_I2C1		0x04
#define	PINMODE1_KSCAN5		0x08
#define	PINMODE1_KPORT6		0x10
#define	PINMODE1_KPORT7		0x20
#define	PINMODE1_SIU1		0x40

extern int vr4181a_get_pinmode2(void);
extern void vr4181a_set_pinmode2(int mode);

#define PINMODE2_PROHIBIT	0x01
#define PINMODE2_SIU2		0x02
#define PINMODE2_SIU2_IRDA	0x04
#define PINMODE2_I2S		0x08
#define PINMODE2_AC97		0x10
#define PINMODE2_USB_ENABLE	0x20

extern void vr4181a_power_active_usb_function(void);
extern void vr4181a_power_save_usb_function(void);

extern void vr4181a_power_active_usb_host(void);
extern void vr4181a_power_save_usb_host(void);

/*
 * Platform
 */
extern unsigned long vr4181a_platform_init(int argc, char **argv, char **envp,
                                           int *prom_vec, char *arcs_cmdline);

#endif /* __NEC_VR4181A */
