/*
 *	Definitions for the SECurity layer
 *
 *	Author:
 *		Robert Muchsel <muchsel@acm.org>
 *
 *	This program is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU General Public License
 *	as published by the Free Software Foundation; either version
 *	2 of the License, or (at your option) any later version.
 */
 
#ifndef _LINUX_IPSEC_H
#define _LINUX_IPSEC_H

#include <linux/config.h>
#include <linux/socket.h>
#include <net/sock.h>
#include <net/route.h>
#include <linux/skbuff.h>
#ifdef CONFIG_IP_IPSEC
#include <net/sadb.h>
#include <net/spd.h>
#endif

/* Values for the set/getsockopt calls */

/* These defines are compatible with NRL IPv6, however their semantics
   is different */

#define IPSEC_LEVEL_NONE	-1	/* send plaintext, accept any */
#define IPSEC_LEVEL_DEFAULT	0	/* encrypt/authenticate if possible */
					/* the default MUST be 0, because a */
					/* socket is initialized with 0's */
#define IPSEC_LEVEL_USE		1	/* use outbound, don't require inbound */
#define IPSEC_LEVEL_REQUIRE	2	/* require both directions */
#define IPSEC_LEVEL_UNIQUE	2	/* for compatibility only */

#ifdef __KERNEL__

/* skb bit flags set on packet input processing */

#define RCV_SEC			0x0f	/* options on receive */
#define RCV_AUTH		0x01	/* was authenticated */
#define RCV_CRYPT		0x02	/* was encrypted */
#define RCV_TUNNEL		0x04	/* was tunneled */
#define SND_SEC			0xf0	/* options on send, these are */
#define SND_AUTH		0x10	/* currently unused */
#define SND_CRYPT		0x20
#define SND_TUNNEL		0x40

/*
 *	FIXME: ignores network encryption for now..
 */
 
#ifdef CONFIG_NET_SECURITY
static __inline__ int ipsec_sk_policy(struct sock *sk, struct sk_buff *skb)
{
	return ((sk->authentication < IPSEC_LEVEL_REQUIRE) ||
		(skb->security & RCV_AUTH)) &&
		((sk->encryption < IPSEC_LEVEL_REQUIRE) ||
		(skb->security & RCV_CRYPT));
}

#else

static __inline__ int ipsec_sk_policy(struct sock *sk, struct sk_buff *skb)
{
	return 1;
}
#endif /* CONFIG */

/* return value for ipsec[46]_output_check() -mk */
#define IPSEC_ACTION_BYPASS		0x0
#define IPSEC_ACTION_AUTH		0x1
#define IPSEC_ACTION_ESP		0x2
#define IPSEC_ACTION_COMP		0x4
#define IPSEC_ACTION_DROP		0x8

#ifdef CONFIG_SYSCTL
extern  int sysctl_ipsec_replay_window;
#ifdef CONFIG_IPSEC_DEBUG
extern	int sysctl_ipsec_debug_ipv4;
extern  int sysctl_ipsec_debug_ipv6;
extern  int sysctl_ipsec_debug_pfkey;
extern  int sysctl_ipsec_debug_sadb;
extern  int sysctl_ipsec_debug_spd;
#endif /* CONFIG_IPSEC_DEBUG */
#endif /* CONFIG_SYSCTL */

#ifdef CONFIG_IP_IPSEC
int ipsec4_out_ah_calc(struct iphdr *iph, struct ip_auth_hdr *authhdr, struct ipsec_sp *policy);

int ipsec4_out_get_ahsize(struct ipsec_sp *policy);
int ipsec4_out_get_espsize(struct ipsec_sp *policy);
static inline int ipsec4_out_get_hdrsize(struct ipsec_sp *policy)
{
	return ipsec4_out_get_ahsize(policy) + ipsec4_out_get_espsize(policy);
}

void ipsec4_out_enc(const void *data, unsigned length, u8 proto,
		void **newdata, unsigned *newlength, struct ipsec_sp *policy);
void ipsec4_out_finish(struct ipsec_sp *policy_ptr);

int ipsec4_input_check(struct sk_buff **skb); 
int ipsec4_output_check(struct sock *sk, struct rtable *rt, const void *frag, struct ipsec_sp **policy_ptr);
#if defined(CONFIG_IPV4_IPSEC_TUNNEL) && defined(CONFIG_IP_IPSEC)
int ipsec4_ipip_output_check(struct rtable *rt, const u8 *data, struct ipsec_sp **policy_ptr);
#endif

#ifdef CONFIG_IPSEC_DEBUG
# ifdef CONFIG_SYSCTL
#  define IPSEC4_DEBUG(fmt, args...) 					\
do {									\
	if (sysctl_ipsec_debug_ipv4) {					\
		printk(KERN_DEBUG "%s: " fmt, __FUNCTION__ , ## args);	\
	}								\
} while(0)
# else
#  define IPSEC4_DEBUG(fmt, args...) printk(KERN_DEBUG "%s: " fmt, __FUNCTION__ , ## args)
# endif /* CONFIG_SYSCTL */
#else
#  define IPSEC4_DEBUG(fmt, args...)
#endif
#endif /* CONFIG_IP_IPSEC */


#endif	/* __KERNEL__ */
#endif	/* _LINUX_IPSEC_H */
