/* $USAGI: ipsec4_output.c,v 1.13 2003/01/14 13:43:57 mk Exp $ */
/*
 * Copyright (C)2001 USAGI/WIDE Project
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Authors:
 *   Kazunori MIYAZAWA <miyazawa@linux-ipv6.org> / USAGI
 *   Mitsuru KANDA <mk@linux-ipv6.org> / USAGI
 *
 */
#include <linux/config.h>
#include <linux/module.h>
#include <linux/init.h>
#include <asm/byteorder.h>
#include <linux/kernel.h>
#include <linux/errno.h>
#include <linux/sysctl.h>
#include <linux/inet.h>
#include <linux/tcp.h>
#include <linux/udp.h>
#include <linux/smp.h>
#include <linux/list.h>
#include <linux/random.h>
#include <linux/slab.h>
#include <asm/uaccess.h>
#include <net/ipv6.h>
#include <net/sadb.h>
#include <net/spd.h>
#include <net/addrconf.h>
#include <net/snmp.h>  
#include <linux/in.h>
#include <linux/ipsec.h>
#include <linux/pfkeyv2.h> /* sa proto type */
#include <linux/pfkey.h>
#include <net/ipcomp.h>

/*
	We assume iph and authder is on continuous buffer. 
*/
int ipsec4_out_ah_calc(struct iphdr *iph, struct ip_auth_hdr *authhdr, struct ipsec_sp *policy)
{
	struct ipsec_sa *sa = NULL;
	struct ip_auth_hdr *pseudo_authhdr = NULL;
	char* pseudo_packet = NULL;
	int packetlen = 0;
	__u8* authdata = NULL;
	
	IPSEC4_DEBUG("called.\n");
	if(!policy){
		return -EINVAL;
	}

	if (!iph) {
		IPSEC4_DEBUG("iph is NULL!\n");
		return -EINVAL;
	}

	if (!authhdr) {
		authhdr = (struct ip_auth_hdr*)(((char*)iph) + iph->ihl*4);
	}

	read_lock_bh(&policy->lock);

	if (!policy->auth_sa_idx || !policy->auth_sa_idx->sa) {
		if (net_ratelimit())
			printk(KERN_WARNING "%s: ipsec(ah) SA missing.\n", __FUNCTION__);
		read_unlock_bh(&policy->lock);
		return -EINVAL;
	}

	ipsec_sa_hold(policy->auth_sa_idx->sa);
	sa = policy->auth_sa_idx->sa;

	read_unlock_bh(&policy->lock);

	packetlen = ntohs(iph->tot_len);

	pseudo_packet = kmalloc(packetlen,GFP_ATOMIC);
	if (!pseudo_packet) {
		ipsec_sa_put(sa);
		return -ENOMEM;
	}

	pseudo_authhdr = (struct ip_auth_hdr*)(pseudo_packet + (((char*)authhdr) - ((char*)iph)));

	authdata=kmalloc((sa->auth_algo.dx)->di->blocksize, GFP_ATOMIC);
	if (!authdata) {
		kfree(pseudo_packet);
		ipsec_sa_put(sa);
		return -ENOMEM;
	}

	write_lock_bh(&sa->lock);

	/* authhdr->spi = htonl(sa->spi); */
	IPSEC4_DEBUG("spi is 0x%x\n", ntohl(sa->spi));
	authhdr->spi = sa->spi; /* -mk */
	authhdr->seq_no = htonl(++sa->replay_window.seq_num);

	memcpy(pseudo_packet,iph,packetlen);
	memset(&pseudo_authhdr->auth_data[0], 0, (authhdr->hdrlen -1) << 2);

	pseudo_authhdr->spi = authhdr->spi;
	pseudo_authhdr->seq_no = authhdr->seq_no;

	((struct iphdr*)pseudo_packet)->tos      = 0;
	((struct iphdr*)pseudo_packet)->frag_off = 0;
	((struct iphdr*)pseudo_packet)->ttl      = 0;
	((struct iphdr*)pseudo_packet)->check    = 0;

	sa->auth_algo.dx->di->hmac_atomic(sa->auth_algo.dx,
			sa->auth_algo.key,
			sa->auth_algo.key_len,
			pseudo_packet, packetlen, authdata);

	memcpy(authhdr->auth_data, authdata, sa->auth_algo.digest_len);
	
	if (!sa->fuse_time) {
		sa->fuse_time = jiffies;
		sa->lifetime_c.usetime = (sa->fuse_time)/HZ;
		ipsec_sa_mod_timer(sa);
		IPSEC4_DEBUG("set fuse_time = %lu\n", sa->fuse_time);
	}

	sa->lifetime_c.bytes += packetlen;
	IPSEC4_DEBUG("sa->lifetime_c.bytes=%-9u %-9u\n",	/* XXX: %-18Lu */
			(__u32)((sa->lifetime_c.bytes) >> 32), (__u32)(sa->lifetime_c.bytes));

	if (sa->lifetime_c.bytes >= sa->lifetime_s.bytes && sa->lifetime_s.bytes) {
		IPSEC4_DEBUG("change sa state DYING\n");
		sa->state = SADB_SASTATE_DYING;
	} 

	if (sa->lifetime_c.bytes >= sa->lifetime_h.bytes && sa->lifetime_h.bytes) {
		sa->state = SADB_SASTATE_DEAD;
		IPSEC4_DEBUG("change sa state DEAD\n");
	}

	write_unlock_bh(&sa->lock);
	ipsec_sa_put(sa);

	kfree(authdata);
	kfree(pseudo_packet); 
	return 0;
}

int ipsec4_out_get_ahsize(struct ipsec_sp *policy)
{
	int result = 0;
	struct ipsec_sa *sa_ah = NULL;

	IPSEC4_DEBUG("called.\n");

	if (!policy) return 0;

	write_lock_bh(&policy->lock);
	if (policy->auth_sa_idx && policy->auth_sa_idx->sa) {
		ipsec_sa_hold(policy->auth_sa_idx->sa);
		sa_ah = policy->auth_sa_idx->sa;
	}

	write_unlock_bh(&policy->lock);

	if (sa_ah) {
		read_lock_bh(&sa_ah->lock);
		if ( sa_ah->auth_algo.algo != SADB_AALG_NONE) {
			result += (offsetof(struct ip_auth_hdr, auth_data) + 
					sa_ah->auth_algo.digest_len + 7) & ~7;	/* 64 bit alignment */
		}
		read_unlock_bh(&sa_ah->lock);
		ipsec_sa_put(sa_ah);
	}

	IPSEC4_DEBUG("Calculated size is %d.\n", result);
	return result;
}

int ipsec4_out_get_espsize(struct ipsec_sp *policy)
{
	int result = 0;
	struct ipsec_sa *sa_esp = NULL;

	IPSEC4_DEBUG("called.\n");

	if (!policy) return 0;

	write_lock_bh(&policy->lock);

	if (policy->esp_sa_idx && policy->esp_sa_idx->sa) {
		ipsec_sa_hold(policy->esp_sa_idx->sa);
		sa_esp = policy->esp_sa_idx->sa;
	}
	write_unlock_bh(&policy->lock);

	if (sa_esp) {
		read_lock_bh(&sa_esp->lock);
		if ( sa_esp->esp_algo.algo != SADB_EALG_NONE){
			result += sizeof(struct ip_esp_hdr) - 8;
			result += sa_esp->esp_algo.cx->ci->ivsize;
			result += (sa_esp->esp_algo.cx->ci->blocksize + 3) & ~3;
			result += 4;	/* included pad_len and next_hdr  32 bit align */
		}else{
			read_unlock_bh(&sa_esp->lock);
			ipsec_sa_put(sa_esp);
			return 0;
		}
		if ( sa_esp->auth_algo.algo != SADB_AALG_NONE) {
			result += (sa_esp->auth_algo.digest_len + 3) & ~3;	/* 32 bit alignment */
		}
		read_unlock_bh(&sa_esp->lock);
		ipsec_sa_put(sa_esp);
	}
	IPSEC4_DEBUG("Calculated size is %d.\n", result);
	return result;
}

/*** ESP ***/

void ipsec4_out_enc(const void *data, unsigned length, u8 proto,
		void **newdata, unsigned *newlength, struct ipsec_sp *policy)
{
	struct ipsec_sa *sa = NULL;
	struct ip_esp_hdr *esphdr = NULL;
	u8* srcdata = NULL;
	u8* authdata = NULL;
	int encblocksize = 0;
	int encsize = 0, hashsize = 0, totalsize = 0;
	int i;

	IPSEC4_DEBUG("called.\nData ptr is %p, data length is %d.\n",data,length);

	if (!policy) {
		if (net_ratelimit())
			printk(KERN_WARNING "%s; ipsec policy is NULL\n", __FUNCTION__);
		return;
	}

	read_lock_bh(&policy->lock);
	if (!policy->esp_sa_idx || !policy->esp_sa_idx->sa) {
		if (net_ratelimit())
			printk(KERN_WARNING "%s: ipsec(esp) SA missing.\n", __FUNCTION__);
		read_unlock_bh(&policy->lock);
		return;
	}
	ipsec_sa_hold(policy->esp_sa_idx->sa);
	sa = policy->esp_sa_idx->sa;
	read_unlock_bh(&policy->lock);

	write_lock_bh(&sa->lock);
	/* Get algorithms */
	if (sa->esp_algo.algo == SADB_EALG_NONE) {
		if (net_ratelimit())
			printk(KERN_WARNING "%s: ipsec(esp) encryption algorithm not present.\n", __FUNCTION__);
		goto unlock_finish;
		return;
	}


	if (!(sa->esp_algo.cx->ci)){
		if (net_ratelimit())
			printk(KERN_WARNING "%s: ipsec(esp) cipher_implementation not present.\n", __FUNCTION__);
		goto unlock_finish;
		return;
	}

	/* Calculate size */


	encblocksize = (sa->esp_algo.cx->ci->blocksize + 3) & ~3;
	encsize = length + 2 + (encblocksize - 1);
	encsize -= encsize % encblocksize;

	/* The tail of payload does not have to be aligned with a multiple number of 64 bit.	*/
	/* 64 bit alignment is adapted to the position of top of header. 			*/

	if (sa->auth_algo.algo != SADB_AALG_NONE)
		hashsize = sa->auth_algo.digest_len;


	totalsize = sizeof(struct ip_esp_hdr) - 8 + sa->esp_algo.cx->ci->ivsize + encsize + hashsize;
	IPSEC4_DEBUG("IV size=%d, enc size=%d hash size=%d, total size=%d\n",
			 sa->esp_algo.cx->ci->ivsize, encsize, hashsize, totalsize);
	
	/* Get memory */
	esphdr = kmalloc(totalsize, GFP_ATOMIC);
	srcdata = kmalloc(encsize, GFP_ATOMIC);
	if (!esphdr || !srcdata) {
		if (net_ratelimit())
			printk(KERN_WARNING "ipsec6_enc: Out of memory.\n");
		if (esphdr) kfree(esphdr);
		if (srcdata) kfree(srcdata);
		goto unlock_finish;
		return;
	}

	memset(esphdr, 0, totalsize);
	memset(srcdata, 0, encsize);
	/* Handle sequence number and fill in header fields */
	esphdr->spi = sa->spi;
	esphdr->seq_no = htonl(++sa->replay_window.seq_num);

	/* Get source data, fill in padding and trailing fields */

	memcpy(srcdata, data, length);
	for (i = length; i < encsize-2; i++) 
		srcdata[i] = (u8)(i-length+1);
	srcdata[encsize-2] = (encsize-2)-length;
	IPSEC4_DEBUG("length=%d, encsize=%d\n", length, encsize);
	IPSEC4_DEBUG("encsize-2=%d\n", srcdata[encsize-2]);

	srcdata[encsize-1] = proto;

	/* Do encryption */

	if (!(sa->esp_algo.iv)) { /* first packet */
		sa->esp_algo.iv = kmalloc(sa->esp_algo.cx->ci->ivsize, GFP_ATOMIC); /* kfree at SA removed */
		get_random_bytes(sa->esp_algo.iv, sa->esp_algo.cx->ci->ivsize);
		IPSEC4_DEBUG("IV initilized.\n");
	}  /* else, had inserted a stored iv (last packet block) */

#ifdef CONFIG_IPSEC_DEBUG
	{
		int i;
		IPSEC4_DEBUG("IV is 0x");
		if (sysctl_ipsec_debug_ipv6) {
			for (i=0; i < sa->esp_algo.cx->ci->ivsize ; i++) {
				printk(KERN_DEBUG "%x", (u8)(sa->esp_algo.iv[i]));
			}
		}
	}
#endif /* CONFIG_IPSEC_DEBUG */
	sa->esp_algo.cx->ci->encrypt_atomic_iv(sa->esp_algo.cx, srcdata,
					(u8 *)&esphdr->enc_data + sa->esp_algo.cx->ci->ivsize, encsize, sa->esp_algo.iv);
	memcpy(esphdr->enc_data, sa->esp_algo.iv, sa->esp_algo.cx->ci->ivsize);
	kfree(srcdata);
	srcdata=NULL;
	/* copy last block for next IV (src: enc_data + ivsize + encsize - ivsize) */
	memcpy(sa->esp_algo.iv, esphdr->enc_data + encsize, sa->esp_algo.cx->ci->ivsize);
	/* if CONFIG_IPSEC_DEBUG isn't defined here is finish of encryption process */

	if(sa->auth_algo.algo){
		authdata = kmalloc(sa->auth_algo.dx->di->blocksize, GFP_ATOMIC);
		if (!authdata) {
			if (net_ratelimit())
				printk(KERN_WARNING "ipsec6_enc: Out of memory.\n");
			kfree(esphdr);
			goto unlock_finish;
			return;
		}
		memset(authdata, 0, sa->auth_algo.dx->di->blocksize);
		sa->auth_algo.dx->di->hmac_atomic(sa->auth_algo.dx,
				sa->auth_algo.key,
				sa->auth_algo.key_len,
				(char*)esphdr, totalsize-hashsize, authdata);
		memcpy(&((char*)esphdr)[8 + sa->esp_algo.cx->ci->ivsize + encsize],
			authdata, sa->auth_algo.digest_len);

		kfree(authdata);
	}	

	if (!sa->fuse_time) {
		sa->fuse_time = jiffies;
		sa->lifetime_c.usetime = (sa->fuse_time)/HZ;
		ipsec_sa_mod_timer(sa);
		IPSEC4_DEBUG("set fuse_time = %lu\n", sa->fuse_time);
	}
	sa->lifetime_c.bytes += totalsize;
	IPSEC4_DEBUG("sa->lifetime_c.bytes=%-9u %-9u\n",	/* XXX: %-18Lu */
			(__u32)((sa->lifetime_c.bytes) >> 32), (__u32)(sa->lifetime_c.bytes));
	if (sa->lifetime_c.bytes >= sa->lifetime_s.bytes && sa->lifetime_s.bytes) {
		sa->state = SADB_SASTATE_DYING;
		IPSEC4_DEBUG("change sa state DYING\n");
	} 
	if (sa->lifetime_c.bytes >= sa->lifetime_h.bytes && sa->lifetime_h.bytes) {
		sa->state = SADB_SASTATE_DEAD;
		IPSEC4_DEBUG("change sa state DEAD\n");
	}

	write_unlock_bh(&sa->lock);
	ipsec_sa_put(sa);

	authdata = NULL;
	/* Set return values */
	*newdata = esphdr;
	*newlength = totalsize;
	return;

unlock_finish:
	write_unlock_bh(&sa->lock);
	ipsec_sa_put(sa);
	return;
}

void ipsec4_out_finish(struct ipsec_sp *policy)
{
	if (policy) {
		ipsec_sp_put(policy);
	}
}

static int ipsec4_output_check_core(struct selector *selector, struct ipsec_sp **policy_ptr)
{
	int error = 0;
	struct ipsec_sp *policy = NULL;
	int result = IPSEC_ACTION_BYPASS; 	/* default */

	IPSEC4_DEBUG("called\n");

	if (!selector) {
		IPSEC4_DEBUG("selector is NULL\n");
		error = -EINVAL;
		goto err;
	}

	policy = ipsec_sp_get(selector);
	if (!policy) { /* not match ! */
		IPSEC4_DEBUG("no policy exists.\n");
		result = IPSEC_ACTION_BYPASS;
		goto err;
	}

	read_lock_bh(&policy->lock);
	if (policy->policy_action == IPSEC_POLICY_DROP) {
		result = IPSEC_ACTION_DROP;
		read_unlock_bh(&policy->lock);
		goto err;
	}  else if (policy->policy_action == IPSEC_POLICY_BYPASS) {
		result = IPSEC_ACTION_BYPASS;
		read_unlock_bh(&policy->lock);
		goto err;
	}
	
	/* policy must then be to apply ipsec */
	if (policy->auth_sa_idx) {
		if (policy->auth_sa_idx->sa) {
			struct ipsec_sa *sa = NULL;
			ipsec_sa_hold(policy->auth_sa_idx->sa);
			sa = policy->auth_sa_idx->sa;
			read_unlock_bh(&policy->lock);

			read_lock_bh(&sa->lock);
			switch (sa->state) {
			case SADB_SASTATE_MATURE:
			case SADB_SASTATE_DYING:
				result |= IPSEC_ACTION_AUTH;
				break;
			default:
				result = IPSEC_ACTION_DROP;
			}
			read_unlock_bh(&sa->lock);
			ipsec_sa_put(sa);
		} else {
			/* copy sa_idx in policy to avoid to lock SADB and SPD at the same time */
			struct sa_index sa_idx;
			sa_index_init(&sa_idx);
			sa_index_copy(&sa_idx, policy->auth_sa_idx);
			read_unlock_bh(&policy->lock);

			sa_idx.sa = sadb_find_by_sa_index(&sa_idx);
			if (sa_idx.sa) {
				write_lock_bh(&policy->lock);
				policy->auth_sa_idx->sa = sa_idx.sa;
				ipsec_sa_hold(policy->auth_sa_idx->sa);
				write_unlock_bh(&policy->lock);
				ipsec_sa_put(sa_idx.sa);
				result |= IPSEC_ACTION_AUTH;
			} else {
				/* SADB_ACQUIRE message should be thrown up to KMd */
				result = IPSEC_ACTION_DROP;
			}
		}
	} else {
		read_unlock_bh(&policy->lock);
	}

	read_lock_bh(&policy->lock);
	if (policy->esp_sa_idx) {
		if (policy->esp_sa_idx->sa) {
			struct ipsec_sa *sa = NULL;
			ipsec_sa_hold(policy->esp_sa_idx->sa);
			sa = policy->esp_sa_idx->sa;
			read_unlock_bh(&policy->lock);

			read_lock_bh(&sa->lock);
			switch (sa->state) {
			case SADB_SASTATE_MATURE:
			case SADB_SASTATE_DYING:
				result |= IPSEC_ACTION_ESP;
				break;
			default:
				result = IPSEC_ACTION_DROP;
			}
			read_unlock_bh(&sa->lock);
			ipsec_sa_put(sa);
		} else {
			/* copy sa_idx in policy to avoid to lock SADB and SPD at the same time */
			struct sa_index sa_idx;
			sa_index_init(&sa_idx);
			sa_index_copy(&sa_idx, policy->esp_sa_idx);
			read_unlock_bh(&policy->lock);

			sa_idx.sa = sadb_find_by_sa_index(&sa_idx);
			if (sa_idx.sa) {
				write_lock_bh(&policy->lock);
				policy->esp_sa_idx->sa = sa_idx.sa;
				ipsec_sa_hold(policy->esp_sa_idx->sa);
				write_unlock_bh(&policy->lock);
				ipsec_sa_put(sa_idx.sa);
				result |= IPSEC_ACTION_ESP;
			} else {
				/* SADB_ACQUIRE message should be thrown up to KMd */
				result = IPSEC_ACTION_DROP;
			}
		}
	} else {
		read_unlock_bh(&policy->lock);
	}

	read_lock_bh(&policy->lock);
	if (policy->comp_sa_idx) {
		if (policy->comp_sa_idx->sa) {
			struct ipsec_sa *sa = NULL;
			ipsec_sa_hold(policy->comp_sa_idx->sa);
			sa = policy->comp_sa_idx->sa;
			read_unlock_bh(&policy->lock);

			read_lock_bh(&sa->lock);
			switch (sa->state) {
			case SADB_SASTATE_MATURE:
			case SADB_SASTATE_DYING:
				result |= IPSEC_ACTION_COMP;
				break;
			default:
				result |= IPSEC_ACTION_DROP;
			}
			read_unlock_bh(&sa->lock);
			ipsec_sa_put(sa);
		} else {
			/* copy sa_idx in policy to avoid to lock SADB and SPD at the same time */
			struct sa_index sa_idx;
			sa_index_init(&sa_idx);
			sa_index_copy(&sa_idx, policy->comp_sa_idx);
			read_unlock_bh(&policy->lock);

			sa_idx.sa = sadb_find_by_sa_index(&sa_idx);
			if (sa_idx.sa) {
				write_lock_bh(&policy->lock);
				policy->comp_sa_idx->sa = sa_idx.sa;
				ipsec_sa_hold(policy->comp_sa_idx->sa);
				write_unlock_bh(&policy->lock);
				ipsec_sa_put(sa_idx.sa);
				result |= IPSEC_ACTION_COMP;
			} else {
				/* SADB_ACUIRE message should be thrown up to KMd */
				result |= IPSEC_ACTION_DROP;
			}
		}
	} else {
		read_unlock_bh(&policy->lock);
	}

	*policy_ptr= policy;

	IPSEC4_DEBUG("end\n");	

err:
	return result;
}

int ipsec4_output_check(struct sock *sk, struct rtable *rt, const void *frag, struct ipsec_sp **policy_ptr)
{
	struct in_addr saddr,daddr;
	u16 sport,dport;
	unsigned char proto;
	struct selector selector;
	int result = IPSEC_ACTION_BYPASS; 	/* default */

	IPSEC4_DEBUG("called\n");
	if (!sk) {
		printk(KERN_ERR "flowi and sock are NULL\n");
		result = -EINVAL;
		goto err;
	}
	
	if (rt) {
		saddr.s_addr = rt->rt_src;
	}else if (sk) {
		saddr.s_addr = sk->saddr; 
	} else {
		result = -EINVAL;
		goto err;
	}

	if (rt) {
		daddr.s_addr = rt->rt_dst;
	} else if (sk) {
		daddr.s_addr = sk->daddr;
	} else { 
		result = -EINVAL;
		goto err;
	}

	if (sk) {
		sport=sk->sport;
		dport=sk->dport;
		proto=sk->protocol;
	} else {
		result = -EINVAL;
		goto err;
	}

	/* for ISKAMP see RFC2408 */
	/* in UDP sock, sk->dport is not yet inserted any port number.
	 * so we have to check uhp->dest.
	 */
	if (proto == IPPROTO_UDP) {
		if (sk->dport == 0 && frag) {
			struct udphdr *uhp = (struct udphdr *)frag;
			dport = uhp->dest;
		}
		if (sport == htons(500) && dport == htons(500)) {
			IPSEC4_DEBUG("match IKE packet.\n");
			result = IPSEC_ACTION_BYPASS; 	/* default */
			goto err;
		}
	}

	if (proto == IPPROTO_ICMP) {
		sport = 0;
		dport = 0;
	}

	memset(&selector, 0, sizeof(struct selector));

	/* these may be realy useless, 
	 * because ipip tunnel build outer ip header by itself :-< 
	 * -mk
	 */
#if 0 /* #ifdef CONFIG_IPV4_IPSEC_TUNNEL */
	if (proto == IPPROTO_IP) {
		selector.mode = IPSEC_MODE_TUNNEL;
	} else {
#endif
		((struct sockaddr_in *)&selector.src)->sin_port = sport;
		((struct sockaddr_in *)&selector.dst)->sin_port = dport;
#if 0 /* #ifdef CONFIG_IPV4_IPSEC_TUNNEL */
	}
	selector.proto = proto;

	if (selector.mode == IPSEC_MODE_TUNNEL) {
		const struct iphdr *h = (struct iphdr *) data;
		((struct sockaddr_in *)&selector.src)->sin_family = AF_INET;
		((struct sockaddr_in *)&selector.src)->sin_addr.s_addr = h->saddr;
		((struct sockaddr_in *)&selector.dst)->sin_family = AF_INET;
		((struct sockaddr_in *)&selector.dst)->sin_addr.s_addr = h->daddr;
	} else { /* IPSEC_MODE_TRANSPORT */
#endif
		((struct sockaddr_in *)&selector.src)->sin_family = AF_INET;
		((struct sockaddr_in *)&selector.src)->sin_addr = saddr;
		((struct sockaddr_in *)&selector.dst)->sin_family = AF_INET;
		((struct sockaddr_in *)&selector.dst)->sin_addr = daddr;
		selector.proto = proto;
#if 0 /* #ifdef CONFIG_IPV4_IPSEC_TUNNEL */
	}
#endif
	selector.prefixlen_d = 32;
	selector.prefixlen_s = 32;

#ifdef CONFIG_IPSEC_DEBUG
	IPSEC4_DEBUG("original src addr: %d.%d.%d.%d\n", NIPQUAD(saddr));
	IPSEC4_DEBUG("original src port: %u\n", ntohs(sport));
	IPSEC4_DEBUG("original dst addr: %d.%d.%d.%d\n", NIPQUAD(daddr));
	IPSEC4_DEBUG("original dst port: %u\n", ntohs(dport));

	IPSEC4_DEBUG("selector src addr: %d.%d.%d.%d\n", 
			NIPQUAD(((struct sockaddr_in *)&selector.src)->sin_addr));
	IPSEC4_DEBUG("selector src port: %u\n", 
			ntohs(((struct sockaddr_in *)&selector.src)->sin_port));
	IPSEC4_DEBUG("selector dst addr: %d.%d.%d.%d\n", 
			NIPQUAD(((struct sockaddr_in *)&selector.dst)->sin_addr));
	IPSEC4_DEBUG("selector dst port: %u\n", 
			ntohs(((struct sockaddr_in *)&selector.dst)->sin_port));
	IPSEC4_DEBUG("selector proto: %u\n", selector.proto);
#endif /* CONFIG_IPSEC_DEBUG */

	result = ipsec4_output_check_core(&selector, policy_ptr);

 err:
		return result;
}

#if defined(CONFIG_IPV4_IPSEC_TUNNEL) && defined(CONFIG_IP_IPSEC)
int ipsec4_ipip_output_check(struct rtable *rt, const u8 *data, struct ipsec_sp **policy_ptr)
{
	struct in_addr saddr,daddr;
	struct selector selector;
	int result = IPSEC_ACTION_BYPASS; 	/* default */
	const struct iphdr *h = (struct iphdr *)data;

	IPSEC4_DEBUG("called\n");
	
	if (rt) {
		saddr.s_addr = rt->rt_src;
		daddr.s_addr = rt->rt_dst;
	} else {
		IPSEC4_DEBUG("rt is NULL\n");
		result = -EINVAL;
		goto err;
	}

	memset(&selector, 0, sizeof(struct selector));

	selector.mode = IPSEC_MODE_TUNNEL;
	selector.proto = IPPROTO_IPIP;

	/* outer ip addr */
	((struct sockaddr_in *)&selector.src)->sin_family = AF_INET;
	((struct sockaddr_in *)&selector.src)->sin_addr.s_addr = h->saddr;
	((struct sockaddr_in *)&selector.dst)->sin_family = AF_INET;
	((struct sockaddr_in *)&selector.dst)->sin_addr.s_addr = h->daddr;
	
	selector.prefixlen_d = 32;
	selector.prefixlen_s = 32;

#ifdef CONFIG_IPSEC_DEBUG
	IPSEC4_DEBUG("original src addr: %d.%d.%d.%d\n", NIPQUAD(h->saddr));
	IPSEC4_DEBUG("original dst addr: %d.%d.%d.%d\n", NIPQUAD(h->daddr));

	IPSEC4_DEBUG("selector src addr: %d.%d.%d.%d\n", 
			NIPQUAD(((struct sockaddr_in *)&selector.src)->sin_addr));
	IPSEC4_DEBUG("selector dst addr: %d.%d.%d.%d\n", 
			NIPQUAD(((struct sockaddr_in *)&selector.dst)->sin_addr));
	IPSEC4_DEBUG("selector proto: %u\n", selector.proto);
#endif /* CONFIG_IPSEC_DEBUG */

	result = ipsec4_output_check_core(&selector, policy_ptr);

 err:
		return result;
}
#endif
