/*
 *	Home Agent IOCTL Control device
 *
 *	Authors:
 *	Henrik Petander		<lpetande@tml.hut.fi>
 *
 *	$Id$
 *
 *	This program is free software; you can redistribute it and/or
 *      modify it under the terms of the GNU General Public License
 *      as published by the Free Software Foundation; either version
 *      2 of the License, or (at your option) any later version.
 */
/*
 *	Changes:
 *
 *	nakam@USAGI		: based on ioctl_mn.c(MIPL) :-)
 */

#include <linux/fs.h>
#include <linux/module.h>
#include <asm/uaccess.h>
#include "debug.h"
#include "ha.h"
#include "mipv6_ioctl.h"

/* The maximum length of the message for the device */
#define BUF_LEN 80

/* The message the device will give when asked */
static char message[BUF_LEN];

/* How far did the process reading the message get?  Useful if the
 * message is larger than the size of the buffer we get to fill in
 * device_read.
 */
static char *message_ptr;

/* This function is called whenever a process which has already opened
 * the device file attempts to read from it.
 */
static ssize_t mipv6_ioctl_read(struct file *file, 
				char *buffer, /* The buffer to fill with the data */
				size_t length,	/* The length of the buffer */
				loff_t * offset)	/* offset to the file */
{
	/* Number of bytes actually written to the buffer */
	int bytes_read = 0;

	DEBUG(DBG_DATADUMP, "(%p,%p,%d)", file, buffer, length);

	/* If we're at the end of the message, return 0 (which
	 * signifies end of file) */
	if (*message_ptr == 0)
		return 0;

	/* Actually put the data into the buffer */
	while (length && *message_ptr) {

		/* Because the buffer is in the user data segment, 
		 * not the kernel data segment, assignment wouldn't 
		 * work. Instead, we have to use put_user which 
		 * copies data from the kernel data segment to the 
		 * user data segment. */
		put_user(*(message_ptr++), buffer++);
		length--;
		bytes_read++;
	}


	DEBUG(DBG_INFO, "Read %d bytes, %d left", bytes_read, length);

	/* Read functions are supposed to return the number 
	 * of bytes actually inserted into the buffer */
	return bytes_read;
}

/* This function is called when somebody tries to write into our
 * device file.
 */
static ssize_t mipv6_ioctl_write(struct file *file,
				 const char *buffer,
				 size_t length, loff_t * offset)
{
	int i;

	DEBUG(DBG_DATADUMP, "%p,%s,%d", file, buffer, length);

	for (i = 0; i < length && i < BUF_LEN; i++)
		get_user(message[i], buffer + i);
	message_ptr = message;

	/* Again, return the number of input characters used */
	return i;
}

extern void ha_set_anycast_addr(unsigned long arg);
extern void ha_del_anycast_addr(unsigned long arg);

static int mipv6_ioctl_ha(struct inode *inode, struct file *file, 
			  unsigned int ioctl_num,	/* The number of the ioctl */
			  unsigned long arg)	/* The parameter to it */
{
	/* Switch according to the ioctl called */
	switch (ioctl_num) {
	case IOCTL_SET_HA_ANYCAST:
                DEBUG(DBG_INFO,"IOCTL_SET_HA_ANYCAST");
                ha_set_anycast_addr(arg);
                break;

        case IOCTL_DEL_HA_ANYCAST:
                DEBUG(DBG_INFO,"IOCTL_DEL_HA_ANYCAST");
                ha_del_anycast_addr(arg);
                break;

	default:
		DEBUG(DBG_WARNING, "Unknown ioctl cmd (%d)", ioctl_num);
		return -ENOENT;
	}
	return 0;
}

struct file_operations fops = {
	owner: THIS_MODULE,
	read: mipv6_ioctl_read,
	write: mipv6_ioctl_write,
	poll: NULL,
	ioctl: mipv6_ioctl_ha,
	open: mipv6_ioctl_open,
	release: mipv6_ioctl_close
};
