/* $USAGI: pfkey_v2_build.c,v 1.9 2002/10/04 19:07:45 mk Exp $ */
/*
 * RFC2367 PF_KEYv2 Key management API message parser
 * Copyright (C) 1999  Richard Guy Briggs.
 * 
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.  See <http://www.fsf.org/copyleft/gpl.txt>.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 */

/*
 *		Template from klips/net/ipsec/ipsec/ipsec_parser.c.
 */


/*
 * Some ugly stuff to allow consistent debugging code for use in the
 * kernel and in user space
*/

#include <linux/config.h>
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/slab.h>
#include <linux/errno.h>
#include <linux/types.h>
#include <linux/sysctl.h>
#include <linux/net.h>
#include <linux/in.h>
#include <linux/ip.h>
#if defined(CONFIG_IPV6) || defined(CONFIG_IPV6_MODULE)
#include <linux/ipv6.h>
#endif
#include <linux/inet.h>
#include <linux/ipsec.h>
#include <linux/pfkeyv2.h>
#include <linux/pfkey.h>
#include <net/sadb.h>
#include "sockaddr_utils.h"

#define ADDRTOT_BUF 128

#define SENDERR(_x) do { error = -(_x); goto errlab; } while (0)

void
pfkey_extensions_init(struct sadb_ext *extensions[SADB_EXT_MAX + 1])
{
	int i;
	
	for (i = 0; i != SADB_EXT_MAX + 1; i++) {
		extensions[i] = NULL;
	}
}

void
pfkey_extensions_free(struct sadb_ext *extensions[SADB_EXT_MAX + 1])
{
	int i;
	
	if (!extensions) {
		return;
	}

	if (extensions[0]) {
		memset(extensions[0], 0, sizeof(struct sadb_msg));
		kfree(extensions[0]);
		extensions[0] = NULL;
	}
	
	for (i = 1; i != SADB_EXT_MAX + 1; i++) {
		if (extensions[i]) {
			memset(extensions[i], 0, extensions[i]->sadb_ext_len * IPSEC_PFKEYv2_ALIGN);
			kfree(extensions[i]);
			extensions[i] = NULL;
		}
	}
}

void
pfkey_msg_free(struct sadb_msg **pfkey_msg)
{
	if (*pfkey_msg) {
		memset(*pfkey_msg, 0, (*pfkey_msg)->sadb_msg_len * IPSEC_PFKEYv2_ALIGN);
		kfree(*pfkey_msg);
		*pfkey_msg = NULL;
	}
}

/* Default extension builders taken from the KLIPS code */

int
pfkey_msg_hdr_build(struct sadb_ext**	pfkey_ext,
		    uint8_t		msg_type,
		    uint8_t		satype,
		    uint8_t		msg_errno,
		    uint32_t		seq,
		    uint32_t		pid)
{
	int error = 0;
	struct sadb_msg *pfkey_msg = (struct sadb_msg *)*pfkey_ext;

	PFKEY_DEBUG("called\n");
	PFKEY_DEBUG("on_entry &pfkey_ext=%p pfkey_ext=%p *pfkey_ext=%p.\n",
		&pfkey_ext, pfkey_ext, *pfkey_ext);
	/* sanity checks... */
	if (pfkey_msg) {
		PFKEY_DEBUG("why is pfkey_msg already pointing to something?\n");
		SENDERR(EINVAL);
	}

	if (!msg_type) {
		PFKEY_DEBUG("msg type not set, must be non-zero.\n");
		SENDERR(EINVAL);
	}

	if (msg_type > SADB_MAX) {
		PFKEY_DEBUG("msg type too large:%d.\n", msg_type);
		SENDERR(EINVAL);
	}

	if (satype > SADB_SATYPE_MAX) {
		PFKEY_DEBUG("satype %d > max %d\n", satype, SADB_SATYPE_MAX);
		SENDERR(EINVAL);
	}

	if (!(*pfkey_ext = (struct sadb_ext*)
	     pfkey_msg = (struct sadb_msg*)
	     kmalloc(sizeof(struct sadb_msg), GFP_ATOMIC))) {
		PFKEY_DEBUG("memory allocation failed\n");
		SENDERR(ENOMEM);
	}
	memset(pfkey_msg, 0, sizeof(struct sadb_msg));

	pfkey_msg->sadb_msg_len = sizeof(struct sadb_msg) / IPSEC_PFKEYv2_ALIGN;

	pfkey_msg->sadb_msg_type = msg_type;
	pfkey_msg->sadb_msg_satype = satype;

	pfkey_msg->sadb_msg_version = PF_KEY_V2;
	pfkey_msg->sadb_msg_errno = msg_errno;
	pfkey_msg->sadb_msg_reserved = 0;
	pfkey_msg->sadb_msg_seq = seq;
	pfkey_msg->sadb_msg_pid = pid;
	PFKEY_DEBUG("on_exit &pfkey_ext=%p pfkey_ext=%p *pfkey_ext=%p.\n",
		&pfkey_ext, pfkey_ext, *pfkey_ext);
errlab:
	return error;
}	

int
pfkey_sa_build(struct sadb_ext **	pfkey_ext,
	       uint16_t			exttype,
	       uint32_t			spi, /* in network order */
	       uint8_t			replay_window,
	       uint8_t			sa_state,
	       uint8_t			auth,
	       uint8_t			encrypt,
	       uint32_t			flags)
{
	int error = 0;
	struct sadb_sa *pfkey_sa = (struct sadb_sa *)*pfkey_ext;

	PFKEY_DEBUG("spi=%08x replay=%d sa_state=%d auth=%d encrypt=%d flags=%d\n",
		    ntohl(spi), replay_window, sa_state, auth, encrypt, flags);
	/* sanity checks... */
	if (pfkey_sa) {
		PFKEY_DEBUG("why is pfkey_sa already pointing to something?\n");
		SENDERR(EINVAL);
	}

	if (exttype != SADB_EXT_SA &&
	   exttype != SADB_X_EXT_SA2) {
		PFKEY_DEBUG("invalid exttype=%d.\n", exttype);
		SENDERR(EINVAL);
	}

	if (replay_window > 64) {
		PFKEY_DEBUG("replay window size: %d -- must be 0 <= size <= 64\n", 
				replay_window);
		SENDERR(EINVAL);
	}

	if (auth > SADB_AALG_MAX) {
		PFKEY_DEBUG("auth=%d > SADB_AALG_MAX=%d.\n", auth, SADB_AALG_MAX);
		SENDERR(EINVAL);
	}

	if (encrypt > SADB_EALG_MAX) {
		PFKEY_DEBUG("encrypt=%d > SADB_EALG_MAX=%d.\n", encrypt, SADB_EALG_MAX);
		SENDERR(EINVAL);
	}

	if (sa_state > SADB_SASTATE_MAX) {
		PFKEY_DEBUG("sa_state=%d exceeds MAX=%d.\n", sa_state, SADB_SASTATE_MAX);
		SENDERR(EINVAL);
	}

	if (!(*pfkey_ext = (struct sadb_ext*)
	     pfkey_sa = (struct sadb_sa*)
	     kmalloc(sizeof(struct sadb_sa), GFP_ATOMIC))) {
		PFKEY_DEBUG("memory allocation failed\n");
		SENDERR(ENOMEM);
	}
	memset(pfkey_sa, 0, sizeof(struct sadb_sa));
	
	pfkey_sa->sadb_sa_len = sizeof(*pfkey_sa) / IPSEC_PFKEYv2_ALIGN;
	pfkey_sa->sadb_sa_exttype = exttype;
	pfkey_sa->sadb_sa_spi = spi;
	pfkey_sa->sadb_sa_replay = replay_window;
	pfkey_sa->sadb_sa_state = sa_state;
	pfkey_sa->sadb_sa_auth = auth;
	pfkey_sa->sadb_sa_encrypt = encrypt;
	pfkey_sa->sadb_sa_flags = flags;

errlab:
	return error;
}	

int
pfkey_lifetime_build(struct sadb_ext **	pfkey_ext,
		     uint16_t		exttype,
		     uint32_t		allocations,
		     uint64_t		bytes,
		     uint64_t		addtime,
		     uint64_t		usetime)
{
	int error = 0;
	struct sadb_lifetime *pfkey_lifetime = (struct sadb_lifetime *)*pfkey_ext;

	PFKEY_DEBUG(
		"pfkey_lifetime_build:\n");
	/* sanity checks... */
	if (pfkey_lifetime) {
		PFKEY_DEBUG("why is pfkey_lifetime already pointing to something?\n");
		SENDERR(EINVAL);
	}

	if (exttype != SADB_EXT_LIFETIME_CURRENT &&
	   exttype != SADB_EXT_LIFETIME_HARD &&
	   exttype != SADB_EXT_LIFETIME_SOFT) {
		PFKEY_DEBUG("invalid exttype=%d.\n", exttype);
		SENDERR(EINVAL);
	}

	if (!(*pfkey_ext = (struct sadb_ext*)
	     pfkey_lifetime = (struct sadb_lifetime*)
	     kmalloc(sizeof(struct sadb_lifetime), GFP_ATOMIC))) {
		PFKEY_DEBUG("memory allocation failed\n");
		SENDERR(ENOMEM);
	}
	memset(pfkey_lifetime, 0, sizeof(struct sadb_lifetime));

	pfkey_lifetime->sadb_lifetime_len = sizeof(struct sadb_lifetime) / IPSEC_PFKEYv2_ALIGN;
	pfkey_lifetime->sadb_lifetime_exttype = exttype;
	pfkey_lifetime->sadb_lifetime_allocations = allocations;
	pfkey_lifetime->sadb_lifetime_bytes = bytes;
	pfkey_lifetime->sadb_lifetime_addtime = addtime;
	pfkey_lifetime->sadb_lifetime_usetime = usetime;

errlab:
	return error;
}

int
pfkey_address_build(struct sadb_ext**	pfkey_ext,
		    uint16_t		exttype,
		    uint8_t		proto,
		    uint8_t		prefixlen,
		    struct sockaddr*	address)
{
	int error = 0;
	int saddr_len = 0;
	char ipaddr_txt[ADDRTOT_BUF];
	struct sadb_address *pfkey_address = (struct sadb_address *)*pfkey_ext;
	
	PFKEY_DEBUG(
		"pfkey_address_build: exttype=%d proto=%d prefixlen=%d\n", exttype, proto, prefixlen);
	/* sanity checks... */
	if (pfkey_address) {
		PFKEY_DEBUG("why is pfkey_address already pointing to something?\n");
		SENDERR(EINVAL);
	}

	if (!address)  {
			PFKEY_DEBUG("address is NULL\n");
			SENDERR(EINVAL);
	}
	
	switch (exttype) {	
	case SADB_EXT_ADDRESS_SRC:
	case SADB_EXT_ADDRESS_DST:
	case SADB_EXT_ADDRESS_PROXY:
		break;
	default:
		PFKEY_DEBUG("unrecognised ext_type=%d.\n", 
			exttype); 
		SENDERR(EINVAL); 
	}

	switch (address->sa_family) {
	case AF_INET:
		PFKEY_DEBUG("found address family AF_INET.\n");
		saddr_len = sizeof(struct sockaddr_in);
		sockaddrtoa(address, ipaddr_txt, sizeof(ipaddr_txt));
		break;
	case AF_INET6:
		PFKEY_DEBUG("found address family AF_INET6.\n");
		saddr_len = sizeof(struct sockaddr_in6);
		sockaddrtoa(address, ipaddr_txt, sizeof(ipaddr_txt));
		break;
	default:
		PFKEY_DEBUG("address->sa_family=%d not supported.\n",
			address->sa_family);
		SENDERR(EPFNOSUPPORT);
	}

	PFKEY_DEBUG("found address=%s.\n", ipaddr_txt);

	if (!(*pfkey_ext = (struct sadb_ext*)
	     pfkey_address = (struct sadb_address*)
	     kmalloc(ALIGN_N(sizeof(struct sadb_address) + saddr_len, IPSEC_PFKEYv2_ALIGN), GFP_ATOMIC))) {
		PFKEY_DEBUG("memory allocation failed\n");
		SENDERR(ENOMEM);
	}
	memset(pfkey_address,
	       0,
	       ALIGN_N(sizeof(struct sadb_address) + saddr_len,
		     IPSEC_PFKEYv2_ALIGN));
	       
	pfkey_address->sadb_address_len = DIVUP(sizeof(struct sadb_address) + saddr_len,
						IPSEC_PFKEYv2_ALIGN);
	
	pfkey_address->sadb_address_exttype = exttype;
	pfkey_address->sadb_address_proto = proto;
	pfkey_address->sadb_address_prefixlen = prefixlen;
	pfkey_address->sadb_address_reserved = 0;

	memcpy((char*)pfkey_address + sizeof(struct sadb_address),
	       address,
	       saddr_len);

	PFKEY_DEBUG("successful.\n");

 errlab:
	return error;
}

int
pfkey_key_build(struct sadb_ext**	pfkey_ext,
		uint16_t		exttype,
		uint16_t		key_bits,
		char*			key)
{
	int error = 0;
	struct sadb_key *pfkey_key = (struct sadb_key *)*pfkey_ext;

	PFKEY_DEBUG(
		"pfkey_key_build:\n");
	/* sanity checks... */
	if (pfkey_key) {
		PFKEY_DEBUG("why is pfkey_key already pointing to something?\n");
		SENDERR(EINVAL);
	}

	if (!key_bits) {
		PFKEY_DEBUG("key_bits is zero, it must be non-zero.\n");
		SENDERR(EINVAL);
	}

	if ( !((exttype == SADB_EXT_KEY_AUTH) || (exttype == SADB_EXT_KEY_ENCRYPT))) {
		PFKEY_DEBUG("unsupported extension type=%d.\n", exttype);
		SENDERR(EINVAL);
	}

	if (!(*pfkey_ext = (struct sadb_ext*)
	     pfkey_key = (struct sadb_key*)
	     kmalloc(sizeof(struct sadb_key) +
				    DIVUP(key_bits, 64) * IPSEC_PFKEYv2_ALIGN, GFP_ATOMIC))) {
		PFKEY_DEBUG("memory allocation failed\n");
		SENDERR(ENOMEM);
	}
	memset(pfkey_key,
	       0,
	       sizeof(struct sadb_key) +
	       DIVUP(key_bits, 64) * IPSEC_PFKEYv2_ALIGN);
	
	pfkey_key->sadb_key_len = DIVUP(sizeof(struct sadb_key) * IPSEC_PFKEYv2_ALIGN +	key_bits,
					64);
	pfkey_key->sadb_key_exttype = exttype;
	pfkey_key->sadb_key_bits = key_bits;
	pfkey_key->sadb_key_reserved = 0;
	memcpy((char*)pfkey_key + sizeof(struct sadb_key),
	       key,
	       DIVUP(key_bits, 8));

errlab:
	return error;
}

int
pfkey_ident_build(struct sadb_ext**	pfkey_ext,
		  uint16_t		exttype,
		  uint16_t		ident_type,
		  uint64_t		ident_id,
		  char*			ident_string)
{
	int error = 0;
	struct sadb_ident *pfkey_ident = (struct sadb_ident *)*pfkey_ext;

	PFKEY_DEBUG(
		"pfkey_ident_build:\n");
	/* sanity checks... */
	if (pfkey_ident) {
		PFKEY_DEBUG("why is pfkey_ident already pointing to something?\n");
		SENDERR(EINVAL);
	}

	if ( ! ((exttype == SADB_EXT_IDENTITY_SRC) ||
	       (exttype == SADB_EXT_IDENTITY_DST))) {
		PFKEY_DEBUG("unsupported extension type=%d.\n", exttype);
		SENDERR(EINVAL);
	}

	if ((ident_type == SADB_IDENTTYPE_RESERVED)) {
		PFKEY_DEBUG("ident_type must be non-zero.\n");
		SENDERR(EINVAL);
	}

	if (ident_type > SADB_IDENTTYPE_MAX) {
		PFKEY_DEBUG("identtype=%d out of range.\n", ident_type);
		SENDERR(EINVAL);
	}

	if (((ident_type == SADB_IDENTTYPE_PREFIX) ||
	    (ident_type == SADB_IDENTTYPE_FQDN)) &&
	   !ident_string) {
		PFKEY_DEBUG("string required to allocate size of extension.\n");
		SENDERR(EINVAL);
	}
	
	if (!(*pfkey_ext = (struct sadb_ext*)
	     pfkey_ident = (struct sadb_ident*)
	     kmalloc(ALIGN_N(sizeof(struct sadb_key) + strlen(ident_string), IPSEC_PFKEYv2_ALIGN), GFP_ATOMIC))) {
		PFKEY_DEBUG("memory allocation failed\n");
		SENDERR(ENOMEM);
	}
	memset(pfkey_ident,
	       0,
	       ALIGN_N(sizeof(struct sadb_ident) + strlen(ident_string),
		     IPSEC_PFKEYv2_ALIGN));
	
	pfkey_ident->sadb_ident_len = DIVUP(sizeof(struct sadb_ident) + strlen(ident_string),
					    IPSEC_PFKEYv2_ALIGN);
	
	pfkey_ident->sadb_ident_exttype = exttype;
	pfkey_ident->sadb_ident_type = ident_type;
	pfkey_ident->sadb_ident_reserved = 0;
	pfkey_ident->sadb_ident_id = ident_id;
	memcpy((char*)pfkey_ident + sizeof(struct sadb_ident),
	       ident_string,
	       strlen(ident_string));

	memset(((char*)pfkey_ident) + sizeof(struct sadb_ident) + strlen(ident_string),
	       0,
	       ALIGN_N(sizeof(struct sadb_ident) + strlen(ident_string), IPSEC_PFKEYv2_ALIGN) -
	       sizeof(struct sadb_ident) + strlen(ident_string));

errlab:
	return error;
}

int
pfkey_sens_build(struct sadb_ext**	pfkey_ext,
		 uint32_t		dpd,
		 uint8_t		sens_level,
		 uint8_t		sens_len,
		 uint64_t*		sens_bitmap,
		 uint8_t		integ_level,
		 uint8_t		integ_len,
		 uint64_t*		integ_bitmap)
{
	int error = 0;
	struct sadb_sens *pfkey_sens = (struct sadb_sens *)*pfkey_ext;
	int i;
	uint64_t* bitmap;

	PFKEY_DEBUG(
		"pfkey_sens_build:\n");
	/* sanity checks... */
	if (pfkey_sens) {
		PFKEY_DEBUG("why is pfkey_sens already pointing to something?\n");
		SENDERR(EINVAL);
	}

	PFKEY_DEBUG("Sorry, I can't build exttype=%d yet.\n", (*pfkey_ext)->sadb_ext_type);
	SENDERR(EINVAL); /* don't process these yet */

	if (!(*pfkey_ext = (struct sadb_ext*)
	     pfkey_sens = (struct sadb_sens*)
	     kmalloc(sizeof(struct sadb_sens) +
		    (sens_len + integ_len) * sizeof(uint64_t), GFP_ATOMIC))) {
		PFKEY_DEBUG("memory allocation failed\n");
		SENDERR(ENOMEM);
	}
	memset(pfkey_sens,
	       0,
	       sizeof(struct sadb_sens) +
	       (sens_len + integ_len) * sizeof(uint64_t));
	
	pfkey_sens->sadb_sens_len = (sizeof(struct sadb_sens) +
		    (sens_len + integ_len) * sizeof(uint64_t)) / IPSEC_PFKEYv2_ALIGN;
	pfkey_sens->sadb_sens_exttype = SADB_EXT_SENSITIVITY;
	pfkey_sens->sadb_sens_dpd = dpd;
	pfkey_sens->sadb_sens_sens_level = sens_level;
	pfkey_sens->sadb_sens_sens_len = sens_len;
	pfkey_sens->sadb_sens_integ_level = integ_level;
	pfkey_sens->sadb_sens_integ_len = integ_len;
	pfkey_sens->sadb_sens_reserved = 0;

	bitmap = (uint64_t*)((char*)pfkey_ext + sizeof(struct sadb_sens));
	for (i = 0; i < sens_len; i++) {
		*bitmap = sens_bitmap[i];
		bitmap++;
	}
	for (i = 0; i < integ_len; i++) {
		*bitmap = integ_bitmap[i];
		bitmap++;
	}

errlab:
	return error;
}

int
pfkey_prop_build(struct sadb_ext**	pfkey_ext,
		 uint8_t		replay,
		 unsigned int		comb_num,
		 struct sadb_comb*	comb)
{
	int error = 0;
	int i;
	struct sadb_prop *pfkey_prop = (struct sadb_prop *)*pfkey_ext;
	struct sadb_comb *combp;

	PFKEY_DEBUG(
		"pfkey_prop_build:\n");
	/* sanity checks... */
	if (pfkey_prop) {
		PFKEY_DEBUG("why is pfkey_prop already pointing to something?\n");
		SENDERR(EINVAL);
	}

	if (!(*pfkey_ext = (struct sadb_ext*)
	     pfkey_prop = (struct sadb_prop*)
	     kmalloc(sizeof(struct sadb_prop) +
		    comb_num * sizeof(struct sadb_comb), GFP_ATOMIC))) {
		PFKEY_DEBUG("memory allocation failed\n");
		SENDERR(ENOMEM);
	}
	memset(pfkey_prop,
	       0,
	       sizeof(struct sadb_prop) +
		    comb_num * sizeof(struct sadb_comb));
	
	pfkey_prop->sadb_prop_len = (sizeof(struct sadb_prop) +
		    comb_num * sizeof(struct sadb_comb)) / IPSEC_PFKEYv2_ALIGN;

	pfkey_prop->sadb_prop_exttype = SADB_EXT_PROPOSAL;
	pfkey_prop->sadb_prop_replay = replay;

	for (i=0; i<3; i++) {
		pfkey_prop->sadb_prop_reserved[i] = 0;
	}

	combp = (struct sadb_comb*)((char*)*pfkey_ext + sizeof(struct sadb_prop));
	for (i = 0; i < comb_num; i++) {
		memcpy (combp, &comb[i], sizeof(struct sadb_comb));
		combp++;
	}

errlab:
	return error;
}

int
pfkey_supported_build(struct sadb_ext**	pfkey_ext,
		      uint16_t		exttype,
		      unsigned int	alg_num,
		      struct sadb_alg*	alg)
{
	int error = 0;
	unsigned int i;
	struct sadb_supported *pfkey_supported = (struct sadb_supported *)*pfkey_ext;
	struct sadb_alg *pfkey_alg;

	/* sanity checks... */
	if (pfkey_supported) {
		PFKEY_DEBUG("why is pfkey_supported already pointing to something?\n");
		SENDERR(EINVAL);
	}

	if ( !((exttype == SADB_EXT_SUPPORTED_AUTH) || (exttype == SADB_EXT_SUPPORTED_ENCRYPT))) {
		PFKEY_DEBUG("unsupported extension type=%d.\n", exttype);
		SENDERR(EINVAL);
	}

	if (!(*pfkey_ext = (struct sadb_ext*)
	     pfkey_supported = (struct sadb_supported*)
	     kmalloc(sizeof(struct sadb_supported) +
					       alg_num *
					       sizeof(struct sadb_alg), GFP_ATOMIC))) {
		PFKEY_DEBUG("memory allocation failed\n");
		SENDERR(ENOMEM);
	}
	memset(pfkey_supported,
	       0,
	       sizeof(struct sadb_supported) +
					       alg_num *
					       sizeof(struct sadb_alg));
	
	pfkey_supported->sadb_supported_len = (sizeof(struct sadb_supported) +
					       alg_num *
					       sizeof(struct sadb_alg)) /
						IPSEC_PFKEYv2_ALIGN;
	pfkey_supported->sadb_supported_exttype = exttype;
	pfkey_supported->sadb_supported_reserved = 0;

	pfkey_alg = (struct sadb_alg*)((char*)pfkey_supported + sizeof(struct sadb_supported));
	for (i = 0; i < alg_num; i++) {
		memcpy (pfkey_alg, &alg[i], sizeof(struct sadb_alg));
		pfkey_alg->sadb_alg_reserved = 0;
		pfkey_alg++;
	}
	
errlab:
	return error;
}

int
pfkey_spirange_build(struct sadb_ext**	pfkey_ext,
		     uint16_t		exttype,
		     uint32_t		min, /* in network order */
		     uint32_t		max) /* in network order */
{
	int error = 0;
	struct sadb_spirange *pfkey_spirange = (struct sadb_spirange *)*pfkey_ext;
	
	/* sanity checks... */
	if (pfkey_spirange) {
		PFKEY_DEBUG("why is pfkey_spirange already pointing to something?\n");
		SENDERR(EINVAL);
	}
	
        if (ntohl(max) < ntohl(min)) {
		PFKEY_DEBUG("minspi=%08x must be < maxspi=%08x.\n", ntohl(min), ntohl(max));
                SENDERR(EINVAL);
        }
	
	if (ntohl(min) <= 255) {
		PFKEY_DEBUG("minspi=%08x must be > 255.\n", ntohl(min));
		SENDERR(EEXIST);
	}
	
	if (!(*pfkey_ext = (struct sadb_ext*)
	     pfkey_spirange = (struct sadb_spirange*)
	     kmalloc(sizeof(struct sadb_spirange), GFP_ATOMIC))) {
		PFKEY_DEBUG("memory allocation failed\n");
		SENDERR(ENOMEM);
	}
	memset(pfkey_spirange,
	       0,
	       sizeof(struct sadb_spirange));
	
        pfkey_spirange->sadb_spirange_len = sizeof(struct sadb_spirange) / IPSEC_PFKEYv2_ALIGN;

	pfkey_spirange->sadb_spirange_exttype = SADB_EXT_SPIRANGE;
	pfkey_spirange->sadb_spirange_min = min;
	pfkey_spirange->sadb_spirange_max = max;
	pfkey_spirange->sadb_spirange_reserved = 0;
 errlab:
	return error;
}

int
pfkey_x_kmprivate_build(struct sadb_ext**	pfkey_ext)
{
	int error = 0;
	struct sadb_x_kmprivate *pfkey_x_kmprivate = (struct sadb_x_kmprivate *)*pfkey_ext;

	/* sanity checks... */
	if (pfkey_x_kmprivate) {
		PFKEY_DEBUG("why is pfkey_x_kmprivate already pointing to something?\n");
		SENDERR(EINVAL);
	}
	
	pfkey_x_kmprivate->sadb_x_kmprivate_reserved = 0;

	PFKEY_DEBUG("Sorry, I can't build exttype=%d yet.\n", (*pfkey_ext)->sadb_ext_type);
	SENDERR(EINVAL); /* don't process these yet */

	if (!(*pfkey_ext = (struct sadb_ext*)
	     pfkey_x_kmprivate = (struct sadb_x_kmprivate*)
	     kmalloc(sizeof(struct sadb_x_kmprivate), GFP_ATOMIC))) {
		PFKEY_DEBUG("memory allocation failed\n");
		SENDERR(ENOMEM);
	}
	memset(pfkey_x_kmprivate,
	       0,
	       sizeof(struct sadb_x_kmprivate));
	
        pfkey_x_kmprivate->sadb_x_kmprivate_len =
		sizeof(struct sadb_x_kmprivate) / IPSEC_PFKEYv2_ALIGN;

        pfkey_x_kmprivate->sadb_x_kmprivate_exttype = SADB_X_EXT_KMPRIVATE;
        pfkey_x_kmprivate->sadb_x_kmprivate_reserved = 0;
errlab:
	return error;
}

int
pfkey_x_satype_build(struct sadb_ext**	pfkey_ext,
		     uint8_t		satype)
{
	int error = 0;
	int i;
	struct sadb_x_satype *pfkey_x_satype = (struct sadb_x_satype *)*pfkey_ext;

	PFKEY_DEBUG("called.\n");
	/* sanity checks... */
	if (pfkey_x_satype) {
		PFKEY_DEBUG("why is pfkey_x_satype already pointing to something?\n");
		SENDERR(EINVAL);
	}
	
	if (!satype) {
		PFKEY_DEBUG("SA type not set, must be non-zero.\n");
		SENDERR(EINVAL);
	}

	if (satype > SADB_SATYPE_MAX) {
		PFKEY_DEBUG("satype %d > max %d\n", 
			satype, SADB_SATYPE_MAX);
		SENDERR(EINVAL);
	}

	if (!(*pfkey_ext = (struct sadb_ext*)pfkey_x_satype = (struct sadb_x_satype*)
	     kmalloc(sizeof(struct sadb_x_satype), GFP_ATOMIC))) {
		PFKEY_DEBUG("memory allocation failed\n");
		SENDERR(ENOMEM);
	}
	memset(pfkey_x_satype,
	       0,
	       sizeof(struct sadb_x_satype));
	
        pfkey_x_satype->sadb_x_satype_len = sizeof(struct sadb_x_satype) / IPSEC_PFKEYv2_ALIGN;

	pfkey_x_satype->sadb_x_satype_exttype = SADB_X_EXT_SATYPE2;
	pfkey_x_satype->sadb_x_satype_satype = satype;
	for (i=0; i<3; i++) {
		pfkey_x_satype->sadb_x_satype_reserved[i] = 0;
	}

errlab:
	return error;
}


int
pfkey_msg_build(struct sadb_msg **pfkey_msg, struct sadb_ext *extensions[], int dir)
{
	int error = 0;
	int ext;
	int total_size;
	struct sadb_ext *pfkey_ext;
	int extensions_seen = 0;
	
	if (!extensions[0]) {
		PFKEY_DEBUG("extensions[0] must be specified (struct sadb_msg).\n");
		SENDERR(EINVAL);
	}

	total_size = sizeof(struct sadb_msg) / IPSEC_PFKEYv2_ALIGN;
	for (ext = 1; ext <= SADB_EXT_MAX; ext++) {
		if (extensions[ext]) {
			total_size += (extensions[ext])->sadb_ext_len;
		}
        }                

	if (!(*pfkey_msg = (struct sadb_msg*)kmalloc(total_size * IPSEC_PFKEYv2_ALIGN, GFP_ATOMIC))) {
		PFKEY_DEBUG("memory allocation failed\n");
		SENDERR(ENOMEM);
	}

	PFKEY_DEBUG("pfkey_msg=%p allocated %d bytes, &extensions[0]=%p\n", 
			*pfkey_msg, total_size * IPSEC_PFKEYv2_ALIGN, &extensions[0]);
	memcpy(*pfkey_msg,
	       extensions[0],
	       sizeof(struct sadb_msg));
	(*pfkey_msg)->sadb_msg_len = total_size;
	(*pfkey_msg)->sadb_msg_reserved = 0;
	extensions_seen =  1 ;

	pfkey_ext = (struct sadb_ext*)(((char*)(*pfkey_msg)) + sizeof(struct sadb_msg));

	for (ext = 1; ext <= SADB_EXT_MAX; ext++) {
		/* copy from extension[ext] to buffer */
		if (extensions[ext]) {    
			memcpy(pfkey_ext,
			       extensions[ext],
			       (extensions[ext])->sadb_ext_len * IPSEC_PFKEYv2_ALIGN);
			((char*)pfkey_ext) += (extensions[ext])->sadb_ext_len * IPSEC_PFKEYv2_ALIGN;
			/* Mark that we have seen this extension and remember the header location */
			extensions_seen |= ( 1 << ext );
		}
	}

errlab:

	return error;
}

/*
 * $Log: pfkey_v2_build.c,v $
 * Revision 1.1.2.1  2003/04/14 17:53:22  mbellon
 *
 *
 * USAGI 4.1 merge
 *
 * Revision 1.1.1.1  2001/05/22 06:14:05  miyazawa
 * kernel for ipsec without FS
 *
 * Revision 1.21  2000/11/17 18:10:30  rgb
 * Fixed bugs mostly relating to spirange, to treat all spi variables as
 * network byte order since this is the way PF_KEYv2 stored spis.
 *
 * Revision 1.20  2000/10/12 00:02:39  rgb
 * Removed 'format, ##' nonsense from debug macros for RH7.0.
 *
 * Revision 1.19  2000/10/10 20:10:20  rgb
 * Added support for debug_ipcomp and debug_verbose to klipsdebug.
 *
 * Revision 1.18  2000/09/12 18:59:54  rgb
 * Added Gerhard's IPv6 support to pfkey parts of libfreeswan.
 *
 * Revision 1.17  2000/09/12 03:27:00  rgb
 * Moved DEBUGGING definition to compile kernel with debug off.
 *
 * Revision 1.16  2000/09/08 19:22:12  rgb
 * Fixed pfkey_prop_build() parameter to be only single indirection.
 * Fixed struct alg copy.
 *
 * Revision 1.15  2000/08/20 21:40:01  rgb
 * Added an address parameter sanity check to pfkey_address_build().
 *
 * Revision 1.14  2000/08/15 17:29:23  rgb
 * Fixes from SZI to untested pfkey_prop_build().
 *
 * Revision 1.13  2000/06/02 22:54:14  rgb
 * Added Gerhard Gessler's struct sockaddr_storage mods for IPv6 support.
 *
 * Revision 1.12  2000/05/10 19:24:01  rgb
 * Fleshed out sensitivity, proposal and supported extensions.
 *
 * Revision 1.11  2000/03/16 14:07:23  rgb
 * Renamed ALIGN macro to avoid fighting with others in kernel.
 *
 * Revision 1.10  2000/01/24 21:14:35  rgb
 * Added disabled pluto pfkey lib debug flag.
 *
 * Revision 1.9  2000/01/21 06:27:32  rgb
 * Added address cases for eroute flows.
 * Removed unused code.
 * Dropped unused argument to pfkey_x_satype_build().
 * Indented compiler directives for readability.
 * Added klipsdebug switching capability.
 * Fixed SADB_EXT_MAX bug not permitting last extension access.
 *
 * Revision 1.8  1999/12/29 21:17:41  rgb
 * Changed pfkey_msg_build() I/F to include a struct sadb_msg**
 * parameter for cleaner manipulation of extensions[] and to guard
 * against potential memory leaks.
 * Changed the I/F to pfkey_msg_free() for the same reason.
 *
 * Revision 1.7  1999/12/09 23:12:20  rgb
 * Removed unused cruft.
 * Added argument to pfkey_sa_build() to do eroutes.
 * Fixed exttype check in as yet unused pfkey_lifetime_build().
 *
 * Revision 1.6  1999/12/07 19:54:29  rgb
 * Removed static pluto debug flag.
 * Added functions for pfkey message and extensions initialisation
 * and cleanup.
 *
 * Revision 1.5  1999/12/01 22:20:06  rgb
 * Changed pfkey_sa_build to accept an SPI in network byte order.
 * Added <string.h> to quiet userspace compiler.
 * Moved pfkey_lib_debug variable into the library.
 * Removed SATYPE check from pfkey_msg_hdr_build so FLUSH will work.
 * Added extension assembly debugging.
 * Isolated assignment with brackets to be sure of scope.
 *
 * Revision 1.4  1999/11/27 11:57:35  rgb
 * Added ipv6 headers.
 * Remove over-zealous algorithm sanity checkers from pfkey_sa_build.
 * Debugging error messages added.
 * Fixed missing auth and encrypt assignment bug.
 * Add argument to pfkey_msg_parse() for direction.
 * Move parse-after-build check inside pfkey_msg_build().
 * Consolidated the 4 1-d extension bitmap arrays into one 4-d array.
 * Add CVS log entry to bottom of file.
 *
 */
